from io import BytesIO

from docx.table import _Cell, Table as DocxTable

from dataiku.base.utils import safe_unicode_str
from .document_handler import DocumentHandler, TableHandler
from dataiku.doctor.docgen.extractor.docx_parser import DocxParser
from docx.enum.style import WD_STYLE_TYPE
import logging
import json

logger = logging.getLogger(__name__)


class Text(object):
    def __init__(self, text):
        self.inner_text = text

    def apply(self, paragraph, document, initial_run, table_style = None):
        """
        Insert the placeholder data as a text inside the location defined by the initial_run inside the paragraph.
        :param paragraph: the location of the placeholder
        :param document: the current document, it will be used to retrieve style (unused)
        :param initial_run: the run of the placeholder
        :param table_style: a potential style to apply to the current table. (unused)
        :return: a new paragraph after the text insertion.
        """
        try:
            if initial_run:
                run = paragraph.add_run(self.inner_text, initial_run.style)
                DocumentHandler.copy_font(initial_run, run)
            else:
                paragraph.add_run(self.inner_text)
        except Exception as err:
            logger.debug("Error while inserting text placeholder " + json.dumps(self.inner_text) + " - " + str(err))

        return paragraph


class Table(object):
    def __init__(self, cells):
        self.cells = cells

    @staticmethod
    def move_table_after(table, paragraph):
        tbl, p = table._tbl, paragraph._p
        p.addnext(tbl)

    def apply(self, paragraph, document, initial_run, table_style = None):
        """
        Insert the placeholder data as a table inside the location defined by the initial_run inside the paragraph.
        :param paragraph: the location of the placeholder
        :param document: the current document, it will be used to retrieve style
        :param initial_run: the run of the placeholder
        :param table_style: a potential style to apply to the current table.
        :return: a new paragraph after the table insertion.
        """
        output_paragraph = DocumentHandler.insert_paragraph_after(paragraph, style=paragraph.style)

        if len(self.cells) < 1:
            table = document.add_table(rows=1, cols=1)
        else:
            max_col_size = 1
            for r in range(0, len(self.cells)):
                if len(self.cells[r]) > max_col_size:
                    max_col_size = len(self.cells[r])
            table = document.add_table(rows=len(self.cells), cols=max_col_size)

        table = CachedTable.from_table(table)

        if table_style is not None:
            table.style = table_style.style.name
            TableHandler.copy_table_style(table, table_style)
        elif hasattr(initial_run, "style") and initial_run.style and initial_run.style.type == WD_STYLE_TYPE.TABLE:
            table.style = initial_run.style

        # we have to split the current paragraph in two, to place the table in the middle, in its own format
        self.move_table_after(table, paragraph)
        for r in range(0, len(self.cells)):
            for c in range(0, len(self.cells[r])):
                value = self.cells[r][c]
                table.cell(r, c).text = '' if value is None else str(value)  # if we assign a None value here we get an npe in the .text setter
        return output_paragraph


class CachedTable(DocxTable):
    """ Keep the cells of a table in memory to avoid very expensive calls to _cells()

    This is to be used when not altering the structure of the table, don't add rows or columns
    """
    def __init__(self, tbl, parent):
        super(DocxTable, self).__init__(parent)
        self._element = self._tbl = tbl
        self._cached_cells = None

    @property
    def _cells(self):
        if self._cached_cells is None:
            self._cached_cells = super(CachedTable, self)._cells
        return self._cached_cells

    @staticmethod
    def from_table(table):
        cached_table = CachedTable(table._tbl, table._parent)
        return cached_table


class PuppeteerContent(object):

    def __init__(self, puppeteer_extractor, puppeteer_config_name):
        self.puppeteer_extractor = puppeteer_extractor
        self.puppeteer_config_name = puppeteer_config_name

    def apply(self, paragraph, document, initial_run, table_style):
        """
        Insert the placeholder data inside the location defined by the initial_run inside the paragraph.
        The placeholder will be replaced by the information retrieved by Puppeteer, which can be a text, a table or an image.
        :param paragraph: the location of the placeholder
        :param document: the current document, it will be used to retrieve style
        :param initial_run: the run of the placeholder
        :param table_style: a potential style to apply to the current table.
        :return: a new paragraph after the element insertion.
        """
        if isinstance(document, _Cell):
            page_width = document.width
        else:
            first_section = document.sections[0]
            page_width = first_section.page_width - first_section.left_margin - first_section.right_margin
        contents = self.puppeteer_extractor.get_contents(self.puppeteer_config_name)

        # Corner case for model.other_algorithms_search_strategy.image: the first element of each group is a title.
        is_title = self.puppeteer_config_name == "DESIGN_OTHER_ALGORITHMS_SEARCH_STRATEGY_IMAGE" or \
                            self.puppeteer_config_name == "DESIGN_OTHER_ALGORITHMS_SEARCH_STRATEGY_TABLE" or \
                            self.puppeteer_config_name == "ROC_CURVE_MULTICLASS" or \
                            self.puppeteer_config_name == "DENSITY_GRAPH_MULTICLASS" or \
                            self.puppeteer_config_name == "CALIBRATION_CHART_MULTICLASS" or \
                            self.puppeteer_config_name == "BINARY_C_DETAILED_METRICS_TABLE"

        if contents:
            for element_index, content in sorted(contents.items()):
                # Some content are a set of images
                for image_index, extracted_content in sorted(content.items()):
                    if extracted_content["type"] == "txt":
                        paragraph = self.apply_templated_text(extracted_content["data"], paragraph, initial_run, document, is_title)
                        # model.other_algorithms_search_strategy.image is one title, one normal text and one image => next one is not a title
                        is_title = False
                    elif extracted_content["type"] == "png":
                        paragraph = self.apply_chart(extracted_content["data"], paragraph, initial_run, page_width)
                        # model.other_algorithms_search_strategy.image is one title, one normal text and one image => next one is a title
                        is_title = True
                    elif extracted_content["type"] == "json":
                        paragraph = self.apply_templated_json(extracted_content["data"], paragraph, initial_run, document, table_style)
                        # model.other_algorithms_search_strategy.image is one title, one normal text and one image => next one is a title
                        is_title = True
        return paragraph

    def apply_templated_text(self, extracted_text, paragraph, initial_run, document, is_title):
        if not extracted_text:
            return paragraph

        lines = extracted_text.splitlines()
        lines = list(filter(None, lines)) # Removes empty lines

        BEFORE_H1_PLACEHOLDER = "[#@#]"  # This value should match the one used in export-charts.js
        lines = list(map(lambda l: safe_unicode_str(l).replace(BEFORE_H1_PLACEHOLDER, ""), lines))  # Add an empty line before each <h1> to better separate text sections

        first_line = lines.pop(0)

        paragraph_style = paragraph.style
        title_style = paragraph.style
        if is_title:
            # if we are on a title, we want have an empty paragraph and then create another paragraph just for the title
            paragraph = DocumentHandler.insert_paragraph_after(paragraph, style=paragraph_style)
            styles = document.styles

            # use Heading 5 style if this exists.
            paragraph_styles = [
                s for s in styles if s.type == WD_STYLE_TYPE.PARAGRAPH
            ]
            for style in paragraph_styles:
                if "Heading 5" == style.name:
                    title_style = "Heading 5"
            paragraph = DocumentHandler.insert_paragraph_after(paragraph, style=title_style)
            run = paragraph.add_run("")
        else:
            # First line on the current paragraph
            run = paragraph.add_run("", initial_run.style)


        DocumentHandler.copy_font(initial_run, run)
        run.text += first_line

        # Create a new paragraph for each next line
        for line in lines:
            paragraph = DocumentHandler.insert_paragraph_after(paragraph, style=paragraph_style)
            run = paragraph.add_run("", initial_run.style)
            DocumentHandler.copy_font(initial_run, run)
            run.text += line

        if is_title:
            if title_style != "Heading 5":
                # If Heading 5 does not exists, fake the style of the title: bolder, underlined and bigger.
                for run in paragraph.runs:
                    run.font.bold = True
                    run.font.underline = True
                    if paragraph.style.font.size:
                        run.font.size = paragraph.style.font.size + 63500
            # Force new paragraph for next element
            paragraph = DocumentHandler.insert_paragraph_after(paragraph, style=paragraph_style)
        return paragraph

    def apply_chart(self, image, paragraph, initial_run, page_width):
        new_paragraph = DocumentHandler.insert_paragraph_after(paragraph, style=paragraph.style)
        run = new_paragraph.add_run("", initial_run.style)
        DocumentHandler.copy_font(initial_run, run)
        logger.debug("Format for %s is %s", self.puppeteer_config_name, DocxParser.debug_format(paragraph.paragraph_format))
        inline_shape = run.add_picture(BytesIO(image))

        # Because the Puppeteer browser takes high resolution screenshots (deviceScaleFactor=2),
        # we need to divide the image dimensions by 2 in the docx so that it appears at the normal size to the user
        inline_shape.height = int(inline_shape.height / 2)
        inline_shape.width = int(inline_shape.width / 2)

        # If the image is bigger than the page, reduce it
        if inline_shape.width > page_width:
            if inline_shape.width != 0:
                inline_shape.height = inline_shape.height * page_width // inline_shape.width
            inline_shape.width = page_width

        return new_paragraph

    def apply_templated_json(self, extracted_text, paragraph, initial_run, document, table_style):
        if not extracted_text:
            return paragraph

        # The format of the json is:
        # [
        #   ["line1 col1", "line1 col2", "line1 col3 subline1\nline1 col3 subline2"],
        #   ["line2 col1", "line2 col2"]
        # ]
        values = json.loads(extracted_text)
        cells = []
        for line in values:
            cells.append(line)
        table = Table(cells)
        return table.apply(paragraph, document, initial_run, table_style)
