/**
 * This script use wikiExportToolbox object that expose a set of functions needed for this feature:
 * - checkLoading() to check whether a dashboard is loading.
 * - goToArticle(articleId) to navigate to another article in the same project.
 *
 * To access this object, a dummy span has been added with id wiki-export-toolbox-anchor so we can access the scope.
 *
 * If you want to do modifications or understand the code better, wikiExportToolbox object is defined in wiki article controller.
 *
 * Usage
 * node export-wiki.js magicHeadlessAuth scriptConfigFile
 *
 * Arguments
 * - magicHeadlessAuth (String): API Key generated by DSS that allows authentication.
 * - scriptConfigFile (String): Path to the JSON file containing the arguments of the script (see WikiExportScriptRunner.Config Java class for details)
 */

const utils = require('./utils');
const log = require('./log');
const fs = require('fs');

const magicHeadlessAuth = process.argv[2].toString();
const scriptConfigFile = process.argv[3].toString();

log.info("Reading Wiki export script configuration from " + scriptConfigFile);
const config = JSON.parse(fs.readFileSync(scriptConfigFile));

wikiHomeURL = config.wikiHomeURL;
outputDirectory = config.outputDirectory;
const urlsRequiringAuthentication = config.urlsRequiringAuthentication;
enforceSandboxing = config.browserSandBoxing;
const pageDefaultTimeout = config.pageDefaultTimeout;

try {
    log.info("Wiki export script started");
    utils.createBrowser(enforceSandboxing).then(function (browser) {
        return exportWiki(browser, config.articles).then(function () {
            log.info("Closing browser");
            return browser.close();
        });
    }).then(function () {
        log.info("Wiki export script completed");
    }).catch(function (err) {
        utils.exit(utils.ERR_GENERIC, "Error while running export script", err);
    });
} catch (err) {
    utils.exit(utils.ERR_GENERIC, "Error while running export script", err);
}

/**
 * Export wiki article and its children if there are.
 *
 * @param browser returned by the method createBrowser.
 * @return {Promise.<Void>}
 */
function exportWiki(browser, articles) {
    let page = null;
    return utils.newBrowserPage(browser, 1000, 1000, urlsRequiringAuthentication, magicHeadlessAuth, pageDefaultTimeout)
        .then(p => {page = p; return utils.navigateTo(page, wikiHomeURL)})
        .then(() => utils.waitForPageToLoad(page, getToolbox, 0))
        .then(() => exportArticleList(page, articles));
}

/**
 * Export a list of article, then recursively export each children the articles have.
 *
 * @param page Browser page opened on a wiki article.
 * @param articles Wiki articles to export (and their children).
 * @return {Promise.<Void>}
 */
function exportArticleList(page, articles) {
    return articles.reduce(function(p, article) {
        return p.then(function () {
            return exportArticle(page, article).then(function() {
                return exportArticleList(page, article.children);
            })
        });
    }, Promise.resolve());
}

/**
 * Navigate to an article, then export it.
 *
 * @param page Browser page opened on a wiki article.
 * @param article Wiki article to export.
 * @return {Promise.<Void>}
 */
function exportArticle(page, article) {
    return goToArticle(page, article.id)
        .then(() => utils.waitForPageToLoad(page, getToolbox, 0))
        .then(() => exportArticleToPDF(page, article));
}

/**
 * Download the PDF to the outputDirectory and log the progress of the script.
 *
 * @param page Browser page opened on a wiki article.
 * @param article Wiki article to export.
 * @return {Promise.<Void>}
 */
function exportArticleToPDF(page, article) {
    return page.pdf({
        path: outputDirectory + "/" + article.id + ".pdf",
        format: config.format,
        printBackground: true,
        margin: {
            top: "40px",
            bottom: "40px"
        }
    }).then(function () {
        // Beware: Do not modify or remove the following line as it is used by the backend to report on the progress of the script.
        // See `WikiExportScriptRunner` in the backend for the details.
        console.log("Successfully exported wiki article");
    });
}

/**
 * Navigate to an article using the toolbox JS handle.
 *
 * @param page Browser page opened on a wiki article.
 * @param articleId Id of the article to navigate to.
 * @return {Promise.<Void>}
 */
function goToArticle(page, articleId) {
    return getToolbox(page).then(function(toolbox) {
        return page.evaluate(function (toolbox, articleId) {
            return toolbox.goToArticle(articleId);
        }, toolbox, articleId);
    });
}

/**
 * Retrieve the toolbox JS handle (defined in wikis.js)
 *
 * @param page Browser page handle opened on a wiki article.
 * @return {*} toolbox JS handle
 */
function getToolbox(page) {
    return page.evaluateHandle(function () {
        let toolboxAnchor = document.querySelector('#wiki-export-toolbox-anchor');
        return angular.element(toolboxAnchor).scope().wikiExportToolBox;
    });
}
