(function() {
    "use strict";

    const explanationModalBody = {
        templateUrl: "static/dataiku/ai-explanations/explanation-modal/explanation-modal-body.component.html",
        bindings: {
            objectType: "<",
            object: "<",
            mode: "<",
            canWriteProject: "<",
            explanationRequested: "=",
            lastJobId: "=",
            abortRequested: "=",
            resolveModal: "&",
            dismiss: "&",
        },
        controller: function(
            $scope,
            $rootScope,
            $filter,
            AI_EXPLANATION_MODAL_MODES,
            AIExplanationService,
            DataikuAPI,
            FlowGraphSelection,
            ActivityIndicator,
            ClipboardUtils
        ) {
            const ctrl = this;

            ctrl.MODES = AI_EXPLANATION_MODAL_MODES;

            let generatedAt;

            ctrl.$onInit = function() {
                ctrl.availableExplanationOptions = AIExplanationService.createAvailableOptions();
                ctrl.explanationOptions = AIExplanationService.loadOptions();
                ctrl.generate();
            };

            ctrl.generate = function() {
                resetErrorInScope($scope);
                AIExplanationService.explainObject(
                    ctrl.objectType,
                    ctrl.object,
                    ctrl.explanationOptions,
                    function(initialResponse) {
                        ctrl.lastJobId = initialResponse.jobId;
                    }
                )
                .then(function(data) {
                    if (data.aborted) {
                        return;
                    }

                    ctrl.explanationResponse = data.result;
                    generatedAt = new Date();
                    ctrl.explanationRequested = false;
                    AIExplanationService.storeOptions(ctrl.explanationOptions);
                })
                .catch(function(r) {
                    setErrorInScope.bind($scope)(r.data, r.status, r.headers);
                    ctrl.explanationRequested = false;
                });
                ctrl.explanationRequested = true;
                ctrl.lastJobId = null;
            };

            ctrl.abortGenerate = function() {
                AIExplanationService.abortExplain(ctrl.lastJobId)
                .then(function() {
                    ctrl.explanationRequested = false;
                    ctrl.abortRequested = false;
                })
                .catch(function(r) {
                    setErrorInScope.bind($scope)(r.data, r.status, r.headers);
                    ctrl.abortRequested = false;
                });
                ctrl.abortRequested = true;
            };

            ctrl.gotOkResponse = function() {
                return ctrl.explanationResponse && ctrl.explanationResponse.ok && !ctrl.explanationRequested;
            };

            ctrl.explanationIsEmpty = function() {
                return (
                    ctrl.gotOkResponse() &&
                    (ctrl.objectType === "PROJECT" || ctrl.objectType === "FLOW_ZONE") &&
                    ctrl.explanationResponse.reason === "EMPTY_FLOW"
                );
            };

            ctrl.explanationIsReady = function() {
                return ctrl.gotOkResponse() && !ctrl.explanationIsEmpty();
            };

            ctrl.copy = function() {
                ClipboardUtils.copyToClipboard(ctrl.explanationResponse.explanation, "Explanation copied!");
            };

            ctrl.getObjectTypeLabel = function() {
                switch (ctrl.objectType) {
                    case "PROJECT":
                        return "project";
                    case "FLOW_ZONE":
                        return "zone";
                }
            };

            ctrl.getExplainedObjectLabel = function() {
                switch (ctrl.objectType) {
                    case "PROJECT":
                        return "Flow";
                    case "FLOW_ZONE":
                        return "zone";
                    case "DATASET":
                        return "dataset"
                }
            };

            ctrl.getEmptyExplanationMessage = function() {
                switch (ctrl.objectType) {
                    case "PROJECT":
                    case "FLOW_ZONE":
                        return "There has to be at least one recipe or one dataset in order to explain a " +
                            `${ctrl.getExplainedObjectLabel()}.`;
                }
            };

            ctrl.getDefaultEmptyAIMessage = AIExplanationService.getDefaultEmptyMessage("AI_SECTION")

            ctrl.getErrorMessage = function() {
                switch (ctrl.objectType) {
                    case "PROJECT":
                    case "FLOW_ZONE":
                        return `Unfortunately, AI was not able to explain the ${ctrl.getExplainedObjectLabel()}`;
                }
            };

            ctrl.getNewObjectDescription = function() {
                return ctrl.explanationResponse.explanation +
                    `\n\nDescription generated on ${generatedAt.toLocaleDateString()}`;
            };

            ctrl.useAsDescription = function() {
                let apiPromise;
                if (ctrl.objectType === "PROJECT") {
                    const projectSummary = angular.copy(ctrl.object);
                    projectSummary.description = ctrl.getNewObjectDescription();
                    apiPromise = DataikuAPI.projects.saveSummary(ctrl.object.projectKey, projectSummary);
                    apiPromise.success(function() {
                        $rootScope.$broadcast("projectDescriptionChanged", projectSummary.description);
                        FlowGraphSelection.clearSelection();
                    });
                } else {
                    const update = {
                        shortDesc: ctrl.object.shortDesc,
                        description: ctrl.getNewObjectDescription(),
                        customFields: ctrl.object.customFields,
                        tags: ctrl.object.tags,
                    };
                    apiPromise = DataikuAPI.taggableObjects.setMetaData(
                        {
                            type: ctrl.objectType,
                            projectKey: ctrl.object.projectKey,
                            id: ctrl.object.id,
                        },
                        update
                    );
                    apiPromise.success(
                        $rootScope.$broadcast.bind($rootScope, "objectMetaDataChanged", update)
                    );
                }
                apiPromise
                    .success(function() {
                        ctrl.resolveModal();
                        const text = `${$filter("capitalize")(ctrl.getObjectTypeLabel())} description saved`;
                        ActivityIndicator.success(text);
                    })
                    .error(setErrorInScope.bind($scope));
            };
        },
    };

    angular.module("dataiku.aiExplanations").component("explanationModalBody", explanationModalBody);
})();
