(function() {
'use strict';

var app = angular.module('dataiku.datasets');


// All Row-Oriented DS (including SQL)
app.controller("BaseRowDatasetController", function($scope, LoggerProvider, DataikuAPI, DatasetUtils, WT1, withConsistency) {
    var Logger = LoggerProvider.getLogger('datasets.row');

    // Set error message when future are aborted
    $scope.setErrorMessageForFuture = function(res, httpCode, c) {
        $scope.testing = false;
        if ('aborted' in res && res.aborted) {
            $scope.testResult = {
                connectionOK: false,
                connectionErrorMsg: "Connection aborted",
                connectionError: false,
                configOK: true
            }
            res.$customMessage = true
            res.message = "Aborted !"
        } else {
            setErrorInScope.bind($scope)(res, httpCode, c);
        }
    }

    $scope.dimensionTypes = [
        { type: 'time',  label: 'Time range' },
        { type: 'value', label: 'Discrete values' }
    ];
    $scope.timeDimensionPeriods = ["YEAR", "MONTH", "DAY", "HOUR"];

    // Consistency methods
    if (!withConsistency) {
        Logger.warn("Consistency not implemented");
        return;
    }
    $scope.setSchemaUserModified = function() {
        $scope.schemaJustModified = true;
        $scope.dataset.schema.userModified = true;
    };

    $scope.discardSchemaChanges = function() {
        // To discard schema changes, we mark the schema as not modified and trigger a new preview.
        // The backend will thus discard our schema changes but keep our comments when applicable.
        Logger.info("Discarding schema changes");
        if ($scope.dataset.schema) {
            $scope.dataset.schema.userModified = false;
        }
        $scope.test(false);
    };

    $scope.pushColumnDescriptionsToTable = function() {
        Logger.info("pushing descriptions to table and saving dataset");
        $scope.saveDataset().then(function() {
            DataikuAPI.datasets.pushColumnDescriptionsToTable(
                $scope.dataset,
                $scope.consistency?.result?.newSchema?.columns,
                $scope.consistency?.result?.detectedSchema?.columns,
            ).success((resp) => {
                $scope.warningMessages = resp?.anyMessage ? resp : null;
                $scope.checkConsistency();
            }).error(setErrorInScope.bind($scope));
        },setErrorInScope.bind($scope));
    }

    $scope.checkConsistency = function() {
        Logger.info('Checking consistency');

        $scope.schemaJustModified = false;
        $scope.consistency = null;
        DataikuAPI.datasets.testSchemaConsistency($scope.dataset).success(function (data) {
            Logger.info("Got consistency result", data);
            $scope.consistency = data;
            $scope.consistency.kind = DatasetUtils.getKindForConsistency($scope.dataset);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.discardConsistencyError = function() {
        $scope.consistency = null;
    };

    $scope.overwriteSchema = function(newSchema) {
        WT1.event("dataset-discard-schema-changed", { datasetType: $scope.dataset.type, datasetManaged: $scope.dataset.managed });
        $scope.dataset.schema = angular.copy(newSchema);
        $scope.schemaJustModified = false;
        $scope.consistency = null;
        $scope.checkConsistency();
    };

    $scope.searchConnectionsWithDescriptions = function(term, connection) { // Connection should be as defined in ConnectionTypeAndName in java
        if (connection.name.includes(term)) {
            return true;
        }
        if (connection.description?.includes(term)) {
            return true;
        }
        return false;
    };
});


// NoSQL Row-Oriented DS
    app.controller("BaseNoSQLRowDatasetController", function($scope, $stateParams, $controller, LoggerProvider, DataikuAPI, Dialogs, withConsistency, MonoFuture) {
    $controller("BaseRowDatasetController", {$scope: $scope, withConsistency: withConsistency});

    var Logger = LoggerProvider.getLogger('datasets.row');

    // Finish dataset initialization if needed
    if (! $scope.dataset.schema) {
        $scope.dataset.schema = { columns: [] };
    }

    $scope.$watch('dataset.type', function (nv, ov) {
        DataikuAPI.connections.getTypeAndNames($scope.dataset.type).success(function (data) {
            $scope.connections = data.map(connection => connection['name']);
            $scope.connectionsWithDescriptions = data;
            if (!$scope.dataset.params.connection && data.length) {
                $scope.dataset.params.connection = data[0]['name'];
                $scope.test(true);
            }
        }).error(setErrorInScope.bind($scope));
    });

    $scope.inferStorageTypesFromData = function(){
        Dialogs.confirm($scope,
            "Infer storage types from data",
            "This only takes into account a very small sample of data, and could lead to invalid data. "+
            "For safer typing of data, use a prepare recipe.").then(function(){
                $scope.dataset.schema.userModified = false;
                $scope.test(false, true);
        });
    };


    // requires $scope.testAPI and $scope.testCallback
    $scope.test = function (connectionOnly, inferStorageTypes, listTables) {
        $scope.testing = true;
        $scope.testResult = null;
        $scope.schemaJustModified = false;
        if (angular.isUndefined($scope.dataset.params.connection)) {
            Logger.info('Not testable');
            $scope.testing = false;
            $scope.testResult = { connectionOK: false, connectionErrorMsg: "no connection defined" };
            return;
        }

        MonoFuture($scope).wrap($scope.testAPI)($stateParams.projectKey, $scope.dataset, connectionOnly, inferStorageTypes, listTables)
            .success(function(data) {
                    Logger.info('Got test result');
                    $scope.testing = false;
                    if (typeof $scope.formatTestResult === "function") {
                        data.result = $scope.formatTestResult(data.result);
                    }
                    $scope.testResult = data.result;
                    $scope.testCallback(connectionOnly, data);
                }
            )
            .error(function(res, httpCode, c) {
                $scope.setErrorMessageForFuture(res, httpCode, c);
            });
    };

    $scope.onLoadComplete = function() {
        if ($scope.$eval('dataset.params.connection')) {
            $scope.test(true);
        }
    };

    $scope.suggestDatasetName = function(name) {
        if (!$scope.dataset.name && !$scope.uiState.new_dataset_name_manually_edited) {
            $scope.uiState.new_dataset_name = name;
        }
    };
});


app.controller("BaseNoSQLRowDatasetControllerWithSingleColumnPartitioning", function($scope, $controller, withConsistency) {
    $controller("BaseNoSQLRowDatasetController", {$scope: $scope, withConsistency: withConsistency});

    $scope.$watch('dataset.params', function (nv, ov) {
        $scope.dataset.partitioning = $scope.dataset.partitioning || {};
        $scope.dataset.partitioning.dimensions = $scope.dataset.partitioning.dimensions || [];
        const dimensions = $scope.dataset.partitioning.dimensions;
        if ($scope.dataset.params.partitioned) {
            if (!dimensions.length) {
                dimensions.push({
                    name: 'time',
                    type: 'value',
                    params: {}
                });
            }
            if (angular.isDefined($scope.dataset.params.partitioningColumn)) {
                dimensions[0].name = $scope.dataset.params.partitioningColumn;
            }
        }
        else {
            $scope.dataset.partitioning.dimensions = [];
            delete $scope.dataset.params.partitioningColumn;
            delete $scope.dataset.params.explicitPartitionsList;
        }
    }, true);

});

app.controller("SharePointOnlineListDatasetController", function($scope, $stateParams, $controller, Dialogs, DataikuAPI, DatasetUtils, MonoFuture) {
    //$controller("BaseRowDatasetController", {$scope: $scope, withConsistency: true}); // Todo: ALX: put that back and clean the code
    $scope.$watch('dataset.type', function (nv, ov) {
        DataikuAPI.connections.getTypeAndNames("SharePointOnline").success(function (data) {
            $scope.connections = data.map(connection => connection['name']);
            $scope.connectionsWithDescriptions = data;
            if (!$scope.dataset.params.connection && data.length) {
                $scope.dataset.params.connection = data[0]['name'];
                $scope.test(true);
            }
        }).error(setErrorInScope.bind($scope));
    });
    $scope.testAPI = DataikuAPI.datasets.sharepointOnlineList.test;
    $scope.testCallback = function (connectionOnly, data) {
        if (!connectionOnly) {
            $scope.dataset.schema = (data.schemaDetection || {}).newSchema;
            $scope.consistency = { empty : false, result : data.schemaDetection };
            $scope.consistency.kind = DatasetUtils.getKindForConsistency($scope.dataset);
            $scope.suggestDatasetName($scope.testResult.suggestedName);
        }
    };
    $scope.test = function (connectionOnly, inferStorageTypes, listTables) {
        $scope.testing = true;
        $scope.schemaJustModified = false;
        if (angular.isUndefined($scope.dataset.params.connection)) {
            $scope.testing = false;
            $scope.testResult = { connectionOK: false, connectionErrorMsg: "no connection defined" };
            return;
        }
        if ($scope.uiState.siteSearch != null) {
            $scope.dataset.params.siteSearch = $scope.uiState.siteSearch + "*";
        } else {
            $scope.dataset.params.siteSearch = $scope.uiState.siteSearch;
        }
        MonoFuture($scope).wrap($scope.testAPI)($stateParams.projectKey, $scope.dataset, connectionOnly, inferStorageTypes, listTables, "test")
            .success(function(futureResult) {
                const MAX_DISPLAYED_SITES = 1000;
                $scope.testing = false;
                let data = futureResult.result;
                if (data.sharePointSites.length > MAX_DISPLAYED_SITES) {
                    data.previewWarningMsg = "Your search returned "
                                            + data.sharePointSites.length + " sites. Only "
                                            + MAX_DISPLAYED_SITES + " can be displayed. Please refine your search.";
                    data.sharePointSites.length = MAX_DISPLAYED_SITES;
                    data.connectionOK = true;
                }
                if (data.sharePointSites.length == 0) {
                    if ($scope.uiState.siteSearch == null) {
                        data.sharePointSites = null;
                    } else {
                        data.sharePointSites = [];
                    }
                    data.sharePointLists = null;
                } else if (data.sharePointSites.length == 1) {
                    // if one site's name exactly matches the search, preselect it
                    $scope.dataset.params.siteId = data.sharePointSites[0].id;
                } else if ($scope.dataset.params.siteId == null) {
                    for (const site of data.sharePointSites) {
                        if (site.name == $scope.uiState.siteSearch) {
                            $scope.dataset.params.siteId = site.id;
                            break;
                        }
                    }
                }
                if (data.connectionErrorMsg) {
                    $scope.errorMessage = data.connectionErrorMsg.split(": Error code")[0];
                }
                $scope.testResult = data;
                $scope.testCallback(connectionOnly, data);
                $scope.testResult.connectionOK = true;
            })
            .error(function(res, httpCode, c) {
                $scope.setErrorMessageForFuture(res, httpCode, c);
            })
            .finally(function() {
                if ("siteSearch" in $scope.dataset.params) {
                    delete($scope.dataset.params.siteSearch);
                }
            });
    };
    $scope.searchAllSites = function() {
        $scope.uiState.siteSearch = "";
        $scope.dataset.params.siteSearch = "";
        return $scope.test(true);
    };
    $scope.searchSites = function(){
        if(("siteSearch" in $scope.uiState)  &&
                $scope.uiState.siteSearch != null &&
                $scope.uiState.siteSearch != '*' &&
                $scope.uiState.siteSearch != '') {
            $scope.test(true);
        } else {
            Dialogs.confirmUnsafeHTML($scope, "Really list all sites?",
                "<p>This will list <strong>all sites</strong> of your tenant.</p>"+
                "<p> On large entreprise SharePoints, this is likely "+
                "to be long and to cause your browser to become unresponsive.</p>").then($scope.searchAllSites.bind())
        };
    }
    $scope.editSearch = function(){
        var siteSearch = null;
        if ("sharePointSites" in $scope.testResult && $scope.testResult.sharePointSites.length > 0) {
            for (var site of $scope.testResult.sharePointSites) {
                if (site.id == $scope.dataset.params.siteId) {
                    siteSearch = site.name;
                    break;
                };
            }
            $scope.uiState.siteSearch = siteSearch;
        }
        $scope.testResult.sharePointSites = null;
    };
    $scope.$watch('dataset.params.siteId', function (nv, ov) {
        if (nv !== ov) {
            $scope.test(false);
        }
    });
    $scope.onLoadComplete = function() {
        if ($scope.$eval('dataset.params.connection')) {
            $scope.test(true);
        }
    };
    $scope.checkConsistency = function() {
        $scope.schemaJustModified = false;
        DataikuAPI.datasets.testSchemaConsistency($scope.dataset).success(function (data) {
            $scope.consistency = data;
            $scope.consistency.kind = DatasetUtils.getKindForConsistency($scope.dataset);
        }).error(setErrorInScope.bind($scope));
    };
    $scope.overwriteSchema = function(newSchema) {
        //WT1.event("dataset-discard-schema-changed", { datasetType: $scope.dataset.type, datasetManaged: $scope.dataset.managed });
        $scope.dataset.schema = angular.copy(newSchema);
        $scope.schemaJustModified = false;
        $scope.consistency = null;
        $scope.checkConsistency();
    };
    $scope.suggestDatasetName = function(name) {
        if (!$scope.dataset.name && !$scope.uiState.new_dataset_name_manually_edited) {
            $scope.uiState.new_dataset_name = name;
        }
    };
    $scope.setSchemaUserModified = function() {
        $scope.schemaJustModified = true;
        $scope.dataset.schema.userModified = true;
    };
    $scope.discardConsistencyError = function() {
        $scope.consistency = null;
    };
    $scope.inferStorageTypesFromData = function(){
        Dialogs.confirm($scope,
            "Infer storage types from data",
            "This only takes into account a very small sample of data, and could lead to invalid data. "+
            "For safer typing of data, use a prepare recipe.").then(function(){
                $scope.dataset.schema.userModified = false;
                $scope.test(false, true);
        });
    };
    $scope.onListSelect = function onListSelect() {
        // Had to do this trick because the <select> does not play well with ng-model, ng-options and ng-change all together
        $scope.dataset.params.sharePointListDisplayName = $scope.testResult.sharePointLists.find(list => list.name === $scope.dataset.params.sharePointList).displayName;
        $scope.dataset.params.listId = $scope.testResult.sharePointLists.find(list => list.name === $scope.dataset.params.sharePointList).id;
    }
});

app.controller("MongoDBDatasetController", function($scope, $controller, DataikuAPI, DatasetUtils) {
    $controller("BaseNoSQLRowDatasetControllerWithSingleColumnPartitioning", {$scope: $scope, withConsistency: true});

    $scope.testAPI = DataikuAPI.datasets.mongoDB.test;

    $scope.testCallback = function (connectionOnly, data) {
        if (!connectionOnly) {
            $scope.dataset.schema = (data.schemaDetection || {}).newSchema;
            $scope.consistency = { empty : false, result : data.schemaDetection };
            $scope.consistency.kind = DatasetUtils.getKindForConsistency($scope.dataset);

            $scope.suggestDatasetName($scope.testResult.suggestedName);
        }
    };

    /* For managed only */
    $scope.createCollection = function() {
        DataikuAPI.datasets.mongoDB.createCollection($scope.dataset).success(function (data) {
            $scope.test(false);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.deleteCollection = function() {
        DataikuAPI.datasets.mongoDB.deleteCollection($scope.dataset).success(function (data) {
            $scope.test(false);
        }).error(setErrorInScope.bind($scope));
    };
});

app.controller("DynamoDBDatasetController", function($scope, $controller, DataikuAPI, DatasetUtils, Dialogs, $state) {
    $controller("BaseNoSQLRowDatasetControllerWithSingleColumnPartitioning", {$scope: $scope, withConsistency: true});


    $scope.testAPI = DataikuAPI.datasets.dynamoDB.test;

    $scope.testCallback = function (connectionOnly, data) {
        if (!connectionOnly) {
            $scope.dataset.schema = (data.schemaDetection || {}).newSchema;
            $scope.consistency = { empty : false, result : data.schemaDetection };
            $scope.consistency.kind = DatasetUtils.getKindForConsistency($scope.dataset);

            $scope.suggestDatasetName($scope.testResult.suggestedName);
        }
    };

    $scope.createTable = function() {
        DataikuAPI.datasets.dynamoDB.createTable($scope.dataset).success(function (data) {
            $scope.test(false);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.deleteTable = function() {
        DataikuAPI.datasets.dynamoDB.deleteTable($scope.dataset).success(function (data) {
            $scope.test(false);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.updateIndex = function() {
        $scope.saveDataset().then(function() {
            DataikuAPI.datasets.dynamoDB.updateIndex($scope.dataset).success(function (data) {
                $scope.test(false);
            }).error(setErrorInScope.bind($scope));
        },setErrorInScope.bind($scope));
    };

    // TODO : check if this function is still in use
    $scope.gotoSearchAndImport = function () {
            $scope.uiState.bypassDirtinessCheck = true;
            DataikuAPI.connections.countIndexedAndUnindexed().success(function (data) {
                if (data.indexedConnections > 0) {
                    $state.go('projects.project.datacatalog.datasources', {selectedTab: 'external-tables'});
                } else {
                    $state.go("projects.project.datacatalog.database-explorer");
                }

            }).error(setErrorInScope.bind($scope));
    };

    $scope.getAllTables = function(){
                Dialogs.confirmUnsafeHTML($scope, "Really list all tables?",
                    "<p>This will list <strong>all tables in all schemas</strong> of your database.</p>"+
                    "<p> On large entreprise databases, this is likely "+
                    "to be extremely long and to cause your browser to become unresponsive.</p>").then($scope.test.bind(this,false,false,true));
    };

});


app.controller("BaseCassandraDatasetController", function($scope, $controller, DataikuAPI) {
    $controller("BaseNoSQLRowDatasetControllerWithSingleColumnPartitioning", {$scope: $scope, withConsistency: false});

    $scope.overwriteSchemaFromTable = function () {
        $scope.dataset.schema = {
            userModified: false,
            columns: $scope.testResult.tableSchema.columns
        };
        $scope.test(true);
    };

    $scope.testAPI = DataikuAPI.datasets.cassandra.test;

    $scope.testCallback = function (connectionOnly, data) {
        if (! $scope.dataset.managed && (!$scope.dataset.schema || !$scope.dataset.schema.columns || $scope.dataset.schema.columns.length == 0) && $scope.testResult.tableSchema) {
            // Overwrite schema using detected schema, if we have none
            $scope.dataset.schema = {
                userModified: false,
                columns: $scope.testResult.tableSchema.columns
            };
            $scope.testResult.schemaMatchesTable = true;

            $scope.suggestDatasetName($scope.testResult.suggestedName);
        }
    };
});


app.controller("ManagedCassandraDatasetController", function($scope, $controller, DataikuAPI, Dialogs) {
    $controller("BaseCassandraDatasetController", {$scope: $scope});

    $scope.createTable = function() {
        DataikuAPI.datasets.cassandra.createTable($scope.dataset)
            .success($scope.test.bind($scope, false))
            .error(setErrorInScope.bind($scope));
    };

    $scope.dropTable = function() {
        Dialogs.confirm($scope,'Drop table','Are you sure you want to drop the Cassandra table?').then(function(){
            DataikuAPI.datasets.cassandra.dropTable($scope.dataset)
                .success($scope.test.bind($scope, false))
                .error(setErrorInScope.bind($scope));
        });
    };
});


app.controller("ExternalCassandraDatasetController", function($scope, $controller) {
    $controller("BaseCassandraDatasetController", {$scope: $scope});
});


app.controller("ElasticSearchDatasetController", function($scope, $controller, DataikuAPI, $stateParams, Logger) {
    $controller("BaseNoSQLRowDatasetController", {$scope: $scope, withConsistency: true});

    $scope.getPane = (panes, tab) => {
        return panes.filter(p => p.title === tab)[0];
    }

    $scope.generatedPartitionName = null;

    $scope.generatePartitionPattern = () => {
        if ($scope.dataset.params.index) {
            $scope.generatedPartitionName = $scope.dataset.params.index.toLowerCase() + "." + $scope.dataset.partitioning.dimensions.map(d => `%{${d.name}}`).join(".");
        }
    }

    $scope.$watch('dataset.params', function (nv, ov) {
        if ($scope.dataset.params.readTemporalMode == undefined) {
            $scope.dataset.params.readTemporalMode = 'AS_IS';
        }
        if ($scope.testResult && !$scope.testResult.testedConnectionOnly && (nv.index !== ov.index || nv.type !== ov.type)) {
            $scope.testResult.testedConnectionOnly = true;
        }
        /* For ElasticSearch, we have:
         *    - managed = partition-by-alias, only partitioning definition in dataset config
         *    - not-managed = partition-by-column, with "partitioned" and "partitioningColumn" and unmanagedPartitioningType === "COLUMN"
         *                    in the dataset config
         *                    OR
         *                    partition-by-index, with "partitioned" and unmanagedPartitioningType === "INDEX"
         *
         * In the managed case, we DO NOT do anything to the dataset config.
         */
        $scope.dataset.partitioning = $scope.dataset.partitioning || {};
        if (!$scope.dataset.params.partitioned || !$scope.dataset.partitioning.dimensions) {
            $scope.dataset.partitioning.dimensions = [];
        }

        if ($scope.dataset.params.partitioned && !$scope.dataset.managed) {
            if ($scope.dataset.params.unmanagedPartitioningType === "COLUMN" && $scope.dataset.partitioning.dimensions.length === 0) {
                $scope.dataset.partitioning.dimensions = [{ name: 'time', type: 'value', params: {}}];
            }
            if (angular.isDefined($scope.dataset.params.partitioningColumn)) {
                $scope.dataset.partitioning.dimensions[0].name = $scope.dataset.params.partitioningColumn;
            }
        }
        $scope.generatePartitionPattern();
    }, true);

    const resolveUnmanagedPartitionedIndexPattern = (partitioningIndexPattern, dimensions) => {
        let pattern = partitioningIndexPattern
        for (const dim of dimensions) {
            if (pattern != null) {
                switch (dim.type) {
                    case "value":
                        pattern = pattern.replace(`%${dim.name}`, "*");
                        pattern = pattern.replace(`%{${dim.name}}`, "*");
                        break;
                    case "time":
                        for (const p of ["%Y", "%M", "%D", "%H"]) {
                            pattern = pattern.replace(p, "*");
                        }
                        break;
                    default:
                        Logger.warn(`Unknown partitioning type: ${dim.type}`);
                }
            }
        }
        return pattern;
    };

    const getExpandedIndexPattern = () => {
        if ($scope.dataset.managed || $scope.dataset.params.unmanagedPartitioningType !== "INDEX") {
            return;
        }
        const pattern = resolveUnmanagedPartitionedIndexPattern($scope.dataset.params.partitioningIndexPattern, $scope.dataset.partitioning.dimensions);
        for (const dim of $scope.dataset.partitioning.dimensions) {
            getPatternsToInsert($scope.dataset.params.partitioningIndexPattern, dim);
        }
        return pattern;
    }

    const getPatternsToInsert = (pattern, dim) => {
        if (!pattern) {
            pattern = "";
        }
        let ret = [];

        switch (dim.type) {
            case "value":
                ret = [{pattern: `%{${dim.name}}`, name: `${dim.name} (%{${dim.name}})`}];
                break;
            case "time":
                switch (dim.params.period) {
                    case "HOUR": // fallthrough
                        ret.push({pattern: "%H", name: "HOUR (%H)"});
                    case "DAY": // fallthrough
                        ret.push({pattern: "%D", name: "DAY (%D)"});
                    case "MONTH": // fallthrough
                        ret.push({pattern: "%M", name: "MONTH (%M)"});
                    case "YEAR":
                        ret.push({pattern: "%Y", name: "YEAR (%Y)"});
                        break;
                    default:
                        Logger.warn(`Unknown period type: ${dim.params.period}`);
                }
                ret.reverse(); // year to hour
                break;
            default:
                Logger.warn(`Unknown partitioning type: ${dim.type}`);
        }

        dim.$patterns = ret.filter(p => !pattern.includes(p.pattern));
    };

    $scope.hasTimeDimension = function () {
        if (!$scope.dataset.params.partitioned) {
            return false;
        }
        return $.grep($scope.dataset.partitioning.dimensions, (item) => item.type === 'time').length > 0;
    };

    $scope.$watch("dataset.partitioning.dimensions", (nv, ov) => {
        $scope.generatePartitionPattern();
        $scope.expandedIndexPattern = getExpandedIndexPattern();
    }, true);

    $scope.$watch("dataset.params.partitioningIndexPattern", (nv, ov) => {
        $scope.expandedIndexPattern = getExpandedIndexPattern();
    });

    $scope.partitionTemplate = { type: 'value' };
    $scope.checkPartitionDimension = function(it) {
        return it.type && it.name && (it.type !== 'time' || it.params.period);
    };

    $scope.testAPI = DataikuAPI.datasets.elasticsearch.test

    $scope.formatTestResult = function(res) {
        if (res && res.defaultMapping) { // prettify
            res.defaultMapping = JSON.stringify(JSON.parse(res.defaultMapping), null, '  ');
        }
        return res
    }

    $scope.testCallback = function (connectionOnly, data) {
        if (!connectionOnly && data.schemaDetection) {
            $scope.dataset.schema = data.schemaDetection.newSchema;
        }
        $scope.documentTypeNeeded = ['ES_LE_2', 'ES_5'].includes($scope.testResult.dialect);
        if ($scope.testResult.version != undefined) { // server unreachable will not return any ES version
            var terms = $scope.testResult.version.split('.');
            if(terms.length >= 2){
                var major = parseInt(terms[0]);
                var minor = parseInt(terms[1]);
                if(major < 6 || (major == 6 && minor <2)) {
                    $scope.indexDocumentTypePattern = new RegExp('^(?!_).+$');
                } else if (major == 6 && minor >= 2) {
                    $scope.indexDocumentTypePattern = new RegExp('^((?!_).+)$|^_doc$');
                }
            } else {
                $scope.indexDocumentTypePattern = new RegExp('^(?!_).+$');
            }
        }

        $scope.suggestDatasetName($scope.testResult.suggestedName);
    };


    const listIndices = () => {
        return DataikuAPI.datasets.elasticsearch.listIndices($stateParams.projectKey, $scope.dataset)
            .then(function({data}) {
                const foundIndices = [];
                for (const [name, count] of Object.entries(data)) {
                    foundIndices.push({name, count});
                }
                return foundIndices;
            }).catch(setErrorInScope.bind($scope));
    };

    $scope.foundIndices = null;
    $scope.listIndices = function() {
        return listIndices().then((foundIndices) => {
            $scope.foundIndices = foundIndices;
        });
    };

    $scope.foundPartitions = null;
    $scope.listPartitions = function() {
        return listIndices().then((foundIndices) => {
            $scope.foundPartitions = foundIndices;
        });
    };

    $scope.isUnmanagedPartitionedOnIndex = () => {
        const dataset = $scope.dataset;
        return !dataset.managed && dataset.params.partitioned && dataset.params.unmanagedPartitioningType === 'INDEX';
    };

    $scope.$watch('dataset.params.connection', function (){
        //Wait init before watching changes.
        if($scope.testResult){
            $scope.test(true);
        }
    });
    $scope.documentTypeNeeded = false;
});

}());
