(function() {
'use strict';

var app = angular.module('dataiku.notebooks', ['dataiku.services', 'dataiku.filters', 'dataiku.controllers']);

app.controller('NotebooksCommons', ($scope) => {
    $scope.getCodeEnvHint = item => {
        if (item.type === 'JUPYTER' ) {
            if (item.displayKernelSpec && item.displayKernelSpec.envName) {
                return ' in ' + item.displayKernelSpec.envName;
            }
        }
        return null;
    };
    $scope.getContainerHint = item => {
        if (item.type === 'JUPYTER' && item.displayKernelSpec) {
            if (item.displayKernelSpec.containerConf) {
                return ' in ' + item.displayKernelSpec.containerConf;
            }
        }
        return null;
    };
});

app.controller('NotebooksController', function($scope, $rootScope, $stateParams, $controller, TaggableObjectsUtils, StateUtils, $q,
               DataikuAPI, CreateModalFromTemplate, TaggableObjectsService, TaggingService, Dialogs, TopNav, DatasetUtils, NotebooksUtils, FutureProgressModal) {

    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});
    $controller('NotebooksCommons', { $scope: $scope });

    TopNav.setLocation(TopNav.TOP_NOTEBOOKS, 'notebooks', TopNav.TABS_NONE, null);
    TopNav.setNoItem();

    $scope.projectKey = $stateParams.projectKey

    $scope.sortBy = [
        { value: 'name', label: 'Name' },
        { value: 'type', label: 'Type' },
        { value: '-lastModifiedOn', label: 'Last modified'},
        { value: 'niceConnection', label: 'SQL connection'}
    ];
    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            },
            analyzedDataset: [],
            gitReference: ''
        },
        filterParams: {
            userQueryTargets: ["name","type","language","connection","tags"],
            propertyRules: {tag: "tags",conn:"connection",lang:"language"},
            exactMatch: ['analyzedDataset']
        },
        orderQuery: "-lastModifiedOn",
        orderReversed: false,
    }, $scope.selection || {});
    $scope.sortCookieKey = 'notebooks';
    $scope.maxItems = 20;
    $scope.useNewIcons = true; // used during the icons migration, in some directives, to determine which icons to use

    if ($stateParams.datasetId) {
        $scope.selection.filterQuery.analyzedDataset.push($stateParams.datasetId);
    }

    DatasetUtils.listDatasetsUsabilityForAny($stateParams.projectKey).success(data => {
    	// Move the usable flag where it's going to be read
        data.forEach(x => {
            x.usable = x.usableAsInput;
            x.usableReason = x.inputReason;
        });
        $scope.availableDatasets = data;
    }).error(setErrorInScope.bind($scope));

    $scope.getNotebookIcon = function(item) {
    	return NotebooksUtils.getNotebookIcon(item);
    };


    function setTaggableType() {
        //since we cannot get notebook taggable type from angular state in taggable_objects.js, set it here to use in tag filter
        if (!$scope.listItemType && $scope.listItems.length) $scope.listItemType = TaggableObjectsUtils.taggableTypeFromAngularState($scope.listItems[0]);
    }

    $scope.list = function() {
        $scope.uiState = {activeTab : "actions"};
        let items = [];
    	const promisedSql = DataikuAPI.sqlNotebooks.listHeads($stateParams.projectKey, $rootScope.tagFilter).then(({data}) => {
            data.items.forEach((sqln) => {
                const parsedConnection = NotebooksUtils.parseConnection(sqln.connection);
                sqln.type = parsedConnection.type;
                sqln.niceConnection = parsedConnection.niceConnection;
            });
            return data.items;
        });

        const promisedES = DataikuAPI.searchNotebooks.listHeads($stateParams.projectKey, $rootScope.tagFilter).then(({data}) => {
            data.items.forEach((item) => {
                item.type = "SEARCH";
                item.niceConnection = item.connection;
            });
            return data.items;
        });

        let promised = $q.all([promisedSql, promisedES]).then(([sqlConn, esConn]) => {
            items = sqlConn.concat(esConn);
        });
        if ($scope.mayCreateActiveWebContent()) { // Only list jupyter notebooks if user has rights to create web content
            promised = promised.then(() => DataikuAPI.jupyterNotebooks.listHeads($stateParams.projectKey, $rootScope.tagFilter))
                .then(({data}) => {
                    data.items.forEach((ipn) => {
                        ipn.type = "JUPYTER";
                        ipn.id = ipn.name;
                    });
                    items = items.concat(data.items);
                });
        }
        promised.then(() => {
            $scope.listItems = items;
            $scope.restoreOriginalSelection();
            setTaggableType();
        })
        .catch(setErrorInScope.bind($scope));
    };
    $scope.list();

    /* Tags handling */

    $scope.$on('selectedIndex', function(e, index){
        // an index has been selected, we unselect the multiselect
        $scope.$broadcast('clearMultiSelect');
    });

    /* Specific actions */
    $scope.goToItem = function(notebook) {
        const notebookId = notebook.type === "JUPYTER" ? notebook.name : notebook.id;
        StateUtils.go.notebook(notebook.type, notebookId, $scope.projectKey);
    };

    $scope.newNotebookHub = function () {
        CreateModalFromTemplate("/templates/notebooks/new-notebook-selector-modal.html", $scope);
    }

    $scope.newNotebook = function() {
        CreateModalFromTemplate("/templates/notebooks/new-notebook-modal.html", $scope);
    };

    $scope.newNotebookFromFile = function() {
        CreateModalFromTemplate("/templates/notebooks/new-notebook-from-file-modal.html", $scope);
    };

    $scope.newNotebookFromGit = function() {
        CreateModalFromTemplate("/templates/notebooks/new-notebook-from-git-modal.html", $rootScope, null, newScope => {
            newScope.gitRef = {
            };

            newScope.isItemSelectable = item => {
                return item && item.isValid && item.nbFormat >= 4;
            }

            newScope.listNotebooks = () => {
                DataikuAPI.jupyterNotebooks.git.listRemoteNotebooks(newScope.gitRef.remote, newScope.gitRef.remoteLogin, newScope.gitRef.remotePassword, newScope.gitRef.ref)
                .success(data => {
                    FutureProgressModal.show(newScope, data, "List remote notebooks").then(notebooks => {
                        if (notebooks) {
                            newScope.remoteNotebooks = notebooks
                                .filter(notebook => {
                                    // Notebooks that are not valid do not have language
                                    // Notebooks without identified language are also shown
                                    if (!notebook.isValid || !notebook.language) {
                                        return true;
                                    }
                                    if(notebook.language === 'R' || notebook.language === 'ir') {
                                        return $rootScope.appConfig.uiCustomization.showR;
                                    } else if(notebook.language === 'scala' || notebook.language === 'toree') {
                                        return $rootScope.appConfig.uiCustomization.showScala;
                                    } else if(notebook.language.toLowerCase().startsWith('julia')) {
                                        return $rootScope.featureFlagEnabled('julia');
                                    } else {
                                        return true; // other languages are not hidden
                                    }
                                })
                                .map(notebook => Object.assign({$selected: notebook.nbFormat >= 4, type: "JUPYTER"}, notebook));
                        }
                    });
                }).error(setErrorInScope.bind(newScope));
            }

            newScope.importNotebooks = () => {
                DataikuAPI.jupyterNotebooks.git.importNotebooks($stateParams.projectKey, newScope.gitRef.remote, newScope.gitRef.remoteLogin, newScope.gitRef.remotePassword, newScope.gitRef.ref, newScope.selection.selectedObjects).success(data => {
                    const parentScope = newScope.$parent;
                    newScope.dismiss();
                    FutureProgressModal.show(parentScope, data, "Import remote notebooks").then(() => {
                        $scope.list();
                    })
                }).error(setErrorInScope.bind(newScope));
            }
        });
    };

	$scope.unloadJupyterNotebook = function(session_id) {
        Dialogs.confirm($scope, 'Unload Jupyter kernel', 'Are you sure you want to stop this notebook?').then(function () {
    		DataikuAPI.jupyterNotebooks.unload(session_id).success(function(data) {
    			$scope.list();
    		}).error(setErrorInScope.bind($scope));
        });
	};

    $scope.startApplyTagging = function() {
        var items =  $scope.selection.selectedObjects.map(function(item) {
            return {
                id : getNotebookId(item),
                displayName: item.name,
                type: $scope.getTaggableType(item),
                projectKey: $stateParams.projectKey
            };
        });
        TaggingService.startApplyTagging(items).then($scope.list, setErrorInScope.bind($scope));
    };

    $scope.copyNotebook = (notebook) => NotebooksUtils.copySqlOrSearchNotebook($stateParams.projectKey, notebook.type, notebook.id, notebook.name, $scope);

    $scope.getTaggableType = (item) => {
        switch (item.type) {
            case "SEARCH":
                return "SEARCH_NOTEBOOK";
            case "JUPYTER":
                return "JUPYTER_NOTEBOOK";
            default:
                return "SQL_NOTEBOOK";
        }
    }

    function getNotebookId(item) {
        return item.type == 'JUPYTER' ? item.name : item.id;
    }

    // Not the generic handling for all other types since we have two taggable types on that page
    $scope.deleteSelected = function() {
        var deletionRequests;
        if($scope.selection.single) {
            var item = $scope.selection.selectedObject;
            deletionRequests = [{
                type: $scope.getTaggableType(item),
                projectKey: $stateParams.projectKey,
                id: item.id,
                displayName: item.name,
                activeSessions: item.activeSessions //TODO @flow hack, remove
            }];
        } else {
            deletionRequests = $scope.selection.selectedObjects.map(function(item){
                return {
                    type: $scope.getTaggableType(item),
                    projectKey: $stateParams.projectKey,
                    id: item.id,
                    displayName: item.name,
                    activeSessions: item.activeSessions //TODO @flow hack, remove
                };
            });
        }

        TaggableObjectsService.delete(deletionRequests)
            .then($scope.list, setErrorInScope.bind($scope));
    };

    $scope.allRemoteNotebooks = selectedObjects => {
        return selectedObjects.every(obj => obj.gitReference && obj.type == 'JUPYTER');
    }

    $scope.pushNotebooksToRemote = NotebooksUtils.pushNotebooksToRemote($scope);
    $scope.pullNotebooksFromRemote = NotebooksUtils.pullNotebooksFromRemote($scope);
    $scope.editNotebookReference = NotebooksUtils.editNotebookReference($scope)($scope.list);
    $scope.unlinkNotebookReference = NotebooksUtils.unlinkNotebookReference($scope)($scope.list);
    $scope.hasRemoteNotebooks = list => list.some(o => o.gitReference && o.type == 'JUPYTER');
});


app.controller('NewNotebookModalController', function($scope, $rootScope, $stateParams, DataikuAPI, WT1, DatasetUtils, GlobalProjectActions, NotebooksUtils, StateUtils) {
    // choose-type, sql, hive, spark, impala, python, r, scala, customjupyter
    $scope.uiState = {
        step : "choose-type"
    }

    function getSQLNotebookType(step) {
        return ['hive', 'impala'].includes(step) ? step + '-jdbc' : step;
    }

    function createSQLNotebookWithoutDataset() {
        DataikuAPI.sqlNotebooks.create($scope.newNotebook.projectKey, $scope.newNotebook.connection, $scope.newNotebook.name).success(data => {
            WT1.event('notebook-sql-create', {
                notebookId : data.id,
                type: getSQLNotebookType($scope.uiState.step),
                withDataset: false
            });
            $scope.dismiss();
            StateUtils.go.sqlNotebook(data.id, $scope.newNotebook.projectKey);
        }).error(setErrorInScope.bind($scope));
    }

    function createSQLNotebookWithDataset() {
        const type = getSQLNotebookType($scope.uiState.step);
        DataikuAPI.sqlNotebooks.createForDataset($stateParams.projectKey, $scope.datasetSmartName, type, $scope.newNotebook.name)
            .success(data => {
                WT1.event('notebook-sql-create', {
                    notebookId : data.id,
                    type: type,
                    withDataset: true
                });
                StateUtils.go.sqlNotebook(data.id, $stateParams.projectKey);
                $scope.dismiss();
            }).error(setErrorInScope.bind($scope));
    }

    function createJupyterNotebookWithoutDataset() {
        DataikuAPI.jupyterNotebooks.newNotebookWithTemplate($stateParams.projectKey,
            $scope.newNotebook.name,
            $scope.newNotebook.template, $scope.newNotebook.codeEnv, $scope.newNotebook.containerConf).success(data => {

            WT1.event('notebook-jupyter-create', {
                notebookId : data.name,
                language: $scope.newNotebook.template.language,
                template : $scope.newNotebook.template.id,
                withDataset: false,
                useCodeEnv: !!data.displayKernelSpec.name
            });

            $scope.dismiss();
            StateUtils.go.jupyterNotebook(data.name, $stateParams.projectKey);
        }).error(setErrorInScope.bind($scope));
    }

    function createJupyterNotebookWithDataset() {
        DataikuAPI.jupyterNotebooks.newNotebookForDataset($stateParams.projectKey,
            $scope.newNotebook.name,
            $scope.datasetSmartName, $scope.newNotebook.template, $scope.newNotebook.codeEnv, $scope.newNotebook.containerConf)
            .success(data => {
                WT1.event('notebook-jupyter-create', {
                    language: data.language,
                    template : $scope.newNotebook.template.id,
                    withDataset: true,
                    useCodeEnv: !!data.displayKernelSpec.name
                });
                StateUtils.go.jupyterNotebook(data.name, $stateParams.projectKey);
                $scope.dismiss();
            }).error(setErrorInScope.bind($scope));
    }

    function createNotebookWithDataset() {
        if (NotebooksUtils.sqlNotebooksTypes.includes($scope.uiState.step)) {
            createSQLNotebookWithDataset();
        } else {
            createJupyterNotebookWithDataset();
        }
    }

    function installSQLFunctions() {
        let autoNotebookName = 'New notebook';
        let userModifiedName = false;
        $scope.connections = [];

        $scope.newNotebook = {
            projectKey: $stateParams.projectKey,
            name: autoNotebookName
        };

        $scope.$watch("newNotebook.name", (nv) => {
            if (nv != null && nv != autoNotebookName) {
                userModifiedName = true;
            }
        });

        //  When creating a sql notebook from a dataset (if datasetSmartName), the connection is automatically chosen by the backend so we do not fetch the list.
        if (!$scope.datasetSmartName) {
            DataikuAPI.sqlNotebooks.listConnections($stateParams.projectKey).success(data => {
                $scope.connections = data.nconns;
                $scope.hiveError = data.hiveError;
            }).error(setErrorInScope.bind($scope));

            $scope.$watch("newNotebook.connection", (nv) => {
                if (!userModifiedName) {
                    for(var k in $scope.connections) {
                        var conn = $scope.connections[k];
                        if (conn.name == nv) {
                            autoNotebookName = $scope.appConfig.login + "'s notebook on " + conn.label;
                            $scope.newNotebook.name = autoNotebookName;
                        }
                    }
                }
            });
        } else {
            $scope.newNotebook.name = `${$scope.appConfig.login}'s sql notebook on ${$scope.datasetSmartName}`;
        }

        $scope.createAndRedirect = function() {
            if ($scope.datasetSmartName) {
                createNotebookWithDataset();
            } else {
                createSQLNotebookWithoutDataset();
            }
        };
    }

    const NICE_TYPES = {r: "R", python: "Python", scala: "Scala (Spark)", julia: "Julia"};

    function installJupyterStd(type) {
        const niceType = NICE_TYPES[type] || type;
        const autoNotebookName = $scope.appConfig.user.login.replace(/\.+/g, ' ') + "'s " + niceType + " notebook";

        $scope.newNotebook = {
            name : autoNotebookName,
            language: type,
            containerConf: ''
        };

        if ($scope.datasetSmartName) {
            $scope.newNotebook.name += ' on ' + $scope.datasetSmartName.replace(/\.+/g, ' ');
        }

        DataikuAPI.notebooks.listTemplates($scope.datasetSmartName ? 'DATASET' : 'STANDALONE', type).success(data => {
            $scope.availableTemplates = data.templates;
            $scope.newNotebook.template = $scope.availableTemplates[0];
        }).error(setErrorInScope.bind($scope));

        $scope.$watch("newNotebook", function(nv, ov) {
            if (!nv) return;
            if ($scope.availableCodeEnvs == null) {
            	if (nv.language == "python" || nv.language == "r" || nv.language == "julia") {
                    $scope.availableCodeEnvs = []; // so that you only do the call once
                    DataikuAPI.codeenvs.listNamesWithDefault(nv.language.toUpperCase(), $stateParams.projectKey).success(function(data){
                        $scope.availableCodeEnvs =
                            [["__BUILTIN__", "Default builtin env"]].concat(data.envs.map(function(ce){
                                return [ce.envName, ce.envName];
                            }))
                        $scope.newNotebook.codeEnv = data.resolvedInheritDefault || "__BUILTIN__";
                    }).error(setErrorInScope.bind($scope));
            	} else {
            		$scope.availableCodeEnvs = [];
                }
            }
            if ($scope.containerConfs == null) {
                if (nv.language == "python" || nv.language == "r" || nv.language == "julia") {
                    $scope.containerConfs = [{id:'', label:"Run locally"}]
                    DataikuAPI.containers.listNamesWithDefault($stateParams.projectKey, null, "USER_CODE").success(function(data) {
                        data.containerNames.forEach(function(n) { $scope.containerConfs.push({id:n, label:n});});

                        if (data.resolvedInheritValue) {
                            $scope.newNotebook.containerConf = data.resolvedInheritValue;
                        }

                    }).error(setErrorInScope.bind($scope));
            	} else {
            		$scope.containerConfs = []
                }
            }
        }, true);

        $scope.createAndRedirect = function() {
            if ($scope.datasetSmartName) {
                createNotebookWithDataset();
            } else {
                createJupyterNotebookWithoutDataset();
            }
        };
    }


    function setUpSearchNotebook(type) {
        $scope.elasticSearchConnections = [];
        const buildNotebookName = (connection) => $scope.appConfig.user.login.replace(/\.+/g, ' ') + "'s " + `search notebook${connection ? " on " + connection : ""}`;

        $scope.$watch("newNotebook.connection", (nv, ov) => {
            if (!nv) {
                return;
            }

            const oldAutoNotebookName = buildNotebookName(ov);
            if (oldAutoNotebookName === $scope.newNotebook.name) {
                $scope.newNotebook.name = buildNotebookName(nv);
            }
        });

        $scope.newNotebook = {
            name : buildNotebookName($scope.newNotebook.connection),
            connection: null,
            language: "ES_QUERY_STRING",
        };

        DataikuAPI.searchNotebooks.listConnections().success(data => {
            $scope.elasticSearchConnections = data.nconns;
        }).error(setErrorInScope.bind($scope));

        $scope.createAndRedirect = function() {
           DataikuAPI.searchNotebooks.create($stateParams.projectKey, $scope.newNotebook.connection, $scope.newNotebook.name).success(data => { //
               WT1.event('notebook-search-create', {
                   notebookId : data.id,
                   withDataset: false
               });
               $scope.dismiss();
               StateUtils.go.searchNotebook(data.id, $stateParams.projectKey);
           }).error(setErrorInScope.bind($scope));
       }
    }

    $scope.$watch("uiState.step", function(nv, ov) {
        switch (nv) {
            case "sql":
            case "impala":
            case "hive":
            case "sparksql":
                installSQLFunctions(nv);
                break;
            case "python":
            case "julia":
            case "r":
            case "scala":
                installJupyterStd(nv);
                break;
            case "search":
                setUpSearchNotebook(nv);
                break;
        }
    });

    if ($scope.datasetSmartName) {
        DatasetUtils.listDatasetsUsabilityForAny($stateParams.projectKey).success(data => {
            // move the usable flag where it's going to be read
            data.forEach(x => {
                x.usable = x.usableAsInput;
                x.usableReason = x.inputReason;
            });
            $scope.availableDatasets = data;
            // set the usable flag here instead of in the UsabilityComputer, like the other places seem to do
            angular.forEach($scope.availableDatasets, x => {
                x.usable = true;
            });
        }).error(setErrorInScope.bind($scope));

        $scope.usability = {};

        const parts = $scope.datasetSmartName.match(/([^.]+)\.(.+)/) || [$scope.datasetSmartName, $stateParams.projectKey, $scope.datasetSmartName]; // [smart, project, dataset]

        DataikuAPI.datasets.getFullInfo($stateParams.projectKey, parts[1], parts[2])
            .success(data => {
                var hasSql = false;

                // Check which languages are available
                ['sql', 'hive', 'impala', 'pig', 'sql99'].forEach(thing => {
                    $scope.usability[thing] = GlobalProjectActions.specialThingMaybePossibleFromDataset(data.dataset, thing);
                    hasSql = hasSql || $scope.usability[thing].ok;
                });

                $scope.usability.spark = { ok: true };

                if (!$rootScope.appConfig.sparkEnabled) {
                    if (!$rootScope.addLicInfo.sparkLicensed) {
                        $scope.usability.spark.ok = false;
                        $scope.usability.spark.reason = 'Spark is not licensed';
                    } else {
                        $scope.usability.spark.ok = false;
                        $scope.usability.spark.reason = 'Spark is not configured';
                    }
                }
            }).error(setErrorInScope.bind($scope));
    }

    $scope.canUseLanguage = function(language) {
        if (!$scope.usability || !$scope.usability[language]) {
            return true;
        } else {
            return $scope.usability[language].ok;
        }
    }

    $scope.getDisabledReason = function(language) {
        if (!$scope.usability || !$scope.usability[language] || !$scope.usability[language].reason) {
            return;
        } else {
            return $scope.usability[language].reason.iconDisabledReason;
        }

    }
});

app.controller('NewNotebookFromTemplateModalController', function($scope, $rootScope, StateUtils, $stateParams, DataikuAPI, WT1) {

    $scope.newTemplatedNotebook = {};

    let templatedBaseName = $rootScope.appConfig.login + "'s "
        + "analysis of " + $scope.datasetSmartName;

    $scope.newTemplatedNotebook.baseName = templatedBaseName.replace(/\.+/g, ' ');

    DataikuAPI.notebooks.listTemplates("DATASET", "pre-built").success(data => {
        $scope.newTemplatedNotebook.availableTemplates = data.templates;
        $scope.newTemplatedNotebook.template = $scope.newTemplatedNotebook.availableTemplates[0];
    }).error(setErrorInScope.bind($scope));

    if ($scope.containerConfs == null) {
        $scope.containerConfs = [{id:'', label:"Run locally"}]
        DataikuAPI.containers.listNamesWithDefault($stateParams.projectKey, null, "USER_CODE").success(data => {
            data.containerNames.forEach(n => { $scope.containerConfs.push({ id: n, label: n }); });
        }).error(setErrorInScope.bind($scope));
    }

    $scope.$watch('newTemplatedNotebook.template', () => {
        if ($scope.newTemplatedNotebook.template != null) { // something was selected
            if (templatedBaseName == $scope.newTemplatedNotebook.baseName) { // name was not modified by user (yet)
                templatedBaseName = ($scope.newTemplatedNotebook.template.title || $scope.newTemplatedNotebook.template.label) + " on "  + $scope.datasetSmartName + " (" + $rootScope.appConfig.login + ")";
                $scope.newTemplatedNotebook.baseName = templatedBaseName.replace(/\.+/g, ' ');
            }
        }
    });

    $scope.createTemplatedNotebook = () => {
        DataikuAPI.jupyterNotebooks.newNotebookForDataset($stateParams.projectKey,
            $scope.newTemplatedNotebook.baseName,
            $scope.datasetSmartName, $scope.newTemplatedNotebook.template, $scope.newTemplatedNotebook.codeEnv, $scope.newTemplatedNotebook.containerConf)
            .success(data => {
                WT1.event("notebook-jupyter-create", {
                    notebookId : data.name,
                    // Anonymize custom template ids
                    template:
                        $scope.newTemplatedNotebook.template.origin === 'PLUGIN'
                        ? $scope.newTemplatedNotebook.template.id.dkuHashCode()
                        : $scope.newTemplatedNotebook.template.id,
                    withDataset: true,
                    useCodeEnv: !!data.displayKernelSpec.name
                });
                StateUtils.go.jupyterNotebook(data.name, $stateParams.projectKey);
                $scope.dismiss();
            }).error(setErrorInScope.bind($scope));
    };
});

app.controller('NewNotebookFromFileModalController', function($scope, $stateParams, DataikuAPI, WT1, StateUtils) {

    $scope.newNotebook = {
        name : $scope.appConfig.user.login.replace(/\.+/g, ' ') + "'s notebook from file"
    };

    function parseNotebook(event) {
        $scope.isParsing = true;
        $scope.$apply();
        // Adding a timeout only for User Experience to display a loader while parsing! It requires using apply for quick refresh.
        window.setTimeout(() => {
            try {
                let fileContent;
                fileContent = JSON.parse(event.target.result);

                $scope.isParsing = false;
                $scope.hasParsingFailed = false;

                let parsedLanguage = null;

                if (fileContent.metadata && fileContent.metadata.kernelspec && fileContent.metadata.kernelspec.language) {
                    parsedLanguage = fileContent.metadata.kernelspec.language;
                } else if (fileContent.metadata && fileContent.metadata.language_info && fileContent.metadata.language_info.name) {
                    parsedLanguage = fileContent.metadata.language_info.name;
                } else {
                    parsedLanguage = "python"; // By far the most likely. Accounts for Databricks notebooks
                }


                $scope.newNotebook.language = parsedLanguage.toLowerCase();
                $scope.isLanguageSupported = !(['python', 'r', 'scala'].includes($scope.newNotebook.language));
                $scope.$apply();
            } catch (exception) {
                $scope.hasParsingFailed = true;
                $scope.isParsing = false;
                $scope.$apply();
            }
        }, 50);
    }

    $scope.uploadAndRedirect = function() {
        DataikuAPI.jupyterNotebooks.newNotebookFromFile($stateParams.projectKey,
            $scope.newNotebook.name,
            $scope.newNotebook.language,
            $scope.datasetSmartName,
            $scope.newNotebook.file)
                .then(data => {
                    data = JSON.parse(data);

                    WT1.event('notebook-jupyter-upload', {
                        notebookId : data.name,
                        language: $scope.newNotebook.language,
                        withDataset: !!$scope.datasetSmartName
                    });

                    $scope.dismiss();
                    StateUtils.go.jupyterNotebook(data.name, $stateParams.projectKey);
                }, (error) => {
                    setErrorInScope2.call($scope, error);
                });
    };

    $scope.$watch("newNotebook.file", function(newValue) {
        $scope.newNotebook.language = null;
        $scope.isLanguageSupported = false;
        $scope.hasParsingFailed = false;

        if (newValue) {
            const reader = new FileReader();
            reader.onload = parseNotebook;
            reader.readAsText(newValue);
            $scope.newNotebook.name = newValue.name.replace('.ipynb', '');
        }
    });
});

app.controller('NotebookGitPushWithConflictModalController', function($scope, $stateParams, DataikuAPI, FutureProgressModal, Dialogs, DKUtils) {
    $scope.hasNoNotebooksSelectedForPush = () => {
        return !$scope.notebookConflictStatus.conflictingNotebooks.some(notebook => notebook.selected)
            && !$scope.notebookConflictStatus.nonConflictingNotebooks.some(notebook => notebook.selected)
            && !$scope.notebookConflictStatus.noLongerOnRemoteNotebooks.some(notebook => notebook.selected);
    }

    // User have chosen notebooks, push force them.
     $scope.forcePushNotebooks = () => {
        const nonConflictingNotebooks = $scope.notebookConflictStatus.nonConflictingNotebooks;
        const conflictingNotebooks =  $scope.notebookConflictStatus.conflictingNotebooks.filter(notebook => notebook.selected);
        const noLongerOnRemoteNotebooks =  $scope.notebookConflictStatus.noLongerOnRemoteNotebooks.filter(notebook => notebook.selected);

        if (!$scope.commitMessage.title) {
            $scope.commitMessage.title = $scope.getDefaultCommmitMessage();
        }
        let commitMessage = $scope.commitMessage.title;
        if ($scope.commitMessage.content && $scope.commitMessage.title) {
            commitMessage = `${$scope.commitMessage.title}\n${$scope.commitMessage.content}`;
        }
        DataikuAPI.jupyterNotebooks.git.pushNotebooksToGit($stateParams.projectKey,
            [...nonConflictingNotebooks, ...conflictingNotebooks, ...noLongerOnRemoteNotebooks],
            commitMessage)
        .success(data => {
            const scope = $scope.$parent;
            $scope.dismiss();
            FutureProgressModal.show(scope, data, "Push notebooks to remote").then(pushReports => {
                if (pushReports) {
                    Dialogs.infoMessagesDisplayOnly(scope, "Report", pushReports, pushReports.futureLog).then(() => {
                        if ($stateParams && $stateParams.notebookId) {
                            DKUtils.reloadState();
                        }
                    })
                }
            });
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller('NotebookGitPullWithConflictModalController', function($scope, $stateParams, DataikuAPI, FutureProgressModal, DKUtils, Dialogs) {
    $scope.hasNoNotebooksSelectedForPull = () => {
        return !$scope.notebookConflictStatus.conflictingNotebooks.some(notebook => notebook.selected)
            && !$scope.notebookConflictStatus.nonConflictingNotebooks.some(notebook => notebook.selected);
    }

    // User have chosen notebooks, push force them.
     $scope.forcePullNotebooks = () => {
        const nonConflictingNotebookIds = $scope.notebookConflictStatus.nonConflictingNotebooks.map(notebook => notebook.notebookName);
        const conflictingNotebookIds =  $scope.notebookConflictStatus.conflictingNotebooks.filter(notebook => notebook.selected).map(notebook => notebook.notebookName);

        DataikuAPI.jupyterNotebooks.git.pullNotebooks($stateParams.projectKey, [...nonConflictingNotebookIds, ...conflictingNotebookIds])
        .success(data => {
            const parentScope = $scope.$parent;
            $scope.dismiss();
            FutureProgressModal.show(parentScope, data, "Pull notebooks from remote").then(pullReports => {
                if (pullReports) {
                    Dialogs.infoMessagesDisplayOnly(parentScope, "Report", pullReports, pullReports.futureLog).then(() => {
                        if ($stateParams && $stateParams.notebookId) {
                            DKUtils.reloadState();
                        }
                    })
                }
            });
        }).error(setErrorInScope.bind($scope));
    }
});

app.directive('notebookConflictHit', function() {
    return {
        scope: {
          notebook: '=',
          noSelection: '=?' // When we are not enabling the selection, reduce the margin-left to 36px to have a better style
        },
        template: `
        <div class="hit h100">
            <div class="illustration">
                <span ng-if="notebook.language && notebook.language.toLowerCase().startsWith('python')">
                    <i class="dku-icon-python-circle-24 universe-color notebook"></i>
                </span>
                <span ng-if="notebook.language=='ir' || notebook.language === 'R'">
                    <i class="dku-icon-recipe-r-circle-fill-24 universe-color notebook"></i>
                </span>
                <span ng-if="notebook.language.toLowerCase().startsWith('julia')">
                    <i class="dku-icon-recipe-julia-circle-fill-24 universe-color notebook"></i>
                </span>
                <span ng-if="notebook.language=='toree' || notebook.language.toLowerCase() === 'scala'">
                    <i class="dku-icon-recipe-sparkscala-circle-fill-24 universe-color notebook"></i>
                </span>
            </div>
            <div class="h100 hitContent" ng-class="{'hitContent__no-selection': noSelection}">
                <div class="hit-content-main" style="max-width: calc(100% - 40px)">
                    <p class="hit-content-main__title" show-tooltip-on-text-overflow text-tooltip="notebook.notebookName">{{notebook.notebookName}}</p>
                    <p class="hit-content-main__subtitle" show-tooltip-on-text-overflow text-tooltip="notebook.gitUrl+' - '+notebook.gitBranch">
                        {{notebook.gitUrl}} - {{notebook.gitBranch}}
                    </p>
                </div>
                <i class="icon-info-sign text-prompt" style="float: right; margin-top: 5px;" toggle="tooltip" container="body" title="Name on the remote Git: {{notebook.remoteNotebookName}}" />
            </div>
        </div>`
    };
});

app.service('NotebooksUtils', function(CreateModalFromTemplate, DataikuAPI, DKUtils, $stateParams, $rootScope, FutureProgressModal, WT1, StateUtils) {
    var svc = {
        parseConnection: function(connection) {
            var virtualPattern = /^@virtual\((.+)\):(connection:)?(.+)$/;
            var parsed = virtualPattern.exec(connection);
            var niceConnection = connection;
            var type = "SQL";
            if(parsed) {
                if(parsed[1].indexOf('impala')>=0) {
                    niceConnection = parsed[3] + ' (Impala)';
                    type = "IMPALA";
                } else if(parsed[1].indexOf('hive')>=0) {
                    niceConnection = parsed[3] + ' (Hive)';
                    type = "HIVE";
                } else {
                    niceConnection = parsed[3] + ' (SparkSQL)';
                    type = "SPARKSQL";
                }
            }
            return {type: type, niceConnection:niceConnection};
        },

        /**
         * Notebook icon
         * @param {*} item
         * @param {number} size - Either 16 | 20 | 24 | 32 | 48
         * @returns
         */
        getNotebookIcon: function(item, size = 24) {
            const lowerCaseLanguage = item.language.toLowerCase();
            switch (lowerCaseLanguage) {
                case "python2":
                case "python3":
                case "python":
                    return `dku-icon-python-circle-${size}`;
                case "ir":
                case "r":
                    return `dku-icon-recipe-r-circle-fill-${size}`;
                case "toree":
                case "scala":
                    return `dku-icon-recipe-sparkscala-circle-fill-${size}`;
                case "es_query_string":
                    return `dku-icon-dataset-search-${size}`;
            }
            if (lowerCaseLanguage.startsWith('julia')) {
                return `dku-icon-recipe-julia-circle-fill-${size}`;
            }
            const nbType = item.notebookType || item.type || item.language;
            if (nbType == 'SQL') {return `dku-icon-recipe-sql-circle-fill-${size}`;}
            else if (nbType == 'HIVE') {return `dku-icon-recipe-hive-circle-fill-${size}`;}
            else if (nbType == 'IMPALA') {return `dku-icon-recipe-impala-circle-fill-${size}`;}
            else if (nbType == 'SPARKSQL') {return `dku-icon-recipe-sparksql-circle-fill-${size}`;}
            else {return `dku-icon-edit-note-${size}`}
        },

        pushNotebooksToRemote: function(scope) {
            return function(selectedNotebooks) {
                // Start by investigating if there is some remote notebooks that have changed
                DataikuAPI.jupyterNotebooks.git.getConflictingNotebooks($stateParams.projectKey, selectedNotebooks.filter(n => n.gitReference).map(n => n.id), false)
                .success(data => {
                    FutureProgressModal.show(scope, data, "Checking conflicts").then(notebookConflictStatus => {
                        if (notebookConflictStatus) {
                            // We have conflicting remove notebooks, let's pop a modal to ask the user what we do next
                            CreateModalFromTemplate("/templates/notebooks/notebook-git-push-with-conflict-modal.html", scope, "NotebookGitPushWithConflictModalController", newScope => {
                                newScope.notebookConflictStatus = notebookConflictStatus
                                newScope.getDefaultCommmitMessage = () => {
                                    if (notebookConflictStatus.nonConflictingNotebooks.length + notebookConflictStatus.conflictingNotebooks.length === 1) {
                                        return 'Export ' + notebookConflictStatus.nonConflictingNotebooks.concat(notebookConflictStatus.conflictingNotebooks)[0].notebookName;
                                    } else {
                                        return 'Export %filename%';
                                    }
                                }
                                newScope.commitMessage = {
                                    title: "",
                                    content: ""
                                };
                            });
                        }
                    })
                }).error(setErrorInScope.bind(scope));
            };
        },
        pullNotebooksFromRemote: function(scope) {
            return function(selectedNotebooks) {
                DataikuAPI.jupyterNotebooks.git.getConflictingNotebooks($stateParams.projectKey, selectedNotebooks.filter(n => n.gitReference).map(n => n.id), true)
                .success(data => {
                    FutureProgressModal.show(scope, data, "Checking conflicts").then(notebookConflictStatus => {
                        if (notebookConflictStatus) {
                            // We have conflicting remove notebooks, let's pop a modal to ask the user what we do next
                            CreateModalFromTemplate("/templates/notebooks/notebook-git-pull-with-conflict-modal.html", scope, "NotebookGitPullWithConflictModalController", newScope => {
                                newScope.notebookConflictStatus = notebookConflictStatus
                            });
                        }
                    })
                }).error(setErrorInScope.bind(scope));
            };
        },
        editNotebookReference: function(scope) {
            return function(endCallback = () => {}) {
                return function(notebook) {
                    CreateModalFromTemplate("/templates/notebooks/notebook-git-edit-reference-modal.html", scope, null, newScope => {
                        newScope.isEditMode = notebook.gitReference !== undefined;
                        newScope.notebook = notebook;
                        newScope.gitRef = Object.assign({}, notebook.gitReference);
                        if (!newScope.gitRef.remotePath) {
                            newScope.gitRef.remotePath = notebook.name + ".ipynb";
                        }
                        newScope.save = () => {
                            DataikuAPI.jupyterNotebooks.git.editReference(notebook.projectKey, notebook.name, newScope.gitRef)
                            .success(() => {
                                endCallback();
                                newScope.dismiss();
                            })
                            .error(setErrorInScope.bind(newScope));
                        }
                    });
                };
            };
        },
        unlinkNotebookReference: function(scope) {
            return function(endCallback = () => {}) {
                return function(notebook) {
                    CreateModalFromTemplate("/templates/notebooks/notebook-git-unlink-reference-modal.html", scope, null, newScope => {
                        // transform into NotebooksPushGitDTO
                        newScope.notebook = {
                            notebookName: notebook.name,
                            remoteNotebookName: notebook.gitReference.remotePath,
                            gitUrl: notebook.gitReference.remote,
                            gitBranch: notebook.gitReference.checkout,
                            language: notebook.language
                        };

                        newScope.unlink = () => {
                            DataikuAPI.jupyterNotebooks.git.unlinkReference(notebook.projectKey, notebook.name)
                            .success(() => {
                                endCallback();
                                newScope.dismiss();
                            })
                            .error(setErrorInScope.bind(newScope));
                        }
                    });
                };
            };
        },

        getSqlSearchNotebooksApiPrefix: (objectType) => {
            switch (objectType) {
                case "SEARCH_NOTEBOOK":
                    return DataikuAPI.searchNotebooks;
                case "SQL_NOTEBOOK":
                    return DataikuAPI.sqlNotebooks;
                default:
                    throw new Error('Not a supported notebook type: ' + objectType);
            }
        },

        copySqlOrSearchNotebook: (projectKey, type, id, name, scope) => {
            if (type === "SEARCH") {
                WT1.event("search-notebook-copy", {});
                DataikuAPI.searchNotebooks.copy(projectKey, id, name+'_copy').success(function(data) {
                    StateUtils.go.searchNotebook(data.id, projectKey);
                }).error(setErrorInScope.bind(scope));
            } else {
                WT1.event("sql-notebook-copy", {});
                DataikuAPI.sqlNotebooks.copy(projectKey, id, name+'_copy').success(function(data) {
                    StateUtils.go.sqlNotebook(data.id, projectKey);
                }).error(setErrorInScope.bind(scope));
            }
        },

        sqlNotebooksTypes: [
            'sql',
            'impala',
            'hive',
            'sparksql'
        ]
    };
    return svc;
});

app.directive('sqlSearchNotebookRightColumnSummary', function($rootScope, DataikuAPI, HistoryService, TopNav, QuickView, NotebooksUtils, $controller,
                                                              ActiveProjectKey, ActivityIndicator, WT1, CreateModalFromTemplate, Assert, $stateParams) {
    return {
        templateUrl :'/templates/notebooks/sql-search-notebook-right-column-summary.html',

        link : function(scope) {
            Assert.inScope(scope, "objectType");

            const apiPrefix = NotebooksUtils.getSqlSearchNotebooksApiPrefix(scope.objectType);
            scope.statePrefix = scope.objectType === "SEARCH_NOTEBOOK" ? "projects.project.notebooks.search_notebook" : "projects.project.notebooks.sql_notebook";

            $controller('_TaggableObjectsMassActions', {$scope: scope});
            $controller('_TaggableObjectsCapabilities', {$scope: scope});

            scope.QuickView = QuickView;

            /* Auto save when summary is modified */
            scope.$on("objectSummaryEdited", function(){
                apiPrefix.save(scope.notebook).success(function(data){
                    ActivityIndicator.success("Notebook saved");
                })
                .error(setErrorInScope.bind(scope));
            });

            const refreshTimeline = function(projectKey) {
                DataikuAPI.timelines.getForObject($stateParams.projectKey, scope.objectType, scope.notebook.id).success(function(data) {
                    scope.objectTimeline = data;
                }).error(setErrorInScope.bind(scope));
            };

            scope.saveCustomFields = function(newCustomFields) {
                WT1.event('custom-fields-save', {objectType: scope.objectType});
                let oldCustomFields = angular.copy(scope.notebook.customFields);
                scope.notebook.customFields = newCustomFields;
                return apiPrefix.save(scope.notebook).success(function(data) {
                        refreshTimeline();
                        $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), scope.notebook.customFields);
                    }).error(function(a, b, c) {
                        scope.notebook.customFields = oldCustomFields;
                        setErrorInScope.bind(scope)(a, b,c);
                    });
            };

            scope.editCustomFields = function() {
                if (!scope.notebook) {
                    return;
                }
                let modalScope = angular.extend(scope, {objectType: scope.objectType, objectName: scope.notebook.name, objectCustomFields: scope.notebook.customFields});
                CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(customFields) {
                    scope.saveCustomFields(customFields);
                });
            };

            scope.refreshData = function(){
                apiPrefix.getSummary(ActiveProjectKey.get(), scope.selection.selectedObject.id).success(function(data){
                    scope.notebookData = {notebook: data.object, timeline: data.timeline, interest: data.interest};
                    scope.notebook = data.object;
                }).error(setErrorInScope.bind(scope));
            }

            scope.$watch("selection.selectedObject", function(nv, ov) {
                if (!nv) return;
                scope.notebookData = {notebook: nv, timeline: scope.notebookTimeline};
                if(scope.selection.confirmedItem != scope.selection.selectedObject) {
                    scope.notebookTimeline = null;
                }
            });

            scope.$watch("selection.confirmedItem", function(nv, ov) {
                if (!nv) return;
                if (ov && nv.id !== ov.id || !scope.notebook) {
                    scope.refreshData();
                }
            });

            scope.$watch('notebook', function(nv, ov){
                if (ov && nv && nv.name == ov.name && !angular.equals(nv.tags, ov.tags)) {
                    apiPrefix.save(scope.notebook).success(function(data){
                        ActivityIndicator.success("Notebook saved");
                    }).error(setErrorInScope.bind(scope));
                }
            }, true);

            scope.getNotebookIcon = function(item) {
                return NotebooksUtils.getNotebookIcon(item);
            };

            scope.updateUserInterests = () => {
                DataikuAPI.interests.getForObject($rootScope.appConfig.login, scope.objectType, ActiveProjectKey.get(), scope.selection.selectedObject.id)
                    .success(function(data) {
                        scope.selection.selectedObject.interest = data;
                        scope.notebookData.interest = data;
                    }).error(setErrorInScope.bind(scope));
            };
            const interestsListener = $rootScope.$on('userInterestsUpdated', scope.updateUserInterests);
            scope.$on("$destroy", interestsListener);
        }
    }
});

app.controller("sqlSearchNotebookPageRightColumnActions", async function($controller, $scope, $stateParams, ActiveProjectKey, Assert, NotebooksUtils, HistoryService) {
    Assert.inScope($scope, "objectType");
    Assert.inScope($scope, "notebookParams");

    const apiPrefix = NotebooksUtils.getSqlSearchNotebooksApiPrefix($scope.objectType);
    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.selection = {
        selectedObject: {
            projectKey: ActiveProjectKey.get(),
            name: $scope.notebookParams.name,
            id: $scope.notebookParams.id,
            nodeType: $scope.objectType,
            interest: $scope.objectInterest,
            language: $scope.notebookParams.language
        },
        confirmedItem: {
            projectKey: ActiveProjectKey.get(),
            name: $stateParams.name,
            id: $scope.notebookParams.id,
            nodeType: $scope.objectType,
            interest: $scope.objectInterest,
            language: $scope.notebookParams.language
        }
    };

    $scope.renameObjectAndSave = function(newName) {
        $scope.notebook.name = newName;
        return apiPrefix.save($scope.notebook).success(() => {
            $scope.refreshTimeline();
            HistoryService.notifyRenamed({
               type: $scope.objectType,
               id: $scope.notebookParams.id,
               projectKey: $scope.notebookParams.projectKey
            }, $scope.notebookParams.name);
        });
    };
});


})();
