(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/maps/maps_base.js
    app.factory('_MapCharts', function(Assert, Logger, ChartLegendUtils, localStorageService, $state, $stateParams, ChartZoomControlAdapter, CHART_ZOOM_CONTROL_TYPES, BuiltinMapBackgrounds) {
        function getLSKey(elt) {
            let lsKey = 'DSSMapPosition.';
            const mapScope = angular.element(elt).scope();
            if ($state.current.name.startsWith('projects.project.datasets.dataset.visualize')) {
                lsKey += 'explore.' + $stateParams.projectKey + '.' + $stateParams.datasetName + '.' + mapScope.currentChart.index;
            } else if ($state.current.name.startsWith('projects.project.analyses.analysis')) {
                lsKey += 'analysis.' + $stateParams.projectKey + '.' + $stateParams.analysisId + '.' + mapScope.currentChart.index;
            } else if ($state.current.name.startsWith('projects.project.dashboards.insights.insight')) {
                lsKey += 'insight.' + $stateParams.projectKey + '.' + $stateParams.insightId;
            } else if ($state.current.name.startsWith('projects.project.dashboards.dashboard')) {
                lsKey += 'insight.' + $stateParams.projectKey + '.' + mapScope.insight.id;
            } else {
                lsKey += 'other.' + $state.current.name;
            }
            return lsKey;
        }

        const svc = {
            createMapIfNeeded: function(elt, chartSpecific, chartDef) {
                let map = elt.data('leaflet-map');

                if (!map) {
                    Logger.info('Creating map');
                    map = L.map(elt[0], { zoomControl: false }).setView([20, 0], 2);
                    chartSpecific.leafletMap = map;
                    elt.data('leaflet-map', map);
                    map.$justCreated = true;
                } else {
                    map.$justCreated = false;
                }
                const prevLayerId = elt.data('leaflet-tile-layer-id');

                let layerId = 'cartodb-positron';
                if (chartDef.mapOptions && chartDef.mapOptions.tilesLayer) {
                    layerId = chartDef.mapOptions.tilesLayer;
                }
                let foundLayer = BuiltinMapBackgrounds.getBackgrounds().find(b => b.id === layerId);
                if (!foundLayer) {
                    layerId = 'cartodb-positron';
                    foundLayer = BuiltinMapBackgrounds.getBackgrounds().find(b => b.id === layerId);
                }

                Logger.info('New layer', layerId, 'Previous layer', prevLayerId);

                if (prevLayerId && layerId != prevLayerId) {
                    Logger.info('Removing previous layer');
                    const prevLayer = elt.data('leaflet-tile-layer');
                    map.removeLayer(prevLayer);
                }
                if (!prevLayerId || layerId != prevLayerId) {
                    Logger.info('Adding layer');
                    Assert.trueish(foundLayer, 'layer not found');
                    const layer = foundLayer.getTileLayer();
                    map.addLayer(layer);
                    elt.data('leaflet-tile-layer-id', layerId);
                    elt.data('leaflet-tile-layer', layer);
                }
                svc.configureZoom(map, chartDef, elt);

                return map;
            },

            repositionMap: function(map, elt, data) {
                if (!elt.data('leaflet-map-positioned') && data.minLat > -90.0) {
                    elt.data('leaflet-map-positioned', 1);
                    const previousPosition = localStorageService.get(getLSKey(elt));
                    if (previousPosition) {
                        map.setView(previousPosition.center, previousPosition.zoom);
                    } else {
                        map.fitBounds([[data.minLat, data.minLon], [data.maxLat, data.maxLon]], { padding: [10, 10] });
                    }
                }
            },

            getOrCreateMapContainer: function($container) {
                let elt = $container.find('.map-container');
                if (!elt.length) {
                    // make sure to cleanup any mainzone not related to map
                    $container.find('.mainzone').remove();
                    elt = $('<div class="map-container mainzone w100 h100" data-qa-screenshot-scope__chart>').appendTo($container);
                }
                return elt;
            },

            adjustLegendPlacement: function(chartDef, $container) {
                const legendZone = $container.find('.legend-zone');
                if (legendZone) {
                    legendZone.css('z-index', 1);
                }

                const margins = { top: 5, bottom: 5, left: 5, right: 5 };

                // Avoid collision with leafleft controls
                switch (chartDef.legendPlacement) {
                    case 'INNER_TOP_LEFT':
                        margins.top = 10;
                        margins.left = 45;
                        break;
                    case 'INNER_BOTTOM_RIGHT':
                        margins.bottom = 20;
                        break;
                }

                return ChartLegendUtils.adjustLegendPlacement(chartDef, $container, margins);
            },

            configureZoom: function(map, chartDef, elt) {
                chartDef.$zoomControlInstanceId = ChartZoomControlAdapter.create(CHART_ZOOM_CONTROL_TYPES.LEAFLET, map, () => {
                    localStorageService.set(getLSKey(elt), { center: map.getCenter(), zoom: map.getZoom() });
                });
            }
        };

        return svc;
    });

})();
