import asyncio
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass
from typing import Any, AsyncGenerator, Generator, Generic, Iterable, Optional, overload, TypeVar, Union

T = TypeVar("T")


@dataclass
class _Item(Generic[T]):
    result: T


@dataclass
class _ItemException:
    error: BaseException


SendType = TypeVar("SendType")
YieldType = TypeVar("YieldType")


@overload
def _wrap_in_generator(sync_generator_or_iterable: Generator[YieldType, SendType, Any]) -> Generator[YieldType, SendType, Any]:
    ...
@overload
def _wrap_in_generator(sync_generator_or_iterable: Iterable[YieldType]) -> Generator[YieldType, None, None]:
    ...
def _wrap_in_generator(sync_generator_or_iterable):
    """
    Wraps any synchronous iterable into a generator delegating to it.
    If the wrapped iterable is already a generator, the returned generator will behave exactly the same as the wrapped generator.
    """
    value = yield from sync_generator_or_iterable
    return value


@overload
def generate_in_executor(sync_generator_or_iterable: Generator[YieldType, SendType, Any], executor: Optional[ThreadPoolExecutor] = None) -> AsyncGenerator[YieldType, SendType]:
    ...
@overload
def generate_in_executor(sync_generator_or_iterable: Iterable[YieldType], executor: Optional[ThreadPoolExecutor] = None) -> AsyncGenerator[YieldType, None]:
    ...
def generate_in_executor(sync_generator_or_iterable, executor=None):
    """
    Creates an asynchronous generator wrapping a synchronous generator/iterator/iterable.
    If the wrapped iterable is a generator, bi-directional communication with the returned async generator via asend/athrow/aclose is forwarded to the wrapped sync generator.
    """
    return _AsyncGeneratorWrapper(sync_generator_or_iterable, executor).__aiter__()


class _AsyncGeneratorWrapper(Generic[YieldType, SendType]):
    """
    Creates an asynchronous generator wrapping a synchronous generator/iterator/iterable.
    It also works on synchronous iterators or even iterables, by wrapping them into a synchronous generator first.
    If the wrapped iterable is a generator, bi-directional communication with the returned async generator via asend/athrow/aclose is forwarded to the wrapped sync generator.
    The implementation relies on running each iteration of the wrapped synchronous generator in a separate thread.

    executor: optional ThreadPoolExecutor in which the threads iterating the synchronous generator will run
    If executor=None, it will rely by default on the default ThreadPoolExecutor set on the event loop.
    """

    @overload
    def __init__(self, sync_generator_or_iterable: Generator[YieldType, SendType, Any], executor: Optional[ThreadPoolExecutor] = None):
        ...
    @overload
    def __init__(self, sync_generator_or_iterable: Iterable[YieldType], executor: Optional[ThreadPoolExecutor] = None):
        ...
    def __init__(self, sync_generator_or_iterable, executor: Optional[ThreadPoolExecutor] = None):
        self.sync_generator: Generator[YieldType, SendType, Any] = _wrap_in_generator(sync_generator_or_iterable)
        self.executor = executor

        # lifecycle flags for testing purposes
        self.started = False
        self.stopped = False

    def _producer(self, received: Union[None, _Item[SendType], _ItemException]) -> Union[_Item[YieldType], _ItemException]:
        try:
            if received is None:
                # start the generator
                result = next(self.sync_generator)
            elif isinstance(received, _ItemException):
                # an exception was raised by the consumer of the async generator
                if isinstance(received.error, GeneratorExit):
                    # early exit requested by the consumer (break / exception raised in async for loop / task cancelled)
                    self.sync_generator.close()
                    # forward GeneratorExit back outwards to the async generator so it can raise it
                    return _ItemException(received.error)
                else:
                    # error explicitely sent from the consumer with athrow
                    # the sync generator may either
                    # - catch it and yield a value
                    # - catch it and exit, ie raise StopIteration
                    # - catch it and raise another exception
                    # - do not catch it: the exception should propagate to the caller
                    result = self.sync_generator.throw(received.error)
            else:
                result = self.sync_generator.send(received.result)

            # forward produced data outwards to the async generator so it can yield it
            return _Item(result)
        except BaseException as e:
            # an exception was raised from within the sync generator, forward it outwards to the async generator and exit
            #  - StopIteration: normal termination, the synchronous generator is exhausted
            #  - or another exception was raised due to an error
            return _ItemException(e)

    async def __aiter__(self) -> AsyncGenerator[YieldType, SendType]:
        self.started = True
        try:
            # start synchronous generator in producer thread
            received: Union[None, _Item[SendType], _ItemException] = None

            while True:
                # consumes values produced by synchronous generator in the producer thread
                produced = await asyncio.get_event_loop().run_in_executor(self.executor, self._producer, received)

                if isinstance(produced, _ItemException):
                    # an exception was raised from within the sync generator, producer thread already exited on its own
                    #  - StopIteration: normal termination, the synchronous generator is exhausted --> terminate normally the async generator with StopAsyncIteration
                    #  - or another exception was raised due to an error --> terminate the async generator abnormally with same error
                    if isinstance(produced.error, StopIteration):
                        return  # raise StopAsyncIteration
                    raise produced.error

                try:
                    # yield the result and forward the received data inwards to the sync generator
                    _received = yield produced.result
                    received = _Item(_received)
                except BaseException as e:
                    # an exception was raised by the consumer of the async generator
                    # forward the exception inwards to the sync generator
                    received = _ItemException(e)
        finally:
            self.stopped = True
