#
# nginx configuration file generation
#
import os
import base
import argparse


design_template = '''
# This file is automatically generated. Manual edits will be lost upon DSS updates.

error_log stderr;
pid "{{DIP_HOME}}/run/nginx/nginx.pid";
daemon off;
working_directory "{{DIP_HOME}}/run/nginx";

events {
    worker_connections 10000;
}

http {
    # make these variables have a default value, they're used in some dynamically added locations
    map $auth_studiourl $auth_studiourl {}
    map $auth_accessforbiddenreason $auth_accessforbiddenreason {}

    gzip on;
    gzip_types text/javascript application/json text/css image/svg+xml;
    client_max_body_size 0;
    large_client_header_buffers 8 256K;
    server_tokens off;

    types {
        text/html       html htm shtml;
        text/css        css;
        text/javascript js;
        text/css        less;
        audio/mpeg      mp3;
        image/svg+xml   svg;
        application/pdf pdf;
    }
    # define the format so to never expose remote_user which may contains API keys
    log_format anonymous '$remote_addr - - [$time_local] ' '"$request" $status $body_bytes_sent ' '"$http_referer" "$http_user_agent"';
    
    access_log "{{DIP_HOME}}/run/nginx/access.log" anonymous;
    client_body_temp_path "{{DIP_HOME}}/run/nginx";
    proxy_temp_path "{{DIP_HOME}}/run/nginx";
    # Define these even if we don't use them to avoid permission issues
    fastcgi_temp_path "{{DIP_HOME}}/run/nginx";
    scgi_temp_path "{{DIP_HOME}}/run/nginx";
    uwsgi_temp_path "{{DIP_HOME}}/run/nginx";

    proxy_http_version 1.1;
    proxy_next_upstream off; # Don't retry
    proxy_read_timeout 3600; # We have long queries
    proxy_set_header Host $http_host;
    proxy_set_header Upgrade $http_upgrade;
    proxy_set_header Connection "upgrade";
    proxy_set_header x-dss-nginx-client-ip $remote_addr;
    proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
    proxy_pass_header X-Accel-Buffering;

    server {
{{LISTEN_DIRECTIVES}}
        root "{{DKUINSTALLDIR}}/frontend";

        {{INCLUDES}}

        {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}

        location ^~ /dip/api/webapps/view {
            # header used for standard webapp redirection after login
            proxy_set_header "X-DKU-Redirect-To" "$request_uri";
            proxy_set_header Host $http_host;
            proxy_set_header Upgrade $http_upgrade;
            proxy_set_header Connection "upgrade";
            proxy_set_header x-dss-nginx-client-ip $remote_addr;
            proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
            proxy_pass http://127.0.0.1:{{DKU_BACKEND_PORT}}/dip/api/webapps/view;
            {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}
            {{API_SECURITY_HEADERS_DIRECTIVES}}
        }
        location ^~ /dip/ {
            proxy_pass http://127.0.0.1:{{DKU_BACKEND_PORT}}/dip/;
            {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}
            {{API_SECURITY_HEADERS_DIRECTIVES}}
        }
        location ^~ /public/packages/ {
            alias "{{DKUINSTALLDIR}}/public/packages/";
        }
        location /public/packages/dataiku-scoring-libs.jar {
            proxy_pass http://127.0.0.1:{{DKU_BACKEND_PORT}}/dip/publicapi/resources/scoring-lib-jar/;
        }
        location ^~ /public/api/ {
            proxy_pass http://127.0.0.1:{{DKU_BACKEND_PORT}}/dip/publicapi/;
        }
        location ^~ /jupyter/ {
            proxy_pass http://127.0.0.1:{{DKU_IPYTHON_PORT}}/jupyter/;
            proxy_redirect off;
            proxy_read_timeout 600;
            proxy_send_timeout 600;
        }

        {{EVENT_SERVER_LOCATION}}

        location ~ ^/plugins/([^/]+)/resource/(.*)$ {
            root /;
            try_files {{DIP_HOME}}/plugins/installed/$1/resource/$2 {{DIP_HOME}}/plugins/dev/$1/resource/$2
            {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}
            add_header Cache-Control private;
            expires 0;
            location ~ ^/plugins/([^/]+)/resource/(.*\.(html|css|js))$ {
                try_files {{DIP_HOME}}/plugins/installed/$1/resource/$2 {{DIP_HOME}}/plugins/dev/$1/resource/$2
                {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}
                add_header Cache-Control private;
                expires -1;    # Equivalent to cache-control: no-cache
            }
        }

        {{SERVE_RESOURCES_LOCATION}}

        {{STORY_PROXY_PASS}}

        location / {
            rewrite ^/(home|profile|project-list|projects|workspaces|wikis|login|logged-out|sso-error|sso|admin|apps|docportal|plugins-explore|plugins|libedition|feature-store|search-dss-items|data-lineage|data-catalog|catalog|inbox|automation|deployer|api-deployer|project-deployer|unified-monitoring|alation-open|external-table|meanings|licensing|data-quality|oauth2-callback).*$ /index.html break;
            index index.html;
            add_header Cache-Control "private, no-cache, max-age=0";
            expires 0;
            {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}
            {{STATIC_SECURITY_HEADERS_DIRECTIVES}}
            location ~ ^/themes/builtin/([^/]+)/(.*)$ {
                alias "{{DKUINSTALLDIR}}/resources/themes/$1/$2";
            }
            location ~ ^/themes/user/([^/]+)/(.*)$ {
                alias "{{DIP_HOME}}/resources/themes/$1/$2";
            }
        }
        location ~ ^/ngxdist/(.*)$ {
            {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}
            {{STATIC_SECURITY_HEADERS_DIRECTIVES}}
            add_header Cache-Control "private, no-cache, max-age=0";
            expires -1;    # Equivalent to cache-control: no-cache
            alias "{{DKUINSTALLDIR}}/frontend/ngxdist/$1";
        }
        location ~ ^/.*\.(html|css|js)$ {
            rewrite ^/(home|profile|project-list|projects|workspaces|wikis|login|logged-out|sso-error|sso|admin|apps|docportal|plugins-explore|plugins|libedition|feature-store|search-dss-items|data-lineage|data-catalog|inbox|automation|deployer|api-deployer|project-deployer|unified-monitoring|alation-open|external-table|meanings|licensing|data-quality|oauth2-callback).*$ /index.html break;
            index index.html;
            {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}
            {{STATIC_SECURITY_HEADERS_DIRECTIVES}}
            add_header Cache-Control "private, no-cache, max-age=0";
            expires -1;    # Equivalent to cache-control: no-cache
        }
        location ^~ /local/static/ {
            alias "{{DIP_HOME}}/local/static/";
        }
    }
}
'''
design_tmpl_vars = [
    "DKUINSTALLDIR", "DIP_HOME", "DKU_BACKEND_PORT", "DKU_IPYTHON_PORT", "DKU_EVENTSERVER_PORT", "DKU_STORIES_PORT"
]


# the /ping and /invocations location directives are for deployment on a
# SageMaker endpoint
apinode_template = '''
# This file is automatically generated. Manual edits will be lost upon DSS updates.

error_log stderr;
pid "{{DIP_HOME}}/run/nginx/nginx.pid";
daemon off;
working_directory "{{DIP_HOME}}/run/nginx";

events {
    worker_connections 768;
}

http {
    gzip on;
    gzip_types application/json;
    client_max_body_size 10000m;
    server_tokens off;

    # define the format so to never expose remote_user which may contains API keys
    log_format anonymous '$remote_addr - - [$time_local] ' '"$request" $status $body_bytes_sent ' '"$http_referer" "$http_user_agent"';
    
    access_log "{{DIP_HOME}}/run/nginx/access.log" anonymous;
    client_body_temp_path "{{DIP_HOME}}/run/nginx";
    proxy_temp_path "{{DIP_HOME}}/run/nginx";
    # Define these even if we don't use them to avoid permission issues
    fastcgi_temp_path "{{DIP_HOME}}/run/nginx";
    scgi_temp_path "{{DIP_HOME}}/run/nginx";
    uwsgi_temp_path "{{DIP_HOME}}/run/nginx";

    proxy_http_version 1.1;
    proxy_next_upstream off; # Don't retry
    proxy_read_timeout 3600; # We have long queries
    proxy_set_header X-Real-IP $remote_addr;
    proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
    proxy_pass_header X-Accel-Buffering;

    server {
{{LISTEN_DIRECTIVES}}

        # Vertex AI has troubles handling a gzipped reponse. See sc-143149.
        location /public/api/v1/vertex-ai/ {
            gzip off;
            proxy_pass http://127.0.0.1:{{DKU_APIMAIN_PORT}}/public/api/v1/vertex-ai/;
        }

        location /public/ {
            proxy_pass http://127.0.0.1:{{DKU_APIMAIN_PORT}}/public/;
        }

        location /admin/ {
            proxy_pass http://127.0.0.1:{{DKU_APIMAIN_PORT}}/admin/;
        }

        location /monitoring/ {
            proxy_pass http://127.0.0.1:{{DKU_APIMAIN_PORT}}/monitoring/;
        }

        location /isAlive {
            proxy_pass http://127.0.0.1:{{DKU_APIMAIN_PORT}}/isAlive/;
        }

        location /ping {
            proxy_pass http://127.0.0.1:{{DKU_APIMAIN_PORT}}/monitoring/api/isFullyAlive;
        }

        location /invocations {
            proxy_pass http://127.0.0.1:{{DKU_APIMAIN_PORT}}/public/api/v1/sagemaker/invocations;
        }
    }
}
'''
apinode_tmpl_vars = [
    "DKUINSTALLDIR", "DIP_HOME", "DKU_APIMAIN_PORT"
]


fm_template = '''
# This file is automatically generated. Manual edits will be lost upon FM updates.

error_log stderr;
pid "{{DIP_HOME}}/run/nginx/nginx.pid";
daemon off;
working_directory "{{DIP_HOME}}/run/nginx";

events {
    worker_connections 768;
}

http {
    gzip on;
    gzip_types text/javascript application/json text/css image/svg+xml;
    client_max_body_size 0;
    server_tokens off;
    large_client_header_buffers 8 64k;

    types {
        text/html       html htm shtml;
        text/css        css;
        text/javascript js;
        text/css        less;
        audio/mpeg      mp3;
        image/svg+xml   svg;
        application/pdf pdf;
    }

    # define the format so to never expose remote_user which may contains API keys
    log_format anonymous '$remote_addr - - [$time_local] ' '"$request" $status $body_bytes_sent ' '"$http_referer" "$http_user_agent"';
    
    access_log "{{DIP_HOME}}/run/nginx/access.log" anonymous;
    client_body_temp_path "{{DIP_HOME}}/run/nginx";
    proxy_temp_path "{{DIP_HOME}}/run/nginx";
    # Define these even if we don't use them to avoid permission issues
    fastcgi_temp_path "{{DIP_HOME}}/run/nginx";
    scgi_temp_path "{{DIP_HOME}}/run/nginx";
    uwsgi_temp_path "{{DIP_HOME}}/run/nginx";

    proxy_http_version 1.1;
    proxy_next_upstream off; # Don't retry
    proxy_read_timeout 3600; # We have long queries
    proxy_set_header X-Real-IP $remote_addr;
    proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
    proxy_pass_header X-Accel-Buffering;

    server {
{{LISTEN_DIRECTIVES}}
        root "{{DKUINSTALLDIR}}/fm-frontend";

        location /api/ {
            proxy_pass http://127.0.0.1:{{DKU_FMMAIN_PORT}}/api/;
        }

        location / {
            rewrite ^/(home|instances|load-balancers|tenant|virtual-networks|instance-templates|accounts|users|login|logged-out|sso|security|blueprints|maintenance|licensing|alerting).*$ /index.html break;
            index index.html;
            add_header Cache-Control "private, no-cache, max-age=0";
            expires 0;
            {{STATIC_SECURITY_HEADERS_DIRECTIVES}}
        }

        location ~ ^/(static|templates)/(.*)$ {
            {{STATIC_SECURITY_HEADERS_DIRECTIVES}}
            add_header Cache-Control "private, no-cache, max-age=0";
            expires -1;    # Equivalent to cache-control: no-cache
            alias "{{DKUINSTALLDIR}}/fm-frontend/$1/$2";
        }

        location ~ ^/.*\.(html|css|js)$ {
            rewrite ^/(home|instances|load-balancers|tenant|virtual-networks|instance-templates|accounts|users|login|logged-out|sso|security|blueprints|maintenance|licensing|alerting).*$ /index.html break;
            index index.html;
            {{STATIC_SECURITY_HEADERS_DIRECTIVES}}
            add_header Cache-Control "private, no-cache, max-age=0";
            expires -1;    # Equivalent to cache-control: no-cache
        }
    }
}
'''
fm_tmpl_vars = [
    "DKUINSTALLDIR", "DIP_HOME", "DKU_FMMAIN_PORT"
]



govern_template = '''
# This file is automatically generated. Manual edits will be lost upon Dataiku Govern updates.

error_log stderr;
pid "{{DIP_HOME}}/run/nginx/nginx.pid";
daemon off;
working_directory "{{DIP_HOME}}/run/nginx";

events {
    worker_connections 10000;
}

http {
    gzip on;
    gzip_types text/javascript application/json text/css image/svg+xml;
    client_max_body_size 0;
    large_client_header_buffers 8 64k;
    server_tokens off;

    types {
        text/html       html htm shtml;
        text/css        css;
        text/javascript js;
        text/css        less;
        audio/mpeg      mp3;
        image/svg+xml   svg;
        application/pdf pdf;
    }

    # define the format so to never expose remote_user which may contains API keys
    log_format anonymous '$remote_addr - - [$time_local] ' '"$request" $status $body_bytes_sent ' '"$http_referer" "$http_user_agent"';
    
    access_log "{{DIP_HOME}}/run/nginx/access.log" anonymous;
    client_body_temp_path "{{DIP_HOME}}/run/nginx";
    proxy_temp_path "{{DIP_HOME}}/run/nginx";
    # Define these even if we don't use them to avoid permission issues
    fastcgi_temp_path "{{DIP_HOME}}/run/nginx";
    scgi_temp_path "{{DIP_HOME}}/run/nginx";
    uwsgi_temp_path "{{DIP_HOME}}/run/nginx";

    proxy_http_version 1.1;
    proxy_next_upstream off; # Don't retry
    proxy_read_timeout 3600; # We have long queries
    proxy_set_header Host $http_host;
    proxy_set_header Upgrade $http_upgrade;
    proxy_set_header Connection "upgrade";
    proxy_set_header x-dss-nginx-client-ip $remote_addr;
    proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
    proxy_pass_header X-Accel-Buffering;

    server {
{{LISTEN_DIRECTIVES}}
        root "{{DKUINSTALLDIR}}/govern-frontend";

        {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}

        location ^~ /dip/ {
            proxy_pass http://127.0.0.1:{{DKU_GOVERNSERVER_PORT}}/dip/;
            {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}
            {{API_SECURITY_HEADERS_DIRECTIVES}}
        }

        location ^~ /public/api/ {
            proxy_pass http://127.0.0.1:{{DKU_GOVERNSERVER_PORT}}/dip/publicapi/;
        }

        location / {
            rewrite ^/(login|logged-out|sso-error|artifact|artifacts|pages|blueprint-designer|admin|roles-and-permissions|custom-page-designer|governable-items|model-registry|gen-ai-registry|bundle-registry|business-initiatives|governed-projects|timeline|governance-settings|action-designer).*$ /index.html break;
            index index.html;
            add_header Cache-Control "private, no-cache, max-age=0";
            expires 0;
            {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}
            {{STATIC_SECURITY_HEADERS_DIRECTIVES}}
        }
        location ~ ^/.*\.(html|css|js)$ {
            rewrite ^/(login|logged-out|sso-error|artifact|artifacts|pages|blueprint-designer|admin|roles-and-permissions|custom-page-designer|governable-items|model-registry|gen-ai-registry|bundle-registry|business-initiatives|governed-projects|timeline|governance-settings|action-designer).*$ /index.html break;
            index index.html;
            {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}
            {{STATIC_SECURITY_HEADERS_DIRECTIVES}}
            add_header Cache-Control "private, no-cache, max-age=0";
            expires -1;    # Equivalent to cache-control: no-cache
        }
    }
}
'''
govern_tmpl_vars = [
    "DKUINSTALLDIR", "DIP_HOME", "DKU_GOVERNSERVER_PORT"
]

#
# Quote a nginx string value to be included into a double-quoted string param
#
def quote(s):
    return s.replace('\\', '\\\\').replace('"', '\\"')

#
# Generate the listen directives
#
def nginx_listen_directives(config, default_ecdh_curves):
    # Assumes DKU_NGINX_PORT == DKU_BASE_PORT
    main_port = config.getServerPort()
    ssl = config.getBoolOption('server', 'ssl', False)
    ipv6 = config.getBoolOption('server', 'ipv6', False)
    listen_on_ipv4 = config.getOption('server','listen_on')
    listen_on_ipv6 = config.getOption('server','listen_on_ipv6','[::]')

    additional_ports = config.getOption("server", "additional_ports")
    if additional_ports is not None:
        additional_ports = additional_ports.split(",")
    else:
        additional_ports = []

    all_ports = [main_port] + additional_ports

    result = ""
    for port in all_ports:
        result += '        listen %s%s%s;\n' % ( '{}:'.format(listen_on_ipv4) if listen_on_ipv4 else '', port, ' ssl' if ssl else '')
        if ipv6:
            result += '        listen %s:%s%s;\n' % (listen_on_ipv6, port, ' ssl' if ssl else '')

    if ssl:
        ssl_certificate = config.getOption('server', 'ssl_certificate')
        ssl_ecdh_curve = config.getOption('server', 'ssl_ecdh_curve', None)
        if ssl_ecdh_curve is None and default_ecdh_curves is not None:
            ssl_ecdh_curve = default_ecdh_curves
        if not ssl_certificate:
            raise Exception("missing mandatory ssl_certificate option in config file: %s" % config.filename)
        ssl_certificate_key = config.getOption('server', 'ssl_certificate_key')
        if not ssl_certificate_key:
            raise Exception("missing mandatory ssl_certificate_key option in config file: %s" % config.filename)
        result += '        ssl_certificate "%s";\n' % quote(ssl_certificate)
        result += '        ssl_certificate_key "%s";\n' % quote(ssl_certificate_key)

        ssl_client_certificate = config.getOption('server','ssl_client_certificate')
        if ssl_client_certificate:
            result += '        ssl_client_certificate "%s";\n' % quote(ssl_client_certificate)
            result += '        ssl_verify_client on;\n'
            ssl_verify_depth = config.getIntOption('server','ssl_verify_depth')
            if ssl_verify_depth is not None:
                result += '        ssl_verify_depth %u;\n' % ssl_verify_depth

        ssl_ciphers = config.getOption('server', 'ssl_ciphers', 'default')
        if ssl_ciphers == 'recommended':
            result += '        ssl_protocols TLSv1.2 TLSv1.3;\n'
            result += '        ssl_ciphers ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-ECDSA-CHACHA20-POLY1305:ECDHE-RSA-CHACHA20-POLY1305;\n'
            if ssl_ecdh_curve:
                result += '        ssl_ecdh_curve %s;\n' % ssl_ecdh_curve
            result += '        ssl_session_timeout 1d;\n'
            result += '        ssl_session_cache shared:DSS:5m;\n'
            result += '        ssl_session_tickets off;\n'
        elif ssl_ciphers == 'tls13only':
            result += '        ssl_protocols TLSv1.3;\n'
            result += '        ssl_conf_command Ciphersuites TLS_AES_128_GCM_SHA256:TLS_AES_256_GCM_SHA384:TLS_CHACHA20_POLY1305_SHA256;\n'
            if ssl_ecdh_curve:
                result += '        ssl_ecdh_curve %s;\n' % ssl_ecdh_curve
            result += '        ssl_prefer_server_ciphers off;\n'
            result += '        ssl_session_timeout 1d;\n'
            result += '        ssl_session_cache shared:DSS:5m;\n'
            result += '        ssl_session_tickets off;\n'
            pass
        elif ssl_ciphers == 'default':
            pass
        else:
            raise Exception("invalid value for ssl_ciphers option in config file: %s" % config.filename)

        # HTTP to HTTPS automatic redirection
        nodeType = config.getNodeType()
        if nodeType == 'design' or nodeType == "automation":
            result += '        error_page 497 https://$host:$server_port$request_uri;\n'

    return result

def nginx_eventserver_location(config):
    if config.getBoolOption("eventserver", "enabled", False):
        return """
        location /events {
            proxy_pass http://127.0.0.1:{{DKU_EVENTSERVER_PORT}}/events;
        }"""
    else:
        return ""

def nginx_include_directives():
    res = 'include "{{DIP_HOME}}/install-support/python-backends.d/*.conf"; #legacy location\n'
    if base.is_os_windows():
        res += '        # include "{{DIP_HOME}}/install-support/backends.d/*/*.conf";' + os.linesep
        res += '        # On Windows, /*/*.conf is not supported, so we are forcing legacy mode.'
    else:
        res += '        include "{{DIP_HOME}}/install-support/backends.d/*/*.conf";'
    return res

def nginx_serve_resources(serve=False):
    if serve:
        return """
        location = /local/projects/auth {
            proxy_pass http://127.0.0.1:{{DKU_BACKEND_PORT}}/dip/api/projects/$dku_project_key/check-access;
            proxy_pass_request_body off;
            proxy_set_header Content-Length "";
            proxy_set_header Sec-WebSocket-Protocol "";
            proxy_set_header X-Original-URI $request_uri;
        }

        location @error_location_local_project_static {
            try_files /local-projects-static-error-401.html /local-projects-static-error-401.html;
            sub_filter "http://localhost:10000" $auth_studiourl;
            sub_filter_once on;
        }
        location @reject_location_local_project_static {
            try_files /local-projects-static-error-403.html /local-projects-static-error-403.html;
            sub_filter "http://localhost:10000" $auth_studiourl;
            sub_filter "ERROR_MESSAGE_PLACEHOLDER" $auth_accessforbiddenreason;
            sub_filter_once on;
        }

        location ~ ^/local/projects/([^/]+)/static/(.*)$ {
            alias "{{DIP_HOME}}/config/projects/$1/lib/static/$2";
            {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}
            add_header Cache-Control private;
            expires -1;    # Equivalent to cache-control: no-cache
            set $dku_project_key $1; # because you can't have variables in the auth_request directive
            auth_request "/local/projects/auth";
            auth_request_set $auth_studiourl $upstream_http_x_dku_studiourl;
            auth_request_set $auth_accessforbiddenreason $upstream_http_x_dku_accessforbiddenreason;
            error_page 502 /local-projects-static-error-502.html;
            error_page 401 @error_location_local_project_static;
            error_page 403 @reject_location_local_project_static;
        }

        location ~ ^/local/projects/([^/]+)/resources/(.*)$ {
            alias "{{DIP_HOME}}/lib/projects/$1/static/$2";
            {{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}
            add_header Cache-Control private;
            expires -1;    # Equivalent to cache-control: no-cache
            set $dku_project_key $1; # because you can't have variables in the auth_request directive
            auth_request "/local/projects/auth";
            auth_request_set $auth_studiourl $upstream_http_x_dku_studiourl;
            auth_request_set $auth_accessforbiddenreason $upstream_http_x_dku_accessforbiddenreason;
            error_page 502 /local-projects-static-error-502.html;
            error_page 401 @error_location_local_project_static;
            error_page 403 @reject_location_local_project_static;
        }

        location ~ ^/local/projects/([^/]+)/public-resources/(.*)$ {
            alias "{{DIP_HOME}}/lib/projects/$1/local-static/$2";
        }
"""
    else:
        return '\n'
    
def nginx_story_proxy_pass(config):
    ssl = config.getBoolOption('server', 'ssl', False)
    protocol = 'https' if ssl else 'http'

    return """
        location ^~ /stories/ {
            proxy_pass %s://127.0.0.1:{{DKU_STORIES_PORT}}/stories/;
        }
""" % protocol

#
# Generates the nginx configuration corresponding to dipHome
# Assumes the environment variables in env-default have been sourced
#
def generate_nginx_config(dipHome, default_ecdh_curves):
    config = dipHome.get_install_config()
    nodeType = config.getNodeType()
    serve_resources = False
    if nodeType == 'design' or nodeType == "automation":
        template = design_template
        tmpl_vars = design_tmpl_vars
        serve_resources = config.getBoolOption("server", "serve_resources", True)
    elif nodeType == 'api':
        template = apinode_template
        tmpl_vars = apinode_tmpl_vars
    elif nodeType == 'fm':
        template = fm_template
        tmpl_vars = fm_tmpl_vars
    elif nodeType == 'govern':
        template = govern_template
        tmpl_vars = govern_tmpl_vars

    else:
        raise Exception("Node type not supported: %s" % nodeType)

    result = template.replace("{{LISTEN_DIRECTIVES}}", nginx_listen_directives(config,default_ecdh_curves))

    result = result.replace("{{EVENT_SERVER_LOCATION}}", nginx_eventserver_location(config))

    result = result.replace("{{INCLUDES}}", nginx_include_directives())

    result = result.replace("{{SERVE_RESOURCES_LOCATION}}", nginx_serve_resources(serve_resources))

    result = result.replace("{{STORY_PROXY_PASS}}", nginx_story_proxy_pass(config))

    global_sec_headers = ""
    static_sec_headers = ""
    api_sec_headers = ""

    if config.getBoolOption('server', 'ssl', False):
        hsts = config.getOption("server", "hsts-max-age")
        if hsts:
            global_sec_headers += '      add_header Strict-Transport-Security "max-age=%s";\n' % hsts

    xfo = config.getOption("server", "x-frame-options")
    if xfo:
        static_sec_headers += '      add_header "X-Frame-Options" %s;\n' % xfo
    csp = config.getOption("server", "content-security-policy")
    if csp:
        static_sec_headers += '      add_header "Content-Security-Policy" "%s";\n' % csp
    xxss = config.getOption("server", "X-XSS-Protection")
    if xxss:
        static_sec_headers += '      add_header "X-XSS-Protection" "%s";\n' % xxss

    xcto = config.getOption("server", "x-content-type-options")
    if xcto:
        static_sec_headers += '      add_header "X-Content-Type-Options" "%s";\n' % xcto
        api_sec_headers += '      add_header "X-Content-Type-Options" "%s";\n' % xcto

    rp = config.getOption("server", "referrer-policy")
    if rp:
        static_sec_headers += '      add_header "Referrer-Policy" "%s";\n' % rp

    pp = config.getOption("server", "permissions-policy")
    if pp:
        static_sec_headers += '      add_header "Permissions-Policy" "%s";\n' % pp
    else:
        static_sec_headers += '      add_header "Permissions-Policy" "fullscreen=*";\n'

    coep = config.getOption("server", "cross-origin-embedder-policy")
    if coep:
        static_sec_headers += '      add_header "Cross-Origin-Embedder-Policy" "%s";\n' % coep

    coop = config.getOption("server", "cross-origin-opener-policy")
    if coop:
        static_sec_headers += '      add_header "Cross-Origin-Opener-Policy" "%s";\n' % coop

    corp = config.getOption("server", "cross-origin-resource-policy")
    if corp:
        static_sec_headers += '      add_header "Cross-Origin-Resource-Policy" "%s";\n' % corp

    result = result.replace("{{GLOBAL_SECURITY_HEADERS_DIRECTIVES}}", global_sec_headers)
    result = result.replace("{{STATIC_SECURITY_HEADERS_DIRECTIVES}}", static_sec_headers)
    result = result.replace("{{API_SECURITY_HEADERS_DIRECTIVES}}", api_sec_headers)

    for var in tmpl_vars:
        result = result.replace("{{%s}}" % var, quote(os.environ.get(var, "")))
    return result

#
# Prints the nginx configuration on standard output
#
if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Generates nginx configuration")
    parser.add_argument("--ecdh-curves",type=str, default=None, help="Colon separated list of elliptic curves suitable for ECH ciphers")
    args = parser.parse_args()
    print(generate_nginx_config(base.DipHome(os.environ["DIP_HOME"]),args.ecdh_curves))
