(function () {
    "use strict";

    const app = angular.module("dataiku.coloring");

    app.component("conditionalFormattingRuleEditor", {
        bindings: {
            columns: "<",
            rule: "=",
        },
        templateUrl:
            "/static/dataiku/coloring/conditional-formatting/cf-group-editor/cf-rule-editor/cf-rule-editor.component.html",
        controller: function conditionalFormattingRuleEditorController(
            $scope,
            ChartColorUtils,
            ConditionalFormattingEditorService,
            Expressions,
            Operators,
            translate
        ) {
            /*
            Initializations
            */
            const ctrl = this;
            ctrl.$onInit = function () {
                // Read attributes and save them into scope
                const defaults = {
                    columns: [],
                    rule: {},
                };
                for (const [param, value] of Object.entries(defaults)) {
                    if (param in ctrl) {
                        $scope[param] = ctrl[param] || value;
                    } else {
                        $scope[param] = value;
                    }
                }
                $scope.ConditionalFormattingEditorService = ConditionalFormattingEditorService;
                $scope.Operators = Operators;

                $scope.condition = getCondition();
                $scope.showList = false;
                $scope.dateUnits = Expressions.dateUnits;

                $scope.ruleStyles = ConditionalFormattingEditorService.getRuleStyles();
                $scope.styleClasses = $scope.ruleStyles.map(s => s.styleClass);

                const colorPalette = ChartColorUtils.getDiscretePalette("dku_explore");
                $scope.colors = [];
                if (colorPalette && colorPalette.colors) {
                    $scope.colors = colorPalette.colors;
                }

                updateBorderStyle();
            };

            function getCondition() {
                if (
                    $scope.rule.filterDesc.uiData &&
                    $scope.rule.filterDesc.uiData.conditions &&
                    $scope.rule.filterDesc.uiData.conditions.length
                ) {
                    return $scope.rule.filterDesc.uiData.conditions[0];
                }
                // Can happen if user modifies the json directly
                return null;
            }

            /*
            Scope methods
            */
            $scope.getOperator = function (name) {
                return Expressions.getOperators().find(op => op.name === name);
            };

            $scope.operatorHasParam = function (paramName) {
                const operatorName = $scope.condition.operator;
                if (!operatorName) return false;

                const operator = $scope.getOperator(operatorName);
                if (!operator || !operator.params) return false;

                return operator.params.includes(paramName);
            };

            $scope.operatorHasSubParam = function (paramName) {
                const operatorName = $scope.condition.operator;
                if (!operatorName) return false;

                const operator = $scope.getOperator(operatorName);
                if (!operator || !operator.subParams) return false;

                return operator.subParams.includes(paramName);
            };

            $scope.getValueListDescription = function () {
                const nbValues = $scope.condition.items ? $scope.condition.items.length : 0;
                if (nbValues === 0) {
                    return "";
                } else if (nbValues === 1) {
                    return translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.CONDITION.ADD_VALUE.ONE_VALUE", "1 value");
                }
                return (
                    $scope.condition.items.length +
                    translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.CONDITION.ADD_VALUE._VALUES", " values")
                );
            };

            $scope.toggleList = function () {
                $scope.showList = !$scope.showList;
            };

            $scope.addConditionValue = function () {
                if ($scope.condition.items === undefined) {
                    $scope.condition.items = [];
                }
                $scope.condition.items.push({ num: 0 });

                $scope.applyInputChange();
            };
            $scope.removeConditionValue = function (index) {
                if ($scope.condition.items.length !== 0) {
                    $scope.condition.items.splice(index, 1);

                    $scope.applyInputChange();
                }
            };

            /**
             * Colors the border line next to the rule definition
             */
            function updateBorderStyle() {
                const borderColor = ConditionalFormattingEditorService.getBorderColorFromRule($scope.rule.filterDesc);
                $scope.rule.borderStyle = { "border-left-color": borderColor };
            }

            $scope.onColorChange = function () {
                updateBorderStyle();

                $scope.applyChanges();
            };

            $scope.onCustomBackgroundColorChange = function ($event) {
                $scope.rule.filterDesc.uiData.color.styleCustomBackgroundColor = $event;
                updateBorderStyle();

                $scope.applyChanges();
            };

            $scope.onCustomFontColorChange = function ($event) {
                $scope.rule.filterDesc.uiData.color.styleCustomFontColor = $event;
                updateBorderStyle();

                $scope.applyChanges();
            };

            $scope.applyInputChange = function () {
                $scope.rule.description = ConditionalFormattingEditorService.getFilterDescriptionString(
                    $scope.rule.filterDesc
                );

                $scope.applyChanges();
            };
            $scope.applyChanges = function () {
                $scope.$parent.applyChanges();
            };
        },
    });

    app.service("Operators", function (Expressions, translate) {
        var annotations = [];
        var labels = [];

        function init() {
            const operators = Expressions.getOperatorsPrepareRecipe();
            const columnTranslated = translate("FILTER.CONDITION.COLUMN", "column");

            for (let i = 0; i < operators.length; i++) {
                const op = operators[i];
                labels.push({
                    group: op.group,
                    label: op.label,
                    name: op.name,
                });
                annotations.push(op.params.includes("col") ? columnTranslated : "");
            }
        }

        init();

        return {
            getAnnotations: function () {
                return annotations;
            },
            getLabels: function () {
                return labels;
            },
        };
    });
})();
