(function(){
'use strict';


const app = angular.module('dataiku.projects.settings');



app.service('QuickSharingWT1EventsService', function(WT1) {
    const QUICK_SHARING = 'quick-sharing';
    const MANAGED_SHARING = 'managed-sharing';
    const OBJECT_SHARING_MODAL_OPEN = 'object-sharing-modal-open';
    const OBJECT_SHARING_MODAL_CANCEL = 'object-sharing-modal-cancel';
    const OBJECT_SHARING_MODAL_SAVE = 'object-sharing-modal-save';
    const OBJECT_SHARING_ENABLE_SHARING = 'object-sharing-enable-sharing';
    const OBJECT_SHARING_DISABLE_SHARING = 'object-sharing-disable-sharing';
    const OBJECT_SHARING_TYPE_CHANGE = 'object-sharing-type-change';
    const OBJECT_SHARING_OBJECT_SHARE = 'object-sharing-object-share';
    const OBJECT_SHARING_OBJECT_UNSHARE = 'object-sharing-object-unshare';

    function getSharingType(object) {
        return object.quickSharingEnabled ? QUICK_SHARING : MANAGED_SHARING;
    }

    function commonPayload(obj) {
        return {
            sharingType: getSharingType(obj),
            objectType: obj.type,
            objecth: md5(obj.projectKey+"."+obj.localName),
            originProjecth: md5(obj.projectKey),
            ...obj.wt1Context,
        }
    }

    /**
     * Sends WT1 event when a new rule (exposition) is added to an object
     * @param obj the object (minimal properties: 'projectKey', 'quickSharingEnabled', 'type', 'localName', optional: wt1Context)
     * @param targetProject the project we share to
     */
    this.onRuleAdded = function (obj, targetProjectKey) {
        WT1.event(OBJECT_SHARING_OBJECT_SHARE, {
            ...commonPayload(obj),
            targetProjecth: md5(targetProjectKey)
        });
    };

    /**
     * Sends WT1 event when a rule (exposition) is removed from an object
     * @param obj the object (minimal properties: 'projectKey', 'quickSharingEnabled', 'type', 'localName', optional: wt1Context)
     * @param targetProject the project we share to
     */
    this.onRuleRemoved = function(obj, targetProjectKey) {
        WT1.event(OBJECT_SHARING_OBJECT_UNSHARE, {
            ...commonPayload(obj),
            targetProjecth: md5(targetProjectKey)
        });
    }

    /**
     * Sends WT1 event when quick sharing state has been changed
     * @param obj the object (minimal properties: 'projectKey', 'quickSharingEnabled', 'type', 'localName', optional: wt1Context)
     */
    function onSharingTypeChanged(obj) {
        WT1.event(OBJECT_SHARING_TYPE_CHANGE, {
            ...commonPayload(obj),
            numberOfUsage: obj.rules.length
        });
    }

    /**
     * Sends WT1 event when sharing is enabled
     * @param obj the object (minimal properties: 'projectKey', 'quickSharingEnabled', 'type', 'localName', optional: wt1Context)
     */
    this.onSharingEnabled = function(obj) {
        WT1.event(OBJECT_SHARING_ENABLE_SHARING, {
            ...commonPayload(obj)
        });
    }

    /**
     * Sends WT1 event when sharing is disabled
     * @param obj the object (minimal properties: 'projectKey', 'quickSharingEnabled', 'type', 'localName', optional: wt1Context)
     */
    this.onSharingDisabled = function(obj) {
        WT1.event(OBJECT_SHARING_DISABLE_SHARING, {
            ...commonPayload(obj),
            numberOfUsage: obj.rules.length
        });
    };

    /**
     * Sends WT1 for sharing previously unshared objects
     * @param projectKey the projectKey for the project containing the object (source project)
     * @param added array of objects (minimal properties: 'quickSharingEnabled', 'type', 'localName', 'rules')
     *
     * Triggers onSharingEnabled if required, and onRuleAdded for each newly created rule
     */
    this.onObjectsAdded = function(projectKey, added) {
        added.forEach(newObject => {
            if(newObject.rules.length > 0 || newObject.quickSharingEnabled) {
                // if newObject is not quickshared and empty, it's not considered as enabled even though it is present in the list
                this.onSharingEnabled({projectKey, ...newObject});
            }
            newObject.rules.forEach(rule => this.onRuleAdded({projectKey, ...newObject}, rule.targetProject));
        });
    };

    /**
     * Sends WT1 for unsharing previously shared objects
     * @param projectKey the projectKey for the project containing the object (source project)
     * @param added array of objects (minimal properties: 'quickSharingEnabled', 'type', 'localName', 'rules')
     *
     * Triggers onSharingDisabled if required, and onRuleRemoved for each previously existing rules
     */
    this.onObjectsRemoved = function(projectKey, removed) {
        removed.forEach(oldObject => {
            oldObject.rules.forEach(rule => this.onRuleRemoved({projectKey, ...oldObject}, rule.targetProject));
            if(oldObject.rules.length > 0 || oldObject.quickSharingEnabled) {
                // if oldObject was already not quickshared and empty, it was already considered as disabled even if present in the list
                this.onSharingDisabled({projectKey, ...oldObject});
            }
        });
    };

    /**
     * Sends WT1 for a changed object
     * @param projectKey the projectKey for the project containing the object (source project)
     * @param oldStatus the previous state
     * @param newStatus the new state
     *
     * Triggers onSharingEnabled, onSharingDisabled or onSharingTypeChanged if required, and onRuleAdded / onRuleRemoved for each rule that was added / removed
     */
    this.onObjectChanged = function (projectKey, oldStatus, newStatus) {
        const addedRules = newStatus.rules.filter(newRule =>
            !oldStatus.rules.find(oldRule => oldRule.targetProject === newRule.targetProject)
        );
        const removedRules = oldStatus.rules.filter(oldRule =>
            !newStatus.rules.find(newRule => oldRule.targetProject === newRule.targetProject)
        );
        addedRules.forEach(rule => this.onRuleAdded({projectKey, ...newStatus}, rule.targetProject));
        removedRules.forEach(rule => this.onRuleRemoved({projectKey, ...newStatus}, rule.targetProject));
        
        if(oldStatus.rules.length === 0 && !oldStatus.quickSharingEnabled) {
            // if there was no rules and it was not quickshared, we just enabled sharing
            this.onSharingEnabled({projectKey, ...newStatus});
        } else if(newStatus.rules.length === 0 && !newStatus.quickSharingEnabled) {
            this.onSharingDisabled({projectKey, ...oldStatus}); // we use oldStatus to have the rule count before change
        } else if(oldStatus.quickSharingEnabled !== newStatus.quickSharingEnabled) {
            onSharingTypeChanged({projectKey, ...newStatus});
        }
    };


    /**
     * Sends WT1 when modal is open for sharing
     * @param objectType objectType or MULTIPLE if multiple objects are selected
     * @param projectKey the projectKey for the project containing the selected object (can be skipped if MULTIPLE)
     * @param objectId localName/Id of the selected object (can be skipped if MULTIPLE)
     */
    this.onModalOpen = function (objectType, projectKey, objectId) {
        WT1.event(OBJECT_SHARING_MODAL_OPEN, {
            objectType: objectType,
            objecth: objectType === 'MULTIPLE' ? 'MULTIPLE' : md5(projectKey+"."+objectId)
        });
    };

    /**
     * Sends WT1 when modal is cancel for sharing
     * @param objectType objectType or MULTIPLE if multiple objects are selected
     * @param projectKey the projectKey for the project containing the selected object (can be skipped if MULTIPLE)
     * @param objectId localName/Id of the selected object (can be skipped if MULTIPLE)
     */
    this.onModalCancel = function (objectType, projectKey, objectId) {
        WT1.event(OBJECT_SHARING_MODAL_CANCEL, {
            objectType: objectType,
            objecth: objectType === 'MULTIPLE' ? 'MULTIPLE' : md5(projectKey+"."+objectId)
        });
    };

    /**
     * Sends WT1 when modal is save for sharing
     * @param objectType objectType or MULTIPLE if multiple objects are selected
     * @param projectKey the projectKey for the project containing the selected object (can be skipped if MULTIPLE)
     * @param objectId localName/Id of the selected object (can be skipped if MULTIPLE)
     */
    this.onModalSave = function (objectType, projectKey, objectId) {
        WT1.event(OBJECT_SHARING_MODAL_SAVE, {
            objectType: objectType,
            objecth: objectType === 'MULTIPLE' ? 'MULTIPLE' : md5(projectKey+"."+objectId)
        });
    }

});

app.controller("RequestSharingModalController", function($scope, DataikuAPI, ActivityIndicator, RequestCenterService) {
    $scope.actionVerb = $scope.actionVerb || 'share';
    $scope.sendRequest = (requestMessage, sharingTargetProjectKey) => {
        DataikuAPI.requests.createSharingRequest($scope.objectType, $scope.objectProjectKey, $scope.objectId, sharingTargetProjectKey, requestMessage).success((data) => {
            RequestCenterService.WT1Events.onRequestSent($scope.objectType, $scope.objectProjectKey, $scope.objectId, requestMessage, data.id, sharingTargetProjectKey, $scope.wt1Context);
            ActivityIndicator.success('Request created');
        }).error(() => {
            ActivityIndicator.error('Failed to create request');
            setErrorInScope.bind($scope);
        });
        $scope.dismiss();
    };
});


app.controller("ExposeSingleObjectModalController", function($scope, $rootScope, DataikuAPI, ActivityIndicator, CatalogItemService, QuickSharingWT1EventsService, $q, $filter, translate, Ng2ProjectsService, StateUtils) {
    $scope.init = function(objectType, objectId, objectDisplayName, objectProjectKey, canManageExposedElements, displayMode, wt1Context) {
        $scope.tor = {
            id: objectId,
            projectKey: objectProjectKey,
            type: objectType
        };
        $scope.objectProjectKey = objectProjectKey;
        $scope.objectDisplayName = objectDisplayName;
        $scope.notFlowItem = ["WEB_APP", "SCENARIO", "JUPYTER_NOTEBOOK", "REPORT"].includes(objectType); // those items types will show a warning saying they will only be visible for dashboards + hide quick-share option (as quickshare is only supported to flow items)
        $scope.canManageExposedElements = canManageExposedElements;
        $scope.showQuickSharingUI = !$scope.notFlowItem && canManageExposedElements && $rootScope.appConfig.quickSharingElementsEnabled && displayMode === 'share';
        const globalPermissions = $rootScope.appConfig.globalPermissions;
        $scope.canCreateProject = (globalPermissions.mayCreateProjects || globalPermissions.mayCreateProjectsFromMacros);

        // Wording
        $scope.modalTitle = displayMode === 'share'
            ? translate('MODALS.SHARE_TO_PROJECTS.TITLE', 'Share "' + objectDisplayName + '" to other project(s)', {name: objectDisplayName})
            : translate('MODALS.SHARE_TO_PROJECTS.TITLE.USE', 'Use "' + objectDisplayName + '" in other project(s)', {name: objectDisplayName});

        $scope.uiState = {
            finishedLoading: false,
            quickSharingEnabled: false,
            newProjectsSharedTo: [],
            useInProjectOption: 'EXISTING',
            hadShareErrorAfterProjectCreation: false
        }

        let hasBeenSaved = false;
        let originalQuickSharing;
        let originalRules;
        let originalTargetProjectKeys = [];

        QuickSharingWT1EventsService.onModalOpen(objectType, objectProjectKey, objectId);

        function refreshAvailableProjects() {
            DataikuAPI.taggableObjects.listAccessibleObjects(null, 'PROJECT').then((resp) => {
                $scope.availableProjects = updateAvailableProjects(resp.data, objectProjectKey, originalTargetProjectKeys)
            }).catch(setErrorInScope.bind($scope));
        };

        function updateAvailableProjects(accessibleProjects, objectProjectKey, originalTargetProjectKeys) {
            // setup for the project dropdown
            return accessibleProjects
                .filter(item => item.object.canWriteProjectContent) // theoretically a manage sharing user could share to a readable project only, but not through that UI / api route.
                .map(item => {
                    if (item.smartId === objectProjectKey) {
                        item.usable = false;
                        item.usableReason = `Cannot share to the project this ${$filter('niceTaggableType')(objectType)} originates from`;
                    } else if (originalTargetProjectKeys.includes(item.smartId)) {
                        item.usable = false;
                        item.usableReason = 'Already shared to this project';
                    } else {
                        item.usable = true;
                    }
                    return item;
                });
        }

        $q.all([
            DataikuAPI.projects.getObjectExposition(objectProjectKey, objectType, objectId),
            DataikuAPI.taggableObjects.listAccessibleObjects(null, 'PROJECT'),
        ]).then(function([{data: objectExposition}, {data: accessibleProjects}]) {
            // setup initial state of the modal
            originalQuickSharing = objectExposition.quickSharingEnabled;
            originalTargetProjectKeys = objectExposition.rules.map(rule => rule.targetProject);
            originalRules = objectExposition.rules.slice();
            $scope.uiState.quickSharingEnabled = originalQuickSharing;

            $scope.availableProjects = updateAvailableProjects(accessibleProjects, objectProjectKey, originalTargetProjectKeys);
            $scope.uiState.finishedLoading = true;
        }).catch(setErrorInScope.bind($scope))

        $scope.canSave = function() {
            return $scope.uiState.finishedLoading && (
                    ($scope.uiState.quickSharingEnabled != originalQuickSharing || $scope.uiState.newProjectsSharedTo.length > 0)
            );
        }

        $scope.canNext = function() {
            return $scope.uiState.finishedLoading && angular.isDefined($scope.uiState.projectCreationOption) && $scope.canCreateProject;
        }

        $scope.save = function() {
            hasBeenSaved = true;
            $scope.uiState.hadShareErrorAfterProjectCreation = false;
            QuickSharingWT1EventsService.onModalSave(objectType, objectProjectKey, objectId);

            const newProjectsSharedTo = $scope.uiState.newProjectsSharedTo.map(item => item.smartId);
            const quickSharingEnabled = $scope.showQuickSharingUI && $scope.shareModalForm.quickSharingEnabled.$dirty ? $scope.uiState.quickSharingEnabled : null; // we only save if changed
            DataikuAPI.projects.appendObjectExposition(objectProjectKey, objectType, objectId, newProjectsSharedTo, quickSharingEnabled)
                .success(function() {
                    QuickSharingWT1EventsService.onObjectChanged(objectProjectKey, {
                            type: objectType,
                            localName: objectId,
                            quickSharingEnabled: originalQuickSharing,
                            rules: originalRules,
                            wt1Context
                        }, {
                            type: objectType,
                            localName: objectId,
                            wt1Context,
                            quickSharingEnabled: $scope.uiState.quickSharingEnabled,
                            rules: originalRules.concat(newProjectsSharedTo.map(key => ({targetProject: key})))
                    });

                    function buildLinkToExposedObject(targetProjectKey) {
                        const href = CatalogItemService.getFlowLink(objectType.toLowerCase(), {
                            id: objectId,
                            projectKey: objectProjectKey
                        }, targetProjectKey);
                        const linkMessage = translate("MODALS.SHARE_TO_PROJECTS.SHARE_OBJECTS_UPDATED.VIEW_IN_PROJECT", `View in ${targetProjectKey} flow`, {project: targetProjectKey});
                        return `<a href='${href}'>${linkMessage}</a>.`;
                    }

                    const links = `<div>${newProjectsSharedTo.map(buildLinkToExposedObject).join('<br>')}</div>`;

                    ActivityIndicator.success(translate("MODALS.SHARE_TO_PROJECTS.SHARE_OBJECTS_UPDATED", "Shared objects updated") + links, 5000);
                    $rootScope.$emit('objectMetaDataChanged');
                    $scope.resolveModal();
                })
                .error(setErrorInScope.bind($scope));
        };

        $scope.next = function() {
            // create a new project
            const runnable = $scope.uiState.projectCreationOption.runnable;
            const wt1Context = { createdFromShareModal: true};
            const openNewProjectModal = runnable
            ? Ng2ProjectsService.openNewProjectCreationMacroModal(runnable, false, wt1Context)
            : Ng2ProjectsService.openNewBlankProjectModal('', false, wt1Context);

            openNewProjectModal.then((newProjectKey) => {
                hasBeenSaved = true;
                $scope.uiState.hadShareErrorAfterProjectCreation = false;
                if (newProjectKey) {
                    DataikuAPI.projects.addExposedObject($scope.tor.projectKey, $scope.tor.type, $scope.tor.id, newProjectKey).then(() => {
                        StateUtils.go.flowLinkFromProps($scope.tor.type, $scope.tor.projectKey, $scope.tor.id, newProjectKey);
                    }).catch((resp) => {
                        setErrorInScope.bind($scope)(resp.data, resp.status, resp.headers);
                        handleShareErrorAfterProjectCreation();
                    });
                } else {
                    // the project has been created but no project key was returned => this is a macro that does not return a project key, we can't share the element to the project
                    // this will never happen in a prod environment though
                    handleShareErrorAfterProjectCreation();
                }
            });
        };

        function handleShareErrorAfterProjectCreation() {
            $scope.uiState.hadShareErrorAfterProjectCreation = true;
            refreshAvailableProjects();
        }

        $scope.$on("$destroy", function() {
            if (!hasBeenSaved) {
                QuickSharingWT1EventsService.onModalCancel(objectType, objectProjectKey, objectId);
            }
        });
    };
});


app.controller("ExposeObjectsModalController", function($scope, $stateParams, translate, $timeout, DataikuAPI, TaggableObjectsUtils, ActivityIndicator, QuickSharingWT1EventsService) {
    $scope.available = {};
    $scope.settings = {
        dashboard: false
    };
    $scope.oldTargetProjects = []
    $scope.newTargetProjects = [];
    $scope.allTargetProjects = {}; //For exclusion in the selector of new projects

    if ($stateParams.projectKey) {
        $scope.allTargetProjects[$stateParams.projectKey] = true;
    }

    $scope.init = function(selectedItems, info) {
        $scope.selectedItems = selectedItems;
        $scope.info = info;
        $scope.itemsType = TaggableObjectsUtils.getCommonType(selectedItems, it => it.type);
        QuickSharingWT1EventsService.onModalOpen('MULTIPLE');

        DataikuAPI.projects.getObjectsExpositions(selectedItems).success(function(currentExpositions) {
            $scope.currentExpositions = currentExpositions;
            $.each($scope.currentExpositions.projects, function(projectKey, _exp){
                $scope.oldTargetProjects[projectKey] = {exposeAll: false};
                $scope.allTargetProjects[projectKey] = true; //exclude
            });
        }).error(setErrorInScope.bind($scope));

        $scope.save = function() {
            $scope.hasBeenSaved = true;
            QuickSharingWT1EventsService.onModalSave('MULTIPLE');

            $scope.settings.projects = $.extend({}, $scope.oldTargetProjects);
            $scope.newTargetProjects.forEach(function(p) {
                $scope.settings.projects[p.projectKey] = {projectKey: p.projectKey, exposeAll: true};
            })

            DataikuAPI.projects.addObjectsExpositions(selectedItems, $scope.settings)
                .success(function() {
                    const refEqual = (a, b) => a.type === b.type && a.projectKey === b.projectKey && a.id === b.id;
                    
                    // handle WT1 events
                    selectedItems.forEach(obj => {
                        const isQuickShared = $scope.currentExpositions.quickSharedObjects.some(ref => refEqual(ref, obj));
                        const wasAlreadySharedTo = Object.entries($scope.currentExpositions.projects)
                            .filter(([, usageInProject]) => usageInProject.items.some(ref => refEqual(ref, obj)))
                            .map(([pkey]) => pkey);
                        const newProjectsSharedTo = Object.entries($scope.settings.projects)
                            // the user just added project OR (user clicked share all for project AND object was not already shared to project)
                            .filter(([pkey, project]) => !$scope.currentExpositions.projects[pkey] || (project.exposeAll && !wasAlreadySharedTo.includes(pkey)))
                            .map(([pkey]) => pkey);
                        
                        const objForWT1 = {
                            projectKey: obj.projectKey,
                            type: obj.type,
                            localName: obj.id,
                            quickSharingEnabled: isQuickShared,
                        };

                        if(newProjectsSharedTo.length > 0 && (!isQuickShared && wasAlreadySharedTo.length === 0)) {
                            QuickSharingWT1EventsService.onSharingEnabled(objForWT1);
                        }
                        newProjectsSharedTo.forEach(pkey => QuickSharingWT1EventsService.onRuleAdded(objForWT1, pkey));
                    });

                    ActivityIndicator.success(translate("MODALS.SHARE_TO_PROJECTS.SHARE_OBJECTS_UPDATED", "Shared objects updated"));
                    $scope.resolveModal();
                })
                .error(setErrorInScope.bind($scope));
        };

        $scope.addProject = function(newProject) {
            $scope.newTargetProjects.push({
                projectKey: newProject.id,
                appearOnFlow: true,
                label: newProject.label
            });

            $scope.allTargetProjects[newProject.id] = true;
            $scope.newProject = null;
        };

        $scope.removeProject = function(projectKey) {
            const idx = $scope.newTargetProjects.findIndex(x => x.projectKey == projectKey);
            if (idx > -1) {
                $scope.newTargetProjects.splice(idx, 1);
            }
            delete $scope.allTargetProjects[projectKey];
        };

        $scope.$watch('newProject', function(nv,ov){
            if (nv != ov && !$.isEmptyObject(nv)) {
                $scope.addProject(nv);
                $scope.projectKey = null;
                $timeout(function(){
                    const element = $('.expose-object-modal .project-select');
                    element[0].scrollTop = element.outerHeight();
                });
            }
        });

        $scope.$on("$destroy", function() {
            if (!$scope.hasBeenSaved) {
                QuickSharingWT1EventsService.onModalCancel('MULTIPLE');
            }
        });
    };
});


app.controller("UnshareModalController", function($scope, $stateParams, DataikuAPI, TaggableObjectsUtils) {

    $scope.init = function(selectedObjects) {
        $scope.selectedObjects = selectedObjects;
        $scope.selectedObjectsType = TaggableObjectsUtils.getCommonType(selectedObjects, it => it.type);

        $scope.ok = function() {
            DataikuAPI.projects.unshare(selectedObjects, $stateParams.projectKey).success(function() {
                $scope.resolveModal();
            }).error(setErrorInScope.bind($scope));
        };
    };
});

})();
