(function () {
'use strict';

var app = angular.module('dataiku.projects.actions');


app.service('DatasetConnectionChangeService', function($rootScope, $q, $stateParams, $filter, CreateModalFromTemplate, DataikuAPI, TaggableObjectsCapabilities, TaggableObjectsUtils, Dialogs, SqlConnectionNamespaceService, ConnectionsService) {
    this.start = function(allSelectedObjects) {

        return CreateModalFromTemplate("/templates/datasets/change-connection-modal.html", $rootScope, null, function(modalScope) {
            modalScope.selectedObjects = allSelectedObjects.filter(TaggableObjectsCapabilities.canChangeConnection);
            modalScope.commonTaggableType = TaggableObjectsUtils.getCommonType(modalScope.selectedObjects, it => it.type);
            modalScope.options = {
                useExistingParams: false,
                specificSettings: {}
            };

            if (modalScope.selectedObjects.length > 1) {
                modalScope.modalTitle = `Change ${modalScope.selectedObjects.length} ${$filter('niceTaggableType')(modalScope.commonTaggableType)} connections`;
            } else {
                const name = modalScope.selectedObjects[0].displayName || modalScope.selectedObjects[0].id;
                modalScope.modalTitle = `Change connection for "${name}"`
            }

            modalScope.searchConnectionsWithDescriptions = ConnectionsService.searchConnectionsWithDescriptions;

            DataikuAPI.flow.refactoring.startChangeConnections($stateParams.projectKey, modalScope.selectedObjects).success(function(data) {
                modalScope.connections = data.connections.filter(c => c.usable); //TODO @flow, display unsable connections with reason

                modalScope.connectionsMap = {};
                modalScope.connections.forEach(function(c) {
                    modalScope.connectionsMap[c.name] = c;
                });

                modalScope.messages = data.messages;
            }).error(function(...args) {
                modalScope.fatalError = true;
                setErrorInScope.apply(modalScope, args);
            });


            modalScope.test = function() {
                delete modalScope.messages;
                resetErrorInScope(modalScope);
                const deferred = $q.defer();
                DataikuAPI.flow.refactoring.testChangeConnections($stateParams.projectKey, modalScope.selectedObjects, modalScope.options).success(function(data) {
                    modalScope.messages = data;
                    if (data.anyMessage) {
                        deferred.reject();
                    } else {
                        deferred.resolve(data)
                    }
                }).error(setErrorInScope.bind(modalScope));
                return deferred.promise;
            };

            modalScope.ok = function(force) {
                if (force) {
                    performChange();
                } else {
                    modalScope.test().then(performChange);
                }
            };

            modalScope.preselectFormatIfOnlyOne = function() {
                const formats = modalScope.connectionsMap[modalScope.options.connection].formats;
                if (formats && formats.length == 1) {
                    modalScope.options.formatOptionId = formats[0].id;
                }
                modalScope.messages = null;
            }

            modalScope.fetchSchemas = function(connectionName) {
                const catalog = modalScope.options.specificSettings.overrideSQLCatalog || 
                    ((modalScope.options.connection && modalScope.connectionsMap[modalScope.options.connection]) ? modalScope.connectionsMap[modalScope.options.connection].unoverridenSQLCatalog : '');
                SqlConnectionNamespaceService.listSqlSchemas(connectionName, modalScope, catalog, 'change-connection-modal', modalScope.connectionsMap[modalScope.options.connection].type);
            }

            modalScope.fetchCatalogs = function(connectionName) {
                SqlConnectionNamespaceService.listSqlCatalogs(connectionName, modalScope, 'change-connection-modal', modalScope.connectionsMap[modalScope.options.connection].type);
            }

            modalScope.fetchNamespaces = function(connectionName) {
                SqlConnectionNamespaceService.listIcebergNamespaces(connectionName, modalScope, 'change-connection-modal', modalScope.connectionsMap[modalScope.options.connection].type);
            }

            modalScope.$watch("options.connection", function() {
                if (modalScope.connectionsMap) {
                    SqlConnectionNamespaceService.setTooltips(modalScope, modalScope.connectionsMap[modalScope.options.connection].type);
                }
                SqlConnectionNamespaceService.resetState(modalScope, modalScope.options.specificSettings);
            });

            function performChange() {
                DataikuAPI.flow.refactoring.changeConnections($stateParams.projectKey, modalScope.selectedObjects, modalScope.options).success(function(data) {
                    if (data.anyMessage) {
                        Dialogs.infoMessagesDisplayOnly($rootScope, 'Change connection action result', data)
                    }
                    modalScope.resolveModal();
                }).error(setErrorInScope.bind(modalScope));
            }
        });
    };
});


app.service('SubFlowCopyService', function($rootScope, $q, $state, $stateParams, WT1, CreateModalFromTemplate, DataikuAPI, TaggingService, PathUtils, LoggerProvider, $compile, $timeout, $interpolate, PromiseService, ProjectFolderService) {
    var logger = LoggerProvider.getLogger('refactoring');
    const DEFAULT_ZONE_NAME = "Default";
	this.start = function(selectedObjects, itemsByZone, onSubflowCopyModalClose) {

		CreateModalFromTemplate("/templates/projects/subflow-copy-modal.html", $rootScope, null, function(modalScope) {
            modalScope.selectedObjects = [...selectedObjects];
            modalScope.projectTags = Object.keys({ ...TaggingService.getProjectTags(), ...TaggingService.getGlobalTags() });
            modalScope.options = {
                copyZoneContent: true,
                targetMode: 'CURRENT_PROJECT',
                targetProjectKey: '',
                targetProjectFolder: null, // will be initialized asynchronously
                datasetNames: {},
                streamingEndpointNames: {},
                flowZoneNames: {}
            };
            if (onSubflowCopyModalClose) {
                modalScope.onSubflowCopyModalClose = () => {
                    onSubflowCopyModalClose();
                    modalScope.dismiss();
                }
            }

            modalScope.dataCanBeCopiedElements = selectedObjects.filter(el =>
                (el.type === 'MANAGED_FOLDER' && el.subType === 'Filesystem') ||
                (el.type === 'DATASET' && ['Inline', 'UploadedFiles'].includes(el.subType)) ||
                el.type === 'SAVED_MODEL' ||
                el.type === 'MODEL_EVALUATION_STORE'
                || el.type === 'LABELING_TASK'
                || el.type === 'RETRIEVABLE_KNOWLEDGE'
            );

            var copyDataTooltipHtml = `<div>
                            <span>Data will be copied for {{dataCanBeCopiedElements.length > 1? 'these elements:' : 'this element:'}} </span>
                            <ul><li ng-repeat="el in dataCanBeCopiedElements">
                                <i class="{{el.type === 'DATASET' ? el.subType : el.type | typeToIcon}}" />
                                {{el.displayName}}
                            </li></ul>
                            <div class="help-inline">
                                <span>Data for editable datasets is always copied.</span>
                                <span>Optional copy is supported for:</span>
                                <ul>
                                    <li>Uploaded datasets</li>
                                    <li>Local filesystem folders</li>
                                    <li>Saved models</li>
                                    <li>Model evaluation stores</li>
                                    <li>Labeling tasks</li>
                                    <li>Knowledge banks</li>
                                </ul>
                            </div>

                        </div>  `;

            var el = angular.element(copyDataTooltipHtml);
            $compile(el)(modalScope);

            $timeout(function() {
                modalScope.copyDataTooltipHtml = el.html();
            });

            function reloadSelectionStats() {
                modalScope.hasAnyDataset = !!modalScope.selectedObjects.filter(to => to.type == 'DATASET').length;
                modalScope.hasAnyStreamingEndpoint = !!modalScope.selectedObjects.filter(to => to.type == 'STREAMING_ENDPOINT').length;
                modalScope.selectedZones = modalScope.selectedObjects.filter(to => to.type == 'FLOW_ZONE');
                modalScope.hasAnyZone = !!modalScope.selectedZones.length;
                modalScope.hasComputables = modalScope.hasAnyDataset || modalScope.hasAnyStreamingEndpoint;
            }

            /** Update renaming mappings for:
             * - Datasets (datasetNames)
             * - Streaming endpoints (streamingEndpointNames)
             * - flowZones (flowZoneNames)
             * 
             * Also set the `renaming` boolean so it is automatically on if a name clash is detected.
             *
             * If the target mode is NEW_PROJECT, `renaming` is only set to true if the Default zone was selected
             * In that case everything aside from the Default zone gets the same name in the remappings - and normally that wil be renamed to `Default_1` 
             * (unless there is also a `Default_1` - it will take the first available `Default_n`)
             * 
             * For EXISTING_PROJECT - `renaming` becomes true if there is a clash of names with the target project
             * For CURRENT_PROJECT - there is always a clash so `renaming` becomes true
             * 
             * The name remappings get populated by object map key value pairs of the format <oldName>:<newName>
             * Except for flow zones, as their display names do not have to be unique and are different from the id - for them we use:
             * id : {oldName:<oldname>, newName : <newName>} 
             * (the value is interpreted as a NameChange object in the back end)
             */
            function updateRenamings() {
                modalScope.options.renaming = false;
                modalScope.options.datasetNames = {};
                modalScope.options.streamingEndpointNames = {};
                modalScope.options.flowZoneNames = {};

                /** Generate a unique name by appending/incrementing numeric suffix
                 * @param {String} name input name.
                 * @param {Array.<String>} existingNames a list of names in the format `name_number`, where name is the base name and number is the numeric suffix.
                 * @returns {string} The generated unique name.
                 */
                function transmogrifyName(name, existingNames) {
                    const re = /^(.*)_([\d]+)$/;

                    let oldName = name;
                    let i = 0;
                    if (re.test(name)) {
                        let matches = re.exec(name);
                        oldName = matches[1]
                        i = parseInt(matches[2])
                    }
                    let newName = name;
                    while (existingNames.includes(newName)) {
                        i++;
                        newName = oldName + '_' + i;
                    }
                    return newName;
                }

                /**
                 * Populates the flowZoneNames given the target projects flowzones (if any)
                 * @param {Array.<Zone>} zonesInTargetProject can be empty
                 */
                function processZoneRemappings(zonesInTargetProject) {
                    const usedNames = zonesInTargetProject.map(zone => zone.name);
                    // When there are no zones yet in the target, we still don't want to create one called "Default" 
                    // as the result will be two "Default" zones
                    if (!usedNames.includes(DEFAULT_ZONE_NAME)) {
                        usedNames.push(DEFAULT_ZONE_NAME);
                    }
                    modalScope.options.flowZoneNames = {};
                    modalScope.selectedObjects
                        .filter(ob => (ob.type === 'FLOW_ZONE'))
                        .forEach(ob => {
                            const newName = transmogrifyName(ob.displayName, usedNames);
                            if (newName !== ob.displayName) {
                                modalScope.options.renaming = true;
                            }
                            modalScope.options.flowZoneNames[ob.id] = {
                                oldName : ob.displayName,
                                newName : newName};
                            usedNames.push(newName);
                        });
                }

               /**
                * Populates datasetNames given the target projects dataset names
                * @param {Array.<String>} namesInTargetProject 
                */
                function processDatasetRemappings(namesInTargetProject) {
                    modalScope.options.datasetNames = {};
                    const usedNames = angular.copy(namesInTargetProject);
                    modalScope.selectedObjects.forEach(function(to) {
                        if (to.type != 'DATASET') {
                            return;
                        }
                        const newName = transmogrifyName(to.id, usedNames);
                        if (newName !== to.id) {
                            modalScope.options.renaming = true;
                        }
                        modalScope.options.datasetNames[to.id] = newName;
                        usedNames.push(newName);
                    });
                }
                
                /**
                 * Populates streamingEndpointNames given the target projects streamingendpoints names
                 * @param {Array.<String>} namesInTargetProject 
                 */
                function processStreamingEndpointRemappings(namesInTargetProject) {
                    modalScope.options.streamingEndpointNames = {};
                    const usedNames = angular.copy(namesInTargetProject);

                    modalScope.selectedObjects.forEach(function(to) {
                        if (to.type != 'STREAMING_ENDPOINT') {
                            return;
                        }
                        const newName = transmogrifyName(to.id, usedNames);
                        if (newName !== to.id) {
                            modalScope.options.renaming = true;
                        }
                        modalScope.options.streamingEndpointNames[to.id] = newName;
                        usedNames.push(newName);
                    });
                };

                let projectKey;
                if (modalScope.options.targetMode == 'CURRENT_PROJECT') {
                    projectKey = $stateParams.projectKey;
                } else if (modalScope.options.targetMode == 'EXISTING_PROJECT') {
                    projectKey = modalScope.options.targetProjectKey;
                } else if (modalScope.options.targetMode == 'NEW_PROJECT') {
                    // Just to rename the default zone so we don't end up with 2 in the new project
                    // also handles it well if they have "Default_1" etc in the set of selected zones
                    processZoneRemappings([]);
                    return;
                }

                if (projectKey) {
                    DataikuAPI.flow.zones.list(projectKey).success(processZoneRemappings);
                    DataikuAPI.datasets.listNames(projectKey).success(processDatasetRemappings);
                    DataikuAPI.streamingEndpoints.listNames(projectKey).success(processStreamingEndpointRemappings);
                }
            }
            updateRenamings();
            reloadSelectionStats()

            modalScope.$watch("options.targetMode", updateRenamings);
            modalScope.$watch("options.targetProjectKey", updateRenamings);

			DataikuAPI.projects.list().success(function (data) {
                modalScope.otherProjects =  data.filter(project => project.projectKey !== $stateParams.projectKey);
	        });

            DataikuAPI.flow.refactoring.startCopySubFlow(selectedObjects).success(function(data) {
                modalScope.messages = data;
            }).error(function(...args) {
                modalScope.fatalError = true;
                setErrorInScope.apply(modalScope, args);
            });

            modalScope.toCopySubflowApiOptions = function(options) {
                return {
                    ...options,
                    targetProjectFolder: undefined,
                    targetProjectFolderId: options.targetProjectFolder ? options.targetProjectFolder.id : '',
                }
            }

            modalScope.test = function() {
                const deferred = $q.defer();
                delete modalScope.messages;
                resetErrorInScope(modalScope);
                DataikuAPI.flow.refactoring.testCopySubFlow(modalScope.selectedObjects, modalScope.toCopySubflowApiOptions(modalScope.options), $stateParams.projectKey).success(function(data) {
                    modalScope.messages = data;
                    if (data.anyMessage) {
                        deferred.reject();
                    } else {
                        deferred.resolve(data)
                    }
                }).error(setErrorInScope.bind(modalScope));
                return deferred.promise;
            };

            modalScope.ok = function(force) {
                if (force) {
                    performCopy();
                } else {
                    modalScope.test().then(performCopy);
                }
            };

            function performCopy() {
                DataikuAPI.flow.refactoring.copySubFlow(modalScope.selectedObjects, modalScope.toCopySubflowApiOptions(modalScope.options), $stateParams.projectKey).success(function(data) {
                    $rootScope.$emit('stopAction');
                    modalScope.resolveModal();

                    if (modalScope.options.targetMode != 'CURRENT_PROJECT' && modalScope.options.targetProjectKey != $stateParams.projectKey) {
                        $state.transitionTo("projects.project.flow", {projectKey: modalScope.options.targetProjectKey});
                    } else {
                        $rootScope.$emit('reloadGraph');
                    }
                    modalScope.selectedObjects.forEach(function(selectedObject){
                        if (selectedObject.type == 'DATASET') {
                            WT1.event("create-dataset", {
                                connectionType: selectedObject.subType
                            });
                        }
                    });                    
                }).error(setErrorInScope.bind(modalScope));
            }

            modalScope.uniqueProjectKey = true;

            DataikuAPI.projects.listAllKeys()
                .success(function(data) {
                    modalScope.allProjectKeys = data;
                })
                .error(setErrorInScope.bind(modalScope));

            function isProjectKeyUnique(value) {
                return !modalScope.allProjectKeys || modalScope.allProjectKeys.indexOf(value) < 0;
            };

            modalScope.$watch("options.targetProjectKey", function(nv, ov) {
                modalScope.uniqueProjectKey = !nv || isProjectKeyUnique(nv);
            });

            /**
             * Update the scope selectedObjects list with the content of the selected zones.
             * Add selected objects to the scope selectedObjects list if option is checked.
             * Remove selected objects from the scope selectedObjects list if option is unchecked.
             * @param {?boolean} copyZoneContent true if copy zone content option is checked, false otherwise.
             */
            function updateCopiedContent(copyZoneContent) {
                modalScope.selectedZones.forEach(to => {
                    const items = itemsByZone[to.id];
                    if (!items) {
                        return
                    }
                    items.forEach(it => {
                        const found = selectedObjects.find(so => it.type === so.type && it.id === so.id);
                        if (found !== undefined) {
                            return;
                        }
                        if (copyZoneContent) {
                            modalScope.selectedObjects.push(it);
                        } else {
                            const index = modalScope.selectedObjects.findIndex(sel => sel.id === it.id);
                            if (index != -1) {
                                modalScope.selectedObjects.splice(index, 1);
                            }
                        }
                    });
                });
                updateRenamings();
                reloadSelectionStats();
            }
            modalScope.$watch("options.copyZoneContent", (newValue, oldValue) => newValue != oldValue && updateCopiedContent(newValue));
            // Update the copied zone content when the modal launches for the first time
            updateCopiedContent(modalScope.options.copyZoneContent);

            modalScope.$watch("options.targetProjectName", function(nv, ov) {
                if (!nv) return;
                var slug = nv.toUpperCase().replace(/\W+/g, ""),
                    cur = slug,
                    i = 0;
                while (!isProjectKeyUnique(cur)) {
                    cur = slug + "_" + (++i);
                }
                modalScope.options.targetProjectKey = cur;
            });

            ProjectFolderService.getDefaultFolderForNewProject().then((folder) => {
                modalScope.options.targetProjectFolder = folder;
            }).catch(setErrorInScope.bind(modalScope));

            modalScope.browse = folderIds => {
                // Use last id in path
                modalScope.destination = PathUtils.makeNLNT(folderIds).split('/').pop();
                // browse-path expects a success-error promise so we need to wrap with qToHttp for now (catch() does not return a monkey-patched promise)
                return PromiseService.qToHttp(ProjectFolderService.getBrowseNode(modalScope.destination).catch(setErrorInScope.bind(modalScope)));
            };

            modalScope.getProjectFolderName = item => item.name;
            modalScope.canSelectFolder = item => item.canWriteContents;
        });
    };
});



})();