(function() {
    'use strict';
    /** @typedef {import('../../../types').ChartDef} ChartDef */
    /** @typedef {import('../../../types').GeneratedSources.PivotTableTensorResponse} PivotTableTensorResponse */
    /** @typedef {import('../../../types').GeneratedSources.DataVisualizationTheme} DataVisualizationTheme */

    /** @typedef {import('../../../../../../../../../../server/src/frontend/node_modules/ag-grid-community').GridOptions} GridOptions */
    /** @typedef {import('../../../../../../../../../../server/src/frontend/node_modules/ag-grid-community')} AgGrid */
    /** @typedef {import('../../../../../../../../../../server/src/frontend/src/app/migration/services/ag-grid-loader.service').AgGridLoaderService} AgGridLoader */

    const app = angular.module('dataiku.charts');
    // (!) This service previously was in static/dataiku/js/simple_report/table/pivot_table.js
    app.factory('PivotTableChart', function(ChartFormatting, ChartDataWrapperFactory, ChartTooltips, ChartCustomMeasures, ChartContextualMenu, ChartStoreFactory, ChartUsableColumns, ColumnAvailability, /** @type {AgGridLoader} */ AgGridLoader, AgGridConverter, ChartDrilldown, PivotTableUtils, ChartFormattingPaneSections, ChartDimension, ChartHierarchyDimension) {
        /** @type JQuery<HTMLElement> */
        let $rootElement;
        /** @type JQuery<HTMLElement> */
        let $container;
        /** @type {Record<string, number>} */
        let lastAxesDefDisplayed;
        const MEASURE_COLUMN_KEY = '__MEASURE__LIST__';
        let contextualMenu;
        let defaultTopHeadersMenuListener;
        let defaultBackgroundMenuListener;
        let topRowHeadersMenuListener;
        let instanceChartHandler;
        /** @type {AgGrid} */
        let AgGrid; // Will be set when AgGridLoader loads the library
        /**
         * Initializes the DOM Element that will be used as a container for the pivot table.
         *
         * @param {JQuery<HTMLElement>} $rootElement
         * @returns
         */
        function initContainer($rootElement) {
            $rootElement.empty();
            $rootElement.addClass('ag-theme-balham');
            const $container = $('<div class="h100 table-pivot-table chart-wrapper" />');
            $rootElement.append($container);
            return $container;
        };

        /**
         * Initializes the pivot table tooltips.
         *
         * @param {JQuery<HTMLElement>} $rootElement
         * @param {JQuery<HTMLElement>} $container
         * @param {PivotTableTensorResponse} chartData
         * @param {ChartDef} chartDef
         * @param {*} chartHandler
         */
        function initTooltips($rootElement, $container, chartData, chartDef, chartHandler) {
            /*
             *  This formatting is not relevant, numValues = 1000 is an arbitrary value and should be modified:
             *  @TODO : https://app.shortcut.com/dataiku/story/110162/enhance-formatting-for-dimensions-and-measures-in-auto-mode
             */
            const tooltipFormatters = ChartFormatting.createMeasureFormatters(chartDef, chartData, 1000);

            const tooltips = ChartTooltips.create($rootElement, chartHandler, chartData, chartDef, tooltipFormatters);
            $container.on('click', function(evt) {
                if (evt.target.closest('[data-legend], [tooltip-el]')) {
                    return;
                }
                tooltips.unfix();
            });
            $container.addClass('has-tooltip');
            return tooltips;
        };

        /**
         * Generates grid options from the given parameters.
         *
         * @param {ChartDef} chartDef
         * @param {Record<string, number>} axesDef
         * @param {PivotTableTensorResponse} data
         * @param {*} chartHandler
         * @param {JQuery<HTMLElement>} $element
         * @param {JQuery<HTMLElement>} $container
         * @returns {GridOptions}
         */
        function generateGridOptions(chartDef, axesDef, data, chartHandler, $element, $container) {
            const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef);
            const chartStore = ChartStoreFactory.get(chartDef.$chartStoreId);
            const tooltips = initTooltips($element, $container, chartData, chartDef, chartHandler);
            contextualMenu = ChartContextualMenu.create(chartData, chartDef, chartStore);
            const agGridConverter = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $element);
            return agGridConverter.convert();
        }

        function updateColumnVisibilities(gridApi, chartDef, store) {
            const show = chartDef.pivotTableOptions.tableFormatting.showRowHeaders;

            chartDef.yDimension.forEach((dimension) => {
                gridApi.setColumnsVisible([store.getDimensionId(dimension)], show);
            });

            const yHierarchyDimension = ChartHierarchyDimension.getCurrentHierarchyDimension(chartDef, 'y');
            if (yHierarchyDimension) {
                gridApi.setColumnVisible(store.getDimensionId(yHierarchyDimension), show);
            }

            gridApi.setColumnsVisible([MEASURE_COLUMN_KEY], show);
        }

        function showRowMainHeaderContextualMenu(event, cellValue, chartDef, mainHeader) {
            const rowHeadersCallbacks = PivotTableUtils.getToggleRowHeadersCallbacks(chartDef, instanceChartHandler.openSection);
            const actions = PivotTableUtils.getRowHeaderContextualMenuActions(chartDef, rowHeadersCallbacks, cellValue, mainHeader);
            event.preventDefault();
            contextualMenu.open({ event, customActions: actions, type: 'pivotTableHeader' });
        }

        function showBackgroundContextualMenu(event, chartDef) {
            const actions = ChartDrilldown.getDrillupActions(chartDef);
            event.preventDefault();
            contextualMenu.open({ event, customActions: actions, type: 'background' });
        }

        function showColumnHeaderContextualMenu(event, cellValue, chartDef, mainHeader) {
            event.preventDefault();
            const columnHeadersCallbacks = {
                openFormatTableSection: () => {
                    instanceChartHandler.openSection(ChartFormattingPaneSections.TABLE);
                }
            };
            contextualMenu.open({ event, customActions: PivotTableUtils.getColumnHeaderActions(chartDef, cellValue, mainHeader, columnHeadersCallbacks), type: 'pivotTableHeader' });
        }

        function updateTopHeadersContextualMenuListeners(chartDef) {
            removeTopHeadersContextualMenuListeners();
            addTopHeadersContextualMenuListeners(chartDef);
        }

        function updateBackgroundContextualMenuListener(chartDef) {
            removeBackgroundContextualMenuListener();
            addBackgroundContextualMenuListener(chartDef);
        }

        function removeTopHeadersContextualMenuListeners() {
            const defaulTopHeaders = $container[0].querySelector('.ag-header-viewport');
            const topRowHeaders = $container[0].querySelector('.ag-pinned-left-header');

            if (defaulTopHeaders) {
                defaulTopHeaders.removeEventListener('contextmenu', defaultTopHeadersMenuListener);
            }

            if (topRowHeaders) {
                topRowHeaders.removeEventListener('contextmenu', topRowHeadersMenuListener);
            }
        }

        function removeBackgroundContextualMenuListener() {
            const gridBody = $container[0].querySelector('.ag-body');

            if (gridBody) {
                gridBody.removeEventListener('contextmenu', defaultBackgroundMenuListener);
            }
        }

        /*
         * Row subheaders not part of agGrid top headers. They are simple cells built with custom renderer, not accessible right away after init like the top header.
         * Consequently, their contextual menu listeners are built in the converter service and must be removed when destroying the pivot.
         */
        function removeRowSubheadersContextualMenuListeners() {
            const rowSubheaders = $container[0].querySelectorAll('.' + PivotTableUtils.ROW_SUBHEADER_CLASSNAME);
            if (rowSubheaders && rowSubheaders.length) {
                rowSubheaders.forEach(subheader => {
                    contextualMenu.removeContextualMenuHandler(subheader);
                });
            }
        }

        function addTopHeadersContextualMenuListeners(chartDef) {
            const defaulTopHeaders = $container[0].querySelector('.ag-header-viewport');
            const areRowHeadersFrozen = chartDef.pivotTableOptions.tableFormatting.freezeRowHeaders;

            if (areRowHeadersFrozen) {
                const rowHeaders = $container[0].querySelector('.ag-pinned-left-header');
                topRowHeadersMenuListener = (event) => {
                    const rowMainHeader = event.target.closest('.' + PivotTableUtils.ROW_MAIN_HEADER_CLASSNAME) || event.target.querySelector('.' + PivotTableUtils.ROW_MAIN_HEADER_CLASSNAME);

                    if (rowMainHeader) {
                        showRowMainHeaderContextualMenu(event, rowMainHeader.innerText, chartDef, true);
                    }
                };
                rowHeaders.addEventListener('contextmenu', topRowHeadersMenuListener);
            }

            defaultTopHeadersMenuListener = (event) => {
                const rowMainHeader = event.target.closest('.' + PivotTableUtils.ROW_MAIN_HEADER_CLASSNAME) || event.target.querySelector('.' + PivotTableUtils.ROW_MAIN_HEADER_CLASSNAME);
                const columnMainHeader = event.target.closest('.' + PivotTableUtils.COLUMN_MAIN_HEADER_CLASSNAME) || event.target.querySelector('.' + PivotTableUtils.COLUMN_MAIN_HEADER_CLASSNAME);
                const columnSubheader = event.target.closest('.' + PivotTableUtils.COLUMN_SUBHEADER_CLASSNAME) || event.target.querySelector('.' + PivotTableUtils.COLUMN_SUBHEADER_CLASSNAME);

                if (rowMainHeader) {
                    showRowMainHeaderContextualMenu(event, rowMainHeader.innerText, chartDef, true);
                } else if (columnMainHeader) {
                    showColumnHeaderContextualMenu(event, columnMainHeader.innerText, chartDef, true);
                } else if (columnSubheader) {
                    showColumnHeaderContextualMenu(event, columnSubheader.innerText, chartDef);
                }
            };

            defaulTopHeaders.addEventListener('contextmenu', defaultTopHeadersMenuListener);
        }

        function addBackgroundContextualMenuListener(chartDef) {
            const xHierarchyLevel = ChartHierarchyDimension.getCurrentHierarchyLevel(chartDef, 'x');
            const yHierarchyLevel = ChartHierarchyDimension.getCurrentHierarchyLevel(chartDef, 'y');
            if (xHierarchyLevel > 0 || yHierarchyLevel > 0) {
                const gridBody = $container[0].querySelector('.ag-body');
                defaultBackgroundMenuListener = (event) => {
                    const isBackgroundClick = (event.target.closest('.ag-body') || event.target.querySelector('.ag-body'))
                        && !event.target.closest('.ag-center-cols-container');
                    if (isBackgroundClick) {
                        showBackgroundContextualMenu(event, chartDef);
                    };
                };

                gridBody.addEventListener('contextmenu', defaultBackgroundMenuListener);
            }
        }

        /**
         * Initializes the AG Grid Grid object.
         *
         * @param {JQuery<HTMLElement>} $container
         * @param {GridOptions} gridOptions
         */
        function initGrid($container, gridOptions, chartDef, store) {
            // @ts-ignore
            const gridApi = AgGrid.createGrid($container[0], gridOptions);
            store.setGridApi(gridApi);
            updateColumnVisibilities(gridApi, chartDef, store);
            chartDef.pivotTableOptions.showSidebar ? gridApi.setSideBarVisible(true): gridApi.setSideBarVisible(false);
            addTopHeadersContextualMenuListeners(chartDef);
            addBackgroundContextualMenuListener(chartDef);
        }

        /**
         * Updates grid options and trigger the needed refresh/redraw for the update to be applied to the UI.
         * @param {GridOptions} newGridOptions
         * @param {ChartDef} chartDef
         */
        function updateGridOptions(newGridOptions, chartDef) {
            const store = ChartStoreFactory.get(chartDef.$chartStoreId);
            const gridApi = store.getGridApi();


            removeRowSubheadersContextualMenuListeners();
            updateTopHeadersContextualMenuListeners(chartDef);
            updateBackgroundContextualMenuListener(chartDef);

            const { sideBar, columnDefs, ...newGridOptionsWithoutColumnDefsAndSideBar } = newGridOptions;
            if (chartDef.pivotTableOptions.showSidebar && typeof newGridOptions.sideBar !== 'string' && typeof newGridOptions.sideBar !== 'boolean') {
                gridApi.getSideBar().toolPanels.forEach((toolpanel, i) => {
                    const instance = gridApi.getToolPanelInstance(toolpanel.id);
                    instance.customRefresh({ ...sideBar.toolPanels[i].toolPanelParams, api: gridApi });
                });
                gridApi.setSideBarVisible(true);
            } else {
                gridApi.setSideBarVisible(false);
            }
            // If not it will merge the previous columnDefs with the new ones, leading to issues when removing measure
            gridApi.setGridOption('columnDefs', columnDefs, { maintainColumnOrder: false });

            gridApi.updateGridOptions(newGridOptionsWithoutColumnDefsAndSideBar);
            gridApi.resetColumnState();

            updateColumnVisibilities(gridApi, chartDef, store);
        }

        /**
         * Returns true if the element passed in parameter is different than the current root element
         * @param {JQuery<HTMLElement>} $element
         * @return {boolean}
         */
        function isNewRootElement($element) {
            return $rootElement === undefined || $element[0] !== $rootElement[0];
        }

        /**
         * Returns true if the axes definition has changed
         * @param {Record<string, number>} axesDef
         * @returns
         */
        function hasChangedAxesDefinition(axesDef) {
            return lastAxesDefDisplayed && !angular.equals(Object.keys(axesDef), Object.keys(lastAxesDefDisplayed));
        }

        /**
         * Returns true if can refresh current view
         * @param {JQuery<HTMLElement>} $element
         * @param {ChartDef} chartDef
         * @param {Record<string, number>} axesDef
         * @return {boolean} true if can refresh current view
         */
        function canRefreshView($element, chartDef, axesDef) {
            const store = ChartStoreFactory.get(chartDef.$chartStoreId);
            const gridApi = store.getGridApi();

            if (gridApi == null) {
                return false;
            }

            /*
             * When changing the "Row headers" toggle from false to true for the first time, we need to switch to treeData mode, which means re-building the pivot from scratch.
             * This is because there is no way as of today in agGrid to hide the main group column so when hiding the row headers, we disable treeData mode entirely.
             */
            const shouldRebuildTreeData = chartDef.pivotTableOptions.tableFormatting.showRowHeaders && gridApi.getGridOption('treeData') === false;
            return !shouldRebuildTreeData &&
                !isNewRootElement($element)
                && !hasChangedAxesDefinition(axesDef);
        }

        /**
         * Purge Ag-Grid instance.
         * @param {ChartDef} chartDef
         */
        function purgeAgGrid(chartDef) {
            const store = ChartStoreFactory.get(chartDef.$chartStoreId);
            const gridApi = store.getGridApi();

            if ($container && $container[0]) {
                removeTopHeadersContextualMenuListeners();
                removeRowSubheadersContextualMenuListeners();
                removeBackgroundContextualMenuListener();
            }

            if (gridApi != null && gridApi.destroy) {
                gridApi.destroy();
            }
            store.setGridApi(null);
        }

        /**
         * Purge expanded status of rows and columns
         * @param {ChartDef} chartDef
         */
        function purgeExpandedStatus(chartDef) {
            chartDef.pivotTableOptions.areColumnExpandedByDefault = true;
            chartDef.pivotTableOptions.areRowsExpandedByDefault = true;
            chartDef.pivotTableOptions.rowIdByCustomExpandedStatus = {};
            chartDef.pivotTableOptions.columnIdByCustomExpandedStatus = {};
        }

        /**
         * Convert (DSS version <= 10) Color Table into regular Pivot Table
         * @param {ChartDef} chartDef
         */
        function convertColorTableToPivotTable(chartDef, data) {
            chartDef.variant = undefined;
            if (chartDef.colorMeasure.length === 0 && chartDef.genericMeasures.length > 0) {
                chartDef.colorMeasure.push(chartDef.genericMeasures[0]);
                data.aggregations.push(data.aggregations[0]);
            }
        }

        /**
         * AngularJS factory building Pivot Table Chart.
         *
         * @param {JQuery<HTMLElement>} $element
         * @param {ChartDef} chartDef
         * @param {any} chartHandler
         * @param {Record<string, number>} axesDef
         * @param {PivotTableTensorResponse} data
         * @param {DataVisualizationTheme} theme
         */
        function PivotTableChart($element, chartDef, chartHandler, axesDef, data) {

            const { store, id } = ChartStoreFactory.getOrCreate(chartDef.$chartStoreId);
            chartDef.$chartStoreId = id;
            store.updateChartStoreMeasureIDsAndDimensionIDs(chartDef);
            instanceChartHandler = chartHandler;

            const dataSpec = chartHandler.getDataSpec();
            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
            const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
            ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);

            if (PivotTableUtils.isColoredTable(chartDef)) {
                convertColorTableToPivotTable(chartDef, data);
            }

            if (hasChangedAxesDefinition(axesDef)) {
                purgeExpandedStatus(chartDef);
            }

            if (canRefreshView($element, chartDef, axesDef)) {
                const newGridOptions = generateGridOptions(chartDef, axesDef, data, chartHandler, $element, $container);
                updateGridOptions(newGridOptions, chartDef);
            } else {
                $rootElement = $element;
                purgeAgGrid(chartDef);
                $container = initContainer($element);
                const gridOptions = generateGridOptions(chartDef, axesDef, data, chartHandler, $element, $container);
                const requestOptions = store.getRequestOptions();
                if (requestOptions && _.isNil(requestOptions.removePivotTableLimitations)) {
                    store.setRequestOptions({ removePivotTableLimitations: false });
                }
                initGrid($container, gridOptions, chartDef, store);
            }

            lastAxesDefDisplayed = axesDef;

            if (typeof (chartHandler.loadedCallback) === 'function') {
                chartHandler.loadedCallback();
            }

            chartHandler.$on('$destroy', function() {
                purgeAgGrid(chartDef);
            });
        };

        return ($element, chartDef, chartHandler, axesDef, data, theme) => {
            AgGridLoader.load()
                .then(loaded => AgGrid = loaded)
                .then(() => PivotTableChart($element, chartDef, chartHandler, axesDef, data, theme));
        };
    });
})();
