(function() {
'use strict';

const app = angular.module('dataiku.controllers');


app.controller('ProfileController', function($scope, $state, $stateParams, $rootScope, AchievementService, DataikuAPI, TopNav, MessengerUtils, WT1, $anchorScroll, translate, UserSettingsService) {
    var tab = $state.current.data ? $state.current.data.selectedTab : null;
    TopNav.setNoItem();
    TopNav.setLocation(TopNav.DSS_HOME, "administration", null, tab);
    $scope.uiState = $scope.uiState || {};

    $scope.$on("$stateChangeSuccess", function() {
        var tab = $state.current.data ? $state.current.data.selectedTab : null;
        TopNav.setNoItem();
        TopNav.setLocation(TopNav.DSS_HOME, "administration", null, tab);
    });

    $scope.requestedProfile = $rootScope.appConfig.login;
    if ($stateParams.userLogin) {
        $scope.requestedProfile = $stateParams.userLogin;
    }

    $scope.profile = {};

    $scope.logout = function() {
        DataikuAPI.logout().success(function(data) {
            // Violent redirect to avoid keeping a cached appConfig
            window.location = "/logged-out";
        }).error(setErrorInScope.bind($scope));
    };

    function loadProfile() {
        DataikuAPI.profile.get($scope.requestedProfile, true).success(function(data) {
            $scope.profile.user = data;
            $anchorScroll();
        }).error(function(data, status, headers, config, statusText, xhrStatus) {
            if (status === 403) {
                $scope.profile.user = { login: $scope.requestedProfile, restricted: true };
            } else {
                setErrorInScope(data, status, headers, config, statusText, xhrStatus);
            }
        });

        DataikuAPI.profile.achievements($scope.requestedProfile).success(function(data) {
            // enrich the data
            var achievements = [];
            for(var k in data.achievements) {
                var x = AchievementService.getAchievement(data.achievements[k].id);
                if (x) {
                    var achievement = data.achievements[k];
                    achievement.icon = x.icon;
                    achievement.title = x.title;
                    achievement.text = x.text;
                    achievements.push(achievement);
                }
            }
            $scope.profile.achievements = data;
            $scope.profile.achievements.achievements = achievements;
        }).error(setErrorInScope.bind($scope));
    }

    WT1.event("user-profile-load", {});
    loadProfile();
    $scope.$on("UPDATED_PROFILE", function() {
        loadProfile();
    });

    $scope.oldUserSettings = angular.copy($rootScope.appConfig.userSettings);
    $scope.userSettings = angular.copy($rootScope.appConfig.userSettings);

    $scope.isDirtyUserSettings = function () {
        return !angular.equals($scope.userSettings, $scope.oldUserSettings);
    };

    $scope.isEnableFlowZoomTracking = function (isOn) { // avoid presenting negative names, but store as 'disable' flag to avoid need to migrate settings
        if (angular.isDefined(isOn)) {
            $scope.userSettings.disableFlowZoomTracking = !isOn ;
        } else {        
            return !$scope.userSettings.disableFlowZoomTracking;
        }
    };

    $scope.saveUserSettings = function () {
        UserSettingsService.saveUserSettings($scope.userSettings).catch(setErrorInScope.bind($scope));
        $scope.oldUserSettings = angular.copy($scope.userSettings);
    };

    $scope.toggleAutoStackUp = function () {
        WT1.event('dashboard-auto-stack-up-toggled-from-user-settings', {
            autoStackUp: $scope.userSettings.dashboards.autoStackUp
        });
    };

    $scope.sampleMessengerNotification = function () {
        MessengerUtils.post({
          message: translate("PROFILE.MY_SETTINGS.NOTIFICATIONS.SAMPLE", "Here is your example! <br/> So, what do you want to be notified of?"),
          hideAfter: 2,
          id: 'sampleNotification'
        });
    };

    $scope.hooks = {save : null, isDirty : null};
});


app.controller('MyProfileEditController', function($scope, $state, $stateParams, DataikuAPI, $rootScope, TopNav, WT1) {
    $scope.user = {};
    $scope.oldUser = {};
    $scope.image = {};

    $scope.canEditEmailAndName = $rootScope.appConfig.enableEmailAndDisplayNameModification || $scope.appConfig.admin;

    DataikuAPI.profile.get()
        .success(function(data) {
            $scope.user = data;
            $scope.user.passwordConfirmation='';
            $scope.user.password='';
            $scope.oldUser = angular.copy($scope.user);
        })
        .error(setErrorInScope.bind($scope));

    $scope.isDirtyUser = function() {
        return !angular.equals($scope.user, $scope.oldUser) && $scope.user.passwordConfirmation == $scope.user.password && !$scope.userDescriptionForm.$invalid;
    };

    $scope.saveUser = function() {
        WT1.event("user-profile-save-user", {});
        DataikuAPI.profile.edit($scope.user).success(function(data) {
             $scope.errorMessage ='';
             if($scope.image.file) {
                 DataikuAPI.profile.uploadPicture($scope.image.file).then(function(data) {
                     $rootScope.$broadcast('UPDATED_PROFILE');
                 }, setErrorInScope.bind($scope));
             } else {
                 $rootScope.$broadcast('UPDATED_PROFILE');
             }
             $scope.oldUser = angular.copy($scope.user);
        }).error(function(a, b, c) {
            if(a.code == "666") {
                // user error
                $scope.errorMessage = a.message;
            } else {
                // api error
                setErrorInScope.bind($scope)(a, b, c)
            }
        });
    }

    $scope.hooks.save = $scope.saveUser;
    $scope.hooks.isDirty = $scope.isDirtyUser;
});

app.controller('MyFilesEditController', function($scope, $state, $stateParams, DataikuAPI, translate) {
    $scope.user = {};
    $scope.uiState = {};
    $scope.uiState.activeZone = $stateParams.tab || $scope.uiState.activeZone || 'versioned';

    DataikuAPI.profile.get()
        .success(function(data) {
            $scope.user = data;
        })
        .error(setErrorInScope.bind($scope));

    $scope.userConfigEmptyCta = {
        title: translate("PROFILE.MY_FILES.USER_CONFIG_CTA.HEADER", "No file for this user."),
        text: translate("PROFILE.MY_FILES.USER_CONFIG_CTA.MESSAGE", "Create and manage your own config files. The contents are accessible only to this user."),
        btnAction: "create",
        btnLabel: translate("PROFILE.MY_FILES.USER_CONFIG_CTA.BUTTON", "Create your first file")
    }

    $scope.userResourcesEmptyCta = {
        title: translate("PROFILE.MY_FILES.USER_RESOURCES_CTA.HEADER", "No file for this user."),
        text: translate("PROFILE.MY_FILES.USER_RESOURCES_CTA.MESSAGE", "Create and manage your own resource files. The contents are accessible only to this user."),
        btnAction: "create",
        btnLabel: translate("PROFILE.MY_FILES.USER_RESOURCES_CTA.BUTTON", "Create your first resource")
    }

    $scope.setFilesTab = function(tabname) {
        $scope.uiState.activeZone = tabname;
    };

});


app.controller('MyProfileAchievementsController', function($scope) {
    $scope.hooks.save = null;
    $scope.hooks.isDirty = null;
});


app.controller('MyProfileStarsController', function($scope, $rootScope, StateUtils, DataikuAPI, InterestsService, WT1) {
    $scope.hooks.save = null;
    $scope.hooks.isDirty = null;

    function getUserInterests(offset) {
        DataikuAPI.interests.getUserInterests($rootScope.appConfig.login, offset, 100, $scope.filters).success(function(data) {
            $scope.results = data;
        }).error(setErrorInScope.bind($scope));
    }
    getUserInterests(0);

    $scope.previousResults = function() {
        getUserInterests($scope.results.offset - $scope.results.pageSize);
    };

    $scope.nextResults = function() {
        getUserInterests($scope.results.offset + $scope.results.pageSize);
    };

    function getTaggableObject(object) {
        return {
            type: object.objectType,
            projectKey: object.projectKey,
            id: object.objectId,
            workspaceKey: object.workspaceKey,
        };
    }

    function getTaggableObjects(object) {
        return [getTaggableObject(object)];
    }

    $scope.getObjectLink = object => StateUtils.href.taggableObject(getTaggableObject(object));

    $scope.starObject = function(object, star) {
        WT1.event("user-profile-star", {});
        InterestsService.star($scope, getTaggableObjects(object), star).success( () => object.starred = star);
    };

    $scope.watchObject = function(object, watch) {
        WT1.event("user-profile-watch", {});
        InterestsService.watch($scope, getTaggableObjects(object), watch).success( () => object.watching = watch);
    };

    $scope.getProjectName = function(projectKey) {
        if (!projectKey || !projectsMap) return;
        return (projectsMap[projectKey] || {}).name;
    };

    var projectsMap;

    $scope.filters = {projectKey: null, taggableType: null, workspaceKey: null};

    DataikuAPI.projects.list().success(function(data) {
        projectsMap = {};
        data.forEach(function(p) {
            projectsMap[p.projectKey] = p;
        });
        $scope.projects = data;
        $scope.projects.push({projectKey: null, name: 'All projects'});
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.workspaces.list().success(data => {
        $scope.workspaces = data;
        $scope.workspaces.push({ workspaceKey: null, displayName: 'All workspaces' });
    }).error(setErrorInScope.bind($scope));

    $scope.$watch("filters", (nv, ov) => {
        if (nv.taggableType !== ov.taggableType) {
            if (nv.taggableType === 'WORKSPACE') {
                $scope.filters.projectKey = null;
            } else if (nv.taggableType === 'PROJECT') {
                $scope.filters.workspaceKey = null;
            }
        }
        getUserInterests($scope.results ? $scope.results.offset : 0);
    }, true);
});


app.controller("MyProfilePersonalAPIKeysController", function ($scope, $state, translate, DataikuAPI, CreateModalFromTemplate, Dialogs, TopNav, WT1, ClipboardUtils) {
    $scope.hooks.save = null;
    $scope.hooks.isDirty = null;

    $scope.refreshApiKeysList = function () {
        DataikuAPI.profile.listPersonalAPIKeys().success(function (data) {
            $scope.apiKeys = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.refreshApiKeysList();

    $scope.isExpired = function (key) {
        return key.expiresOn != 0 && key.expiresOn < new Date().getTime();
    };

    $scope.createAPIKey = function () {
        CreateModalFromTemplate("/templates/admin/security/personal-api-key-modal.html", $scope, null, function (newScope) {
            newScope.apiKey = {
                "label": "Personal for " + $scope.profile.user.displayName ,
                "description": ""
            };
            newScope.creation = true;
        });
    };

    $scope.editPersonalAPIKey = function (apiKey) {
        CreateModalFromTemplate("/templates/admin/security/personal-api-key-modal.html", $scope, null, function (newScope) {
            newScope.apiKey = {
                ...apiKey
            };
            newScope.creation = false;
        });
    };

    $scope.deleteAPIKey = function (keyId) {
        Dialogs.confirm($scope, translate("PROFILE.API_KEYS.REMOVE_DIALOG.TITLE", "Remove API key"), translate("PROFILE.API_KEYS.REMOVE_DIALOG.MESSAGE", "Are you sure you want to remove this API key?")).then(function () {
            WT1.event("user-profile-delete-API-key", {});
            DataikuAPI.profile.deletePersonalAPIKey(keyId).success(function (data) {
                $scope.refreshApiKeysList();
            }).error(setErrorInScope.bind($scope));
        });
    };

    $scope.viewQRCode = function (key) {
        CreateModalFromTemplate("/templates/admin/security/api-key-qrcode-modal.html", $scope, null, function (newScope) {
            newScope.apiKeyQRCode = JSON.stringify({
                k : key.key,
                u : $scope.appConfig.dssExternalURL
            });
        });
    };
});

app.controller("MyProfileConnectionCredentialsController", function ($scope, $state, $location, translate, DataikuAPI, CreateModalFromTemplate, Dialogs, TopNav, WT1, ActivityIndicator, $window, CredentialDialogs) {
    $scope.hooks.save = null;
    $scope.hooks.isDirty = null;

    $scope.selectionConnections = { orderQuery: "connectionType", orderReversed: false };
    $scope.selectionPlugins = { orderQuery: ["connectionType", "connection"], orderReversed: false };

    $scope.refreshCredentials = function () {
        DataikuAPI.profile.listConnectionCredentials()
            .success(({ credentials }) => $scope.credentials = credentials)
            .error(setErrorInScope.bind($scope));
    };

    $scope.returnToPreviousPage = function() {
        window.close();
    }

    $scope.getPluginIcon = function(credential) {
        if ($scope.plugins[credential.pluginCredentialRequestInfo.pluginId]) {
            return $scope.plugins[credential.pluginCredentialRequestInfo.pluginId].installedDesc.desc.meta.icon;
        }
        return "fas fa-puzzle-piece";
    }

    $scope.editCredential = function(credential) {
        WT1.event("user-profile-edit-credentials", {});
        CredentialDialogs.enterCredential($scope, credential).then(function(redirect) {
            if(!redirect) {
                ActivityIndicator.success("Credential saved");
                $scope.refreshCredentials();
            }
        });
    };

    $scope.deleteConnectionCredential = function (credential) {
        WT1.event("user-profile-delete-credentials", {});
        Dialogs.confirm($scope, translate("PROFILE.CREDENTIALS.CONNECTIONS.REMOVE_DIALOG.TITLE", "Remove personal credential"), translate("PROFILE.CREDENTIALS.CONNECTIONS.REMOVE_DIALOG.MESSAGE", "Are you sure you want to remove this connection credential?")).then(function () {
            DataikuAPI.profile.setBasicConnectionCredential(credential.connection, null, null).success(function (data) {
                ActivityIndicator.success(translate("PROFILE.CREDENTIALS.CREDENTIAL_REMOVED", "Credential removed"))
                $scope.refreshCredentials();
            }).error(setErrorInScope.bind($scope));
        });
    };

    $scope.deletePluginCredential = function (pluginCredentialRequestInfo) {
        WT1.event("user-profile-delete-credentials", {});
        Dialogs.confirm($scope, translate("PROFILE.CREDENTIALS.PLUGINS.REMOVE_DIALOG.TITLE", "Remove plugin credential"), translate("PROFILE.CREDENTIALS.PLUGINS.REMOVE_DIALOG.MESSAGE", "Are you sure you want to remove this plugin credential?")).then(function () {
            DataikuAPI.profile.pluginCredentials.setBasicCredential(pluginCredentialRequestInfo.pluginId,
                pluginCredentialRequestInfo.paramSetId, pluginCredentialRequestInfo.presetId,
                pluginCredentialRequestInfo.paramName, null, null).success(function (data) {
                ActivityIndicator.success(translate("PROFILE.CREDENTIALS.CREDENTIAL_REMOVED", "Credential removed"))
                $scope.refreshCredentials();
            }).error(setErrorInScope.bind($scope));
        });
    };

    $scope.isConnectionCredential = function(credential) {
        return credential && ['CONNECTION', 'VIRTUAL_CONNECTION', 'DATABRICKS_INTEGRATION'].includes(credential.requestSource)
    }

    $scope.connectionCredentials = function () {
        if ($scope.credentials) {
            return $scope.credentials.filter($scope.isConnectionCredential);
        }
    }

    $scope.pluginCredentials = function () {
        if ($scope.credentials) {
            return $scope.credentials.filter(credential => credential.requestSource == 'PLUGIN');
        }
    }

    $scope.refreshCredentials();
});


app.controller('MyProfileExportController', function($scope, $rootScope, $state, translate, DataikuAPI, TopNav, WT1, Dialogs) {
    $scope.hooks.save = null;
    $scope.hooks.isDirty = null;

    function list() {
        DataikuAPI.exports.list().success(function(data) {
            $scope.userExports = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.downloadExport = function(exportId) {
        WT1.event("user-profile-download-export", {});
        downloadURL(DataikuAPI.exports.getDownloadURL(exportId));
    };

    $scope.deleteExport = function(exportId) {
        WT1.event("user-profile-delete-export", {});
        DataikuAPI.exports.remove(exportId).error(setErrorInScope.bind($scope)).then(list);
    };


    $scope.clearExports = function() {
        WT1.event("user-profile-clear-exports", {});
        Dialogs.confirm($scope, translate("PROFILE.EXPORTS.REMOVE_DIALOG.TITLE", "Remove all exports"), translate("PROFILE.EXPORTS.REMOVE_DIALOG.MESSAGE", "Are you sure you want to remove all finished exports?")).then(function() {
            DataikuAPI.exports.clear().error(setErrorInScope.bind($scope)).then(list);
        });
    };


    list();
});

app.controller('MyProfileAccountController', function($scope, $rootScope, translate, CodeMirrorSettingService, EXPORTED_TILES, $anchorScroll, $timeout, AvailableLanguages) {
    $timeout($anchorScroll);

    $scope.hooks.save = $scope.saveUserSettings;
    $scope.hooks.isDirty = $scope.isDirtyUserSettings;
    $scope.initCodeEditorSettings = function() {
        if (!$scope.userSettings.codeEditor) {
            $scope.userSettings.codeEditor = {
                theme: 'default',
                fontSize: 13,
                autoCloseBracket: true,
                keyMap: 'default'
            };
        }
    };
    $scope.initCodeEditorSettings();

    /*
     * Preview
     */

    $scope.initEditor = function() {
        $scope.sample = 'def process(dataset, partition_id):\n    parameter1_value = config.get("parameter1", None)\n    # return a dict of the metrics\' values\n    metric_values = {}\n    # type is inferred automatically from the object type\n    metric_values[\'name1\'] = 15\n    # type can be forced using the values in the MetricDataTypes enum\n    metric_values[\'name2\'] = (15, MetricDataTypes.STRING)\n    return metric_values\ntest';
        $scope.editorOptions = CodeMirrorSettingService.get("text/x-python", {onLoad: function(cm){$scope.codeMirror = cm;}});
    };
    $scope.initEditor();

    $scope.updatePreview = function() {
        $scope.codeMirror.setOption('theme', $scope.userSettings.codeEditor.theme);
        $scope.codeMirror.setOption('keyMap', $scope.userSettings.codeEditor.keyMap);
        $scope.codeMirror.setOption('matchBrackets', $scope.userSettings.codeEditor.matchBrackets);
        $($($scope.codeMirror.getTextArea()).siblings('.CodeMirror')[0]).css('font-size', $scope.userSettings.codeEditor.fontSize + 'px');
    };

    /*
     * Options
     */

    $scope.fontSizes = [];
    for (var i = 8; i<=30; i++) {
        $scope.fontSizes.push(i);
    }

    $scope.codeMirrorThemeList = [
        "default",
        "3024-day",
        "3024-night",
        "ambiance",
        "base16-dark",
        "base16-light",
        "blackboard",
        "cobalt",
        "eclipse",
        "elegant",
        "erlang-dark",
        "lesser-dark",
        "mbo",
        "mdn-like",
        "midnight",
        "monokai",
        "neat",
        "neo",
        "night",
        "paraiso-dark",
        "paraiso-light",
        "pastel-on-dark",
        "rubyblue",
        "solarized",
        "the-matrix",
        "tomorrow-night-eighties",
        "twilight",
        "vibrant-ink",
        "xq-dark",
        "xq-light"
    ];

    $scope.languageList = AvailableLanguages;

    // Take the potentially new tiles and override with the existing
    $scope.userSettings.home.rows = $scope.userSettings.home.rows || [];
    EXPORTED_TILES.forEach(tile => {
        const found = $scope.userSettings.home.rows.find(row => row.tileType == tile.type);
        if (found) {
            Object.assign(found, { name: translate(tile.translationId, tile.heading), tileType: tile.type });
        } else {
            $scope.userSettings.home.rows.push({ name: translate(tile.translationId, tile.heading), tileType: tile.type, visible: !tile.hiddenByDefault });
        }
    });
    $scope.oldUserSettings.home.rows = angular.copy($scope.userSettings.home.rows); // Trick for the save button
    $scope.nonVisibleRows = $scope.userSettings.home.rows.filter(r => r.visible === false);

    $scope.removeRow = (row, idx) => {
        row.visible = false;
        $scope.userSettings.home.rows.push($scope.userSettings.home.rows.splice(idx, 1)[0]);
        $scope.nonVisibleRows = $scope.userSettings.home.rows.filter(r => r.visible === false);
    };

    $scope.addNewRow = selectedOption => {
        if (!selectedOption) { return; }
        const found = $scope.userSettings.home.rows.find(r => r.tileType === selectedOption.tileType);
        const foundIndex = $scope.userSettings.home.rows.findIndex(r => r.tileType === selectedOption.tileType);
        if (found !== undefined) {
            found.visible = true;
            $scope.nonVisibleRows = $scope.userSettings.home.rows.filter(r => r.visible === false);
            $scope.userSettings.home.rows.splice(foundIndex, 1);
            const newPosition = $scope.userSettings.home.rows.length - $scope.nonVisibleRows.length;
            $scope.userSettings.home.rows.splice(newPosition, 0, found);
        }
    };

    $scope.treeOptions = {
        dropped: () => {
            // Rebuild position based on index of the list as the component has already updated their index
            $scope.userSettings.home.rows.forEach((r, index) => r.position = index);
        },
    }

    $scope.canEditEmailAndName = $rootScope.appConfig.enableEmailAndDisplayNameModification || $scope.appConfig.admin;

    $scope.showErrorEmailNotFilled = () => {
        // Look if user already fill his email
        if ($scope.profile.user.email) {
            return false;
        }

        // Look if user enabled some notifications
        const EMAILS_TYPE = ["mentionEmails", "discussionEmails", "offlineQueue", "digests", "accessRequestEmails", "grantedAccessEmails"];
        if ($scope.appConfig.admin) {
            EMAILS_TYPE.push("pluginRequestEmails", "codeEnvCreationRequestEmails", "instanceAccessRequestsEmails", "profileUpgradeRequestsEmails");
        } else {
            EMAILS_TYPE.push("grantedPluginRequestEmails", "grantedCodeEnvCreationRequestEmails");
        }
        return EMAILS_TYPE.some((type) => $scope.userSettings[type].enabled); // has any email notification enabled ?
    }
});

app.service("AchievementService", ['$rootScope', function () {
    const achievements = {
        LOL: {
            icon: 'icon-bell',
            title: "I lol on you",
            text: "You clicked the lol button"
        },
        // This is buggy
        // ACTIVE_5_MINUTES : {
        //     icon : 'icon-bell',
        //     title : "Eager discoverer",
        //     text : "Use Dataiku DSS for 5 minutes"
        // },
        // ACTIVE_30_MINUTES : {
        //     icon : 'icon-bell',
        //     title : "Wanna-be addict",
        //     text : "Use Dataiku DSS for 30 minutes"
        // },
        LETS_GET_COOKING: {
            icon: 'icon-book',
            title: "Let's get cooking",
            text: "Create your first recipe"
        },
        APPRENTICE_BARTENDER: {
            icon: 'icon-visual_prep_cleanse_recipe',
            title: "Apprentice bartender",
            text: "Create a Prepare recipe script with 5 steps"
        },
        NOT_QUITE_PARKINSON: {
            icon: 'icon-visual_prep_cleanse_recipe',
            title: "Eidetic memory",
            text: "Create a Prepare recipe with more than 20 steps"
        },
        ALL_ON_BOARD: {
            icon: 'icon-group',
            title: "All on board !",
            text: "Connect at least 3 times to DSS"
        },
        OH_NOES_JOB_FAILED: {
            icon: 'icon-frown',
            title: 'Oh noes ! Job failed !',
            text: 'Fail your first job'
        },
        MY_FIRST_JOB: {
            icon: 'icon-rocket',
            title: 'My first job !',
            text: 'Make your first job success'
        },
        STING_OF_THE_BEE: {
            icon: 'icon-code_hive_recipe',
            title: 'Sting of the bee',
            text: 'Fail a Hive job'
        },
        SLOW_SHAKE_1: {
            icon: 'icon-visual_prep_cleanse_recipe',
            title: 'I almost had to wait',
            text: 'Make a Prepare recipe script last more than 10 seconds'
        },
        SLOW_SHAKE_2: {
            icon: 'icon-visual_prep_cleanse_recipe',
            title: 'I guess I could grab a coffee',
            text: 'Make a Prepare recipe script last more than 30 seconds'
        },
        SLOW_SHAKE_3: {
            icon: 'icon-visual_prep_cleanse_recipe',
            title: "Almost fell asleep",
            text: 'Make a Prepare recipe script last more than 1 minute'
        },
        CODE_SAMPLE_WIZZARD: {
            icon: 'icon-code',
            title: "My first code sample !",
            text: 'Congrats ! Even better when it\'s shared ;) '
        },
        LIGHT_THE_SPARK: {
            icon: 'icon-magic',
            title: "Homo Erectus",
            text: "Light the first Spark"
        },
        IT_HAD_TO_BE_DONE: {
            icon: 'icon-list-ol',
            title: "It had to be done !",
            text: "Complete your first todo list"
        },
        TEAM_PLAYER: {
            icon: 'icon-thumbs-up-alt',
            title: "Teammate of the month",
            text: "Add description and tags to DSS objects"
        },
        COLORS_OF_THE_WIND: {
            icon: 'icon-sun',
            title: "Paint with all the colors of the wind",
            text: "Color the flow based on various metadata"
        },
        WE_ARE_ALCHEMISTS: {
            icon: 'icon-beaker',
            title: "It's science, we're alchemists",
            text: "Create a recipe from a notebook"
        }
    };

    this.getAchievement = function(id) {
        return achievements[id];
    };
}]);


app.directive('achievements', function(AchievementService, Notification, Logger, $http, $templateCache, $compile) {
    const templatePath = "/templates/achievement.html";

    return {
        restrict : 'A',
        link : function($scope, element, attrs) {
            const achievementUnlockedUnsubscribe = Notification.registerEvent("achievement-unlocked", function(evt, message) {
                Logger.info("Achievement unlocked", message);

                $http.get(templatePath, {cache: $templateCache}).then(function(response) {
                    const a = $('<div class="achievement" />').html(response.data).css('pointer-events', 'none');
                    const contents = a.contents();
                    $("body").append(a);
                    const newScope = $scope.$new();

                    const achievement = AchievementService.getAchievement(message.achievementId);
                    newScope.achievementId = message.achievementId;
                    newScope.achievementText = achievement.text;
                    newScope.achievementTitle = achievement.title;
                    newScope.achievementIcon = achievement.icon;

                    $compile(contents)(newScope);
                    window.setTimeout(function() {a.addClass("active");}, 800);
                    window.setTimeout(function() {a.addClass("gone");}, 5000);
                });
            });
            $scope.$on('$destroy', achievementUnlockedUnsubscribe);
        }
    };
});

app.factory('InterestWording', (Ng2InterestService) => {
    return Ng2InterestService.WORDING;
});

app.factory('WatchInterestState', (Ng2InterestService) => {
    const values = {
        YES: 'YES',
        SHALLOW: 'SHALLOW',
        ENO: 'ENO',
        INO: 'INO',
    };

    return {
        values,
        isShallowWatching: Ng2InterestService.isShallowWatching,
        isFullyWatching: Ng2InterestService.isFullyWatching,
        isWatching: Ng2InterestService.isWatching,
    };
});

app.service('InterestsService', function($state, WT1, DataikuAPI, WatchInterestState, ToolBridgeService) {
    this.watch = function(scope, taggableObjects, watch) {
        // Note: watch is not a boolean, please see WatchInterestState
        let w;
        if (watch === true) {
            w = WatchInterestState.values.YES;
        } else if (watch === false) {
            w = WatchInterestState.values.ENO;
        } else {
            w = watch;
        }

        WT1.event("watch-object", {watch: w, state: $state.current.name, objects: taggableObjects.length});
        return DataikuAPI.interests.watch(taggableObjects, w)
            .success(function() {ToolBridgeService.emitRefreshView('WatchView')})
            .error(setErrorInScope.bind(scope));
    };

    this.star = function(scope, taggableObjects, star) {
        WT1.event("star-object", {star: star, state: $state.current.name, objects: taggableObjects.length});
        return DataikuAPI.interests.star(taggableObjects, star)
            .success(function() {ToolBridgeService.emitRefreshView('WatchView')})
            .error(setErrorInScope.bind(scope));
    };

    this.getInterest = function(scope, user, objectType, projectKey, objectId, workspaceKey) {
        return DataikuAPI.interests.getForObject(user, objectType, projectKey, objectId, workspaceKey)
            .error(setErrorInScope.bind(scope));
    };
    
});


app.service('UserSettingsService', function (DataikuAPI, $rootScope, WT1, ActivityIndicator, $translate, $q, translate) {

    this.saveUserSettings = function (userSettings) {
        // We need to refresh the browser when user change languages so that constants that are loaded on bootstrap are correctly reloaded
        let refreshBrowserNeeded = false;
        const oldUserSettings = $rootScope.appConfig.userSettings;

        const result = DataikuAPI.profile.setUserSettings(userSettings).success(function () {
            refreshBrowserNeeded = oldUserSettings && oldUserSettings.uiLanguage !== userSettings.uiLanguage;

            if (!angular.equals((userSettings.home || {}).rows, (oldUserSettings.home || {}).rows)) {
                WT1.event("user-profile-set-home-row", {rows: (userSettings.home || {}).rows});
            }
            $rootScope.appConfig.userSettings = userSettings;

            if (refreshBrowserNeeded) {
                ActivityIndicator.success(translate("USER_SETTINGS.RELOADING_PAGE_IN_NEW_LANGUAGE", "Reloading page in new language. Please wait..."));
                location.reload(true);
            } else if (userSettings.uiLanguage) {
                $translate.use(userSettings.uiLanguage);
            }
            return $q.resolve();
        })
        .error((error) => {
            return $q.reject(error);
        });

        WT1.event("user-profile-save-settings", {
            digests: userSettings.digests.enabled,
            mentionEmails: userSettings.mentionEmails.enabled,
            offlineQueue: userSettings.offlineQueue.enabled,
            disableFlowZoomTracking: userSettings.disableFlowZoomTracking,
            frontendNotifications: userSettings.frontendNotifications,
        });

        return result;
    };

});

    app.component('userSettingsPageSectionForm', {
        templateUrl: '/templates/profile/page-section-form.html',
        bindings: {
            pageSectionSettings: '<',
        },
        controller: function (DashboardPageSectionFormatService, $scope, translate, $timeout) {
            const ctrl = this;

            ctrl.pageSectionFormats = [
                { id: 'FIT_TO_PAGE', label: translate(`DASHBOARD_SETTINGS.ZOOM_MODE.FORM.FIXED_ASPECT_RATIO.LABEL`, 'Fixed aspect ratio') },
                { id: 'FIT_TO_WIDTH', label: translate(`DASHBOARD_SETTINGS.ZOOM_MODE.FORM.FIT_TO_WIDTH.LABEL`, 'Fit to width') },
            ];

            // The Settings page is considered "modified" if its JSON differs from the original JSON.
            // Therefore, we need to reuse the initial pageSectionFormat if it matches the one selected in the frontend 
            // (the frontend’s pageSectionFormat is slightly different from the backend’s).
            // Ideally, the “modified” state of the Settings page should have been more granular than a simple JSON comparison.
            ctrl.initialPageSectionFormatInput = null;

            $scope.$watch('$ctrl.pageSectionSettings.pageSectionFormat', (newPageSectionFormat) => {
                if (newPageSectionFormat == undefined) {
                    return;
                }

                if (ctrl.initialPageSectionFormatInput == null) {
                    ctrl.initialPageSectionFormatInput = DashboardPageSectionFormatService.createPageSectionFormat({
                        ...newPageSectionFormat,
                    });
                }
            });

            ctrl.onPageSectionFormatUpdated = ($event) => {

                const formatUpdated = ctrl.initialPageSectionFormatInput.formatName !== $event.formatName;
                const orientationUpdated = ctrl.initialPageSectionFormatInput.orientation !== $event.orientation;
                const customSizeUpdated = (
                    !formatUpdated
                    && $event.formatName === 'CUSTOM'
                    && (
                        ctrl.initialPageSectionFormatInput.xInPx !== $event.xInPx
                        || ctrl.initialPageSectionFormatInput.yInPx !== $event.yInPx
                    )
                );

                const pageSectionFormatHasChanged = (
                    ctrl.initialPageSectionFormatInput === null
                    || formatUpdated
                    || customSizeUpdated
                    || orientationUpdated
                );

                if (pageSectionFormatHasChanged) {
                    $timeout(() => {
                        ctrl.pageSectionSettings.pageSectionFormat = $event;
                    });
                }
            };
        }
    });

}());
