/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.security.auth;

import com.dataiku.dip.coremodel.InfoMessage;
import com.dataiku.dip.dao.DkuGroup;
import com.dataiku.dip.dao.DkuUser;
import com.dataiku.dip.exceptions.CodedException;
import com.dataiku.dip.exceptions.DKUSecurityException;
import com.dataiku.dip.exceptions.UnauthorizedException;
import com.dataiku.dip.futures.FutureProgress;
import com.dataiku.dip.futures.FutureProgressState;
import com.dataiku.dip.security.PasswordEncryptionService;
import com.dataiku.dip.security.SecurityCodes;
import com.dataiku.dip.security.audit.AuditTrailService;
import com.dataiku.dip.security.auth.AuthSyncSettings;
import com.dataiku.dip.security.auth.ExternalUser;
import com.dataiku.dip.security.auth.FetchedUsersResponse;
import com.dataiku.dip.security.auth.GroupProfile;
import com.dataiku.dip.security.auth.ServerAuthenticationFailure;
import com.dataiku.dip.security.auth.UserAttributes;
import com.dataiku.dip.security.auth.UserAuthenticationException;
import com.dataiku.dip.security.auth.UserCredentialsAuthenticator;
import com.dataiku.dip.security.auth.UserDiff;
import com.dataiku.dip.security.auth.UserIdentity;
import com.dataiku.dip.security.auth.UserNotFoundException;
import com.dataiku.dip.security.auth.UserNotInAuthorizedGroupsException;
import com.dataiku.dip.security.auth.UserQueryFilter;
import com.dataiku.dip.security.auth.UserRemappingRule;
import com.dataiku.dip.security.auth.UserSourceType;
import com.dataiku.dip.security.auth.UserSupplier;
import com.dataiku.dip.security.auth.UserSupplierSettings;
import com.dataiku.dip.server.services.DkuUsersService;
import com.dataiku.dip.server.services.TransactionService;
import com.dataiku.dip.server.services.licensing.DkuLicenseEnforcementService;
import com.dataiku.dip.server.services.licensing.LimitsStatusComputer;
import com.dataiku.dip.transactions.ifaces.RWTransaction;
import com.dataiku.dip.transactions.ifaces.Transaction;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.ExceptionUtils;
import com.dataiku.dip.utils.JSON;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.Validate;

public abstract class UserAuthenticationService {
    private static final DKULogger logger = DKULogger.getLogger((String)"dku.auth");
    private final Map<UserSourceType, UserCredentialsAuthenticator> userAuthenticators;
    private final Map<UserSourceType, UserSupplier> userSuppliers;
    private final DkuUsersService usersService;
    private final TransactionService transactionService;
    private final DkuLicenseEnforcementService licenseEnforcementService;
    private final AuditTrailService auditTrailService;
    private final PasswordEncryptionService symetricCryptoService;
    protected boolean isEmailSafe = false;
    protected AuthSyncSettings authSyncSettings;

    public UserAuthenticationService(AuditTrailService auditTrailService, DkuUsersService usersService, TransactionService transactionService, List<UserCredentialsAuthenticator> userCredentialsAuthenticators, List<UserSupplier> userSuppliers, DkuLicenseEnforcementService licenseEnforcementService, PasswordEncryptionService symetricCryptoService) {
        this.auditTrailService = auditTrailService;
        this.usersService = usersService;
        this.transactionService = transactionService;
        this.userAuthenticators = userCredentialsAuthenticators.stream().collect(Collectors.toMap(u -> u.getAuthenticatorSettings().getUserSourceType(), Function.identity()));
        this.userSuppliers = userSuppliers.stream().collect(Collectors.toMap(u -> u.getUserSupplierSettings().getUserSourceType(), Function.identity()));
        this.licenseEnforcementService = licenseEnforcementService;
        this.symetricCryptoService = symetricCryptoService;
    }

    public DkuUser authenticateWithPassword(String inputLogin, String password) throws ServerAuthenticationFailure, UnauthorizedException, UserDisabledException, UserAuthenticationException {
        DkuUser dkuUser = this.getUserWithCaseSensitivenessRule(inputLogin);
        if (dkuUser != null) {
            logger.debugV("Found already existing user %s for login '%s'", new Object[]{dkuUser, dkuUser.getLogin()});
        } else {
            logger.debugV("No DSS user with login '%s'", new Object[]{inputLogin});
        }
        List authenticators = this.userAuthenticators.values().stream().filter(authenticator -> dkuUser == null || authenticator.getAuthenticatorSettings().getUserSourceType() == dkuUser.getUserSourceType()).filter(u -> u.getAuthenticatorSettings().isEnabled()).filter(u -> u.getAuthenticatorSettings().isAuthenticationEnabled()).sorted(Comparator.comparingInt(u -> u.getAuthenticatorSettings().getOrder())).collect(Collectors.toList());
        for (UserCredentialsAuthenticator authenticator2 : authenticators) {
            try {
                UserIdentity userIdentity = authenticator2.authenticate(inputLogin, password);
                Set<String> authorizedGroups = authenticator2.getAuthenticatorSettings().getAuthorizedGroups();
                if (!authorizedGroups.isEmpty() && Collections.disjoint(authorizedGroups, userIdentity.groupNames)) {
                    if (dkuUser != null) {
                        this.applyNotInAuthorizedGroupsAction(dkuUser);
                        this.syncOrProvisionFromTrustedSourceAtLoginTime(userIdentity, dkuUser);
                    }
                    throw new UserNotInAuthorizedGroupsException(String.format("successfully authenticated the user but none of the user groups (%s) is within authorized groups (%s)", String.join((CharSequence)", ", userIdentity.groupNames), String.join((CharSequence)", ", authorizedGroups)));
                }
                userIdentity.login = inputLogin;
                logger.infoV("Authentication success for login '%s' with authenticator of source '%s'", new Object[]{inputLogin, authenticator2.getAuthenticatorSettings().getUserSourceType()});
                return this.syncOrProvisionFromTrustedSourceAtLoginTime(userIdentity, dkuUser);
            }
            catch (UserAuthenticationException e) {
                logger.infoV("Authenticator for user type '%s' could not authenticate login '%s', trying next authentication method. Reason: %s", new Object[]{authenticator2.getAuthenticatorSettings().getUserSourceType(), inputLogin, e.getMessage()});
            }
            catch (UserNotFoundException e) {
                if (dkuUser != null) {
                    logger.warnV("Authenticator for user type '%s' could not find user '%s', applying missing user action", new Object[]{authenticator2.getAuthenticatorSettings().getUserSourceType(), inputLogin});
                    this.applyMissingUserAction(dkuUser, authenticator2.getAuthenticatorSettings().getUserSourceType());
                    throw new UserAuthenticationException(String.format("Authenticator for user type '%s' could not find user '%s'", new Object[]{authenticator2.getAuthenticatorSettings().getUserSourceType(), inputLogin}), (Throwable)((Object)e));
                }
                logger.infoV("Authenticator for user type '%s' could not find user '%s', trying next authentication method. Reason: %s", new Object[]{authenticator2.getAuthenticatorSettings().getUserSourceType(), inputLogin, e.getMessage()});
            }
            catch (ServerAuthenticationFailure e) {
                logger.errorV((Throwable)((Object)e), "Authenticator for user type '%s' wasn't operational, trying next authentication method", new Object[]{authenticator2.getAuthenticatorSettings().getUserSourceType()});
            }
            catch (UserNotInAuthorizedGroupsException e) {
                logger.infoV("Authenticator for user type '%s' found user '%s' but is not in the authorized groups. Reason: %s", new Object[]{authenticator2.getAuthenticatorSettings().getUserSourceType(), inputLogin, e.getMessage()});
                throw e;
            }
            catch (LimitsStatusComputer.LicenseLimitException e) {
                logger.warnV((Throwable)e, "Could not provision %s user '%s' due to license limit: %s", new Object[]{authenticator2.getAuthenticatorSettings().getUserSourceType(), inputLogin, e.getMessage()});
            }
        }
        throw new UserAuthenticationException("No authenticator was able to authenticate user '" + inputLogin + "'");
    }

    public DkuUser syncOrProvisionFromTrustedSourceAtLoginTime(UserIdentity userIdentity, DkuUser dkuUser) throws UserNotFoundException, ServerAuthenticationFailure, UserDisabledException, UnauthorizedException {
        if (dkuUser != null) {
            logger.debug((Object)("Found already existing user. User=" + String.valueOf(dkuUser)));
            if (dkuUser.getUserSourceType() != UserSourceType.LOCAL) {
                this.readAttributesAndSyncAtLoginTime(userIdentity, dkuUser);
            }
        } else {
            dkuUser = this.readAttributesAndProvisionAtLoginTime(userIdentity);
        }
        if (!dkuUser.isEnabled()) {
            throw new UserDisabledException("User '" + userIdentity.login + "' is disabled");
        }
        return dkuUser;
    }

    public Set<String> getFetchableSourceTypes() {
        return this.userSuppliers.values().stream().filter(userSupplier -> userSupplier.getUserSupplierSettings().isEnabled()).filter(UserSupplier::canFetchUsers).filter(userSupplier -> userSupplier.getUserSupplierSettings().isOnDemandUsersProvisioningEnabled()).map(userSupplier -> userSupplier.getUserSupplierSettings().getUserSourceType().name()).collect(Collectors.toSet());
    }

    public Set<String> getSuppliersForProvisioningAtLoginTime() {
        return this.userSuppliers.values().stream().filter(userSupplier -> userSupplier.getUserSupplierSettings().isEnabled()).filter(userSupplier -> userSupplier.getUserSupplierSettings().isUsersAutoProvisioningAtLoginTimeEnabled()).map(userSupplier -> userSupplier.getUserSupplierSettings().getUserSourceType().name()).collect(Collectors.toSet());
    }

    public InfoMessage.InfoMessages explicitOnDemandSync(List<String> logins) throws IOException, ServerAuthenticationFailure {
        InfoMessage.InfoMessages ret = new InfoMessage.InfoMessages();
        ArrayList<DkuUser> usersToSync = new ArrayList<DkuUser>();
        logger.info((Object)("Starting explicit on-demand sync of users: " + JSON.log(logins)));
        if (logins == null) {
            Iterator allUsers;
            try (Object t = this.transactionService.beginRead();){
                allUsers = this.usersService.listUsers();
            }
            t = allUsers.iterator();
            while (t.hasNext()) {
                user = (DkuUser)t.next();
                userSupplier = this.userSuppliers.get((Object)user.getUserSourceType());
                if (!user.isEnabled() || userSupplier == null || !userSupplier.getUserSupplierSettings().isEnabled() || !userSupplier.canSyncOnDemand() || !userSupplier.getUserSupplierSettings().isOnDemandUsersSyncEnabled()) continue;
                usersToSync.add(user);
            }
        } else {
            for (String login : logins) {
                user = null;
                try (Transaction t = this.transactionService.beginRead();){
                    user = this.usersService.getUserWithCaseSensitiveRule(login);
                }
                catch (Exception e) {
                    logger.warn((Object)"Failed to read users list", (Throwable)e);
                }
                if (user == null) {
                    ret.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_SYNC_NOT_ELIGIBLE, "Not syncing user '%s': Unknown user", new Object[]{login});
                    continue;
                }
                if (!user.isEnabled()) {
                    ret.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_SYNC_NOT_ELIGIBLE, "Not syncing user '%s': User is disabled", new Object[]{login});
                    continue;
                }
                userSupplier = this.userSuppliers.get((Object)user.getUserSourceType());
                if (userSupplier == null) {
                    ret.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_SYNC_NOT_ELIGIBLE, "Not syncing user '%s': supplier for source type %s not found", new Object[]{login, user.getUserSourceType()});
                    continue;
                }
                if (!userSupplier.getUserSupplierSettings().isEnabled()) {
                    ret.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_SYNC_NOT_ELIGIBLE, "Not syncing user '%s': supplier for source type %s disabled", new Object[]{login, user.getUserSourceType()});
                    continue;
                }
                if (!userSupplier.canSyncOnDemand()) {
                    ret.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_SYNC_NOT_ELIGIBLE, "Not syncing user '%s': supplier for source type %s cannot sync on demand", new Object[]{login, user.getUserSourceType()});
                    continue;
                }
                if (!userSupplier.getUserSupplierSettings().isOnDemandUsersSyncEnabled()) {
                    ret.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_SYNC_NOT_ELIGIBLE, "Not syncing user '%s': supplier for source type %s does not allow on-demand sync", new Object[]{login, user.getUserSourceType()});
                    continue;
                }
                usersToSync.add(user);
            }
        }
        try {
            FutureProgress.pushState("Syncing users", usersToSync.size(), FutureProgressState.StateUnit.RECORDS);
            for (DkuUser userToSync : usersToSync) {
                UserSupplier userSupplier = this.userSuppliers.get((Object)userToSync.getUserSourceType());
                try {
                    ret.mergeFrom(this.explicitOnDemandSyncSingleUserInternal(userToSync, userSupplier));
                }
                catch (ServerAuthenticationFailure e) {
                    logger.warnV((Throwable)((Object)e), "Unknown error while syncing user %s", new Object[]{userToSync.getLogin()});
                    ret.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_SYNC_FAILED, "Unknown error while syncing user %s from source %s: %s", new Object[]{userToSync.getLogin(), userSupplier.getUserSupplierSettings().getUserSourceType(), ExceptionUtils.getMessageWithCauses((Throwable)((Object)e))});
                }
                FutureProgress.incrementState(1.0);
            }
        }
        catch (InterruptedException e) {
            ret.withWarningV((InfoMessage.MessageCode)SecurityCodes.WARN_SECURITY_SUPPLIER_SYNC_ABORTED, "Operation aborted, %d/%d users have been synchronized", new Object[]{(int)FutureProgress.getState().cur, (int)FutureProgress.getState().target});
        }
        return ret;
    }

    public void syncUserBeforeAction(String username) throws ServerAuthenticationFailure, IOException {
        DkuUser userToSync;
        try (Transaction t = this.transactionService.beginRead();){
            userToSync = this.usersService.getUserWithCaseSensitiveRule(username);
        }
        Validate.notNull((Object)userToSync, (String)"Could not find user '%' to sync", (Object[])new Object[]{username});
        UserSupplier userSupplier = this.userSuppliers.get((Object)userToSync.getUserSourceType());
        if (userSupplier == null) {
            logger.debug((Object)("Did not find a user supplier for source type '" + String.valueOf((Object)userToSync.getUserSourceType()) + "', bypassing user sync"));
            return;
        }
        if (!userSupplier.getUserSupplierSettings().isEnabled()) {
            throw new ServerAuthenticationFailure("User source type is '" + String.valueOf((Object)userToSync.getUserSourceType()) + "' but supplier '" + String.valueOf((Object)userToSync.getUserSourceType()) + "' is disabled.");
        }
        if (!userSupplier.canSyncOnDemand()) {
            return;
        }
        if (!userSupplier.getUserSupplierSettings().isOnDemandUsersSyncEnabled()) {
            logger.warn((Object)("User source type is '" + String.valueOf((Object)userToSync.getUserSourceType()) + "' but sync on demand is explicitly disabled."));
        } else {
            this.explicitOnDemandSyncSingleUserInternal(userToSync, userSupplier);
        }
    }

    private InfoMessage.InfoMessages explicitOnDemandSyncSingleUserInternal(DkuUser userToSync, UserSupplier userSupplier) throws ServerAuthenticationFailure {
        assert (userSupplier != null);
        assert (userSupplier.getUserSupplierSettings().isEnabled());
        assert (userSupplier.canSyncOnDemand());
        assert (userSupplier.getUserSupplierSettings().isOnDemandUsersSyncEnabled());
        InfoMessage.InfoMessages infoMessages = new InfoMessage.InfoMessages();
        try {
            UserIdentity identityToSync = new UserIdentity(null, userToSync.getLogin());
            if (this.isEmailSafe) {
                identityToSync.email = userToSync.getEmail();
            }
            UserAttributes userAttributes = userSupplier.getUserAttributes(identityToSync);
            logger.infoV("Supplier for user type '%s' retrieved some user attributes %s", new Object[]{userSupplier.getUserSupplierSettings().getUserSourceType(), userAttributes});
            Set<String> authorizedGroups = userSupplier.getUserSupplierSettings().getAuthorizedGroups();
            this.auditTrailService.generic("security-admin-user-on-demand-sync-from-source").with("login", userToSync.getLogin()).emit();
            this.syncLocalUser(userSupplier.getUserSupplierSettings(), userToSync.getLogin(), userAttributes, userSupplier.getDefaultProfile());
            if (!authorizedGroups.isEmpty() && Collections.disjoint(authorizedGroups, userAttributes.sourceGroupNames)) {
                logger.warnV("User %s is not authorized anymore by authorized groups (%s)", new Object[]{userToSync.getLogin(), JSON.log(authorizedGroups)});
                switch (this.authSyncSettings.notInAuthorizedGroupsAction) {
                    case DISABLE_USER: {
                        infoMessages.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_SYNC_USER_NOT_AUTHORIZED_ANYMORE, "User '%s' not authorized anymore by authorized groups, disabling it", new Object[]{userToSync.getLogin()});
                        this.disableDkuUser(userToSync.getLogin(), "not authorized anymore during on-demand resync");
                        this.auditTrailService.generic("security-admin-user-disable").with("login", userToSync.getLogin()).with("reason", "on-demand-sync-not-authorized-anymore").emit();
                        break;
                    }
                    case WARN: {
                        infoMessages.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_SYNC_USER_NOT_AUTHORIZED_ANYMORE, "User '%s' not authorized anymore by authorized groups, not syncing it", new Object[]{userToSync.getLogin()});
                    }
                }
            }
        }
        catch (UserNotFoundException e) {
            logger.warnV((Throwable)((Object)e), "User %s does not exist anymore in source", new Object[]{userToSync.getLogin()});
            switch (this.authSyncSettings.missingUserAction) {
                case DISABLE_USER: {
                    infoMessages.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_SYNC_USER_NOT_FOUND, "User '%s' does not exist anymore in source (%s), disabling it", new Object[]{userToSync.getLogin(), userSupplier.getUserSupplierSettings().getUserSourceType()});
                    this.disableDkuUser(userToSync.getLogin(), "not found anymore in source during on-demand resync");
                    this.auditTrailService.generic("security-admin-user-disable").with("login", userToSync.getLogin()).with("reason", "on-demand-sync-not-found-anymore").emit();
                    break;
                }
                case WARN: {
                    infoMessages.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_SYNC_USER_NOT_FOUND, "User '%s' does not exist anymore in source (%s), not syncing it", new Object[]{userToSync.getLogin(), userSupplier.getUserSupplierSettings().getUserSourceType()});
                }
            }
        }
        return infoMessages;
    }

    public DkuUser authenticateWithNoLoginAuthMethod(boolean noLoginMode, String noLoginImpersonateUserName) throws ServerAuthenticationFailure {
        if (!noLoginMode) {
            throw new SecurityException("No-login mode is disabled");
        }
        DkuUser user = this.getUserWithCaseSensitivenessRule(noLoginImpersonateUserName);
        if (user == null) {
            throw new SecurityException("Invalid impersonation user");
        }
        return user;
    }

    private DkuUser syncLocalUser(UserSupplierSettings userSupplierSettings, String userLogin, UserAttributes userAttributes, String defaultProfile) throws ServerAuthenticationFailure {
        try (RWTransaction t = this.transactionService.beginWriteAsDSS();){
            DkuUser syncedUser = this.getUserWithCaseSensitivenessRule(userLogin);
            try {
                if (this.patchUser(syncedUser, userSupplierSettings, userAttributes, defaultProfile)) {
                    UserDiff userDiff = this.usersService.updateUser(syncedUser);
                    t.commit(String.format("Sync local user '%s'", userLogin));
                    this.auditTrailService.generic("security-admin-user-patch").with("login", syncedUser.getLogin()).withAll(userDiff.getDiff()).emit();
                }
            }
            catch (LimitsStatusComputer.LicenseLimitException e) {
                logger.warnV((Throwable)e, "Could not sync user '%s' with new profile %s due to quota limit: %s", new Object[]{userLogin, syncedUser.getUserProfile(), e.getMessage()});
                throw new ServerAuthenticationFailure(String.format("Could not provision %s user '%s' due to quota limit", userLogin, syncedUser.getUserProfile()), e);
            }
            catch (CodedException | IOException e) {
                throw new ServerAuthenticationFailure("Could not sync local user", e);
            }
            DkuUser dkuUser = syncedUser;
            return dkuUser;
        }
    }

    private void disableDkuUser(String userLogin, String reason) throws ServerAuthenticationFailure {
        try (RWTransaction t = this.transactionService.beginWriteAsDSS();){
            this.usersService.disableDkuUser(userLogin);
            t.commit(String.format("Disabled user '%s':  %s", userLogin, reason));
        }
        catch (CodedException | LimitsStatusComputer.LicenseLimitException | IOException e) {
            throw new ServerAuthenticationFailure("Could not disable user", e);
        }
    }

    private boolean patchUser(DkuUser dkuUser, UserSupplierSettings userSupplierSettings, UserAttributes userAttributes, String defaultProfile) throws IOException {
        boolean modified = false;
        logger.infoV("Patching user '%s' with user attributes %s", new Object[]{dkuUser, userAttributes});
        if (!this.authSyncSettings.syncUserDisplayName) {
            logger.info((Object)"Syncing display name is not enabled");
        } else if (StringUtils.isNotEmpty((String)userAttributes.displayName) && !userAttributes.displayName.equals(dkuUser.getDisplayName())) {
            logger.infoV("Syncing user '%s' display name from '%s' to '%s'", new Object[]{dkuUser.getLogin(), dkuUser.getDisplayName(), userAttributes.displayName});
            dkuUser.setDisplayName(userAttributes.displayName);
            modified = true;
        } else {
            logger.infoV("No update required on user '%s' display name '%s'", new Object[]{dkuUser.getLogin(), dkuUser.getDisplayName()});
        }
        if (!this.authSyncSettings.syncUserEmail) {
            logger.info((Object)"Syncing email is not enabled");
        } else if (StringUtils.isNotEmpty((String)userAttributes.email) && !userAttributes.email.equals(dkuUser.getEmail())) {
            logger.infoV("Syncing user '%s' email from '%s' to '%s'", new Object[]{dkuUser.getLogin(), dkuUser.getEmail(), userAttributes.email});
            dkuUser.setEmail(userAttributes.email);
            modified = true;
        } else {
            logger.infoV("No update required on user '%s' email '%s'", new Object[]{dkuUser.getLogin(), dkuUser.getEmail()});
        }
        if (!this.authSyncSettings.syncUserGroups) {
            logger.info((Object)"Syncing user groups is not enabled");
            userAttributes.dkuGroupNames = new HashSet<String>(dkuUser.getGroups());
        } else {
            List<? extends DkuGroup> allDkuGroups;
            try (Transaction t = this.transactionService.retrieveOrBeginRead();){
                Set<String> dkuGroups = this.toDkuGroups(userAttributes, dkuUser.getUserSourceType());
                userAttributes.dkuGroupNames.addAll(dkuGroups);
                allDkuGroups = this.usersService.listGroups();
            }
            Set groupsWithMapping = allDkuGroups.stream().filter(g -> g.getUserSourceType() == dkuUser.getUserSourceType()).map(DkuGroup::getName).collect(Collectors.toSet());
            Set<String> userGroupsWithMapping = dkuUser.getGroups().stream().filter(groupsWithMapping::contains).collect(Collectors.toSet());
            if (!userGroupsWithMapping.equals(userAttributes.dkuGroupNames)) {
                userGroupsWithMapping.forEach(dkuUser::removeGroupMembership);
                if (logger.isInfoEnabled()) {
                    logger.infoV("Syncing user '%s' groups with mapping from [%s] to [%s]", new Object[]{dkuUser.getLogin(), String.join((CharSequence)",", userGroupsWithMapping), String.join((CharSequence)",", userAttributes.dkuGroupNames)});
                }
                userAttributes.dkuGroupNames.forEach(dkuUser::addGroupMembership);
                modified = true;
            } else if (logger.isInfoEnabled()) {
                logger.infoV("No update required on user '%s' groups with mapping '%s'", new Object[]{dkuUser.getLogin(), String.join((CharSequence)",", userGroupsWithMapping)});
            }
        }
        if (!this.authSyncSettings.syncUserProfile) {
            logger.info((Object)"Syncing user profile is not enabled");
        } else {
            String profile = this.getGrantedProfile(userSupplierSettings, userAttributes, defaultProfile);
            if (profile != null && !profile.equals(dkuUser.getUserProfile())) {
                logger.infoV("Syncing user '%s' profile from '%s' to '%s'", new Object[]{dkuUser.getLogin(), dkuUser.getUserProfile(), profile});
                dkuUser.setUserProfile(profile);
                modified = true;
            } else {
                logger.infoV("No update required on user '%s' profile '%s'", new Object[]{dkuUser.getLogin(), dkuUser.getUserProfile()});
            }
        }
        if (!userAttributes.secrets.isEmpty()) {
            dkuUser.upsertSecrets(this.symetricCryptoService, userAttributes.secrets);
            modified = true;
        }
        return modified;
    }

    public Set<String> toDkuGroups(UserAttributes userAttributes, UserSourceType userSourceType) throws IOException {
        return UserAuthenticationService.toDkuGroups(this.usersService.listGroups(), userAttributes, userSourceType);
    }

    private static Set<String> toDkuGroups(List<? extends DkuGroup> dkuGroups, UserAttributes userAttributes, UserSourceType userSourceType) {
        return dkuGroups.stream().filter(g -> g.getUserSourceType() == userSourceType).filter(g -> {
            Set<String> externalGroupNames = g.getExternalGroupNames();
            externalGroupNames.retainAll(userAttributes.sourceGroupNames);
            return !externalGroupNames.isEmpty();
        }).map(DkuGroup::getName).collect(Collectors.toSet());
    }

    private DkuUser createUser(String login, UserAttributes userAttributes, UserSourceType userSourceType, String grantedProfile) throws ServerAuthenticationFailure, UnauthorizedException {
        try {
            if (!this.usersService.isValidLogin(login)) {
                throw new CodedException((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_CRUD_INVALID_SETTINGS, "Login is invalid or contains special characters");
            }
            userAttributes.dkuGroupNames.addAll(this.toDkuGroups(userAttributes, userSourceType));
            DkuUser provisionedUser = this.usersService.createForeignUser(userSourceType, login, StringUtils.isEmpty((String)userAttributes.displayName) ? login : userAttributes.displayName, userAttributes.email, userAttributes.dkuGroupNames, grantedProfile, userAttributes.secrets);
            logger.infoV("Authenticator for user type '%s' provisioned user %s", new Object[]{userSourceType, provisionedUser});
            return provisionedUser;
        }
        catch (LimitsStatusComputer.LicenseLimitException e) {
            logger.errorV("Could not provision user '%s' due to quota limit: %s", new Object[]{login, e.getMessage()});
            throw new UnauthorizedException(String.format("Could not provision user '%s' due to quota limit", login), "license-limit-quota", e);
        }
        catch (CodedException | IOException e) {
            throw new ServerAuthenticationFailure(e.getMessage(), e);
        }
    }

    private String getGrantedProfile(UserSupplierSettings userSupplierSettings, UserAttributes userAttributes, String defaultProfile) {
        LimitsStatusComputer.LicenseLimitsStatus ls;
        Set profilesGranted;
        try (Transaction t = this.transactionService.retrieveOrBeginRead();){
            Map<String, String> profileMappings = userSupplierSettings.getProfileMappings().stream().filter(groupProfile -> {
                if (StringUtils.isEmpty((String)groupProfile.group)) {
                    logger.warnV("One group to profile mapping for supplier '%s' has an empty group, rule is ignored", new Object[]{userSupplierSettings.getUserSourceType()});
                    return false;
                }
                if (StringUtils.isEmpty((String)groupProfile.profile)) {
                    logger.warnV("One group to profile mapping for supplier '%s' has an empty profile, rule is ignored", new Object[]{userSupplierSettings.getUserSourceType()});
                    return false;
                }
                return true;
            }).collect(Collectors.toMap(GroupProfile::getGroup, GroupProfile::getProfile, (existing, newest) -> {
                logger.warnV("Duplicate group \u2192 profile mapping for %s user supplier", new Object[]{userSupplierSettings.getUserSourceType()});
                return newest;
            }));
            profilesGranted = userAttributes.sourceGroupNames.stream().map(profileMappings::get).filter(Objects::nonNull).collect(Collectors.toSet());
            ls = this.licenseEnforcementService.getLimitsStatus();
        }
        for (LimitsStatusComputer.LicensedProfile licensedProfile : ls.licensedProfiles.values()) {
            if (!profilesGranted.contains(licensedProfile.profile)) continue;
            return licensedProfile.profile;
        }
        if (StringUtils.isNotEmpty((String)defaultProfile)) {
            for (LimitsStatusComputer.LicensedProfile licensedProfile : ls.licensedProfiles.values()) {
                if (!defaultProfile.equals(licensedProfile.profile)) continue;
                return licensedProfile.profile;
            }
        }
        if (!profilesGranted.isEmpty()) {
            logger.warn((Object)("Did not find granted profile: " + JSON.log(profilesGranted)));
        }
        return ls.fallbackProfile;
    }

    private void readAttributesAndSyncAtLoginTime(UserIdentity userIdentity, DkuUser dkuUser) throws ServerAuthenticationFailure, UserNotFoundException, UserNotInAuthorizedGroupsException {
        Set<String> authorizedGroups;
        UserAttributes userAttributes;
        UserSupplier userSupplier = this.userSuppliers.get((Object)dkuUser.getUserSourceType());
        if (!userSupplier.getUserSupplierSettings().isEnabled()) {
            throw new ServerAuthenticationFailure("User source type is '" + String.valueOf((Object)dkuUser.getUserSourceType()) + "' but supplier '" + String.valueOf((Object)dkuUser.getUserSourceType()) + "' is disabled.");
        }
        try {
            userAttributes = userSupplier.getUserAttributes(userIdentity);
        }
        catch (UserNotFoundException e) {
            this.applyMissingUserAction(dkuUser, userSupplier.getUserSupplierSettings().getUserSourceType());
            throw e;
        }
        logger.infoV("Supplier for user type '%s' retrieved some user attributes %s", new Object[]{userSupplier.getUserSupplierSettings().getUserSourceType(), userAttributes});
        if (userSupplier.getUserSupplierSettings().isUsersAutoSyncAtLoginTimeEnabled()) {
            DkuUser syncedUser = this.syncLocalUser(userSupplier.getUserSupplierSettings(), dkuUser.getLogin(), userAttributes, userSupplier.getDefaultProfile());
            logger.infoV("Supplier for user type '%s' synced user %s", new Object[]{userSupplier.getUserSupplierSettings().getUserSourceType(), syncedUser});
        }
        if (!(authorizedGroups = userSupplier.getUserSupplierSettings().getAuthorizedGroups()).isEmpty() && Collections.disjoint(authorizedGroups, userAttributes.sourceGroupNames)) {
            this.applyNotInAuthorizedGroupsAction(dkuUser);
            throw new UserNotInAuthorizedGroupsException(String.format("none of the user groups (%s) is within authorized groups (%s)", String.join((CharSequence)", ", userIdentity.groupNames), String.join((CharSequence)", ", authorizedGroups)));
        }
    }

    private void applyMissingUserAction(DkuUser dkuUser, UserSourceType userSourceType) throws ServerAuthenticationFailure {
        switch (this.authSyncSettings.missingUserAction) {
            case DISABLE_USER: {
                logger.warnV("User '%s' does not exist anymore in source (%s), disabling it", new Object[]{dkuUser.getLogin(), userSourceType});
                this.disableDkuUser(dkuUser.getLogin(), "not found anymore in source during login time resync");
                this.auditTrailService.generic("security-admin-user-disable").with("login", dkuUser.getLogin()).with("reason", "login-time-sync-not-found-anymore").emit();
                break;
            }
            case WARN: {
                logger.warnV("User '%s' does not exist anymore in source (%s), not syncing it", new Object[]{dkuUser.getLogin(), userSourceType});
            }
        }
    }

    private void applyNotInAuthorizedGroupsAction(DkuUser dkuUser) throws ServerAuthenticationFailure {
        switch (this.authSyncSettings.notInAuthorizedGroupsAction) {
            case DISABLE_USER: {
                logger.warnV("User '%s' not authorized anymore by authorized groups, disabling it", new Object[]{dkuUser.getLogin()});
                this.disableDkuUser(dkuUser.getLogin(), "not authorized anymore during login time resync");
                this.auditTrailService.generic("security-admin-user-disable").with("login", dkuUser.getLogin()).with("reason", "login-time-sync-not-authorized-anymore").emit();
                break;
            }
            case WARN: {
                logger.warnV("User '%s' not authorized anymore by authorized groups, not syncing it", new Object[]{dkuUser.getLogin()});
            }
        }
    }

    private DkuUser readAttributesAndProvisionAtLoginTime(UserIdentity userIdentity) throws UserNotFoundException, UnauthorizedException {
        logger.infoV("User '%s' does not exist yet, finding a supplier that is able to provision the user", new Object[]{userIdentity.login});
        List provisioningUserSuppliers = this.userSuppliers.values().stream().filter(u -> u.getUserSupplierSettings().isEnabled()).filter(supplier -> userIdentity.sourceType == UserSourceType.LOCAL_NO_AUTH || supplier.getUserSupplierSettings().getUserSourceType() == userIdentity.sourceType).filter(supplier -> supplier.getUserSupplierSettings().isUsersAutoProvisioningAtLoginTimeEnabled()).sorted(Comparator.comparingInt(supplier -> supplier.getUserSupplierSettings().getOrder())).collect(Collectors.toList());
        for (UserSupplier userSupplier : provisioningUserSuppliers) {
            DkuUser dkuUser;
            block11: {
                UserSourceType userSourceType = userSupplier.getUserSupplierSettings().getUserSourceType();
                logger.infoV("Trying supplier for user type '%s' for user '%s'", new Object[]{userSourceType, userIdentity.login});
                UserAttributes userAttributes = userSupplier.getUserAttributes(userIdentity);
                userAttributes.login = userIdentity.login;
                RWTransaction t = this.transactionService.beginWriteAsDSS();
                try {
                    DkuUser dkuUser2 = this.provisionUserAttributes(userAttributes, userSupplier);
                    t.commit(String.format("Imported a '%s' new user: %s", new Object[]{userSourceType, dkuUser2.getLogin()}));
                    this.usersService.sendWelcomeEmail(dkuUser2);
                    logger.infoV("User '%s' was successfully provisioned from supplier for user type '%s'", new Object[]{userIdentity.login, userSourceType});
                    dkuUser = dkuUser2;
                    if (t == null) break block11;
                }
                catch (Throwable throwable) {
                    try {
                        if (t != null) {
                            try {
                                t.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (UserNotInAuthorizedGroupsException e) {
                        logger.infoV("Supplier for user type '%s' knows user %s but it is not in any of authorized groups. Reason: %s", new Object[]{userSupplier.getUserSupplierSettings().getUserSourceType(), userIdentity.login, e.getMessage()});
                    }
                    catch (UserNotFoundException e) {
                        logger.infoV("Supplier for user type '%s' could not find user %s. Reason: %s", new Object[]{userSupplier.getUserSupplierSettings().getUserSourceType(), userIdentity.login, e.getMessage()});
                    }
                    catch (ServerAuthenticationFailure | IOException e) {
                        logger.warnV((Throwable)e, "Supplier for user type '%s' could not provision user %s", new Object[]{userSourceType, userIdentity.login});
                    }
                }
                t.close();
            }
            return dkuUser;
        }
        logger.infoV("No supplier was able to provision user '%s'. Tried %s suppliers.", new Object[]{userIdentity.login, provisioningUserSuppliers.size()});
        throw new UserNotFoundException("Unknown user '" + userIdentity.login + "' to all suppliers");
    }

    private DkuUser provisionUserAttributes(UserAttributes userAttributes, UserSupplier userSupplier) throws ServerAuthenticationFailure, UnauthorizedException {
        logger.infoV("Supplier for user type '%s' retrieved some user attributes %s", new Object[]{userSupplier.getUserSupplierSettings().getUserSourceType(), userAttributes});
        Set<String> authorizedGroups = userSupplier.getUserSupplierSettings().getAuthorizedGroups();
        if (!authorizedGroups.isEmpty() && Collections.disjoint(authorizedGroups, userAttributes.sourceGroupNames)) {
            throw new UserNotInAuthorizedGroupsException(String.format("none of the user groups (%s) is within authorized groups (%s)", String.join((CharSequence)", ", userAttributes.sourceGroupNames), String.join((CharSequence)", ", authorizedGroups)));
        }
        String grantedProfile = this.getGrantedProfile(userSupplier.getUserSupplierSettings(), userAttributes, userSupplier.getDefaultProfile());
        return this.createUser(userAttributes.login, userAttributes, userSupplier.getUserSupplierSettings().getUserSourceType(), grantedProfile);
    }

    public DkuUser getUserWithCaseSensitivenessRule(String inputLogin) throws ServerAuthenticationFailure {
        DkuUser dkuUser;
        block8: {
            Transaction t = this.transactionService.retrieveOrBeginRead();
            try {
                dkuUser = this.usersService.getUserWithCaseSensitiveRule(inputLogin);
                if (t == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (t != null) {
                        try {
                            t.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new ServerAuthenticationFailure("Could not fetch user", e);
                }
            }
            t.close();
        }
        return dkuUser;
    }

    public Set<String> getOnDemandSyncableSourceTypes() {
        return this.userSuppliers.values().stream().filter(userSupplier -> userSupplier.getUserSupplierSettings().isEnabled()).filter(UserSupplier::canSyncOnDemand).filter(userSupplier -> userSupplier.getUserSupplierSettings().isOnDemandUsersSyncEnabled()).map(userSupplier -> userSupplier.getUserSupplierSettings().getUserSourceType().name()).collect(Collectors.toSet());
    }

    public static String getDesensitizedMessageForUserAuthenticationException(Exception e) {
        return UserAuthenticationService.getDesensitizedExceptionForUserAuthenticationException(e).getMessage();
    }

    public static Exception getDesensitizedExceptionForUserAuthenticationException(Exception e) {
        if (e instanceof UserNotFoundException) {
            return new UserNotFoundException("User not found");
        }
        if (e instanceof UserDisabledException) {
            return new UserDisabledException("User is disabled");
        }
        if (e instanceof UnauthorizedException) {
            return new UnauthorizedException("User is not allowed to login", "unauthorized");
        }
        return new ServerAuthenticationFailure("Login failure");
    }

    public FetchedUsersResponse fetchUsers(UserSourceType userSourceType, UserQueryFilter filter) throws ServerAuthenticationFailure {
        List<? extends DkuGroup> dkuGroups;
        Map<String, DkuUser> dkuUsers;
        UserSupplier userSupplier = this.userSuppliers.get((Object)userSourceType);
        if (!userSupplier.canFetchUsers()) {
            throw new ServerAuthenticationFailure("Trying to fetch users from incompatible supplier");
        }
        try (Transaction t = this.transactionService.beginRead();){
            dkuUsers = this.usersService.listUsers().stream().collect(Collectors.toMap(DkuUser::getLogin, u -> u));
            dkuGroups = this.usersService.listGroups();
        }
        catch (IOException e) {
            throw new ServerAuthenticationFailure("Could not fetch existing users;");
        }
        Set<ExternalUser> externalUsers = userSupplier.fetchUsers(filter).stream().map(userAttributes -> {
            DkuUser dkuUser;
            userAttributes.login = this.remapProvidedLogin(userAttributes.login, userSupplier.getUserSupplierSettings().getUserRemappingRules());
            userAttributes.dkuGroupNames = UserAuthenticationService.toDkuGroups(dkuGroups, userAttributes, userSourceType);
            String profile = this.getGrantedProfile(userSupplier.getUserSupplierSettings(), (UserAttributes)userAttributes, userSupplier.getDefaultProfile());
            ExternalUser.Status status = !dkuUsers.containsKey(userAttributes.login) ? ExternalUser.Status.NOT_PROVISIONED : ((dkuUser = (DkuUser)dkuUsers.get(userAttributes.login)).getUserSourceType() == userSourceType ? (this.isSynced(userSupplier, (UserAttributes)userAttributes, dkuUser) ? ExternalUser.Status.SYNCED : ExternalUser.Status.UNSYNCED) : ExternalUser.Status.SOURCE_CONFLICT);
            return new ExternalUser((UserAttributes)userAttributes, status, profile);
        }).collect(Collectors.toSet());
        return new FetchedUsersResponse(externalUsers, filter.getUnusedFilters());
    }

    private String remapProvidedLogin(String inputLogin, List<UserRemappingRule> userRemappingRules) {
        if (StringUtils.isEmpty((String)inputLogin)) {
            return inputLogin;
        }
        for (UserRemappingRule rule : userRemappingRules) {
            Pattern p = Pattern.compile(rule.ruleFrom.trim());
            Matcher m = p.matcher(inputLogin);
            if (!m.matches()) continue;
            return m.replaceFirst(rule.ruleTo.trim());
        }
        return inputLogin;
    }

    private boolean isSynced(UserSupplier userSupplier, UserAttributes u, DkuUser existingUser) {
        boolean isSynced = !this.authSyncSettings.syncUserEmail || Objects.equals(existingUser.getEmail(), u.email);
        isSynced &= !this.authSyncSettings.syncUserDisplayName || Objects.equals(existingUser.getDisplayName(), u.displayName);
        if (this.authSyncSettings.syncUserGroups) {
            isSynced &= CollectionUtils.isEqualCollection(existingUser.getGroups(), u.dkuGroupNames);
        }
        if (this.authSyncSettings.syncUserProfile) {
            String profile = this.getGrantedProfile(userSupplier.getUserSupplierSettings(), u, userSupplier.getDefaultProfile());
            isSynced &= profile.equals(existingUser.getUserProfile());
        }
        return isSynced;
    }

    public List<String> fetchGroups(UserSourceType userSourceType) throws ServerAuthenticationFailure {
        UserSupplier userSupplier = this.userSuppliers.get((Object)userSourceType);
        if (!userSupplier.canFetchGroups()) {
            throw new ServerAuthenticationFailure("Trying to fetch groups from incompatible supplier");
        }
        return userSupplier.fetchGroups().stream().sorted().collect(Collectors.toList());
    }

    public InfoMessage.InfoMessages provisionUsers(UserSourceType userSourceType, Set<UserAttributes> users) throws InterruptedException {
        InfoMessage.InfoMessages infoMessages = new InfoMessage.InfoMessages();
        UserSupplier userSupplier = this.userSuppliers.get((Object)userSourceType);
        if (!userSupplier.getUserSupplierSettings().isOnDemandUsersProvisioningEnabled()) {
            infoMessages.withError((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_PROVISIONING_FAILED, "On-demand provisioning is not enabled for this user supplier in the settings of your instance");
            return infoMessages;
        }
        try (RWTransaction t = this.transactionService.beginWriteAsDSS();){
            int count = 0;
            ArrayList<AuditTrailService.EmittableAuditObj> auditLogs = new ArrayList<AuditTrailService.EmittableAuditObj>();
            FutureProgress.pushState("Provisioning users", users.size(), FutureProgressState.StateUnit.RECORDS);
            ArrayList<DkuUser> provisionedUsers = new ArrayList<DkuUser>();
            for (UserAttributes user : users) {
                try {
                    DkuUser dkuUser = this.provisionUserAttributes(user, userSupplier);
                    auditLogs.add(this.auditTrailService.generic("security-admin-user-on-demand-provisioning-from-source").with("login", user.login));
                    ++count;
                    provisionedUsers.add(dkuUser);
                }
                catch (UnauthorizedException e) {
                    logger.errorV((Throwable)((Object)e), "Supplier for user type '%s' could not authorize user %s. Reason: %s", new Object[]{userSupplier.getUserSupplierSettings().getUserSourceType(), user.login, e.getMessage()});
                    infoMessages.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_PROVISIONING_FAILED, "Supplier for user type '%s' could not authorize user %s. Reason: %s", new Object[]{userSupplier.getUserSupplierSettings().getUserSourceType(), user.login, e.getMessage()});
                }
                catch (Exception e) {
                    logger.errorV((Throwable)e, "Supplier for user type '%s' could not provision user %s. Reason: %s", new Object[]{userSupplier.getUserSupplierSettings().getUserSourceType(), user.login, e.getMessage()});
                    infoMessages.withWarningV((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_PROVISIONING_FAILED, "Supplier for user type '%s' could not provision user %s. Reason: %s", new Object[]{userSupplier.getUserSupplierSettings().getUserSourceType(), user, e.getMessage()});
                }
                FutureProgress.incrementState(1.0);
            }
            t.commit(String.format("Imported %d/%d new users of source type '%s'", new Object[]{count, users.size(), userSourceType}));
            auditLogs.forEach(AuditTrailService.EmittableAuditObj::emit);
            infoMessages.withInfoV((InfoMessage.MessageCode)SecurityCodes.INFO_SECURITY_SUPPLIER_PROVISIONING_SUMMARY, "Imported %d/%d new users of source type '%s'", new Object[]{count, users.size(), userSourceType});
            this.sendWelcomeEmailsToUsers(provisionedUsers);
        }
        catch (IOException e) {
            logger.errorV((Throwable)e, "Could not provision users. Reason: %s", new Object[]{e.getMessage()});
            infoMessages.withFatal((InfoMessage.MessageCode)SecurityCodes.ERR_SECURITY_SUPPLIER_PROVISIONING_FAILED, String.format("Could not provision users of type '%s'. Reason: %s", new Object[]{userSupplier.getUserSupplierSettings().getUserSourceType(), e.getMessage()}));
        }
        catch (InterruptedException e) {
            logger.warn((Object)"Users were not provisioned. Reason: Operation was aborted by user");
            throw e;
        }
        return infoMessages;
    }

    private void sendWelcomeEmailsToUsers(List<DkuUser> provisionedUsers) {
        provisionedUsers.forEach(u -> this.usersService.sendWelcomeEmail((DkuUser)u));
    }

    public static class UserDisabledException
    extends DKUSecurityException {
        private static final long serialVersionUID = 1L;

        public UserDisabledException(String message) {
            super(message);
        }
    }

    public static class UserAuthInfo {
        public boolean exists;
        public boolean enabled;
        public boolean provisioningAtLogin;
    }
}

