/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.connections;

import com.dataiku.dip.DKUApp;
import com.dataiku.dip.DSSMetrics;
import com.dataiku.dip.connections.AbstractSQLConnection;
import com.dataiku.dip.connections.ConnectionUtils;
import com.dataiku.dip.connections.SQLConnectionFactory;
import com.dataiku.dip.connections.SQLConnectionProvider;
import com.dataiku.dip.connections.SQLConnectionService;
import com.dataiku.dip.exceptions.DKUSecurityException;
import com.dataiku.dip.logging.MainLoggingConfigurator;
import com.dataiku.dip.reports.IReflectedEventsService;
import com.dataiku.dip.security.AuthCtx;
import com.dataiku.dip.server.notifications.DSSEvent;
import com.dataiku.dip.server.notifications.backend.ConnectionChangedEvent;
import com.dataiku.dip.server.notifications.backend.ReflectedEventEvent;
import com.dataiku.dip.server.services.PubSubService;
import com.dataiku.dip.transactions.TransactionContext;
import com.dataiku.dip.utils.DKULogger;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import javax.annotation.PostConstruct;
import org.apache.commons.pool.KeyedPoolableObjectFactory;
import org.apache.commons.pool.impl.GenericKeyedObjectPool;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class SQLPooledConnectionService
implements SQLConnectionService {
    private final KeyedPoolableObjectFactory<SQLConnectionService.SQLConnectionKey, PooledConnection> connectionFactory = new SQLConnectionFactory();
    private final Map<String, GenericKeyedObjectPool<SQLConnectionService.SQLConnectionKey, PooledConnection>> connectionPools = new ConcurrentHashMap<String, GenericKeyedObjectPool<SQLConnectionService.SQLConnectionKey, PooledConnection>>();
    private final ExecutorService asynchronousGiveExecutor = Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("SQLConnectionPool-AsyncGiver-%d").build());
    @Autowired
    private PubSubService pubSub;
    private static final DKULogger logger = DKULogger.getLogger((String)"dku.sql.connection.pool");

    @Override
    public SQLConnectionProvider.SQLConnectionWrapper take(AuthCtx authCtx, SQLConnectionService.SQLConnectionKey key) throws Exception {
        this.checkNoAttachedTransaction();
        try (DSSMetrics.MTimeCtx mt = DSSMetrics.mtimeCtx((String[])new String[]{"dku.connections.sqlPool.overall.take", "dku.connections.sqlPool." + SQLConnectionFactory.connName(key) + ".take"});){
            GenericKeyedObjectPool<SQLConnectionService.SQLConnectionKey, PooledConnection> connectionPool = this.getOrCreatePoolByConnectionKey(authCtx, key);
            logger.debugV("Retrieving SQL connection from pool for wrapper id=%s, idleForKey=%s activeForKey=%s totalIdle=%s totalActive=%s", new Object[]{key.debugId, connectionPool.getNumIdle((Object)key), connectionPool.getNumActive((Object)key), connectionPool.getNumIdle(), connectionPool.getNumActive()});
            PooledConnection pooledConnection = (PooledConnection)connectionPool.borrowObject((Object)key);
            pooledConnection.connectionPool = connectionPool;
            logger.debugV("Borrowed SQL connection id=%s for wrapper id=%s", new Object[]{pooledConnection.connectionId, key.debugId});
            PooledConnectionCloser connectionCloser = new PooledConnectionCloser(key, pooledConnection, this);
            SQLConnectionProvider.SQLConnectionWrapper sQLConnectionWrapper = new SQLConnectionProvider.SQLConnectionWrapper(key.sqlConnectionData, pooledConnection.connection, key.debugId, key.verboseRollback, connectionCloser);
            return sQLConnectionWrapper;
        }
    }

    private synchronized GenericKeyedObjectPool<SQLConnectionService.SQLConnectionKey, PooledConnection> getOrCreatePoolByConnectionKey(AuthCtx authCtx, SQLConnectionService.SQLConnectionKey key) throws SQLException, IOException, DKUSecurityException {
        String connectionName = key.sqlConnectionData.getConnection().name;
        GenericKeyedObjectPool<SQLConnectionService.SQLConnectionKey, PooledConnection> connectionPool = this.connectionPools.get(connectionName);
        if (connectionPool == null) {
            connectionPool = this.createPool(authCtx, connectionName);
            this.connectionPools.put(connectionName, connectionPool);
        }
        return connectionPool;
    }

    private void doGive(SQLConnectionService.SQLConnectionKey key, PooledConnection pooledConnection) throws Exception {
        try (DSSMetrics.MTimeCtx mt = DSSMetrics.mtimeCtx((String[])new String[]{"dku.connections.sqlPool.overall.give", "dku.connections.sqlPool." + SQLConnectionFactory.connName(key) + ".give"});){
            String connectionName = key.sqlConnectionData.getConnection().name;
            GenericKeyedObjectPool<SQLConnectionService.SQLConnectionKey, PooledConnection> connectionPool = this.connectionPools.get(connectionName);
            if (connectionPool != null && connectionPool == pooledConnection.connectionPool) {
                logger.debugV("Returning connection id=%s to pool, pool status before: idleForKey=%s activeForKey=%s totalIdle=%s totalActive=%s", new Object[]{pooledConnection.connectionId, connectionPool.getNumIdle((Object)key), connectionPool.getNumActive((Object)key), connectionPool.getNumIdle(), connectionPool.getNumActive()});
                connectionPool.returnObject((Object)key, (Object)pooledConnection);
            } else {
                if (connectionPool == null) {
                    logger.info((Object)("No existing pool found for '" + connectionName + "', the connection may have been deleted from DSS. Manually closing the JDBC connection"));
                } else {
                    logger.info((Object)("Pool found for '" + connectionName + "' does not match the returned JDBC connection's pool, the pool was likely deleted and recreated. Manually closing the JDBC connection"));
                }
                ConnectionUtils.actuallyCloseJdbcConnection(pooledConnection.connection, key.sqlConnectionData.getDialect());
            }
        }
    }

    public void give(SQLConnectionService.SQLConnectionKey key, PooledConnection pooledConnection) throws Exception {
        this.checkNoAttachedTransaction();
        if (key.getConnectionPoolSettings().asynchronousGive) {
            this.asynchronousGiveExecutor.submit(() -> {
                try {
                    this.doGive(key, pooledConnection);
                }
                catch (Exception e) {
                    logger.error((Object)"Failed to asynchronously give connection back to pool", (Throwable)e);
                }
            });
        } else {
            this.doGive(key, pooledConnection);
        }
    }

    private void checkNoAttachedTransaction() {
        MainLoggingConfigurator.ProcessType processType = DKUApp.getProcessType();
        if (processType != null && processType.equals((Object)MainLoggingConfigurator.ProcessType.BACKEND) && TransactionContext.hasAttachedTransaction()) {
            IReflectedEventsService.ReflectedEvent event = new IReflectedEventsService.ReflectedEvent("SQL connection pool has an attached transaction while taking or giving connection.", (Throwable)new Exception());
            this.pubSub.publish((DSSEvent)new ReflectedEventEvent(event));
            TransactionContext.warnAttachedTransaction();
        }
    }

    private GenericKeyedObjectPool<SQLConnectionService.SQLConnectionKey, PooledConnection> createPool(AuthCtx authCtx, String connectionName) throws SQLException, IOException, DKUSecurityException {
        AbstractSQLConnection connection = SQLConnectionProvider.getDSSConnection(authCtx, connectionName);
        AbstractSQLConnection.ConnectionPoolSettings connectionPoolSettings = connection.getParams().connectionPoolSettings;
        logger.info((Object)("Create connection pool: " + connectionName));
        GenericKeyedObjectPool connectionPool = new GenericKeyedObjectPool(this.connectionFactory);
        connectionPool.setTestOnBorrow(connectionPoolSettings.testOnBorrow);
        connectionPool.setTestOnReturn(false);
        connectionPool.setTestWhileIdle(true);
        connectionPool.setMaxActive(connectionPoolSettings.maxActive);
        connectionPool.setMaxIdle(connectionPoolSettings.maxIdle);
        connectionPool.setWhenExhaustedAction((byte)1);
        connectionPool.setMinEvictableIdleTimeMillis(connectionPoolSettings.minEvictableIdleTimeSeconds * 1000L);
        connectionPool.setTimeBetweenEvictionRunsMillis(connectionPoolSettings.timeBetweenEvictionRunsSeconds * 1000L);
        return connectionPool;
    }

    private void deletePoolOnConnectionChange(ConnectionChangedEvent event) {
        ConnectionChangedEvent.ActionType action = event.getAction();
        String connectionName = event.getConnectionName();
        GenericKeyedObjectPool<SQLConnectionService.SQLConnectionKey, PooledConnection> pool = this.connectionPools.get(connectionName);
        if (pool != null && !action.equals((Object)ConnectionChangedEvent.ActionType.CREATED)) {
            logger.info((Object)("Clearing and removing pool on connection settings change: " + connectionName));
            this.connectionPools.remove(connectionName);
            Thread poolClearingThread = new Thread(() -> {
                logger.debug((Object)("Running connection pool clearing thread to close any idle pooled connections: " + connectionName));
                pool.clear();
            });
            poolClearingThread.setDaemon(true);
            poolClearingThread.start();
        }
    }

    @PostConstruct
    public void init() {
        this.pubSub.subscribe("connections-changed", this::deletePoolOnConnectionChange);
    }

    public static class PooledConnection {
        public final Connection connection;
        public final String connectionId;
        public long lastKnownGood;
        public GenericKeyedObjectPool<SQLConnectionService.SQLConnectionKey, PooledConnection> connectionPool;

        public PooledConnection(Connection connection, String connectionId) {
            this.connection = connection;
            this.connectionId = connectionId;
        }
    }

    public static class PooledConnectionCloser
    implements SQLConnectionProvider.ConnectionCloser {
        private final SQLConnectionService.SQLConnectionKey key;
        private final PooledConnection pooledConnection;
        private final SQLPooledConnectionService pooledConnectionService;
        private boolean hasBeenReturned = false;

        public PooledConnectionCloser(SQLConnectionService.SQLConnectionKey key, PooledConnection pooledConnection, SQLPooledConnectionService pooledConnectionService) {
            this.key = key;
            this.pooledConnection = pooledConnection;
            this.pooledConnectionService = pooledConnectionService;
        }

        @Override
        public void close() throws SQLException {
            if (!this.hasBeenReturned) {
                logger.debug((Object)("Close on PooledConnectionCloser, returning SQL connection id=" + this.pooledConnection.connectionId + " to pool"));
                this.hasBeenReturned = true;
                try {
                    this.pooledConnectionService.give(this.key, this.pooledConnection);
                }
                catch (Exception e) {
                    logger.error((Object)"Failed to return a SQL connection to the pool", (Throwable)e);
                }
            }
        }

        @Override
        public boolean isClosed() throws SQLException {
            return this.hasBeenReturned;
        }
    }
}

