/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.client.core;

import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import net.snowflake.client.core.CachedCredentialType;
import net.snowflake.client.core.Constants;
import net.snowflake.client.core.SFException;
import net.snowflake.client.core.SFLoginInput;
import net.snowflake.client.core.SecureStorageAppleManager;
import net.snowflake.client.core.SecureStorageLinuxManager;
import net.snowflake.client.core.SecureStorageManager;
import net.snowflake.client.core.SecureStorageWindowsManager;
import net.snowflake.client.jdbc.ErrorCode;
import net.snowflake.client.jdbc.SnowflakeUtil;
import net.snowflake.client.log.SFLogger;
import net.snowflake.client.log.SFLoggerFactory;

public class CredentialManager {
    private static final SFLogger logger = SFLoggerFactory.getLogger(CredentialManager.class);
    private SecureStorageManager secureStorageManager;

    private CredentialManager() {
        this.initSecureStorageManager();
    }

    private void initSecureStorageManager() {
        try {
            if (Constants.getOS() == Constants.OS.MAC) {
                this.secureStorageManager = SecureStorageAppleManager.builder();
            } else if (Constants.getOS() == Constants.OS.WINDOWS) {
                this.secureStorageManager = SecureStorageWindowsManager.builder();
            } else if (Constants.getOS() == Constants.OS.LINUX) {
                this.secureStorageManager = SecureStorageLinuxManager.getInstance();
            } else {
                logger.error("Unsupported Operating System. Expected: OSX, Windows, Linux", false);
            }
        }
        catch (NoClassDefFoundError error) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
        }
    }

    static void resetSecureStorageManager() {
        logger.debug("Resetting the secure storage manager", new Object[0]);
        CredentialManager.getInstance().initSecureStorageManager();
    }

    static void injectSecureStorageManager(SecureStorageManager manager) {
        logger.debug("Injecting secure storage manager", new Object[0]);
        CredentialManager.getInstance().secureStorageManager = manager;
    }

    public static CredentialManager getInstance() {
        return CredentialManagerHolder.INSTANCE;
    }

    static void fillCachedIdToken(SFLoginInput loginInput) throws SFException {
        logger.debug("Looking for cached id token for user: {}, host: {}", loginInput.getUserName(), loginInput.getHostFromServerUrl());
        CredentialManager.getInstance().fillCachedCredential(loginInput, loginInput.getHostFromServerUrl(), loginInput.getUserName(), CachedCredentialType.ID_TOKEN);
    }

    static void fillCachedMfaToken(SFLoginInput loginInput) throws SFException {
        logger.debug("Looking for cached mfa token for user: {}, host: {}", loginInput.getUserName(), loginInput.getHostFromServerUrl());
        CredentialManager.getInstance().fillCachedCredential(loginInput, loginInput.getHostFromServerUrl(), loginInput.getUserName(), CachedCredentialType.MFA_TOKEN);
    }

    static void fillCachedOAuthAccessToken(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        logger.debug("Looking for cached OAuth access token for user: {}, host: {}", loginInput.getUserName(), host);
        CredentialManager.getInstance().fillCachedCredential(loginInput, host, loginInput.getUserName(), CachedCredentialType.OAUTH_ACCESS_TOKEN);
    }

    static void fillCachedOAuthRefreshToken(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        logger.debug("Looking for cached OAuth refresh token for user: {}, host: {}", loginInput.getUserName(), host);
        CredentialManager.getInstance().fillCachedCredential(loginInput, host, loginInput.getUserName(), CachedCredentialType.OAUTH_REFRESH_TOKEN);
    }

    static void fillCachedDPoPBundledAccessToken(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        logger.debug("Looking for cached DPoP public key for user: {}, host: {}", loginInput.getUserName(), host);
        CredentialManager.getInstance().fillCachedCredential(loginInput, host, loginInput.getUserName(), CachedCredentialType.DPOP_BUNDLED_ACCESS_TOKEN);
    }

    synchronized void fillCachedCredential(SFLoginInput loginInput, String host, String username, CachedCredentialType credType) throws SFException {
        String base64EncodedCred;
        if (SnowflakeUtil.isNullOrEmpty(username)) {
            logger.debug("Missing username; Cannot read from credential cache", new Object[0]);
            return;
        }
        if (this.secureStorageManager == null) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
            return;
        }
        String cred = null;
        try {
            base64EncodedCred = this.secureStorageManager.getCredential(host, username, credType.getValue());
        }
        catch (NoClassDefFoundError error) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
            return;
        }
        if (base64EncodedCred == null) {
            logger.debug("Retrieved {} is null", new Object[]{credType});
        }
        logger.debug("Setting {}{} token for user: {}, host: {}", base64EncodedCred == null ? "null " : "", credType.getValue(), username, host);
        if (base64EncodedCred != null && credType != CachedCredentialType.DPOP_BUNDLED_ACCESS_TOKEN) {
            try {
                cred = new String(Base64.getDecoder().decode(base64EncodedCred));
            }
            catch (Exception e) {
                this.deleteTemporaryCredential(host, username, credType);
                return;
            }
        }
        switch (credType) {
            case ID_TOKEN: {
                loginInput.setIdToken(cred);
                break;
            }
            case MFA_TOKEN: {
                loginInput.setMfaToken(cred);
                break;
            }
            case OAUTH_ACCESS_TOKEN: {
                loginInput.setOauthAccessToken(cred);
                break;
            }
            case OAUTH_REFRESH_TOKEN: {
                loginInput.setOauthRefreshToken(cred);
                break;
            }
            case DPOP_BUNDLED_ACCESS_TOKEN: {
                this.updateInputWithTokenAndPublicKey(base64EncodedCred, loginInput);
                break;
            }
            default: {
                throw new SFException(ErrorCode.INTERNAL_ERROR, new Object[]{"Unrecognized type {} for local cached credential", credType});
            }
        }
    }

    private void updateInputWithTokenAndPublicKey(String cred, SFLoginInput loginInput) throws SFException {
        if (SnowflakeUtil.isNullOrEmpty(cred)) {
            String[] values = cred.split("\\.");
            if (values.length != 2) {
                throw new SFException(ErrorCode.INTERNAL_ERROR, "Invalid DPoP bundled access token credential format");
            }
            Base64.Decoder decoder = Base64.getDecoder();
            loginInput.setOauthAccessToken(new String(decoder.decode(values[0])));
            loginInput.setDPoPPublicKey(new String(decoder.decode(values[1])));
        }
    }

    static void writeIdToken(SFLoginInput loginInput, String idToken) throws SFException {
        logger.debug("Caching id token in a secure storage for user: {}, host: {}", loginInput.getUserName(), loginInput.getHostFromServerUrl());
        CredentialManager.getInstance().writeTemporaryCredential(loginInput.getHostFromServerUrl(), loginInput.getUserName(), idToken, CachedCredentialType.ID_TOKEN);
    }

    static void writeMfaToken(SFLoginInput loginInput, String mfaToken) throws SFException {
        logger.debug("Caching mfa token in a secure storage for user: {}, host: {}", loginInput.getUserName(), loginInput.getHostFromServerUrl());
        CredentialManager.getInstance().writeTemporaryCredential(loginInput.getHostFromServerUrl(), loginInput.getUserName(), mfaToken, CachedCredentialType.MFA_TOKEN);
    }

    static void writeOAuthAccessToken(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        logger.debug("Caching OAuth access token in a secure storage for user: {}, host: {}", loginInput.getUserName(), host);
        CredentialManager.getInstance().writeTemporaryCredential(host, loginInput.getUserName(), loginInput.getOauthAccessToken(), CachedCredentialType.OAUTH_ACCESS_TOKEN);
    }

    static void writeOAuthRefreshToken(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        logger.debug("Caching OAuth refresh token in a secure storage for user: {}, host: {}", loginInput.getUserName(), host);
        CredentialManager.getInstance().writeTemporaryCredential(host, loginInput.getUserName(), loginInput.getOauthRefreshToken(), CachedCredentialType.OAUTH_REFRESH_TOKEN);
    }

    static void writeDPoPBundledAccessToken(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        logger.debug("Caching DPoP public key in a secure storage for user: {}, host: {}", loginInput.getUserName(), host);
        Base64.Encoder encoder = Base64.getEncoder();
        String tokenBase64 = encoder.encodeToString(loginInput.getOauthAccessToken().getBytes(StandardCharsets.UTF_8));
        String publicKeyBase64 = encoder.encodeToString(loginInput.getDPoPPublicKey().getBytes(StandardCharsets.UTF_8));
        CredentialManager.getInstance().writeTemporaryCredential(host, loginInput.getUserName(), tokenBase64 + "." + publicKeyBase64, CachedCredentialType.DPOP_BUNDLED_ACCESS_TOKEN);
    }

    synchronized void writeTemporaryCredential(String host, String user, String cred, CachedCredentialType credType) {
        if (SnowflakeUtil.isNullOrEmpty(user)) {
            logger.debug("Missing username; Cannot write to credential cache", new Object[0]);
            return;
        }
        if (SnowflakeUtil.isNullOrEmpty(cred)) {
            logger.debug("No {} is given.", new Object[]{credType});
            return;
        }
        if (this.secureStorageManager == null) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
            return;
        }
        try {
            if (credType == CachedCredentialType.DPOP_BUNDLED_ACCESS_TOKEN) {
                this.secureStorageManager.setCredential(host, user, credType.getValue(), cred);
            } else {
                String base64EncodedCred = Base64.getEncoder().encodeToString(cred.getBytes(StandardCharsets.UTF_8));
                this.secureStorageManager.setCredential(host, user, credType.getValue(), base64EncodedCred);
            }
        }
        catch (NoClassDefFoundError error) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
        }
    }

    static void deleteIdTokenCacheEntry(String host, String user) {
        logger.debug("Removing cached id token from a secure storage for user: {}, host: {}", user, host);
        CredentialManager.getInstance().deleteTemporaryCredential(host, user, CachedCredentialType.ID_TOKEN);
    }

    static void deleteMfaTokenCacheEntry(String host, String user) {
        logger.debug("Removing cached mfa token from a secure storage for user: {}, host: {}", user, host);
        CredentialManager.getInstance().deleteTemporaryCredential(host, user, CachedCredentialType.MFA_TOKEN);
    }

    static void deleteOAuthAccessTokenCacheEntry(String host, String user) {
        logger.debug("Removing cached oauth access token from a secure storage for user: {}, host: {}", user, host);
        CredentialManager.getInstance().deleteTemporaryCredential(host, user, CachedCredentialType.OAUTH_ACCESS_TOKEN);
    }

    static void deleteOAuthRefreshTokenCacheEntry(String host, String user) {
        logger.debug("Removing cached OAuth refresh token from a secure storage for user: {}, host: {}", user, host);
        CredentialManager.getInstance().deleteTemporaryCredential(host, user, CachedCredentialType.OAUTH_REFRESH_TOKEN);
    }

    static void deleteDPoPBundledAccessTokenCacheEntry(String host, String user) {
        logger.debug("Removing cached DPoP public key from a secure storage for user: {}, host: {}", user, host);
        CredentialManager.getInstance().deleteTemporaryCredential(host, user, CachedCredentialType.DPOP_BUNDLED_ACCESS_TOKEN);
    }

    static void deleteOAuthAccessTokenCacheEntry(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        CredentialManager.deleteOAuthAccessTokenCacheEntry(host, loginInput.getUserName());
    }

    static void deleteOAuthRefreshTokenCacheEntry(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        CredentialManager.deleteOAuthRefreshTokenCacheEntry(host, loginInput.getUserName());
    }

    static void deleteDPoPBundledAccessTokenCacheEntry(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        CredentialManager.deleteDPoPBundledAccessTokenCacheEntry(host, loginInput.getUserName());
    }

    static String getHostForOAuthCacheKey(SFLoginInput loginInput) throws SFException {
        String oauthTokenRequestUrl = loginInput.getOauthLoginInput().getTokenRequestUrl();
        if (oauthTokenRequestUrl != null) {
            URI parsedUrl = URI.create(oauthTokenRequestUrl);
            return parsedUrl.getHost();
        }
        return loginInput.getHostFromServerUrl();
    }

    synchronized void deleteTemporaryCredential(String host, String user, CachedCredentialType credType) {
        if (this.secureStorageManager == null) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
            return;
        }
        if (SnowflakeUtil.isNullOrEmpty(user)) {
            logger.debug("Missing username; Cannot delete from credential cache", new Object[0]);
            return;
        }
        try {
            this.secureStorageManager.deleteCredential(host, user, credType.getValue());
        }
        catch (NoClassDefFoundError error) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
        }
    }

    private static void logMissingJnaJarForSecureLocalStorage() {
        logger.warn("JNA jar files are needed for Secure Local Storage service. Please follow the Snowflake JDBC instruction for Secure Local Storage feature. Fall back to normal process.", false);
    }

    private static class CredentialManagerHolder {
        private static final CredentialManager INSTANCE = new CredentialManager();

        private CredentialManagerHolder() {
        }
    }
}

