(function () {
    "use strict";

    const app = angular.module("dataiku.coloring");

    app.component("conditionalFormattingGroupEditor", {
        bindings: {
            group: "=",
            initScaleMinMax: "<",
            paneState: "<",
            removeColoringToWholeRows: "<",
            removeColoringToWholeRowsTooltip: "<",
        },
        templateUrl: "/static/dataiku/coloring/conditional-formatting/cf-group-editor/cf-group-editor.component.html",
        controller: function conditionalFormattingGroupEditorController(
            $scope,
            ConditionalFormattingEditorService,
            openDkuPopin,
            translate
        ) {
            /*
            Initializations
            */
            const ctrl = this;
            ctrl.$onInit = function () {
                // Read attributes and save them into scope
                const defaults = {
                    group: {},
                    initScaleMinMax: null,
                    paneState: {},
                    removeColoringToWholeRows: null,
                    removeColoringToWholeRowsTooltip: null,
                };
                for (const [param, value] of Object.entries(defaults)) {
                    if (param in ctrl) {
                        $scope[param] = ctrl[param] || value;
                    } else {
                        $scope[param] = value;
                    }
                }
                $scope.ConditionalFormattingEditorService = ConditionalFormattingEditorService;
                $scope.columns = $scope.paneState.columns;
                $scope.scaleParametersExpanded = false;

                $scope.initColorScalePalette();
            };

            /*
            Scope methods
            */

            $scope.initColorScalePalette = function () {
                const coloringGroup = $scope.group.coloringGroup;
                if (coloringGroup && ConditionalFormattingEditorService.isSchemeScale(coloringGroup)) {
                    const targetedColumns = $scope.group.isAllColumnsBasedOnAnotherColumn
                        ? [coloringGroup.basedOnColumnName]
                        : coloringGroup.targetedColumnNames;

                    // Case where the user changes the coloring mode to scale
                    // Uses the default if undefined depending on meaning
                    if (!coloringGroup.colorScaleDef) {
                        coloringGroup.colorScaleDef = ConditionalFormattingEditorService.newColorScaleDef(
                            targetedColumns,
                            $scope.paneState.tableHeaders
                        );
                        $scope.initScaleMinMax(targetedColumns, $scope.group);
                    }

                    $scope.group.scaleBorderGradient = ConditionalFormattingEditorService.getScaleBorderGradient(
                        coloringGroup.colorScaleDef.sample
                    );
                }
            };

            $scope.getFilterDescs = function () {
                return $scope.group.coloringGroup.rulesGroup.filterDescs;
            };

            $scope.onColorScalePaletteChange = function ($event) {
                $scope.group.coloringGroup.colorScaleDef = {
                    colorPalette: $event.colorPalette,
                    customPalette: $event.customPalette,
                    max: $scope.group.coloringGroup.colorScaleDef.max,
                    min: $scope.group.coloringGroup.colorScaleDef.min,
                    paletteType: $event.paletteType,
                    quantizationMode: $scope.group.coloringGroup.colorScaleDef.quantizationMode,
                    sample: [],
                    useSample2: $scope.group.coloringGroup.colorScaleDef.useSample2,
                };

                const sample = ConditionalFormattingEditorService.getColorScaleDefSample(
                    $scope.group.coloringGroup.colorScaleDef
                );
                $scope.group.scaleBorderGradient = ConditionalFormattingEditorService.getScaleBorderGradient(sample);
                $scope.group.coloringGroup.colorScaleDef.sample = sample;

                $scope.applyChanges(true);
            };

            $scope.onChangeSelectedColumns = function () {
                const coloringGroup = $scope.group.coloringGroup;
                if (coloringGroup && ConditionalFormattingEditorService.isSchemeScale(coloringGroup)) {
                    const targetedColumns = $scope.group.isAllColumnsBasedOnAnotherColumn
                        ? [coloringGroup.basedOnColumnName]
                        : coloringGroup.targetedColumnNames;

                    $scope.initScaleMinMax(targetedColumns, $scope.group);
                }

                $scope.$parent.updateGroupTitle($scope.group);

                $scope.applyChanges(true);
            };

            $scope.toggleRule = function (rule) {
                rule.expanded = !rule.expanded;
            };

            $scope.toggleScaleParameters = function () {
                $scope.scaleParametersExpanded = !$scope.scaleParametersExpanded;
            };

            $scope.removeRule = function (rule) {
                const index = $scope.group.rules.indexOf(rule);
                if (index === -1) {
                    // Should never happen
                    return;
                }

                if (index < $scope.group.rules.length) {
                    // Removes the rule in the model (filterDesc)
                    $scope.getFilterDescs().splice(index, 1);
                    // Removes the rule in the "UI" variable ($scope.group) representing the current group
                    $scope.group.rules.splice(index, 1);

                    $scope.applyChanges(true);
                }
            };

            $scope.addNewRule = function () {
                // Updates model
                const filterDesc = $scope.ConditionalFormattingEditorService.getDefaultFilterDesc();
                $scope.getFilterDescs().push(filterDesc);

                // Updates UI representation
                ConditionalFormattingEditorService.addUiRuleToGroup($scope.group, filterDesc, true);

                $scope.applyChanges();
            };

            $scope.applyChanges = function (force = false) {
                if (
                    force ||
                    $scope.group.coloringGroup.targetedColumnNames.length ||
                    $scope.group.coloringGroup.basedOnColumnName !== null
                ) {
                    $scope.$parent.applyChanges();
                }
            };

            $scope.btnCollapseAllTitle = function () {
                return translate(
                    "SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.CONDITIONS.COLLAPSE_ALL",
                    "Collapse all conditions"
                );
            };

            $scope.btnExpandAllTitle = function () {
                return translate(
                    "SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.CONDITIONS.EXPAND_ALL",
                    "Expand all conditions"
                );
            };

            $scope.setExpandedOnAllRules = function (expand) {
                if ($scope.group.rules) {
                    for (let i = 0; i < $scope.group.rules.length; i++) {
                        $scope.group.rules[i].expanded = expand;
                    }
                }
            };

            /*
             * Options menu
             */
            $scope.btnOptionsTitle = function () {
                return translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.CONDITION.OPTIONS", "More options");
            };

            $scope.openRuleMenu = function ($event, rule) {
                // dismiss existing menu
                if (typeof $scope.activeRuleMenu === "function") {
                    $scope.activeRuleMenu();
                }

                function isElsewhere() {
                    return true;
                }

                const newScope = $scope.$new();
                newScope.rule = rule;

                const template = `
                    <ul class="dropdown-menu z-index-10000">
                        <li>
                            <a ng-click="duplicateRule(rule);">
                                <i class="dku-icon-copy-16 dibvab"/>
                                <span translate="SHAKER.VIEW.CONDITIONAL_FORMATTING.CONDITION.OPTIONS.DUPLICATE">Duplicate this condition</span>
                            </a>
                        </li>
                        <li>
                            <a ng-click="removeRule(rule);" class="text-error">
                                <i class="dku-icon-trash-16 dibvab"/>
                                <span translate="SHAKER.VIEW.CONDITIONAL_FORMATTING.CONDITION.OPTIONS.DELETE">Delete this condition</span>
                            </a>
                        </li>
                    </ul>
                `;

                const dkuPopinOptions = {
                    template: template,
                    isElsewhere: isElsewhere,
                    callback: null,
                    popinPosition: "CLICK",
                    onDismiss: () => {
                        $scope.activeRuleMenu = null;
                    },
                };

                $scope.activeRuleMenu = openDkuPopin(newScope, $event, dkuPopinOptions);
            };

            $scope.duplicateRule = function (rule) {
                const filterDescs = $scope.getFilterDescs();

                // Finds the index of the rule
                let index = filterDescs.indexOf(rule.filterDesc);
                if (index === -1) {
                    // Should never happen
                    return;
                }

                // Inserts the rule after the original
                index++;
                const newRule = angular.copy(rule);
                // Adds the new rule into the model (filterDesc)
                filterDescs.splice(index, 0, newRule.filterDesc);
                // Adds the new rule into the "UI" variable ($scope.group) representing the current group
                $scope.group.rules.splice(index, 0, newRule);

                $scope.applyChanges(true);
            };

            /*
             * Reordering Groups
             */
            function afterRuleMove(event) {
                const destIndex = event.dest.index;
                const srcIndex = event.source.index;

                if (destIndex !== srcIndex) {
                    const filterDescs = $scope.getFilterDescs();
                    const filterDescSrc = filterDescs[srcIndex];
                    filterDescs.splice(srcIndex, 1);
                    filterDescs.splice(destIndex, 0, filterDescSrc);

                    $scope.applyChanges();
                }
            }

            $scope.childNodesCount = function () {
                return $scope.group.rules.length;
            };

            $scope.treeOptions = {
                dropped: afterRuleMove,
                accept: function (sourceNodeScope, destNodesScope, destIndex) {
                    // Authorizes drag and drop in same parent only for now
                    return sourceNodeScope.$parent.$id === destNodesScope.$id && destIndex < $scope.childNodesCount();
                },
            };
        },
    });
})();
