(function () {
    "use strict";

    angular
        .module("dataiku.coloring")
        .service("ConditionalFormattingEditorService", ConditionalFormattingEditorService);

    function ConditionalFormattingEditorService(Expressions, WT1, translate, ChartColorUtils, DKU_PALETTE_NAMES) {
        // borderColor is the color of the border line next to the coloring group definition
        const ruleStyles = [
            {
                label: translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.COLOR_LABEL.RED_BG", "Red background"),
                styleClass: "shaker-color-rule--red-background",
                borderColor: "#f0b8bf",
            },
            {
                label: translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.COLOR_LABEL.ORANGE_BG", "Orange background"),
                styleClass: "shaker-color-rule--orange-background",
                borderColor: "#f9c69b",
            },
            {
                label: translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.COLOR_LABEL.GREEN_BG", "Green background"),
                styleClass: "shaker-color-rule--green-background",
                borderColor: "#a5d6a7",
            },
            {
                label: translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.COLOR_LABEL.RED_TEXT", "Red text"),
                styleClass: "shaker-color-rule--red-text",
                borderColor: "#f0b8bf",
            },
            {
                label: translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.COLOR_LABEL.ORANGE_TEXT", "Orange text"),
                styleClass: "shaker-color-rule--orange-text",
                borderColor: "#f9c69b",
            },
            {
                label: translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.COLOR_LABEL.GREEN_TEXT", "Green text"),
                styleClass: "shaker-color-rule--green-text",
                borderColor: "#a5d6a7",
            },
            {
                label: translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.COLOR_LABEL.CUSTOM", "Custom"),
                styleClass: "shaker-color-rule--custom",
                borderColor: null,
            },
        ];

        // Legacy scale palettes defined shaker.less .cell-value-coloring-theme
        const scalePaletteForGender = ["rgba(255, 192, 203, 0.3)", "#E8F1FA"]; // F, M
        const scalePaletteForBoolean = ["#FFDDDD", "#EEFFEE"]; // false, true
        const scalePaletteForNumber = ["#E8F1FA", "#CDE0F5", "#B3CFEF", "#98BEE9", "#70A5E0"];
        const scalePaletteForString = [
            "rgba(141, 211, 199, 0.32)",
            "rgba(255, 255, 179, 0.32)",
            "rgba(190, 186, 218, 0.32)",
            "rgba(251, 128, 114, 0.32)",
            "rgba(128, 177, 211, 0.32)",
            "rgba(253, 180, 98, 0.32)",
            "rgba(179, 222, 105, 0.32)",
            "rgba(252, 205, 229, 0.32)",
            "rgba(188, 128, 189, 0.32)",
            "rgba(204, 235, 197, 0.32)",
            "rgba(255, 237, 111, 0.32)",
        ];

        /**
         * Adds the associated UI data of a rule to a given UI representation of a coloring group
         *
         * @param uiGroup - UI data of a coloring group
         * @param {FilterDesc} filterDesc - filter description of the targeted rule
         * @param {boolean} expanded - if the group is expanded
         */
        function addUiRuleToGroup(uiGroup, filterDesc, expanded) {
            const description = getFilterDescriptionString(filterDesc);
            const uiRule = {
                borderStyle: { "border-left-color": getBorderColorFromRule(filterDesc) },
                description: description,
                expanded: expanded,
                filterDesc: filterDesc,
            };
            uiGroup.rules.push(uiRule);
        }

        /**
         * Returns the color that should color the border line next to the coloring group definition.
         *
         * Returns an empty string if no color could be found.
         */
        function getBorderColorFromRule(rule) {
            if (rule.uiData.color.styleClass === "shaker-color-rule--custom") {
                if (isNoneColor(rule.uiData.color.styleCustomBackgroundColor)) {
                    // Returns neutral gray if background and font are "none" color
                    if (isNoneColor(rule.uiData.color.styleCustomFontColor)) {
                        return "#F2F2F2";
                    }

                    // Returns the font color if no specific background color
                    return rule.uiData.color.styleCustomFontColor;
                }

                return rule.uiData.color.styleCustomBackgroundColor;
            }

            const ruleStyle = ruleStyles.find(s => s.styleClass === rule.uiData.color.styleClass);
            if (ruleStyle) {
                return ruleStyle.borderColor;
            }

            return "";
        }

        function getDefaultFilterDesc() {
            const filterDesc = {
                distinct: false,
                enabled: true,
                uiData: {
                    mode: "\u0026\u0026",
                    conditions: [
                        {
                            input: "",
                            operator: "== [string]",
                            col: "",
                            string: "",
                            num: 0.0,
                            num2: 0.0,
                        },
                    ],
                    color: {
                        styleClass: "shaker-color-rule--green-background",
                        styleCustomFontColor: "#000000",
                        styleCustomBackgroundColor: "#F2F2F2",
                    },
                },
            };
            return filterDesc;
        }

        function getScaleBorderGradient(paletteSample) {
            if (paletteSample) {
                if (paletteSample.length === 2) {
                    const color1 = paletteSample[0];
                    const color2 = paletteSample[1];
                    return {
                        background: `linear-gradient(to top, ${color1}, ${color1} 50%, ${color2} 50%, ${color2} 100%)`,
                    };
                } else if (paletteSample.length > 2) {
                    let newPalette = null;
                    if (paletteSample.length > 5) {
                        // Fills a palette of 5 colors only
                        newPalette = [paletteSample[0], "", "", "", paletteSample[paletteSample.length - 1]];

                        let indexNewPalette = 1; // newPalette[0] already filled
                        let lastIndexNewPalette = newPalette.length - 1; // newPalette[newPalette.length - 1] already filled
                        let index1 = 0;
                        let index2 = paletteSample.length - 1;
                        while (indexNewPalette < lastIndexNewPalette) {
                            if (lastIndexNewPalette - indexNewPalette > 1) {
                                const rawCoefficient = (index2 - index1 - 1) / (lastIndexNewPalette - indexNewPalette);
                                let coefficient = Math.round(rawCoefficient);
                                if (coefficient >= lastIndexNewPalette - indexNewPalette) {
                                    coefficient = Math.floor(rawCoefficient);
                                }

                                // Fills 2 elements at boundaries
                                index1 = index1 + coefficient;
                                index2 = index2 - coefficient;
                                newPalette[indexNewPalette] = paletteSample[index1];
                                indexNewPalette += 1;
                                lastIndexNewPalette -= 1;
                                newPalette[lastIndexNewPalette] = paletteSample[index2];
                            } else {
                                // Fills last element in center of array
                                const lastIndex = index1 + Math.round((index2 - index1) / 2);
                                newPalette[indexNewPalette] = paletteSample[lastIndex];
                                break;
                            }
                        }
                    } else {
                        newPalette = paletteSample;
                    }

                    let borderBackground = "linear-gradient(to top";
                    for (let i = 0; i < newPalette.length; i++) {
                        borderBackground += ", " + newPalette[i];
                    }
                    borderBackground += ")";

                    return {
                        background: borderBackground,
                    };
                } else if (paletteSample.length === 1) {
                    return {
                        background: paletteSample[0],
                    };
                }
            }
            return getDefaultScaleBorderGradient();
        }

        function getScalePaletteForNumber() {
            return scalePaletteForNumber;
        }

        function getDefaultScaleBorderGradient() {
            const sample = ChartColorUtils.getContinuousPalette("default").sample;
            return sample.length ? getScaleBorderGradient(sample) : {};
        }

        function getFilterDescriptionString(filterDesc) {
            if (!filterDesc || !filterDesc.enabled || !filterDesc.uiData) {
                return "No filter";
            }
            // For conditional formatting only one condition in filterDesc.uiData.conditions (so no subCondition)
            for (let i = 0; i < filterDesc.uiData.conditions.length; i++) {
                const cond = { ...filterDesc.uiData.conditions[i] };
                const op = Expressions.getOperatorByName(cond["operator"]);

                if (!op) {
                    // Can happen if user modifies the json directly with a non-existing operator
                    return "Unknown operator";
                } else {
                    if (op.repr) {
                        // No need to display the input because column(s) are stored in targetedColumnNames
                        if (op.name === ">< [number]" || op.name === "<> [number]" || op.name === ">< [date]") {
                            cond.input = "...";
                        } else {
                            cond.input = "";
                        }
                        const tempDivElement = document.createElement("div");
                        // Set the HTML content with the given value
                        tempDivElement.innerHTML = op.repr(cond);
                        // Retrieve the text property of the element
                        const str = tempDivElement.textContent || tempDivElement.innerText || "";
                        return str.trim();
                    } else {
                        // Case where repr is missing in Expressions.operators
                        return "No representation available";
                    }
                }
            }
            return "";
        }

        function getRuleStyles() {
            return ruleStyles;
        }

        function isAllColumnsBasedOnAnotherColumn(group) {
            return group.scope === "ALL_COLUMNS_BASED_ON_ANOTHER_COLUMN";
        }

        function isSchemeMeaning(group) {
            return group.scheme === "MEANING_AND_STATUS";
        }
        function isSchemeRules(group) {
            return group.scheme === "RULES";
        }
        function isSchemeScale(group) {
            return group.scheme === "COLOR_SCALE";
        }

        /**
         * "None" color is defined as en empty string
         */
        function isNoneColor(color) {
            return color === "";
        }

        /**
         * Migrates the table coloring scheme to COLORING_GROUPS, adding appropriate coloring groups to copy legacy behavior.
         *
         * No migration happens if the coloring scheme is already COLORING_GROUPS or if the coloringGroups list is not empty.
         *
         * 2 special cases of other table coloring schemes still in use (from the "display" menu) with COLORING_GROUPS:
         * - MEANING_AND_STATUS
         * - ALL_COLUMNS_VALUES
         * In both cases, no coloring group is added to ensure consistency between the first (full) migration
         * and the following switches from COLORING_GROUPS to ALL_COLUMNS_VALUES.
         *
         * Returns true if the coloring was migrated or the table coloring scheme was updated and the changes need to be saved.
         * And false if nothing happened or the changes do not need to be saved.
         */
        function migrateToColoringGroups(coloring, tableHeaders) {
            if (!coloring || coloring.scheme === "COLORING_GROUPS") {
                return false;
            }

            let prevScheme = "";
            if (coloring.scheme) {
                prevScheme = coloring.scheme.slice(0); // Deep copy needed for WT1 events
            }

            prepareMigrationToColoringGroups(coloring, tableHeaders);

            coloring.scheme = "COLORING_GROUPS";

            // Special cases for table coloring schemes other than COLORING_GROUPS and still in use.
            // No coloring groups are added for MEANING_AND_STATUS and ALL_COLUMNS_VALUES table coloring schemes.
            // If some rules were defined in the legacy individualColumnsRules, it goes through a full migration.
            if (prevScheme === "MEANING_AND_STATUS" || prevScheme === "ALL_COLUMNS_VALUES") {
                WT1.event("conditional-formatting-migrate", {
                    details: "only switched to coloring groups",
                    fromScheme: prevScheme,
                    nbRules: coloring.coloringGroups.length,
                    toScheme: coloring.scheme,
                });

                // If some coloring groups already exists, the coloring of the table has already been customized and the user might want
                // to just go back to this existing customized coloring. So the coloring needs to be saved.
                // Otherwise, the coloring of the table is "neutral" (no coloring groups) and does not need to be saved yet.
                return coloring.coloringGroups.length > 0;
            }

            WT1.event("conditional-formatting-migrate", {
                fromScheme: prevScheme,
                nbRules: coloring.coloringGroups.length,
                toScheme: coloring.scheme,
            });

            return true;
        }

        function prepareMigrationToColoringGroups(coloring, tableHeaders) {
            if (!coloring.coloringGroups) {
                coloring.coloringGroups = [];
            }

            // Helpers to simplify migration from INDIVIDUAL_COLUMNS_RULES
            const withEnabledRulesColoringGroupIdxs = [];

            // Handles existing conditional formatting rules: creates a "backup" coloring group for each column
            // that had some user-defined rules on them.
            // Copies each group of rules for each column in their associated coloring group.
            // Also enables all those rules as they might have been "disabled" in the legacy color logic.
            if (coloring.individualColumnsRules && coloring.individualColumnsRules.length > 0) {
                // Older groups of rules might still exist in this list, and needs to be discarded
                const alreadyProcessedColumns = {};

                for (const columnRules of coloring.individualColumnsRules) {
                    // Already found more recent rules
                    if (columnRules.column in alreadyProcessedColumns) {
                        continue;
                    }

                    const rulesGroupFilterDescs = [];
                    let hasSomeEnabledRules = false;
                    if (columnRules.rulesDesc && columnRules.rulesDesc.length > 0) {
                        // Copies and enables each rule (filter description)
                        for (const filterDesc of columnRules.rulesDesc) {
                            if (filterDesc.enabled) {
                                hasSomeEnabledRules = true;
                            }

                            rulesGroupFilterDescs.push({ ...filterDesc, enabled: true });
                        }
                    }

                    coloring.coloringGroups.push(
                        newColoringGroup({
                            scheme: "RULES",
                            scope: "COLUMNS",
                            targetedColumnNames: [columnRules.column],
                            rulesGroupFilterDescs,
                            enabled: coloring.scheme === "INDIVIDUAL_COLUMNS_VALUES" ? false : hasSomeEnabledRules,
                        })
                    );

                    // Associates a column with some enabled rules to its coloring group index
                    if (hasSomeEnabledRules) {
                        withEnabledRulesColoringGroupIdxs.push(coloring.coloringGroups.length - 1);
                    }

                    alreadyProcessedColumns[columnRules.column] = true;
                }
            }

            switch (coloring.scheme) {
                // No coloring group is added to ensure consistency between the first (full) migration
                // and the following switches from MEANING_AND_STATUS / ALL_COLUMNS_VALUES to COLORING_GROUPS.
                case "MEANING_AND_STATUS":
                case "ALL_COLUMNS_VALUES":
                    break;

                // Adds a coloring group of scheme COLOR_SCALE for each column already colored by scale
                case "INDIVIDUAL_COLUMNS_VALUES":
                    for (const columnName of coloring.individualColumns) {
                        const coloringGroup = newColoringGroup({
                            scope: "COLUMNS",
                            targetedColumnNames: [columnName],
                            scheme: "COLOR_SCALE",
                        });
                        coloringGroup.colorScaleDef = newColorScaleDef([columnName], tableHeaders, true);
                        coloring.coloringGroups.push(coloringGroup);
                    }
                    break;

                // Re-orders and enables the "backup" RULES scheme coloring groups containing rules that were enabled before,
                // to keep the same formatting than with INDIVIDUAL_COLUMNS_RULES
                case "INDIVIDUAL_COLUMNS_RULES":
                    const reorderedColoringGroups = [];

                    // First adds the coloring groups containing rules that were disabled before
                    for (const [idx, coloringGroup] of coloring.coloringGroups.entries()) {
                        if (!withEnabledRulesColoringGroupIdxs.includes(idx)) {
                            reorderedColoringGroups.push(coloringGroup);
                        }
                    }

                    // Then adds and enables the coloring groups containing rules that were enabled before
                    for (const idx of withEnabledRulesColoringGroupIdxs) {
                        coloring.coloringGroups[idx].enabled = true;
                        reorderedColoringGroups.push(coloring.coloringGroups[idx]);
                    }

                    coloring.coloringGroups = reorderedColoringGroups;
                    break;

                // Adds a coloring group based on the "single column", targeting all columns and of scheme COLOR_SCALE
                case "SINGLE_COLUMN_VALUES":
                    const coloringGroup = newColoringGroup({
                        basedOnColumnName: coloring.singleColumn,
                        scope: "ALL_COLUMNS_BASED_ON_ANOTHER_COLUMN",
                        scheme: "COLOR_SCALE",
                    });
                    coloringGroup.colorScaleDef = newColorScaleDef([coloring.singleColumn], tableHeaders, true);
                    coloring.coloringGroups.push(coloringGroup);
                    break;

                // Moves the coloring group associated to the single column at the top, enables it and updates its scope to "based on another column"
                case "SINGLE_COLUMN_RULES": {
                    const singleColoringGroupIdx = coloring.coloringGroups.findIndex(
                        group =>
                            group.targetedColumnNames.length === 1 &&
                            group.targetedColumnNames[0] === coloring.singleColumn
                    );

                    if (singleColoringGroupIdx === -1) {
                        break;
                    }
                    const singleColoringGroup = coloring.coloringGroups[singleColoringGroupIdx];
                    singleColoringGroup.enabled = true;
                    singleColoringGroup.scope = "ALL_COLUMNS_BASED_ON_ANOTHER_COLUMN";
                    singleColoringGroup.basedOnColumnName = coloring.singleColumn;
                    singleColoringGroup.targetedColumnNames = [];

                    coloring.coloringGroups.splice(singleColoringGroupIdx, 1);
                    coloring.coloringGroups.push(singleColoringGroup);
                    break;
                }

                // No coloring group is added, uses default coloring group logic
                case "SINGLE_VALUE_HIGHLIGHT":
                case "FILL_ONLY":
                default:
                    break;
            }

            // Deletes legacy to avoid migrating them again
            delete coloring.individualColumns;
            delete coloring.individualColumnsRules;
        }

        function getMeaning(targetedColumns, tableHeaders) {
            let firstMeaning = undefined;

            if (targetedColumns && targetedColumns.length && tableHeaders) {
                const column = tableHeaders.find(({ name }) => name === targetedColumns[0]);
                if (column) {
                    firstMeaning = column.meaningLabel;

                    // Check that all targeted columns have the same meaning
                    for (let i = 1; i < targetedColumns.length; i++) {
                        const column = tableHeaders.find(({ name }) => name === targetedColumns[i]);
                        if (!column || column.meaningLabel !== firstMeaning) {
                            return undefined;
                        }
                    }
                }
            }
            return firstMeaning;
        }

        const defaultCustomPalette = {
            colors: ["#66DAEB", "#FFB867"],
            fixedValues: false, // Specify values check box
            id: DKU_PALETTE_NAMES.CUSTOM,
            name: "Custom Palette",
            values: [], // For specify values when fixedValues is true (null for auto or number)
        };

        function newColorScaleDef(targetedColumns, tableHeaders, migration = false) {
            let paletteName = migration ? DKU_PALETTE_NAMES.CUSTOM : "default";
            let customPalette = angular.copy(defaultCustomPalette);
            const meaning = getMeaning(targetedColumns, tableHeaders);
            let useSample2 = false;

            if (meaning === "Gender") {
                // Legacy palette: CG
                customPalette.colors = scalePaletteForGender;
            } else if (meaning === "Boolean") {
                // Legacy palette: CB
                customPalette.colors = scalePaletteForBoolean;
            } else if (
                meaning === "Integer" ||
                meaning === "Decimal" ||
                meaning === "Datetime with tz" ||
                meaning === "Date only" ||
                meaning === "Datetime no tz" ||
                meaning === "Temperature"
            ) {
                // Legacy palette: CN
                customPalette.colors = scalePaletteForNumber;
            } else if (meaning !== undefined) {
                // Legacy palette: CA1
                customPalette.colors = scalePaletteForString;
                useSample2 = true;
            } else {
                paletteName = "default";
            }

            let colorScaleDef = {
                colorPalette: paletteName,
                customPalette: customPalette,
                max: -1,
                min: 0,
                paletteType: "CONTINUOUS",
                quantizationMode: "NONE",
                useSample2: useSample2,
            };
            colorScaleDef.sample = getColorScaleDefSample(colorScaleDef);
            return colorScaleDef;
        }

        function getColorScaleDefSample(colorScaleDef) {
            let sample = [];
            if (colorScaleDef) {
                if (
                    colorScaleDef.colorPalette === DKU_PALETTE_NAMES.CUSTOM &&
                    colorScaleDef.customPalette &&
                    colorScaleDef.customPalette.colors
                ) {
                    sample = colorScaleDef.customPalette.colors;
                } else {
                    const continuousPalette = ChartColorUtils.getContinuousPalette(colorScaleDef.colorPalette);
                    if (continuousPalette) {
                        sample = colorScaleDef.useSample2 ? continuousPalette.sample2 : continuousPalette.sample;
                    }
                }
            }
            return sample;
        }

        function getColumnType(targetedColumns, tableHeaders) {
            const meaning = getMeaning(targetedColumns, tableHeaders);

            if (meaning === "Integer" || meaning === "Decimal" || meaning === "Temperature") {
                // Numerical
                return "NUM";
            }
            if (meaning === "Gender" || meaning === "Boolean") {
                // Binary
                return "BIN";
            }
            // Categorical
            return "CAT";
        }

        /**
         * Inits a ColoringGroup by enforcing a complete structure ready to be persisted
         *
         * The new coloring group is enabled by default.
         */
        function newColoringGroup({
            scheme,
            scope,
            enabled,
            basedOnColumnName,
            targetedColumnNames,
            rulesGroupFilterDescs,
        }) {
            return {
                scope: scope ?? "COLUMNS",
                scheme: scheme ?? "MEANING_AND_STATUS",
                enabled: enabled !== undefined && enabled !== null ? enabled : true,
                basedOnColumnName, // Can be null
                targetedColumnNames: targetedColumnNames ?? [],
                rulesGroup: rulesGroupFilterDescs ? { filterDescs: rulesGroupFilterDescs } : { filterDescs: [] },
            };
        }

        const conditionalFormattingEditorService = {
            addUiRuleToGroup,
            getBorderColorFromRule,
            getColorScaleDefSample,
            getDefaultFilterDesc,
            getFilterDescriptionString,
            getRuleStyles,
            getScaleBorderGradient,
            getScalePaletteForNumber,
            getColumnType,
            isAllColumnsBasedOnAnotherColumn,
            isNoneColor,
            isSchemeMeaning,
            isSchemeRules,
            isSchemeScale,
            migrateToColoringGroups,
            newColoringGroup,
            newColorScaleDef,
            prepareMigrationToColoringGroups,
        };
        return conditionalFormattingEditorService;
    }
})();
