(function () {
    "use strict";
    var app = angular.module("dataiku.admin.globalApiKeyInput", []);
    /* A <span> that display details about an encrypted api key field, and allow to create and bind a new one. Put it between some <label> and a <span class="help-inline"> */
    app.component("globalApiKeyInput", {
        bindings: {
            apiKey: '=',
            newKeyLabel: '<',       // When creating a new key, the label to use
            newKeyDescription: '<', // When creating a new key, the description to use
        },
        template: `
        <span class="global-api-key-input">
            <input id="apikey"
                type="password"
                ng-model="$ctrl.apiKey"/>
            <span class="overlay-disablable-item-container">
                <span toggle="tooltip" container="body" data-placement="right"
                    title="{{apiKeyDescription}}" >
                    <i ng-if="!apiKeyWarning" class="ng-scope icon-info-sign text-info"></i>
                    <i ng-if="apiKeyWarning" class="ng-scope icon-dku-warning text-warning"></i>
                </span>
                <button class="btn btn--secondary"
                        title="New Global API key"
                        ng-click="createGlobalApiKey()">
                        <i class="icon-plus" ></i>
                </button>
                <span class="overlay-disabled-item"
                    ng-if="!isInstanceAdmin"
                    toggle="tooltip"
                    container="body"
                    title="Only instance administrators can check or create an API key">
                </span>
            </span>
        </div>
        `,
    
        controller: function($scope, $rootScope, $timeout, $filter, DataikuAPI, CreateModalFromTemplate, ClipboardUtils) {
            const $ctrl = this;
            $scope.isInstanceAdmin = $rootScope.appConfig.admin;
            var getKeyDebounceTimeout;
        
            $scope.$watch("$ctrl.apiKey", function(nv) {
                $scope.apiKeyWarning = false;
                $scope.apiKeyDescription = "";
                if (nv && $scope.isInstanceAdmin) {
                    if (getKeyDebounceTimeout) {
                        $timeout.cancel(getKeyDebounceTimeout);
                    }
            
                    getKeyDebounceTimeout = $timeout(function() {
                        DataikuAPI.admin.publicApi.getGlobalKeyDetails(nv).then(function ({data}) {
                            $scope.apiKeyDescription = "Key Label: " + data.label + ", Created on: "+ $filter("friendlyDate")(data.createdOn, 'd MMMM yyyy');
                        }).catch(function (err) {
                            $scope.apiKeyDescription = err.data.detailedMessage;
                            $scope.apiKeyWarning = true;
                        });
                    }, 500); // 500ms delay
                } else {
                    $scope.apiKeyWarning = true;
                    $scope.apiKeyDescription = "No key defined";
                }
            });
    
            $scope.createGlobalApiKey = function() {
                let apiKey = {
                    label : $ctrl.newKeyLabel,
                    description : $ctrl.newKeyDescription,
                    globalPermissions : {admin: true}
                };
                DataikuAPI.admin.publicApi.createGlobalKey(apiKey).success(function (data) {
                    $scope.apiKeyWarning = false;
                    $ctrl.apiKey = data.key;
                    CreateModalFromTemplate("/templates/admin/security/new-api-key-modal.html", $scope, null, function(newScope) {
                        newScope.hashedApiKeysEnabled = $rootScope.appConfig.hashedApiKeysEnabled;
                        newScope.key = data;

                        newScope.copyKeyToClipboard = function() {
                            ClipboardUtils.copyToClipboard(data.key, 'Copied to clipboard.');
                        }
        
                        newScope.viewQRCode = function() {
                            CreateModalFromTemplate("/templates/admin/security/api-key-qrcode-modal.html", $scope, null, function (newScope) {
                                newScope.apiKeyQRCode = JSON.stringify({
                                    k : data.key,
                                    u : $rootScope.appConfig.dssExternalURL
                                });
                            });
                        };
                    });
                }).error(setErrorInScope.bind($scope));
            };
        }
    
    });
})();
    