(function () {
    'use strict';
    var app = angular.module('dataiku.credentials', []);

    app.factory("CredentialDialogs", function(CreateModalFromTemplate, $q, $state, $timeout, $window, DataikuAPI) {

        function isConnectionCredential(credential) {
            return credential && ['CONNECTION', 'VIRTUAL_CONNECTION', 'DATABRICKS_INTEGRATION'].includes(credential.requestSource);
        }

        function enterCredential($scope, credential) {
            let deferred = $q.defer();

            resetErrorInScope($scope);

            if (credential.type === "SINGLE_FIELD" || credential.type === "BASIC") {
                CreateModalFromTemplate("/templates/profile/edit-connection-credential-modal.html", $scope, null, function (newScope) {
                    newScope.credential = credential;
                    newScope.plugin = credential.pluginCredentialRequestInfo;
                    newScope.credential.password = "";
                    newScope.passwordFieldTitle = credential.type === 'BASIC' ? "Password" : "Credentials";
                    newScope.isConnectionCredential = isConnectionCredential(credential);

                    newScope.confirm = function() {
                        let apiCall;
                        if (isConnectionCredential(credential)) {
                            apiCall = DataikuAPI.profile.setBasicConnectionCredential(newScope.credential.connection,
                                          newScope.credential.user, newScope.credential.password);
                        } else {
                            apiCall = DataikuAPI.profile.pluginCredentials.setBasicCredential(newScope.plugin.pluginId,
                                          newScope.plugin.paramSetId, newScope.plugin.presetId, newScope.plugin.paramName,
                                          newScope.credential.user, newScope.credential.password);
                        }

                        return apiCall.success(function () {
                                newScope.dismiss();
                                deferred.resolve(false);
                            }).error(deferred.reject);
                    }
                });
            } else if (credential.type === "AZURE_OAUTH_DEVICECODE") {
                CreateModalFromTemplate("/templates/profile/edit-azure-oauth-connection-credential-modal.html", $scope, null, function (newScope) {
                    newScope.uiState = {
                        step: "STARTUP"
                    }
                    newScope.credential = credential;

                    newScope.startStep1 = function() {
                        const state = {
                            name: $state.current.name,
                            params: $state.params
                         }
                        DataikuAPI.profile.connectionCredentials
                            .azureOAuthDeviceCodeDanceStep1(
                                urlWithProtocolAndHost(),
                                JSON.stringify(state),
                                newScope.credential)
                            .success(function (data) {
                                newScope.uiState.step = "STEP1_COMPLETE";
                                newScope.uiState.deviceCode = data;
                            }).error(setErrorInScope.bind($scope));
                    }
                    newScope.startStep2 = function() {
                        const state = {
                            name: $state.current.name,
                            params: $state.params
                         }
                        DataikuAPI.profile.connectionCredentials
                            .azureOAuthDeviceCodeDanceStep2(
                                urlWithProtocolAndHost(),
                                JSON.stringify(state),
                                newScope.credential)
                            .success(function() {
                                newScope.dismiss();
                                deferred.resolve(false);
                            }).error(setErrorInScope.bind($scope));
                    }

                    newScope.startStep1();
                });
            } else if (credential.type === "OAUTH_REFRESH_TOKEN") {
                CreateModalFromTemplate("/templates/profile/edit-oauth-connection-credential-modal.html", $scope, null, function (newScope) {
                    newScope.credential = credential;
                    newScope.plugin = credential.pluginCredentialRequestInfo;
                    newScope.isConnectionCredential = isConnectionCredential(credential);

                    newScope.confirm = function() {
                        const state = {
                           name: $state.current.name,
                           params: $state.params
                        }
                        return DataikuAPI.profile.startOAuth2AuthorizationFlow(
                            urlWithProtocolAndHost(),
                            JSON.stringify(state),
                            newScope.credential)
                        .success(function (data) {
                            // Redirect to begin authorization process
                            $window.location.href = data;
                            deferred.resolve(true);
                        }).error(setErrorInScope.bind($scope));
                    }
                });
            }
            return deferred.promise;
        }

        function oauth2Success(scope, credential, plugin) {
            return CreateModalFromTemplate("/templates/profile/oauth2-callback-success-modal.html", scope, null, function (newScope) {
                newScope.credential = credential;
                newScope.plugin = plugin;
                newScope.isConnectionCredential = isConnectionCredential(newScope.credential);
            });
        }

        function oauth2Error(scope, data, credential, plugin, error) {
            return CreateModalFromTemplate("/templates/profile/oauth2-callback-error-modal.html", scope, null, function (newScope) {
                newScope.credential = credential;
                newScope.plugin = plugin;
                newScope.error = error;
                newScope.isConnectionCredential = isConnectionCredential(newScope.credential);
                newScope.tryAgain = enterCredential.bind(this, newScope, newScope.credential);
            });
        }

        return {
            enterCredential: enterCredential,
            oauth2Success: oauth2Success,
            oauth2Error: oauth2Error
        };
    });

    app.controller('OAuth2CallbackController', function($scope, $rootScope, $timeout, $location, $state, DataikuAPI, CredentialDialogs) {
        const code = $location.search().code;
        const state = $location.search().state;
        const error = $location.search().error;
        const errorDescription = $location.search().error_description;
        const errorUri = $location.search().error_uri;

        $location.search("code", null);
        $location.search("state", null);
        $location.search("error", null);
        $location.search("error_description", null);
        $location.search("error_uri", null);

        if (code || error || state) {
            DataikuAPI.profile.exchangeAuthorizationCode(code, state, error, errorDescription, errorUri)
                .success(function(data) {
                    $timeout(() => data.isSuccess
                        ? CredentialDialogs.oauth2Success($rootScope, data.authorizationRequestInfo.credential, data.authorizationRequestInfo.credential.pluginCredentialRequestInfo)
                        : CredentialDialogs.oauth2Error($rootScope, data, data.authorizationRequestInfo.credential, data.authorizationRequestInfo.credential.pluginCredentialRequestInfo, data)
                    , 100);
                    const userCurrentState = JSON.parse(data.authorizationRequestInfo.userCurrentState);
                    $state.go(userCurrentState.name, userCurrentState.params);
                })
                .error(setErrorInScope.bind($scope));
        }
    });

}());