(function(){
'use strict';

const app = angular.module('dataiku.agentreviews', ['platypus.utils']);

app.service("AgentReviewsService", function(DataikuAPI, $stateParams) {
    var svc = this;
    svc.listItems = [];

    svc.fetchAgentReviews = function() {
        return DataikuAPI.agentreviews.listHeads($stateParams.projectKey)
            .success(function(data) {
                svc.listItems.length = 0; 
                angular.extend(svc.listItems, data);
            });
    };
});

app.controller("AgentReviewsListController", function($scope, $controller, $state, TopNav, AgentReviewsService) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    TopNav.setNoItem();
    TopNav.setLocation(TopNav.TOP_AGENT_REVIEWS, TopNav.LEFT_AGENT_REVIEWS, TopNav.TABS_NONE, null);

    $scope.sortBy = [
        { value: 'name', label: 'Name' },
        { value: 'agentDisplayName', label: 'Agent' },
        { value: 'owner', label: 'Owner'},
        { value: '-lastModifiedOn', label: 'Last modified'},
    ];

    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            },
            agentId: $state.params.filterOn ? [$state.params.filterOn] : []
        },
        filterParams: {
            userQueryTargets: ["name", "owner", "tags"],
            propertyRules: {tag: 'tags'},
            exactMatch: ['agentId']
        },
        orderQuery: "-lastModifiedOn",
        orderReversed: false
    }, $scope.selection || {});

    $scope.sortCookieKey = 'agentreviews';
    $scope.maxItems = 20;

    $scope.agentList = [];

    $scope.list = function() {
        AgentReviewsService.fetchAgentReviews().success(function() {
            $scope.listItems = AgentReviewsService.listItems;
            $scope.restoreOriginalSelection();
            const uniqueAgents = {};
            const targetAgentId = $state.params.agentId;
            const targetVersionId = $state.params.versionId;
            let existingReview = false;

            $scope.listItems.forEach(element => {
                if (!uniqueAgents[element.agentId]) {
                    uniqueAgents[element.agentId] = {
                        id: element.agentId,
                        agentDisplayName: element.agentDisplayName
                    };
                }
                if (targetAgentId && element.agentId === targetAgentId && (!targetVersionId || element.agentVersion === targetVersionId)) {
                    existingReview = true;
                }
            });
            $scope.agentList = Object.values(uniqueAgents).sort((a, b) => a.agentDisplayName.localeCompare(b.agentDisplayName));
            if (targetAgentId) {
                if (existingReview) {
                    $state.go(
                        'projects.project.agentreviews.list',
                        { filterOn: targetAgentId, agentId: null, versionId: null },
                        { location: 'replace', reload: true }
                    );
                } else {
                    $scope.showCreateAgentReviewModal(targetAgentId, targetVersionId);
                }
            }
        });
    }
    $scope.list();

    $scope.setAgentFilter = function(agentId) {
        const index = $scope.selection.filterQuery.agentId.indexOf(agentId);
        if (index !== -1) {
            $scope.selection.filterQuery.agentId.splice(index, 1);
        } else {
            $scope.selection.filterQuery.agentId.push(agentId);
        }
    }
});

app.controller("AgentReviewCreationController", function( $scope, DataikuAPI, StateUtils, SmartId, $stateParams, AgentReviewsService, WT1, $location) {

    $scope.newAgentReview = {};
    $scope.activeVersionAnnotation = function(value) {
        return value.active ? "Active" : undefined;
    };

    $scope.versionLabel = function(value) {
        return value.versionId ?? "Always use active version";
    };

    $scope.onChangeAgent = function() {
        $scope.versionsList = [];
        DataikuAPI.savedmodels.llmCommon.getStatus($stateParams.projectKey, SmartId.create($scope.newAgentReview.agent.id, $scope.newAgentReview.agent.projectKey)).success(function (data) {

            const regex = new RegExp(`^(.*)(\\d+)$`);
            const maxNumber = AgentReviewsService.listItems.map(item => item.name).reduce((max, str) => {
                const match = str.match(regex);
                if (match && match[1] === $scope.newAgentReview.agent.label + " review ") {
                    const num = match[2] ? parseInt(match[2]) : 0;
                    return num > max ? num : max;
                }
                return max;
            }, 0);
            if (!$scope.newAgentReview.name) {
                $scope.newAgentReview.name = $scope.newAgentReview.agent.label + " review " + (maxNumber + 1);
            }
            $scope.versionsList = [{versionId: null}, ...data.versions];
            if ($scope.versionId) {
                const version = $scope.versionsList.find(v => v.versionId === $scope.versionId);
                if (version) {
                    if (version.active) {
                        $scope.newAgentReview.version = $scope.versionsList[0];
                    } else {
                        $scope.newAgentReview.version = version;
                    }
                }
                $scope.versionId = undefined;
            } else {
                $scope.newAgentReview.version = $scope.versionsList[0];
            }
        }).error(function(data, status, headers, config, statusText) {
            if (status === 404) {
                $scope.versionsList = [{versionId: null}];
                $scope.newAgentReview.version = $scope.versionsList[0];
            } else {
                setErrorInScope(data, status, headers, config, statusText);
            }
        });
    }

    DataikuAPI.taggableObjects.listAccessibleObjects($stateParams.projectKey, "SAVED_MODEL", "READ").success(function (data) {
        $scope.agentsList = data.filter(sm => ["TOOLS_USING_AGENT", "PLUGIN_AGENT", "PYTHON_AGENT"].includes(sm.object.savedModelType));

        const labelCounts = $scope.agentsList.reduce((acc, item) => {
            acc[item.label] = (acc[item.label] || 0) + 1;
            return acc;
        }, {});

        $scope.agentsList = $scope.agentsList.map(item => {
            if (labelCounts[item.label] > 1) {
                return {
                ...item,
                labelWithId: `${item.label} (${item.id})`
                };
            }
            return item;
        });

        if ($scope.agentId) {
            $scope.newAgentReview.agent = $scope.agentsList.find(a => a.id === $scope.agentId);
            $scope.onChangeAgent();
            $scope.agentId = undefined;
        }
    }).error(setErrorInScope.bind($scope));

    $scope.createReview = function() {
        WT1.event('agent-review-action', { action: 'review:create-review' });
        $scope.creatingAgentReview = true;
         DataikuAPI.agentreviews.save(
            {
                projectKey: $stateParams.projectKey,
                name: $scope.newAgentReview.name,
                agentSmartId: SmartId.create($scope.newAgentReview.agent.id, $scope.newAgentReview.agent.projectKey),
                agentVersion: $scope.newAgentReview.version.versionId
            }).success(function(data) {
                $scope.creatingAgentReview = false;
                StateUtils.go.agentReview(data.id);
                $scope.resolveModal();
        }).error(setErrorInScope.bind($scope));
    }
});


app.controller("AgentReviewPageRightColumnActions", function($controller, $scope, $rootScope, DataikuAPI, $stateParams, ActiveProjectKey, TopNav) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.selection = {};

    DataikuAPI.agentreviews.get(ActiveProjectKey.get(), $stateParams.agentReviewId).success((data) => {
        $scope.selection = {
            selectedObject : data, confirmedItem : data,
        };
        $scope.selection.selectedObject.nodeType = "AGENT_REVIEW";


        TopNav.setItem(TopNav.ITEM_AGENT_REVIEW, $stateParams.agentReviewId, {
            name : data.name,
        });

    }).error(setErrorInScope.bind($scope));

    $scope.renameObjectAndSave = function(newName) {
        $scope.selection.selectedObject.name = newName;
        return DataikuAPI.agentreviews.save($scope.selection.selectedObject);
    };

    function updateUserInterests() {
        DataikuAPI.interests.getForObject($rootScope.appConfig.login, "AGENT_REVIEW", ActiveProjectKey.get(), $scope.selection.selectedObject.id).success(function(data) {
            $scope.selection.selectedObject.interest = data;
            $scope.agentReviewData.interest = data;
        }).error(setErrorInScope.bind($scope));
    }

    const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
    $scope.$on("$destroy", interestsListener);
});


app.directive('agentReviewRightColumnSummary', function($controller, $state, $stateParams, $rootScope, DataikuAPI, CreateModalFromTemplate, QuickView,
    ActiveProjectKey, ActivityIndicator, TopNav, WT1, GlobalProjectActions) {

    return {
        templateUrl :'/templates/agentreviews/right-column-summary.html',

        link : function(scope) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});
            $controller('_TaggableObjectPageRightColumnActions', {$scope: scope});

            scope.refreshData = function() {
                const projectKey = scope.selection.selectedObject.projectKey;
                const name = scope.selection.selectedObject.name;

                DataikuAPI.agentreviews.getFullInfo(ActiveProjectKey.get(), scope.selection.selectedObject.id).then(function({data}){
                    if (!scope.selection.selectedObject || scope.selection.selectedObject.projectKey != projectKey || scope.selection.selectedObject.name != name) {
                        return;
                    }
                    scope.agentReviewData = data;
                    scope.agentReview = data.agentReview;
                    scope.selection.selectedObject.interest = data.interest;
                }).catch(setErrorInScope.bind(scope));
            };

            function save() {
                return DataikuAPI.agentreviews.save(scope.agentReview, {summaryOnly: true})
                .success(function() {
                    ActivityIndicator.success("Saved");
                }).error(setErrorInScope.bind(scope));
            }

            scope.$on("objectSummaryEdited", save);

            scope.$watch("selection.selectedObject",function() {
                if (scope.selection.selectedObject != scope.selection.confirmedItem) {
                    scope.agentReview = null;
                    scope.objectTimeline = null;
                }
            });

            scope.$watch("selection.confirmedItem", function(nv) {
                if (!nv) return;
                if (!nv.projectKey) {
                    nv.projectKey = ActiveProjectKey.get();
                }
                scope.refreshData();
            });

            scope.deleteAgentReview = function() {
                GlobalProjectActions.deleteTaggableObject(scope, 'AGENT_REVIEW', scope.selection.selectedObject.id, scope.selection.selectedObject.name);
            };
        }
    }
});


})();