(function() {
    'use strict';
    const CONVERSION_FIELD_ICON = 'icon';
    const CONVERSION_FIELD_NAME = 'name';
    const CONVERSION_FIELD_OTHER_NAME = 'otherName';
    const CONVERSION_FIELD_LANGUAGE = 'language';
    const CONVERSION_FIELD_CATEGORY = 'category';

    /**
     *
     * @param {string} icon
     * @param {string} name
     * @param {string} otherName
     * @param {boolean} newIconOnly
     * @returns {Object} type definition
     */
    function buildTypeDefinition(icon, name, otherName, newIconOnly=false) {
        return { [CONVERSION_FIELD_ICON]: icon, [CONVERSION_FIELD_NAME]: name, [CONVERSION_FIELD_OTHER_NAME]: otherName || name, newIconOnly: newIconOnly };
    }

    /**
     *
     * @param {string} icon icon class name
     * @param {string} category category icon is part of (e.g., visual, code, etc.)
     * @param {string} language language of code/continuous recipe
     * @param {string} newIconOnly true if icon is only available in the modern icon set
     * @returns {Object} recipe type definition
     */
    function buildRecipeTypeDefinition(icon, category, language, newIconOnly) {
        return {
            ...buildTypeDefinition(icon, undefined, undefined, newIconOnly),
            [CONVERSION_FIELD_CATEGORY]: category,
            [CONVERSION_FIELD_LANGUAGE]: language
        };
    }

    const FS_PROVIDER_TYPES = {
        'filesystem': buildTypeDefinition('icon-server_file_system_1', `Server's Filesystem`),
        'hdfs': buildTypeDefinition('icon-HDFS', 'Hadoop HDFS'),
        'ftp': buildTypeDefinition('icon-uncached_FTP', 'FTP'),
        'sftp': buildTypeDefinition('icon-dataset-ssh marker-sftp', 'SFTP'),
        'scp': buildTypeDefinition('icon-dataset-ssh marker-scp', 'SCP'),
        'azure': buildTypeDefinition('icon-azure-storage', 'Azure Blob Storage'),
        'gcs': buildTypeDefinition('icon-google-cloud-storage', 'Google Cloud Storage'),
        's3': buildTypeDefinition('icon-amazon_s3', 'Amazon S3'),
        'url': buildTypeDefinition(undefined, 'HTTP or FTP URL'),
        'sharepointonline': buildTypeDefinition('icon-microsoft-sharepoint', 'SharePoint Online'),
    };

    const COMMON_TYPES = {
        ...FS_PROVIDER_TYPES,
        'hiveserver2': buildTypeDefinition('icon-dku-hive', 'Hive'),
        'uploadedfiles': buildTypeDefinition('icon-upload', 'Uploaded Files'),
        'mongodb': buildTypeDefinition('icon-mongo_db', 'MongoDB'),
        'dynamodb': buildTypeDefinition('icon-dynamoDB', 'DynamoDB'),
        'mysql': buildTypeDefinition('icon-mySQL', 'MySQL'),
        'cassandra': buildTypeDefinition('icon-cassandra_1', 'Cassandra'),
        'iceberg': buildTypeDefinition('icon-iceberg', 'Iceberg'),
        'postgresql': buildTypeDefinition('icon-postgreSQL', 'PostgreSQL'),
        'alloydb': buildTypeDefinition('icon-dku-alloydb', 'Google AlloyDB'),
        'vertica': buildTypeDefinition('icon-HP_vertica', 'Vertica'),
        'redshift': buildTypeDefinition('icon-amazon_redshift', 'Amazon Redshift'),
        'greenplum': buildTypeDefinition('icon-greenplum', 'Greenplum'),
        'databricks': buildTypeDefinition('icon-dku-databricks', 'Databricks'),
        'databricksvolume': buildTypeDefinition('icon-dku-databricks-volume', 'Databricks Volume'),
        'databrickslakebase': buildTypeDefinition('icon-dku-databricks-lakebase', 'Databricks Lakebase'),
        'teradata': buildTypeDefinition('icon-teradata', 'Teradata'),
        'oracle': buildTypeDefinition('icon-oracle', 'Oracle'),
        'athena': buildTypeDefinition('icon-athena', 'Athena'),
        'trino': buildTypeDefinition('icon-trino', 'Trino'),
        'denodo': buildTypeDefinition('icon-denodo', 'Denodo'),
        'kdbplus': buildTypeDefinition('dku-icon-clock', 'KDB+', undefined, true),
        'treasuredata': buildTypeDefinition('icon-treasuredata', 'Treasure Data'),
        'sqlserver': buildTypeDefinition('icon-sqlserver', 'MS SQL Server'),
        'synapse': buildTypeDefinition('icon-dku-azure-synapse', 'Azure Synapse'),
        'fabricwarehouse': buildTypeDefinition('icon-microsoft-fabric-warehouse', 'MS Fabric Warehouse'),
        'netezza': buildTypeDefinition('icon-netezza', 'IBM Netezza'),
        'saphana': buildTypeDefinition('icon-sap-hana', 'SAP Hana'),
        'bigquery': buildTypeDefinition('icon-google-bigquery', 'Google BigQuery'),
        'snowflake': buildTypeDefinition('icon-snowflake', 'Snowflake'),
        'jdbc': buildTypeDefinition('icon-other_sql', 'SQL database (JDBC)', 'Other SQL databases'),
        'elasticsearch': buildTypeDefinition('icon-elasticsearch', 'ElasticSearch'),
        'twitter': buildTypeDefinition('icon-twitter', 'Twitter'),
        'sharepointonlinelist': buildTypeDefinition('icon-microsoft-sharepoint', 'SharePoint List'),
        'openai': buildTypeDefinition('icon-openai', 'OpenAI'),
        'azureopenai': buildTypeDefinition('icon-azure', 'Azure OpenAI'),
        'azureaifoundry': buildTypeDefinition('icon-azure', 'Azure AI Foundry'),
        'azurellm': buildTypeDefinition('icon-azure', 'Azure LLM'),
        'azureaisearch': buildTypeDefinition('icon-azure', 'Azure AI Search'),
        'cohere': buildTypeDefinition('icon-cohere', 'Cohere'),
        'anthropic': buildTypeDefinition('icon-anthropic', 'Anthropic'),
        'bedrock': buildTypeDefinition('icon-bedrock', 'AWS Bedrock'),
        'mosaicml': buildTypeDefinition('icon-mosaicml', 'MosaicML (deprecated)'),
        'stabilityai': buildTypeDefinition('dku-icon-stability-ai', 'Stability AI', undefined, true),
        'nvidia-nim': buildTypeDefinition('dku-icon-nvidia', 'NVIDIA NIM'), // key needs to be lowercase, see typeProcessor in services.js
        'sagemaker-genericllm': buildTypeDefinition('icon-model-sagemaker', 'Amazon SageMaker LLM'),
        'vertexaillm': buildTypeDefinition('icon-vertex', 'Vertex Generative AI'),
        'databricksllm': buildTypeDefinition('icon-dku-databricks', 'Databricks Mosaic AI'),
        'snowflakecortex': buildTypeDefinition('icon-snowflake', 'Snowflake Cortex'),
        'mistralai': buildTypeDefinition('icon-mistral', 'MistralAI'),
        'huggingfaceinferenceapi': buildTypeDefinition('icon-hugging-face', 'Hugging Face API'),
        'huggingfacelocal': buildTypeDefinition('icon-hugging-face', 'Local Hugging Face'),
        'pinecone': buildTypeDefinition('icon-pinecone', 'Pinecone'),
        'customllm': buildTypeDefinition('icon-puzzle-piece', 'Custom LLM'),
        'kafka': buildTypeDefinition('icon-kafka', 'Kafka'),
        'sqs': buildTypeDefinition('icon-sqs', 'SQS'),
        'httpsse': buildTypeDefinition('icon-httpsse', 'HTTP Server Sent Events'),
        'remotemcp': buildTypeDefinition('dku-icon-mcp-client-remote', 'Remote MCP', undefined, true),
    };

    const DATASET_TYPES = {
        ...COMMON_TYPES,
        'cachedhttp': buildTypeDefinition('icon-FTP-HTTP-SSH', 'HTTP (with cache)'),
        'filesinfolder': buildTypeDefinition('icon-box', 'Files in Folder'),
        'labels': buildTypeDefinition('icon-screenshot', 'Labels'),
        'http': buildTypeDefinition('icon-FTP-HTTP-SSH', 'HTTP'),
        'inline': buildTypeDefinition('icon-inline', 'Editable'),
        'jobsdb': buildTypeDefinition('icon-bar-chart', 'Metrics'),
        'statsdb': buildTypeDefinition('icon-tasks marker-internal-stats', 'Internal stats'),
        'experimentsdb': buildTypeDefinition('icon-dku-exp-tracking', 'Experiments'),
        'remotefiles': buildTypeDefinition('icon-FTP-HTTP-SSH'),
        'sample': buildTypeDefinition('icon-beaker'),
    };
    const STREAMING_ENDPOINT_TYPES = {
        'kafka': buildTypeDefinition('icon-double-angle-right'),
        'httpsse': buildTypeDefinition('icon-double-angle-right'),
        'sqs': buildTypeDefinition('icon-double-angle-right')
    };
    const CONNECTION_TYPES = {
        ...COMMON_TYPES,
        'ec2': buildTypeDefinition('icon-amazon_s3', 'Amazon S3'),
        'ssh': buildTypeDefinition('icon-FTP-HTTP-SSH', 'SCP/SFTP'),
        'vertexaillm': buildTypeDefinition('icon-vertex', 'Vertex Generative AI'),
        'sagemaker': buildTypeDefinition('icon-model-sagemaker', 'Amazon SageMaker'),
        'azureml': buildTypeDefinition('icon-model-azureml', 'Azure Machine Learning'),
        'vertexaimodeldeployment': buildTypeDefinition('icon-model-google-vertex', 'Google Vertex AI'),
        'databricksmodeldeployment': buildTypeDefinition('icon-dku-databricks', 'Databricks Model Depl.')
    };

    const CREDENTIAL_TYPES = {
        'single_field': buildTypeDefinition('icon-key', 'Secret'),
        'basic': buildTypeDefinition('icon-key', 'Password'),
        'azure_oauth_devicecode': buildTypeDefinition('icon-link', 'OAuth2'),
        'oauth_refresh_token': buildTypeDefinition('icon-link', 'OAuth2'),
    };

    const RECIPE_CATEGORIES = {
        CLUSTERING: 'clustering',
        SCORE: 'score',
        TRAIN: 'train',
        STANDALONE: 'standalone',
        EVALUATION: 'evaluation',
        EDA: 'eda',
        VISUAL: 'visual',
        CODE: 'code',
        NLP: 'nlp',
        CONTINUOUS: 'continuous'
    };
    const RECIPE_TYPES = {
        'clustering_cluster': buildRecipeTypeDefinition('icon-clustering_recipe', RECIPE_CATEGORIES.CLUSTERING),
        'clustering_scoring': buildRecipeTypeDefinition('icon-clustering_recipe', RECIPE_CATEGORIES.SCORE),
        'clustering_training': buildRecipeTypeDefinition('icon-train_recipe', RECIPE_CATEGORIES.TRAIN),
        'distinct': buildRecipeTypeDefinition('icon-visual_prep_distinct_recipe', RECIPE_CATEGORIES.VISUAL),
        'download': buildRecipeTypeDefinition('icon-visual_download_recipe', RECIPE_CATEGORIES.VISUAL),
        'eda_pca': buildRecipeTypeDefinition('icon-eda_pca_recipe', RECIPE_CATEGORIES.EDA), // [eda] fake icon making bridge between the old/new icon system
        'eda_stats': buildRecipeTypeDefinition('icon-eda_stats_recipe', RECIPE_CATEGORIES.EDA), // [eda] fake icon making bridge between the old/new icon system
        'eda_univariate': buildRecipeTypeDefinition('icon-eda_univariate_recipe', RECIPE_CATEGORIES.EDA), // [eda] fake icon making bridge between the old/new icon system
        'evaluation': buildRecipeTypeDefinition('icon-evaluation_recipe', RECIPE_CATEGORIES.EVALUATION),
        'standalone_evaluation': buildRecipeTypeDefinition('icon-standalone_evaluation_recipe', RECIPE_CATEGORIES.STANDALONE),
        'export': buildRecipeTypeDefinition('icon-visual_export_recipe', RECIPE_CATEGORIES.VISUAL),
        'extract_failed_rows': buildRecipeTypeDefinition('dku-icon-recipe-extract-failed-rows', RECIPE_CATEGORIES.VISUAL, undefined, true),
        'grouping': buildRecipeTypeDefinition('icon-visual_prep_group_recipe', RECIPE_CATEGORIES.VISUAL),
        'upsert': buildRecipeTypeDefinition('icon-visual_prep_upsert_recipe', RECIPE_CATEGORIES.VISUAL),
        'hive': buildRecipeTypeDefinition('icon-code_hive_recipe', RECIPE_CATEGORIES.CODE, 'text/x-hivesql'),
        'impala': buildRecipeTypeDefinition('icon-code_impala_recipe', RECIPE_CATEGORIES.CODE, 'text/x-hivesql'),
        'ipython': buildRecipeTypeDefinition('icon-python', 'python'),
        'labeling': buildRecipeTypeDefinition('icon-labeling', RECIPE_CATEGORIES.VISUAL),
        'join': buildRecipeTypeDefinition('icon-visual_prep_join_recipe', RECIPE_CATEGORIES.VISUAL),
        'fuzzyjoin': buildRecipeTypeDefinition('icon-visual_prep_fuzzyjoin_recipe', RECIPE_CATEGORIES.VISUAL),
        'generate_features': buildRecipeTypeDefinition('icon-visual_prep_auto_feature_generation_recipe', RECIPE_CATEGORIES.VISUAL),
        'geojoin': buildRecipeTypeDefinition('icon-visual_prep_geojoin_recipe', RECIPE_CATEGORIES.VISUAL),
        'merge_folder': buildRecipeTypeDefinition('icon-visual_prep_merge_folder_recipe', RECIPE_CATEGORIES.VISUAL),
        'list_folder_contents': buildRecipeTypeDefinition('icon-visual_prep_list_folder_contents_recipe', RECIPE_CATEGORIES.VISUAL),
        'list_access': buildRecipeTypeDefinition('icon-visual_prep_list_access_recipe', RECIPE_CATEGORIES.VISUAL),
        'julia': buildRecipeTypeDefinition('icon-code_julia_recipe', RECIPE_CATEGORIES.CODE, 'text/x-julia'),
        'pig': buildRecipeTypeDefinition('icon-code_pig_recipe', RECIPE_CATEGORIES.CODE, 'text/x-dkupig'),
        'pivot': buildRecipeTypeDefinition('icon-visual_prep_pivot_recipe', RECIPE_CATEGORIES.VISUAL),
        'prediction_scoring': buildRecipeTypeDefinition('icon-score_recipe', RECIPE_CATEGORIES.SCORE),
        'prediction_training': buildRecipeTypeDefinition('icon-train_recipe', RECIPE_CATEGORIES.TRAIN),
        'pyspark': buildRecipeTypeDefinition('icon-code_pyspark_recipe', RECIPE_CATEGORIES.CODE, 'text/x-python'),
        'python': buildRecipeTypeDefinition('icon-code_python_recipe', RECIPE_CATEGORIES.CODE, 'text/x-python'),
        'r': buildRecipeTypeDefinition('icon-code_r_recipe', RECIPE_CATEGORIES.CODE, 'text/x-rsrc'),
        'recipe': buildRecipeTypeDefinition('icon-circle', 'circle'),
        'sampling': buildRecipeTypeDefinition('icon-visual_prep_filter-sample_recipe', RECIPE_CATEGORIES.VISUAL),
        'shaker': buildRecipeTypeDefinition('icon-visual_prep_cleanse_recipe', RECIPE_CATEGORIES.VISUAL),
        'shell': buildRecipeTypeDefinition('icon-code_shell_recipe', RECIPE_CATEGORIES.CODE, 'text/x-sh'),
        'sort': buildRecipeTypeDefinition('icon-visual_prep_sort_recipe', RECIPE_CATEGORIES.VISUAL),
        'spark_scala': buildRecipeTypeDefinition('icon-code_spark_scala_recipe', RECIPE_CATEGORIES.CODE, 'text/x-scala'),
        'spark_sql_query': buildRecipeTypeDefinition('icon-code_sparksql_recipe', RECIPE_CATEGORIES.CODE, 'text/x-sql2'),
        'sparkr': buildRecipeTypeDefinition('icon-code_sparkr_recipe', RECIPE_CATEGORIES.CODE, 'text/x-rsrc'),
        'split': buildRecipeTypeDefinition('icon-visual_prep_split_recipe', RECIPE_CATEGORIES.VISUAL),
        'sql': buildRecipeTypeDefinition('icon-sql', 'sql'),
        'sql_query': buildRecipeTypeDefinition('icon-code_sql_recipe', RECIPE_CATEGORIES.CODE, 'text/x-sql2'),
        'sql_script': buildRecipeTypeDefinition('icon-code_sql_recipe', RECIPE_CATEGORIES.CODE, 'text/x-sql2'),
        'sync': buildRecipeTypeDefinition('icon-visual_prep_sync_recipe', RECIPE_CATEGORIES.VISUAL),
        'topn': buildRecipeTypeDefinition('icon-visual_prep_topn_recipe', RECIPE_CATEGORIES.VISUAL),
        'update': buildRecipeTypeDefinition('icon-visual_push_to_editable_recipe', RECIPE_CATEGORIES.VISUAL),
        'vstack': buildRecipeTypeDefinition('icon-visual_prep_vstack_recipe', RECIPE_CATEGORIES.VISUAL),
        'window': buildRecipeTypeDefinition('icon-visual_prep_window_recipe', RECIPE_CATEGORIES.VISUAL),
        'csync': buildRecipeTypeDefinition('icon-continuous_sync_recipe', RECIPE_CATEGORIES.CONTINUOUS),
        'ksql': buildRecipeTypeDefinition('icon-continuous_ksql_recipe', RECIPE_CATEGORIES.CONTINUOUS, 'text/x-sql'),
        'cpython': buildRecipeTypeDefinition('icon-continuous_python_recipe', RECIPE_CATEGORIES.CONTINUOUS, 'text/x-python'),
        'streaming_spark_scala': buildRecipeTypeDefinition('icon-continuous_spark_scala_recipe', RECIPE_CATEGORIES.CONTINUOUS, 'text/x-scala'),
        'prompt': buildRecipeTypeDefinition('icon-nlp_prompt_recipe', RECIPE_CATEGORIES.NLP),
        'nlp_llm_user_provided_classification': buildRecipeTypeDefinition('icon-nlp_text_classification_recipe', RECIPE_CATEGORIES.NLP),
        'nlp_llm_model_provided_classification': buildRecipeTypeDefinition('icon-nlp_text_classification_recipe', RECIPE_CATEGORIES.NLP),
        'nlp_llm_summarization': buildRecipeTypeDefinition('icon-nlp_summarization_recipe', RECIPE_CATEGORIES.NLP),
        'nlp_llm_finetuning': buildRecipeTypeDefinition('icon-nlp_fine_tuning_recipe', RECIPE_CATEGORIES.NLP),
        'nlp_llm_rag_embedding': buildRecipeTypeDefinition('icon-nlp_rag_embedding_recipe', RECIPE_CATEGORIES.NLP),
        'embed_documents': buildRecipeTypeDefinition('dku-icon-recipe-embed-document-circle-fill', RECIPE_CATEGORIES.NLP, undefined, true),
        'extract_content': buildRecipeTypeDefinition('dku-icon-recipe-extract-content-circle-fill', RECIPE_CATEGORIES.NLP, undefined, true),
        'nlp_llm_evaluation': buildRecipeTypeDefinition('icon-nlp_llm_evaluation_recipe', RECIPE_CATEGORIES.NLP),
        'nlp_agent_evaluation': buildRecipeTypeDefinition('icon-nlp_agent_evaluation_recipe', RECIPE_CATEGORIES.NLP),
    };

    // graph.js (line 418) - recipeFlowIcon filter

    const ML_TYPES = {
        'prediction': buildTypeDefinition('icon-beaker'),
        'regression': buildTypeDefinition('icon-machine_learning_regression'),
        'clustering': buildTypeDefinition('icon-machine_learning_clustering'),
    };
    const OTHER_TAGGABLE_OBJECTS_TYPES = {
        'analysis': buildTypeDefinition('icon-dku-nav_analysis'),
        'managed_folder': buildTypeDefinition('icon-folder-open'),
        'saved_model': buildTypeDefinition('icon-machine_learning_regression'),
        'model_evaluation_store': buildTypeDefinition('icon-model-evaluation-store'),
        'genai_evaluation_store': buildTypeDefinition('icon-model-evaluation-store'),
        'retrievable_knowledge': buildTypeDefinition('dku-icon-cards-stack', undefined, undefined, true),
        'knowledge_bank': buildTypeDefinition('dku-icon-cards-stack', undefined, undefined, true), // IndexableType
        'prompt_studio': buildTypeDefinition('icon-prompt-iterative'),
        'agent_review': buildTypeDefinition('icon-agent_review'),
        'agent_tool': buildTypeDefinition('dku-icon-tool-wrench', undefined, undefined, true),
        'model_comparison': buildTypeDefinition('icon-dku-comparator'),
        'genai_comparison': buildTypeDefinition('icon-dku-comparator'),
        'labeling_task': buildTypeDefinition('icon-labeling-task'),
        'statistics_worksheet': buildTypeDefinition('icon-dku-statistics'),
        'scenario': buildTypeDefinition('icon-list'),
        'article': buildTypeDefinition('icon-dku-wiki'),
        'lambda_service': buildTypeDefinition('icon-cloud'),
        'flow_zone': buildTypeDefinition('icon-zone'),
        'code_studio': buildTypeDefinition('icon-code-studio'),
        'taggable_object': buildTypeDefinition('icon-puzzle-piece'), //Generic (used for heterogeneous groups of taggable objects)
        'workspace_link': buildTypeDefinition('icon-link'),
        'workspace_story': buildTypeDefinition('icon-dku-datastory'),
        'data_collection': buildTypeDefinition('dku-icon-stacked', undefined, undefined, true)
    };
    const NON_TAGGABLE_OBJECTS_TYPES = {
        'column': buildTypeDefinition('icon-list icon-rotate-90'),
        'meaning': buildTypeDefinition('icon-tags'),
        'discussion': buildTypeDefinition('icon-comments'),
    };
    const WEBAPPS_TYPES = {
        'web_app': buildTypeDefinition('icon-code'),
        'bokeh': buildTypeDefinition('icon-bokeh'),
        'dash': buildTypeDefinition('icon-dash'),
        'shiny': buildTypeDefinition('icon-code_r_recipe'),
        'streamlit': buildTypeDefinition('icon-streamlit'),
        'standard': buildTypeDefinition('icon-code'),
        'code_studio_as_webapp': buildTypeDefinition('icon-code-studio'),
    };
    const DASHBOARDS_OR_INSIGHTS_TYPES = {
        'insight': buildTypeDefinition('icon-dku-nav_dashboard'),
        'data-quality': buildTypeDefinition('dku-icon-shield-check', undefined, undefined, true),
        'dashboard': buildTypeDefinition('icon-dku-dashboard'),
        'html': buildTypeDefinition('icon-code'),
        'group': buildTypeDefinition('dku-icon-group', undefined, undefined, true),
        'image': buildTypeDefinition('icon-picture'),
        'text': buildTypeDefinition('icon-font'),
        'static_file': buildTypeDefinition('icon-file-alt'),
        'iframe': buildTypeDefinition('icon-link'),
        'bokeh_export': buildTypeDefinition('icon-bokeh'),
        'dash_export': buildTypeDefinition('icon-dash'),
        'streamlit_export': buildTypeDefinition('icon-streamlit'),
        'static_chart': buildTypeDefinition('icon-bar-chart'),
        'discussions': buildTypeDefinition('icon-comments-alt'),
        'runnable-button': buildTypeDefinition('icon-macro')
    };
    const OTHER_TYPES = {
        'project': buildTypeDefinition('icon-dkubird'),
        'app': buildTypeDefinition('icon-project-app'),
        'report': buildTypeDefinition('icon-DKU_rmd'),
        'new': buildTypeDefinition('icon-plus'),
        'help': buildTypeDefinition('icon-question-sign'),
        'workspace': buildTypeDefinition('icon-dku-workspace'),
    };
    const NOTEBOOKS_TYPES = {
        'notebook': buildTypeDefinition('icon-dku-nav_notebook'),
        'sql_notebook': buildTypeDefinition('icon-sql'),
        'search_notebook': buildTypeDefinition('icon-dku-search-notebook'),
        'jupyter_notebook': buildTypeDefinition('icon-dku-nav_notebook'),
    };
    const BACKEND_TYPES = {
        'h2o': buildTypeDefinition('dku-icon-h2o', undefined, undefined, true),
        'mllib': buildTypeDefinition('dku-icon-sparkmllib', undefined, undefined, true),
        'py_memory': buildTypeDefinition('dku-icon-python-circle', undefined, undefined, true),
        'keras': buildTypeDefinition('dku-icon-keras', undefined, undefined, true),
        'deep_hub': buildTypeDefinition('dku-icon-deep-hub', undefined, undefined, true),
        'vertica': COMMON_TYPES.vertica
    }
    const ALL_TYPES = {
        ...COMMON_TYPES, ...DATASET_TYPES, ...STREAMING_ENDPOINT_TYPES, ...CONNECTION_TYPES, ...CREDENTIAL_TYPES, ...RECIPE_TYPES, ...ML_TYPES, ...OTHER_TAGGABLE_OBJECTS_TYPES,
        ...NON_TAGGABLE_OBJECTS_TYPES, ...WEBAPPS_TYPES, ...DASHBOARDS_OR_INSIGHTS_TYPES, ...OTHER_TYPES, ...NOTEBOOKS_TYPES, ...BACKEND_TYPES
    };

    /*
        Visual analysis + saved model icons
        Do not include in ALL_TYPES (name collision; not a taggable object)
    */
    const VISUAL_ANALYSIS_TYPES = {
        'prediction': buildTypeDefinition('icon-dku-automl-prediction'),
        'clustering': buildTypeDefinition('icon-dku-automl-clustering'),
        'timeseries': buildTypeDefinition('icon-dku-timeseries-forecasting'),
        'causal': buildTypeDefinition('icon-dku-causal'),
        'keras': buildTypeDefinition('icon-dku-deeplearning-prediction'),
        'deephub_image_classification': buildTypeDefinition('icon-dku-deephub-image-classification'),
        'deephub_object_detection': buildTypeDefinition('icon-dku-deephub-object-detection')
    };
    const SAVED_MODEL_TYPES = {
        'prediction': buildTypeDefinition('dku-icon-machine-learning-regression', undefined, undefined, true),
        'clustering': buildTypeDefinition('dku-icon-machine-learning-clustering', undefined, undefined, true),
        'timeseries': buildTypeDefinition('dku-icon-saved-model-timeseries', undefined, undefined, true),
        'causal': buildTypeDefinition('dku-icon-saved-model-causal', undefined, undefined, true),
        'keras': buildTypeDefinition('dku-icon-saved-model-deep-learning', undefined, undefined, true),
        'deephub_image_classification': buildTypeDefinition('dku-icon-saved-model-computer-vision', undefined, undefined, true),
        'deephub_object_detection': buildTypeDefinition('dku-icon-saved-model-computer-vision', undefined, undefined, true),
        'mlflow': buildTypeDefinition('dku-icon-model-mlflow', undefined, undefined, true),
        'fine_tuning': buildTypeDefinition('dku-icon-saved-model-fine-tuning', undefined, undefined, true),
        'python_agent': buildTypeDefinition('dku-icon-ai-agent-code', undefined, undefined, true),
        'plugin_agent': buildTypeDefinition('dku-icon-ai-agent-plugin', undefined, undefined, true),
        'tools_using_agent': buildTypeDefinition('dku-icon-ai-agent-visual', undefined, undefined, true),
        'retrieval_augmented_llm': buildTypeDefinition('dku-icon-llm-augmented', undefined, undefined, true),
        'sagemaker': buildTypeDefinition('dku-icon-model-sagemaker', undefined, undefined, true),
        'vertex': buildTypeDefinition('dku-icon-model-google-vertex', undefined, undefined, true),
        'azureml': buildTypeDefinition('dku-icon-model-azureml', undefined, undefined, true),
        'databricks': buildTypeDefinition('dku-icon-model-databricks', undefined, undefined, true),
    };

    const LLM_TYPES = {
        'saved_model_agent': buildTypeDefinition('dku-icon-ai-agent', undefined, undefined, true),
        'tools_using_agent': buildTypeDefinition('dku-icon-ai-agent-visual', undefined, undefined, true),
        'python_agent': buildTypeDefinition('dku-icon-ai-agent-code', undefined, undefined, true),
        'plugin_agent': buildTypeDefinition('dku-icon-ai-agent-plugin', undefined, undefined, true),
        'fine_tuning': buildTypeDefinition('dku-icon-saved-model-fine-tuning', undefined, undefined, true),
        'saved_model_finetuned_openai': buildTypeDefinition('dku-icon-saved-model-fine-tuning', undefined, undefined, true),
        'saved_model_finetuned_azure_openai': buildTypeDefinition('dku-icon-saved-model-fine-tuning', undefined, undefined, true),
        'saved_model_finetuned_huggingface_transformer': buildTypeDefinition('dku-icon-saved-model-fine-tuning', undefined, undefined, true),
        'saved_model_finetuned_bedrock': buildTypeDefinition('dku-icon-saved-model-fine-tuning', undefined, undefined, true),
        'openai': buildTypeDefinition('dku-icon-openai', undefined, undefined, true),
        'azure_openai_model': buildTypeDefinition('dku-icon-microsoft-azure', undefined, undefined, true),
        'azure_openai_deployment': buildTypeDefinition('dku-icon-microsoft-azure', undefined, undefined, true),
        'azure_llm': buildTypeDefinition('dku-icon-microsoft-azure', undefined, undefined, true),
        'cohere': buildTypeDefinition('dku-icon-cohere', undefined, undefined, true),
        'mistralai': buildTypeDefinition('dku-icon-mistral', undefined, undefined, true),
        'anthropic': buildTypeDefinition('dku-icon-anthropic', undefined, undefined, true),
        'vertex': buildTypeDefinition('dku-icon-google-vertex', undefined, undefined, true),
        'bedrock': buildTypeDefinition('dku-icon-amazon-aws-bedrock', undefined, undefined, true),
        'mosaicml': buildTypeDefinition('dku-icon-mosaicml', undefined, undefined, true),
        'stabilityai': buildTypeDefinition('dku-icon-stability-ai', undefined, undefined, true),
        'nvidia_nim': buildTypeDefinition('dku-icon-nvidia', undefined, undefined, true),
        'huggingface_api': buildTypeDefinition('dku-icon-huggingface', undefined, undefined, true),
        'huggingface_transformer_local': buildTypeDefinition('dku-icon-huggingface', undefined, undefined, true),
        'databricks': buildTypeDefinition('dku-icon-databricks', undefined, undefined, true),
        'snowflake_cortex': buildTypeDefinition('dku-icon-snowflake', undefined, undefined, true),
        'retrieval_augmented': buildTypeDefinition('dku-icon-llm-augmented', undefined, undefined, true),
        'retrieval_augmented_llm': buildTypeDefinition('dku-icon-llm-augmented', undefined, undefined, true),
        'custom': buildTypeDefinition('dku-icon-puzzle-piece', undefined, undefined, true),
        'sagemaker': buildTypeDefinition('dku-icon-amazon-sagemaker', undefined, undefined, true),
        'sagemaker_genericllm': buildTypeDefinition('dku-icon-amazon-sagemaker', undefined, undefined, true),
    };

    // Map the old icons to the new one. Icon migration campaign in 2023
    // All new icons start with 'dku-icon-'
    const NEW_ICON_TYPE_MAPPING = {
        'icon-dkubird': 'dku-icon-dataiku',
        'icon-dataset': 'dku-icon-dataset',
        'icon-project-app': 'dku-icon-application',
        'icon-machine_learning_regression': 'dku-icon-machine-learning-regression',
        'icon-dku-nav_dashboard': 'dku-icon-chart-vertical-bar',
        'icon-dku-dashboard': 'dku-icon-dashboard',
        'icon-picture': 'dku-icon-image',
        'icon-font': 'dku-icon-text-letter-character',
        'icon-file-alt': 'dku-icon-file',
        'icon-file-text-alt': 'dku-icon-file-text',
        'icon-link': 'dku-icon-link',
        'icon-bokeh': 'dku-icon-bokeh',
        'icon-bar-chart': 'dku-icon-chart-vertical-stack-bar',
        'icon-comments-alt': 'dku-icon-comment-multiple-question',
        'icon-code_r_recipe': 'dku-icon-recipe-r-circle-fill',
        'icon-dash': 'dku-icon-plotly',
        'icon-streamlit': 'dku-icon-streamlit',
        'icon-code': 'dku-icon-code',
        'icon-mongo_db': 'dku-icon-mongo',
        'icon-cassandra_1': 'dku-icon-cassandra',
        'icon-google-bigquery': 'dku-icon-google-bigquery',
        'icon-twitter': 'dku-icon-twitter',
        'icon-openai': 'dku-icon-openai',
        'icon-bedrock': 'dku-icon-amazon-aws-bedrock',
        'icon-pinecone': 'dku-icon-pinecone',
        'icon-azure': 'dku-icon-microsoft-azure',
        'icon-cohere': 'dku-icon-cohere',
        'icon-mistral': 'dku-icon-mistral',
        'icon-mosaicml': 'dku-icon-mosaicml',
        'icon-anthropic': 'dku-icon-anthropic',
        'icon-vertex': 'dku-icon-model-google-vertex',
        'icon-hugging-face': 'dku-icon-huggingface',
        'icon-postgreSQL': 'dku-icon-postgresql',
        'icon-snowflake': 'dku-icon-snowflake-brand',
        'icon-server_file_system_1': 'dku-icon-folder-multiple',
        'icon-HDFS': 'dku-icon-hdfs',
        'icon-google-cloud-storage': 'dku-icon-google-gcp-cloud-storage',
        'icon-microsoft-sharepoint': 'dku-icon-microsoft-sharepoint',
        'icon-amazon_s3': 'dku-icon-amazon-web-s3',
        'icon-FTP-HTTP-SSH': 'dku-icon-globe',
        'icon-uncached_FTP': 'dku-icon-data-ftp',
        'icon-other_sql': 'dku-icon-dataset',
        'icon-dku-databricks': 'dku-icon-databricks',
        'icon-dku-databricks-volume': 'dku-icon-databricks-volume',
        'icon-dku-databricks-lakebase': 'dku-icon-databricks-lakebase',
        'icon-sap-hana': 'dku-icon-sap-hana',
        'icon-HP_vertica': 'dku-icon-vertica',
        'icon-athena': 'dku-icon-athena',
        'icon-trino': 'dku-icon-trino',
        'icon-denodo': 'dku-icon-denodo',
        'icon-treasuredata': 'dku-icon-treasure-data',
        'icon-iceberg': 'dku-icon-apache-iceberg',
        'icon-dku-alloydb': 'dku-icon-alloydb',
        'icon-dku-julia': 'dku-icon-julia-brand',
        'icon-greenplum': 'dku-icon-greenplum',
        'icon-mySQL': 'dku-icon-mysql',
        'icon-sqlserver': 'dku-icon-sql-server',
        'icon-microsoft-fabric-warehouse': 'dku-icon-microsoft-fabric-warehouse-diagram',
        'icon-oracle': 'dku-icon-oracle',
        'icon-tableau': 'dku-icon-tableau',
        'icon-teradata': 'dku-icon-teradata',
        'icon-dku-deepl': 'dku-icon-deepl',
        'icon-dropbox': 'dku-icon-dropbox',
        'icon-neo4j': 'dku-icon-neo4j',
        'icon-dku-azure-synapse': 'dku-icon-microsoft-azure-synapse',
        'icon-amazon-comprehend': 'dku-icon-amazon-comprehend',
        'icon-amazon-elastic-kubernetes': 'dku-icon-amazon-elastic-kubernetes',
        'icon-amazon-rekognition': 'dku-icon-amazon-rekognition',
        'icon-amazon_redshift': 'dku-icon-amazon-redshift',
        'icon-amazon-transcribe': 'dku-icon-amazon-transcribe',
        'icon-azure-cognitive-services': 'dku-icon-amazon-cognitive-services',
        'icon-azure-storage': 'dku-icon-microsoft-azure-blob-storage',
        'icon-azure-kubernetes-services': 'dku-icon-microsoft-azure-kubernetes',
        'icon-elasticsearch': 'dku-icon-elastic-search',
        'icon-gcp-cloud-dataproc': 'dku-icon-google-gcp-dataproc',
        'icon-gcp-kubernetes-engine': 'dku-icon-google-gcp-kubernetes',
        'icon-gcp-natural-language-api': 'dku-icon-gcp-naturallang',
        'icon-gcp-cloud-vision-api': 'dku-icon-google-gcp-vision',
        'icon-gcp-translation-api': 'dku-icon-google-gcp-translation',
        'icon-powerbi': 'dku-icon-microsoft-powerbi',
        'icon-netezza': 'dku-icon-netezza',
        'icon-dku-hive': 'dku-icon-hive',
        'icon-upload': 'dku-icon-arrow-circle-dataset-upload',
        'icon-inline': 'dku-icon-edit',
        'icon-score_recipe': 'dku-icon-recipe-score-circle-fill',
        'icon-train_recipe': 'dku-icon-recipe-train-circle-fill',
        'icon-visual_prep_distinct_recipe': 'dku-icon-recipe-distinct-circle-fill',
        'icon-visual_download_recipe': 'dku-icon-recipe-download',
        'icon-eda_pca_recipe': 'dku-icon-recipe-principal-component-analysis',
        'icon-eda_stats_recipe': 'dku-icon-recipe-statistical-test',
        'icon-eda_univariate_recipe': 'dku-icon-recipe-univariate-analysis',
        'icon-evaluation_recipe': 'dku-icon-recipe-evaluation-circle-fill',
        'icon-standalone_evaluation_recipe': 'dku-icon-recipe-ser',
        'icon-visual_export_recipe': 'dku-icon-recipe-export-circle-fill',
        'icon-visual_prep_group_recipe': 'dku-icon-recipe-group-circle-fill',
        'icon-visual_prep_upsert_recipe': 'dku-icon-recipe-upsert-circle-fill',
        'icon-labeling': 'dku-icon-recipe-labeling-circle-fill',
        'icon-visual_prep_fuzzyjoin_recipe': 'dku-icon-recipe-fuzzy-join-circle-fill',
        'icon-visual_prep_merge_folder_recipe': 'dku-icon-recipe-merge-folder',
        'icon-visual_prep_list_folder_contents_recipe': 'dku-icon-recipe-list-folder-contents-circle-fill',
        'icon-visual_prep_list_access_recipe': 'dku-icon-recipe-list-access',
        'icon-visual_prep_pivot_recipe': 'dku-icon-recipe-pivot-circle-fill',
        'icon-circle': 'dku-icon-recipe-type',
        'icon-visual_prep_filter-sample_recipe': 'dku-icon-recipe-filter-circle-fill',
        'icon-visual_prep_split_recipe': 'dku-icon-recipe-split-circle-fill',
        'icon-visual_prep_sync_recipe': 'dku-icon-recipe-sync-right',
        'icon-visual_prep_topn_recipe': 'dku-icon-recipe-top-n-circle-fill',
        'icon-visual_push_to_editable_recipe': 'dku-icon-recipe-push-to-editable',
        'icon-visual_prep_vstack_recipe': 'dku-icon-recipe-stack-circle-fill',
        'icon-visual_prep_window_recipe': 'dku-icon-recipe-window-circle-fill',
        'icon-continuous_sync_recipe': 'dku-icon-recipe-sync-right',
        'icon-continuous_ksql_recipe': 'dku-icon-recipe-ksql-circle-fill',
        'icon-continuous_python_recipe': 'dku-icon-python-circle',
        'icon-continuous_spark_scala_recipe': 'dku-icon-recipe-sparkscala-circle-fill',
        'icon-code_hive_recipe': 'dku-icon-recipe-hive-circle-fill',
        'icon-code_impala_recipe': 'dku-icon-recipe-impala-circle-fill',
        'icon-python': 'dku-icon-python',
        'icon-sql': 'dku-icon-sql',
        'icon-sql marker-database': 'dku-icon-dataset',
        'icon-code_sql_recipe': 'dku-icon-recipe-sql-circle-fill',
        'icon-visual_prep_cleanse_recipe': 'dku-icon-recipe-prepare-circle-fill',
        'icon-code_shell_recipe': 'dku-icon-recipe-shell',
        'icon-visual_prep_sort_recipe': 'dku-icon-recipe-sort-circle-fill',
        'icon-code_spark_scala_recipe': 'dku-icon-recipe-sparkscala-circle-fill',
        'icon-code_sparksql_recipe': 'dku-icon-recipe-sparksql-circle-fill',
        'icon-code_sparkr_recipe': 'dku-icon-recipe-sparkr-circle-fill',
        'icon-code_pyspark_recipe': 'dku-icon-recipe-sparkpython-circle-fill',
        'icon-code_python_recipe': 'dku-icon-python-circle',
        'icon-visual_prep_join_recipe': 'dku-icon-recipe-join-with-circle-fill',
        'icon-code_julia_recipe': 'dku-icon-recipe-julia-circle-fill',
        'icon-code_pig_recipe': 'dku-icon-apache-pig',
        'icon-visual_prep_geojoin_recipe': 'dku-icon-recipe-geo-join-circle-fill',
        'icon-nlp_prompt_recipe': 'dku-icon-recipe-prompt',
        'icon-nlp_fine_tuning_recipe': 'dku-icon-recipe-fine-tuning',
        'icon-nlp_text_classification_recipe': 'dku-icon-recipe-classification',
        'icon-nlp_summarization_recipe': 'dku-icon-recipe-summarization',
        'icon-nlp_rag_embedding_recipe': 'dku-icon-recipe-document-card-circle-fill',
        'icon-nlp_llm_evaluation_recipe': 'dku-icon-recipe-evaluation-llm-circle-fill',
        'icon-nlp_agent_evaluation_recipe': 'dku-icon-recipe-evaluation-agent-circle-fill',
        'icon-retrievable_knowledge': 'dku-icon-cards-stack',
        'icon-prompt-iterative': 'dku-icon-ai-prompt',
        'icon-agent_review':'dku-icon-ai-agent',
        'icon-book': 'dku-icon-book',
        'icon-clustering_recipe': 'dku-icon-recipe-clustering-circle-fill',
        'icon-list': 'dku-icon-list-bulleted',
        'icon-table': 'dku-icon-data-table',
        'icon-file-text': 'dku-icon-file-text',
        'icon-dashboard': 'dku-icon-dashboard',
        'icon-dku-modelize': 'dku-icon-modelize',
        'icon-dku-clustering': 'dku-icon-clustering',
        'icon-external-link': 'dku-icon-arrow-external-link',
        'icon-dku-nav_notebook': 'dku-icon-edit-note',
        'icon-folder-close-alt': 'dku-icon-folder-closed',
        'icon-DKU_rmd': 'dku-icon-rmd',
        'icon-kafka': 'dku-icon-kafka',
        'icon-sqs': 'dku-icon-amazon-sqs',
        'icon-httpsse': 'dku-icon-httpsse',
        'icon-dku-automl-prediction': 'dku-icon-automl-prediction',
        'icon-dku-timeseries-forecasting': 'dku-icon-time-series',
        'icon-dku-deeplearning-prediction': 'dku-icon-prediction-deep-learning',
        'icon-dku-deephub-image-classification': 'dku-icon-image-classification',
        'icon-dku-deephub-object-detection': 'dku-icon-object-detection',
        'icon-dku-automl-clustering': 'dku-icon-automl-clustering',
        'icon-dku-causal': 'dku-icon-causal',
        'icon-dku-nav_analysis': 'dku-icon-ml-analysis',
        'icon-dku-wiki': 'dku-icon-wiki',
        'icon-beaker': 'dku-icon-beaker',
        'icon-box': 'dku-icon-dataset-files-in-folder',
        'icon-calendar': 'dku-icon-calendar',
        'icon-database': 'dku-icon-dataset',
        'icon-dataset-ssh marker-sftp': 'dku-icon-data-sftp',
        'icon-dataset-ssh marker-scp': 'dku-icon-data-scp',
        'icon-dku-exp-tracking': 'dku-icon-experiment-tracking',
        'icon-dku-feature-store': 'dku-icon-feature-store-label',
        'icon-dku-share': 'dku-icon-flow-share',
        'icon-dku-meanings': 'dku-icon-barcode',
        'icon-dku-offline-translation': 'dku-icon-translation',
        'icon-dynamoDB': 'dku-icon-dynamo-db',
        'icon-filter': 'dku-icon-filter',
        'icon-model-evaluation-store': 'dku-icon-model-evaluation-store',
        'icon-signal': 'dku-icon-signal',
        'icon-tasks': 'dku-icon-chart-horizontal-stack-bar',
        'icon-tasks marker-internal-stats': 'dku-icon-statistics',
        'icon-time': 'dku-icon-clock',
        'icon-screenshot': 'dku-icon-image-tag',
        'icon-cloud': 'dku-icon-cloud',
        'icon-code-studio': 'dku-icon-code-studio',
        'icon-comments': 'dku-icon-comment-multiple',
        'icon-dku-comparator': 'dku-icon-model-comparison',
        'icon-dku-statistics': 'dku-icon-statistics',
        'icon-dku-workspace': 'dku-icon-workspaces',
        'icon-double-angle-right': 'dku-icon-chevron-double-right',
        'icon-folder-open': 'dku-icon-dataset-files-in-folder',
        'icon-labeling-task': 'dku-icon-label',
        'icon-list icon-rotate-90': 'dku-icon-columns',
        'icon-machine_learning_clustering': 'dku-icon-machine-learning-clustering',
        'icon-plus': 'dku-icon-plus',
        'icon-puzzle-piece': 'dku-icon-puzzle-piece',
        'icon-question-sign': 'dku-icon-question-circle-fill',
        'icon-tags': 'dku-icon-tag',
        'icon-visual_prep_auto_feature_generation_recipe': 'dku-icon-recipe-auto-feature-circle-fill',
        'icon-zone': 'dku-icon-zone',
        'icon-mail-forward': 'dku-icon-arrow-curve-right',
        'icon-eye': 'dku-icon-eye',
        'icon-globe': 'dku-icon-globe',
        'icon-dku-application-as-recipe': 'dku-icon-application-as-recipe',
        'icon-Filesystem': 'dku-icon-dataset-filesystem',
        'icon-cloud-download': 'dku-icon-cloud-download',
        'icon-cloud-upload': 'dku-icon-cloud-upload',
        'icon-flow_dataset_folder': 'dku-icon-dataset-files-in-folder',
        'recipe_cluster': 'dku-icon-machine-learning-clustering',
        'recipe_distinct': 'dku-icon-recipe-distinct-circle-fill',
        'recipe_download': 'dku-icon-recipe-download',
        'recipe_evaluation': 'dku-icon-recipe-evaluation-circle-fill',
        'recipe_export': 'dku-icon-recipe-export-circle-fill',
        'recipe_filter': 'dku-icon-recipe-filter-circle-fill',
        'recipe_fuzzyjoin': 'dku-icon-recipe-fuzzy-join-circle-fill',
        'recipe_geojoin': 'dku-icon-recipe-geo-join-circle-fill',
        'recipe_group': 'dku-icon-recipe-group-circle-fill',
        'recipe_hive': 'dku-icon-recipe-hive-circle-fill',
        'recipe_impala': 'dku-icon-recipe-impala-circle-fill',
        'recipe_join': 'dku-icon-recipe-join-with-circle-fill',
        'recipe_julia': 'dku-icon-recipe-julia-circle-fill',
        'recipe_ksql': 'dku-icon-recipe-ksql-circle-fill',
        'recipe_list_folder_contents': 'dku-icon-recipe-list-folder-contents-circle-fill',
        'recipe_list_access': 'dku-icon-recipe-list-access',
        'recipe_merge_folder': 'dku-icon-recipe-merge-folder',
        'recipe_pig': 'dku-icon-apache-pig',
        'recipe_pivot': 'dku-icon-recipe-pivot-circle-fill',
        'recipe_prepare': 'dku-icon-broom',
        'recipe_push_to_editable': 'dku-icon-recipe-push-to-editable',
        'recipe_pyspark': 'dku-icon-recipe-sparkpython-circle-fill',
        'recipe_python': 'dku-icon-python-circle',
        'recipe_R': 'dku-icon-recipe-r-circle-fill',
        'recipe_score': 'dku-icon-recipe-score-circle-fill',
        'recipe_shell': 'dku-icon-recipe-shell',
        'recipe_sort': 'dku-icon-recipe-sort-circle-fill',
        'recipe_spark_scala': 'dku-icon-recipe-sparkscala-circle-fill',
        'recipe_spark_sql': 'dku-icon-recipe-sparksql-circle-fill',
        'recipe_sparkr': 'dku-icon-recipe-sparkr-circle-fill',
        'recipe_split': 'dku-icon-recipe-split-circle-fill',
        'recipe_sql': 'dku-icon-recipe-sql-circle-fill',
        'recipe_stack': 'dku-icon-recipe-stack-circle-fill',
        'recipe_standalone_evaluation': 'dku-icon-recipe-ser',
        'recipe_sync': 'dku-icon-recipe-sync-circle-fill',
        'recipe_topn': 'dku-icon-recipe-top-n-circle-fill',
        'recipe_train': 'dku-icon-recipe-train-circle-fill',
        'recipe_window': 'dku-icon-recipe-window-circle-fill',
        'icon-dku-datastory': 'dku-icon-dataiku-story',
        'icon-dku-search-notebook': 'dku-icon-dataset-search',
        'icon-trash': 'dku-icon-trash',
        'icon-model-sagemaker': 'dku-icon-model-sagemaker',
        'icon-model-azureml': 'dku-icon-model-azureml',
        'icon-model-google-vertex': 'dku-icon-model-google-vertex',
        'icon-youtube-play' : 'dku-icon-youtube',
        'icon-envelope-alt' : 'dku-icon-mail',
        'icon-file' : 'dku-icon-file',
        'icon-bullseye': 'dku-icon-target',
        'icon-bug': 'dku-icon-bug',
        'icon-indent-right': 'dku-icon-text-indent',
        'icon-indent-left': 'dku-icon-text-indent',
        'icon-cogs': 'dku-icon-gear',
        'icon-gear': 'dku-icon-gear',
        'icon-gears': 'dku-icon-gear',
        'icon-sitemap': 'dku-icon-sitemap',
        'icon-forward': 'dku-icon-arrow-fast-forward',
        'icon-anchor': 'dku-icon-anchor',
        'icon-align-justify': 'dku-icon-text-align-justified',
        'icon-search': 'dku-icon-search',
        'icon-microphone': 'dku-icon-microphone',
        'icon-envelope': 'dku-icon-mail',
        'icon-list-alt': 'dku-icon-list-bulleted',
        'icon-magic': 'dku-icon-magic-wand',
        'icon-quote-left': 'dku-icon-text-quote',
        'icon-superscript': 'dku-icon-text-superscript',
        'icon-dollar': 'dku-icon-currency-usd',
        'icon-dku-collaborative-filtering': 'dku-icon-person-group',
        'icon-heart': 'dku-icon-heart-fill',
        'icon-barcode': 'dku-icon-barcode',
        'icon-rocket': 'dku-icon-api-connect',
        'icon-lock': 'dku-icon-lock-closed',
        'icon-cut': 'dku-icon-scissors-horizontal',
        'icon-eye-open': 'dku-icon-eye',
        'icon-sun': 'dku-icon-sun',
        'icon-group': 'dku-icon-person-group',
        'icon-agent': 'dku-icon-ai-agent',
        'icon-gen_ai_model': 'dku-icon-llm'
    };

    const app = angular.module('dataiku.constants');

    app.constant('TYPE_MAPPING', {
        CONVERSION_FIELD_ICON,
        CONVERSION_FIELD_NAME,
        CONVERSION_FIELD_OTHER_NAME,
        CONVERSION_FIELD_LANGUAGE,
        CONVERSION_FIELD_CATEGORY,

        FS_PROVIDER_TYPES,
        COMMON_TYPES,
        DATASET_TYPES,
        STREAMING_ENDPOINT_TYPES,
        CONNECTION_TYPES,
        CREDENTIAL_TYPES,
        RECIPE_TYPES,
        RECIPE_CATEGORIES,
        ML_TYPES,
        OTHER_TAGGABLE_OBJECTS_TYPES,
        NON_TAGGABLE_OBJECTS_TYPES,
        WEBAPPS_TYPES,
        DASHBOARDS_OR_INSIGHTS_TYPES,
        OTHER_TYPES,
        NOTEBOOKS_TYPES,
        BACKEND_TYPES,
        ALL_TYPES,

        VISUAL_ANALYSIS_TYPES,
        SAVED_MODEL_TYPES,
        LLM_TYPES,
        NEW_ICON_TYPE_MAPPING,
    });

})();
