(function() {
    'use strict';

    var app = angular.module('dataiku.ml.explainability');

    app.controller("InteractiveScoringController", function($scope, $timeout, $q, DataikuAPI, WT1, Debounce, ExplanationBarUtils, WhatIfRouter, WhatIfView,
                                                            localStorageService, CreateModalFromTemplate, ClipboardUtils, ActivityIndicator, ExportUtils,
                                                            Dialogs, openDkuPopin, epochShift, ExplorationLocalStore, OutcomeOptimizationSpecialTarget,
                                                            FullModelLikeIdUtils, SpinnerService, FutureWatcher, FeaturesDistributionAdapter,
                                                            NO_TARGET_CLASS, InteractiveModelCommand, LoggerProvider, WhatIfFeaturesSortingFactory,
                                                            WhatIfFormattingService) {
        const logger = LoggerProvider.getLogger('ml.interactive-scoring');
        if (!$scope.modelData) {
            return;
        }
        const perFeature = $scope.modelData.preprocessing.per_feature;
        const authorizedTypes = ['CATEGORY', 'NUMERIC', 'TEXT', 'VECTOR', 'IMAGE'];
        
        const authorizedRoles = ['INPUT'];
        const fullModelId = $scope.fmi = FullModelLikeIdUtils.getFmi($scope);
        const backendFormat = "YYYY-MM-DDTHH:mm:ss.SSS";
        const scorePrefix = 'proba_';
        const OTHERS_COLOR = '#ddd';
        const computationStatus = { DONE: 'DONE', COMPUTING: 'COMPUTING', ABORTED: 'ABORTED', ERROR: 'ERROR'};

        const LOCAL_STORAGE_CACHING_ENABLED = true;
        const LOCAL_STORAGE_VERSION = 'v2'; // update this when you modify the format of the data to invalidate previous caches

        const baseKey = `dku.ml.interactivescoring.${LOCAL_STORAGE_VERSION}`;
        let LOCAL_STORAGE_BUCKET_KEY,
            LOCAL_STORAGE_EXPLANATION_PARAMS_KEY,
            LOCAL_STORAGE_UI_STATE_KEY;
        if ($scope.insight && $scope.insight.id) {
            LOCAL_STORAGE_BUCKET_KEY = `${baseKey}.insight.${$scope.insight.id}.bucket`;
            LOCAL_STORAGE_EXPLANATION_PARAMS_KEY = `${baseKey}.insight.${$scope.insight.id}.explanationParams`;
            LOCAL_STORAGE_UI_STATE_KEY = `${baseKey}.insight.${$scope.insight.id}.uiState`;
        } else {
            LOCAL_STORAGE_BUCKET_KEY = `${baseKey}.${fullModelId}.bucket`;
            LOCAL_STORAGE_EXPLANATION_PARAMS_KEY = `${baseKey}.${fullModelId}.explanationParams`;
            LOCAL_STORAGE_UI_STATE_KEY = `${baseKey}.${fullModelId}.uiState`;
        }
        // For dashboard
        $scope.userFeatures = null;

        let currentComputation = {
           COMPARATOR : {
                SCORING: {
                    status: undefined,
                    runningJobId: null,
                    nextComputation: null
                },
                    
                EXPLANATIONS: {
                    status: undefined,
                    runningJobId: null,
                    nextComputation: null
                }
            },
            MAIN : {
                SCORING: {
                    status: undefined,
                    runningJobId: null,
                    nextComputation: null
                },
                    
                EXPLANATIONS: {
                    status: undefined,
                    runningJobId: null,
                    nextComputation: null
                }
            }
        };

        $scope.ignoreFeatureTooltip = "Ignore feature: don't specify a value to the model";

        $scope.formatProba = (proba) => d3.format(",.2f")(proba * 100);

        $scope.getNoScoreReason = function() {
            if ($scope.allFeaturesEmpty()) {
                return "Set at least one feature value to compute the prediction";
            }
            if (!$scope.allEnabledImageFeaturesFilled()) {
                return "Upload an image for each enabled image feature to compute a prediction";
            }
            if ($scope.isCurrentViewAborted(InteractiveModelCommand.SCORING)) {
                return "Aborted task: Change a feature value to recompute the prediction";
            }
            return "This row may have been ignored by the preparation script or the model.";
        } 

        $scope.getNoExplanationReason = function() {
            if ($scope.getExplanationIncompatibility()) {
                return $scope.getExplanationIncompatibility();
            }
            if (!$scope.explanationParams.enabled) {
                return "Check option to compute features importance";
            }
            if ($scope.allFeaturesEmpty()) {
                return "Set at least one feature value to compute features importance";
            }
            if ( $scope.isCurrentViewAborted(InteractiveModelCommand.EXPLANATIONS) ) {
                return "Aborted task: Change a feature value to recompute features importance";
            }
            return "This row may have been ignored by the preparation script or the model.";
        }

        $scope.getExplanationIncompatibility = function() {
            if ($scope.modelData.coreParams.backendType === "KERAS") {
                return "Cannot compute explanations for a Deep learning model";
            }
            if (Object.values($scope.modelData.preprocessing.per_feature).some(
                feature => feature.role == "INPUT" && feature.type == "IMAGE" && feature.image_handling == "EMBEDDING_EXTRACTION"
            )) {
                return "Cannot compute explanations for a model with an image embedding extraction preprocessing";
            }
            if (!$scope.canComputeUfi()) {
                return $scope.ufiNotAvailableMessage($scope.explanationParams ? $scope.explanationParams.method : "");
            }
            return null;
        }

        $scope.uiStateAlreadyLoaded = false;
        if (LOCAL_STORAGE_CACHING_ENABLED && localStorageService.get(LOCAL_STORAGE_UI_STATE_KEY)) {
            logger.info("Retrieving saved uiState");
            $scope.uiState = localStorageService.get(LOCAL_STORAGE_UI_STATE_KEY);
            $scope.uiStateAlreadyLoaded = true;
        } else {
            logger.info("Loading uiState");
            $scope.uiState = {
                features: [],
                preScriptFeatures: [],
                applyPreparationScript: null,
                couldNotRetrieveSchema: false,
                hasPreparationSteps: false,
                featureFilterOptions: [], // will be set once we know if importance is available
            };
        }

        const whatIfRouter = $scope.whatIfRouter = WhatIfRouter.build();
        $scope.WhatIfView = WhatIfView;
        $scope.InteractiveModelCommand = InteractiveModelCommand;
        $scope.WhatIfFormattingService = WhatIfFormattingService;
        $scope.featuresSortingService = WhatIfFeaturesSortingFactory.init(); // will be passed to other what-if pages to avoid setting its attributes again
        $scope.selection = $scope.featuresSortingService.cleanSelection();

        $scope.showOtherFeatures = false;
        $scope.errors = {};

        $scope.labels = [''];
        $scope.barContainerWidth = 350;
        $scope.compareBarContainerWidth = 125;
        $scope.barMaxWidth = 100;
        $scope.compareBarMaxWidth = 50;

        $scope.getExplanationXBarPosition = ExplanationBarUtils.computeExplanationXBarPositionFunc($scope.barMaxWidth, $scope.barContainerWidth);
        $scope.getCompareExplanationXBarPosition = ExplanationBarUtils.computeExplanationXBarPositionFunc($scope.compareBarMaxWidth, $scope.compareBarContainerWidth);
        $scope.pickerFormat = "YYYY-MM-DD HH:mm";

        $scope.score = undefined;
        $scope.explanations = undefined;
        $scope.explanationParams = localStorageService.get(LOCAL_STORAGE_EXPLANATION_PARAMS_KEY) || {
            nbExplanations: 5,
            method: 'ICE',
            enabled: $scope.getExplanationIncompatibility() === null,
        }


        $scope.bucket = localStorageService.get(LOCAL_STORAGE_BUCKET_KEY) || [];
        $scope.bucketCharts = [];
        $scope.scrollToLastItem = false;

        $scope.sendBasicWT1Event = function() {
            WT1.event('interactive-scoring', {
                explanationParams: $scope.explanationParams,
                applyPreparationScript: $scope.uiState.applyPreparationScript
            })
        };

        $scope.hasPrediction = function(score) {
            if (!score) return false;
            if (!score.override) return true;
            return !(score.override.rulePolicy === "DECLINED");
        }

        // ----- UI interactions

        let distributions = null;
        FeaturesDistributionAdapter.init(fullModelId)
            .then(_distributions => { distributions = _distributions; })
            .finally(() => { $scope.$digest(); });

        $scope.getExplorationDisabledReason = () => {
            if ($scope.modelData.backendType === "KERAS") {
                return 'This option is not available for deep learning models';
            }
            if ($scope.isPartitionedModel() && $scope.isOnPartitionedBaseModel()) {
                return 'Switch to a specific partition to use exploration tools';
            }
            if ($scope.getAlgorithm().endsWith('_ENSEMBLE')) {
                return 'This option is not available for ensemble models';
            }
            if ($scope.uiState.applyPreparationScript) {
                return 'This option is not available when "Apply preparation script" is enabled';
            }
            if ($scope.modelData.modeling.skipExpensiveReports) {
                return 'This option is not available because computation of expensive reports was disabled';
            }
            if (!distributions) {
                return 'This model is too old, retrain it to use exploration tools';
            }
            if ($scope.getFeatures().some(feature => feature.editMode === 'UNSET')) {
                return 'This option is not available when some features are ignored';
            }
            if (!$scope.hasPrediction($scope.score)) {
                return 'This option is not available when the prediction is not valid';
            }
            return null;
        }

        $scope.openExplorationConstraints = () => {
            $scope.featuresSortingService.setPerFeatureDistributionType(distributions); // iscor didn't need it because `distributionType` was set already

            const reference = {};
            const explorationLocalStore = $scope.explorationLocalStore = ExplorationLocalStore(fullModelId, $scope.insight);
            for (const feature of $scope.getFeatures()) {
                reference[feature.name] = feature.value;
            }
            if ($scope.isClassification()) {
                const predictionColorMap = Object.fromEntries($scope.predictions.filter(pc => pc.color !== OTHERS_COLOR).map(pc => [pc.name, pc.color]));
                explorationLocalStore.setPredictionColorMap(predictionColorMap);
                explorationLocalStore.setTarget(NO_TARGET_CLASS);
            } else {
                explorationLocalStore.setTarget(OutcomeOptimizationSpecialTarget.MIN);
            }
            const formattedScore = { ...$scope.score };
            formattedScore["prediction"] = WhatIfFormattingService.formatPrediction($scope.modelData, $scope.score.prediction);
            explorationLocalStore.setScore(formattedScore);
            explorationLocalStore.setReference(reference);
            // Open the relevant page
            whatIfRouter.openConstraints();
        };

        $scope.abortAllComputations = () => {
            // killing any computation will kill the interactive scoring kernel (and thus kill all the running jobs). This make sure to abort each of them cleanly:
            for (const view in currentComputation) {
                for (const computationType in currentComputation[view]) {
                    abortComputationIfRunning(view, computationType);
                }
            }
        }

        $scope.openComparator = () => {
            whatIfRouter.openComparator();
            WT1.event('interactive-scoring-compare-button', {
                itemCount: $scope.bucket.length
            });
            if ($scope.bucket.length > 0) {
                computeAll($scope.bucket, null);
            }
            setTimeout(() => $scope.scrollToLastItem = false);
        };

        $scope.changeEditMode = function(feature, newMode, triggerCompute=true) {
            feature.editMode = newMode;
            const featureOldValue = feature.value;
            switch(newMode) {
                case "UNSET":
                    feature.value = null;
                    break;
                case "DOMAIN":
                    if (feature.type === "NUMERIC" || feature.type === "DATE") {
                        if (feature.value === null || isNaN(feature.value)) {
                            feature.value = feature.defaultValue;
                        } else if (feature.value > feature.max) {
                            feature.value = feature.max;
                        } else if (feature.value < feature.min) {
                            feature.value = feature.min;
                        }
                    } else if (feature.type === "CATEGORY") {
                        if (!feature.possibleValues.includes(feature.value)) {
                            feature.value = feature.defaultValue;
                        }
                    }
                    else if (feature.type === "IMAGE"){
                        if (feature.value === null) {
                            feature.value = feature.defaultValue;
                        }
                    }
                    break;
                case "RAW":
                    if (feature.type !== "NUMERIC") {
                        if (feature.value === null) {
                            feature.value = "";
                        }
                    }
                    break;
            }
            if (triggerCompute && featureOldValue !== feature.value) {
                onFeatureChange();
            }
        };

        $scope.toggleOtherFeatures = function() {
            $scope.showOtherFeatures = !$scope.showOtherFeatures;
        };

        function isViewComputing(view, computationType=null) {
            if (computationType !== null) {
                return currentComputation[view][computationType].status === computationStatus.COMPUTING;
            }
            return Object.values(currentComputation[view]).some(computationType => computationType.status  === computationStatus.COMPUTING);
        }
        $scope.isCurrentViewComputing = function(computationType) {
            return isViewComputing(whatIfRouter.getCurrentView(), computationType);
        }        
        $scope.isSomethingComputing = function() {
            return Object.keys(currentComputation).some(view => isViewComputing(view));
        }

        function isViewAborted(view, computationType=null) {
            if (computationType !== null) {
                return currentComputation[view][computationType].status === computationStatus.ABORTED;
            }
            return Object.values(currentComputation[view]).some(computationType => computationType.status  === computationStatus.ABORTED);
        }
        $scope.isCurrentViewAborted = function(computationType=null) {
            return isViewAborted(whatIfRouter.getCurrentView(), computationType);
        }

        $scope.fileUploaded = function(event, feature) {
            const file = event.srcElement.files[0];

            const reader = new FileReader();
            reader.onloadend = function(event) {
                feature.value = event.target.result.split(",")[1]; // remove prefix "data:image/*;base64"
                $scope.onFeatureChange();
            }
            reader.readAsDataURL(file);
        };

        // ----- Feature formatting

        $scope.getFeatures = function() {
            return getFeaturesFromUiState($scope.uiState);
        };

        $scope.allFeaturesEmpty = function() {
            return $scope.getFeatures().every(f => f.value == undefined);
        };

        $scope.allEnabledImageFeaturesFilled = function() {
            return $scope.getFeatures()
                .filter(f => f.type == 'IMAGE')
                .every(f =>  f.editMode == 'UNSET' || f.value != undefined);
        }

        $scope.getPredictedClassProba = function(score) {
            if ($scope.isClassification()) {
                return $scope.formatProba(score[scorePrefix + score.prediction]);
            }
        };

        $scope.getPositiveClassProba = function(score) {
            if ($scope.isBinaryClassification()) {
                return $scope.formatProba(score[scorePrefix + $scope.getPositiveClass()]);
            }
        };

        $scope.getProbaForBucketItem = function(score) {
            if ($scope.isMulticlass()) {
                return $scope.getPredictedClassProba(score);
            } else if ($scope.isBinaryClassification()) {
                return $scope.getPositiveClassProba(score);
            }
        };

        function getOtherFeatures() {
            if ($scope.tile) {
                if ($scope.tile.tileParams.advancedOptions.interactiveScoring) {
                    $scope.userFeatures = $scope.tile.tileParams.advancedOptions.interactiveScoring.featuresOrder;
                    return $scope.getFeatures()
                                 .filter(f => !$scope.userFeatures.includes(f.name) && f.type !== 'PART_DIM')
                                 .map(f => f.name);
                } else {
                    return [];
                }
            }
        }

        $scope.hasFilteredOutFeatures = function() {
            const otherFeatures = getOtherFeatures();
            return $scope.userFeatures && otherFeatures && otherFeatures.length;
        };

        $scope.keepOnlyMainFeatures = function(feature) {
            if ($scope.userFeatures) {
                return $scope.userFeatures.includes(feature.name);
            }
            return !$scope.keepOnlyPartDimensions(feature);
        };

        $scope.keepOnlyPartDimensions = function(feature) {
            return feature.type == "PART_DIM";
        };

        $scope.keepOtherFeatures = function(feature) {
            const otherFeatures = getOtherFeatures();
            if (otherFeatures) {
                return otherFeatures.includes(feature.name);
            }
            return false;
        };

        // for binary classification
        $scope.getPositiveClass = function() {
            return $scope.modelData.classes ? $scope.modelData.classes[1] : '';
        };

        const setUserDefinedOrdersIfNecessary = () => {
            if (!$scope.userFeatures) {
                return;
            }
            if ($scope.uiState.features) {
                $scope.uiState.features.forEach(feature => feature.userDefinedOrder = $scope.userFeatures.indexOf(feature.name));
            }
            if ($scope.uiState.preScriptFeatures) {
                $scope.uiState.preScriptFeatures.forEach(feature => feature.userDefinedOrder = $scope.userFeatures.indexOf(feature.name));
            }
            $scope.$broadcast('refresh-list');
        };

        function getFeaturesFromUiState(uiState) {
            return uiState.applyPreparationScript ? uiState.preScriptFeatures : uiState.features;
        }

        function formatFeatures(features) {
            return features.map(feature => ({
                name: feature.name,
                value: feature.value,
                type: feature.type,
                distributionType: feature.distributionType,
                importance: feature.importance,
            }));
        }

        function roundToXDigits(value, nDigits) {
            return parseFloat(value.toFixed(nDigits));
        }

        function getFeatureInfo(collectorData, featuresStorageType, featureName, featureType) {
            let defaultValue;
            let defaultEditMode = "DOMAIN";
            /* For DSS_MANAGED models, all feature lists are normally in concordance, but this is
             * too strong of an assumption for MLFLOW models, so be more lax */
            if (!(featureName in featuresStorageType) || !(featureName in collectorData.per_feature)) {
                return { value: "", editMode: "RAW", defaultEditMode: "RAW" };
            }
            switch (featureType) {
            case "NUMERIC": {
                const isTemporal = ["date", "dateonly", "datetimenotz"].indexOf(featuresStorageType[featureName]) >= 0;
                const isInteger = featuresStorageType[featureName] === "bigint";

                const defaultNumValue = collectorData.per_feature[featureName].stats.median;
                const min = collectorData.per_feature[featureName].stats.min;
                const max = collectorData.per_feature[featureName].stats.max;
                const nDecimals = isInteger ? 0 : WhatIfFormattingService.getSmartNumberDigits(min, max);
                defaultValue = isTemporal
                    ? moment.unix(defaultNumValue - epochShift).utc().format($scope.pickerFormat)
                    : roundToXDigits(defaultNumValue, nDecimals);
                return {
                    value: defaultValue,
                    min: roundToXDigits(min, nDecimals),
                    max: roundToXDigits(max, nDecimals),
                    type: isTemporal ? "DATE" : featureType,
                    editMode: defaultEditMode,
                    defaultValue,
                    defaultEditMode,
                    nDecimals,
                }
            }
            case "CATEGORY": {
                const possibleValues = [...(collectorData.per_feature[featureName].category_possible_values || [])];
                if (perFeature[featureName].dummy_drop === "DROP") {
                    const doppedModality = collectorData.per_feature[featureName].dropped_modality;
                    // when nan is the dropped modality it has been replaced by "__DKU_N/A__"
                    // and it will not be added to the possible values (can use ignore feature for that)
                    if (doppedModality !== undefined && doppedModality !== "__DKU_N/A__") {
                        possibleValues.push(doppedModality);
                    }
                }
                defaultValue = collectorData.per_feature[featureName].stats.mostFrequentValue;
                defaultEditMode = possibleValues && possibleValues.length ? "DOMAIN" : "RAW";
                return {
                    value: defaultValue,
                    editMode: defaultEditMode,
                    defaultValue,
                    possibleValues,
                    defaultEditMode,
                }
            }
            case "VECTOR": {
                defaultEditMode = "RAW";
                const vector = [];
                vector.length = collectorData.per_feature[featureName].vector_length || 0;
                defaultValue = "["+vector.fill(0).join(", ")+"]";
                return { value: defaultValue, editMode: defaultEditMode, defaultValue, defaultEditMode};
            }
            case "TEXT":
                defaultEditMode = "RAW";
                return { value: "", editMode: defaultEditMode, defaultEditMode };
            case "IMAGE":
                return { value: undefined, defaultValue: undefined, editMode: defaultEditMode, defaultEditMode };
            }
        }

        function formatFeatureDomains(records) {
            return records.map((features) => {
                const formattedRecord = {};
                features.forEach(feature => {
                    if (feature.value !== null) {
                        formattedRecord[feature.name] = feature.type == "DATE" ? moment(feature.value).utc().format(backendFormat) : feature.value;
                    }
                });
                return formattedRecord;
            });
        }

        // Scores/Explanations and charts

        function generateChartData() {
            if (! $scope.score) return;

            if ($scope.isClassification()) {
                const MAX_CLASSES = 6;
                $scope.predictions = getPredictionChartData($scope.score, MAX_CLASSES)
                $scope.threshold = $scope.isBinaryClassification() ? $scope.modelData.userMeta.activeClassifierThreshold : undefined;
            } else if ($scope.isRegression()) {
                $scope.prediction = $scope.score.prediction;
                let predictions;
                if ($scope.modelData.predictionInfo) {
                    predictions = $scope.modelData.predictionInfo.predictions;
                } else {
                    // Overall partitioned models do not have prediction statistics.
                    // We need to use the scatterplot data
                    predictions = $scope.modelData.perf.scatterPlotData.y;
                }
                
                const rangeLimit = { min: 0, max: 1 };
                rangeLimit.max = d3.max(predictions);
                rangeLimit.min = d3.min(predictions);
                const x = d3.scale.linear().domain([rangeLimit.min, rangeLimit.max]).nice();
                const data = d3.layout.histogram()
                    .frequency(0)
                    .bins(x.ticks(20))
                    (predictions);

                $scope.axes = {
                    x: data.map(d => d.x),
                    ys: [data.map(d => d.y)]
                };
                $scope.xm = [$scope.prediction];
                $scope.dataAxes = ["Prediction", "Prediction density"];
            }
        }

        function sortAndFormatExplanations(results) {
            const allFormattedExplanations = []
            for (const result of results) {
                const formattedExplanations = [];
                if (result === null || !result.explanation || !$scope.hasPrediction(result.score))  {
                    allFormattedExplanations.push(null);
                } else {
                    for (const featureName in result.explanation) {
                        formattedExplanations.push({
                            feature: featureName,
                            value: result.explanation[featureName],
                        });
                    }
                    const sortedExplanations = formattedExplanations.sort((exp1, exp2) => Math.abs(exp2.value) - Math.abs(exp1.value));
                    $scope.topExplanationValue = sortedExplanations[0].value;
                    sortedExplanations.forEach((explanation) => {
                        explanation.barWidthRatio =  ExplanationBarUtils.computeExplanationBarWidthFunc($scope.barMaxWidth)($scope.topExplanationValue, explanation.value) / $scope.barMaxWidth;
                    });
                    allFormattedExplanations.push(sortedExplanations);
                }
            }
        return allFormattedExplanations;
        }

        $scope.onFeatureChange = Debounce().withDelay(200, 200).withScope($scope).wrap(onFeatureChange);
        const computeScoreDebounced = Debounce().withDelay(200, 200).withScope($scope).wrap(computeScore);
        const computeExplanationsDebounced = Debounce().withDelay(1000, 1000).withScope($scope).wrap(computeExplanations);

        if ($scope.tile && $scope.tile.tileParams.advancedOptions.interactiveScoring) {
            $scope.userFeatures = $scope.tile.tileParams.advancedOptions.interactiveScoring.featuresOrder;
        }

        if ($scope.uiStateAlreadyLoaded) {
            $scope.featuresSortingService.setPerFeatureIndex($scope.uiState.schemaColumns);
            $scope.featuresSortingService.setPerFeatureImportance($scope.uiState.perFeatureImportance);
            angular.extend($scope.selection, $scope.featuresSortingService.getDefaultSelection())

            if ($scope.userFeatures) {
                $scope.selection.orderQuery = 'userDefinedOrder';
                $scope.selection.orderReversed = false;
            }
            onFeatureChange();
            $scope.sendBasicWT1Event();
        } else {
            Promise.all([DataikuAPI.ml.prediction.getCollectorData(fullModelId),
                        DataikuAPI.ml.prediction.getColumnImportance(fullModelId),
                        DataikuAPI.ml.prediction.getSplitDesc(fullModelId)]).then(
                            ([collectorDataResp, columnImportanceResp, splitDescResp]) => {
                const collectorData = collectorDataResp.data;
                let perFeatureImportance;
                $scope.uiState.schemaColumns = splitDescResp.data.schema.columns;
                if (columnImportanceResp.data) {
                    const columns = columnImportanceResp.data.columns;
                    const importances = columnImportanceResp.data.importances;
                    perFeatureImportance = {};
                    columns.forEach((col, i) => perFeatureImportance[col] = importances[i]);
                    $scope.uiState.perFeatureImportance = perFeatureImportance;
                    $scope.featuresSortingService.setPerFeatureImportance($scope.uiState.perFeatureImportance);
                }
                $scope.featuresSortingService.setPerFeatureIndex($scope.uiState.schemaColumns);
                $scope.uiState.featureFilterOptions = $scope.featuresSortingService.getSortOptions();
                angular.extend($scope.selection, $scope.featuresSortingService.getDefaultSelection());
                $scope.uiState.featuresStorageType = Object.fromEntries(splitDescResp.data.schema.columns.map(x => [x.name, x.type]));

                $scope.uiState.features = splitDescResp.data.schema.columns
                    .map(column => column.name)
                    .filter(featureName => featureName in perFeature)
                    .filter(featureName => authorizedTypes.includes(perFeature[featureName].type) &&
                                            authorizedRoles.includes(perFeature[featureName].role))
                    .map((name, index) => ({
                        index,
                        name,
                        type: perFeature[name].type,
                        role: perFeature[name].role,
                        importance: perFeatureImportance ? perFeatureImportance[name] : null,
                        ...getFeatureInfo(collectorData, $scope.uiState.featuresStorageType, name, perFeature[name].type)
                    }));

                if ($scope.isPartitionedModel() && $scope.isOnPartitionedBaseModel()) {
                    const dimensionNames = $scope.modelData.coreParams.partitionedModel.dimensionNames;
                    const donePartitions = Object.entries($scope.partitionedModelSnippets.partitions.summaries)
                                            .filter(([_, partition]) => partition.state.endsWith("DONE"))
                                            .map(([name, _]) => name.split("|"));

                    for (const [index, dimensionName] of dimensionNames.entries()) {
                        const possibleValues = [...new Set(donePartitions.map(part => part[index]))];
                        $scope.uiState.features.push({
                            name: dimensionName,
                            defaultValue: donePartitions[0][index],
                            value: donePartitions[0][index],
                            type: "PART_DIM",
                            possibleValues,
                        });
                    }
                }
                $scope.uiState.features.forEach(feature => feature.distributionType = feature.type);
                if ($scope.userFeatures) {
                    setUserDefinedOrdersIfNecessary();
                    $scope.selection.orderQuery = 'userDefinedOrder';
                    $scope.selection.orderReversed = false;
                }

                // Check if preparation script contains steps, if yes get the schema of the dataset before script
                DataikuAPI.ml.prediction.getPreparationScript(fullModelId).success((preparationScript) => {
                    $scope.uiState.hasPreparationSteps = preparationScript.steps.some(step => !step.disabled);
                    if ($scope.uiState.hasPreparationSteps) {
                        $scope.uiState.applyPreparationScript = true;
                        DataikuAPI.ml.prediction.getInputDatasetSchema(fullModelId).success((schema) => {
                            $scope.uiState.preScriptFeatures = schema.columns.map((col, index) =>  {
                                // setting importance to -1 because when sorting by importance, these features should be below features that have importance=0
                                const f = $scope.uiState.features.find(f => f.name === col.name) ||
                                    { name: col.name, type: "TEXT", value: "", editMode: "RAW", importance: -1 };
                                f.index = index;
                                return f;
                            });
                            $scope.uiState.preScriptFeatures.forEach(feature => feature.distributionType = feature.type);
                            setUserDefinedOrdersIfNecessary();
                            onFeatureChange();
                        }).catch((error) => {
                            if (error.status != 404) {
                                setErrorInScope.bind($scope)(error.data, error.status, error.headers);
                            } else {
                                $scope.uiState.applyPreparationScript = false;
                                $scope.uiState.couldNotRetrieveSchema = true;
                                onFeatureChange();
                            }
                        }).finally(() => {
                            $scope.sendBasicWT1Event();
                        });
                    } else {
                        onFeatureChange();
                        $scope.sendBasicWT1Event();
                    }
                }).catch(setErrorInScope.bind($scope));

            }).catch(setErrorInScope.bind($scope));
        }


        // ----- Comparator methods

        // We avoid having features before and after preparation script in the same comparator
        // If incompatible features are added to the comparator we offer two choices to the user:
        // cancel this adding or override the comprator content
        function protectBucketFromConflicts(needPreparationScript) {
            return new Promise((resolve, reject) => {
                if ($scope.bucket.length !== 0 &&
                    $scope.bucket[0].applyPreparationScript !== null &&
                    needPreparationScript !== null &&
                    $scope.bucket[0].applyPreparationScript !== needPreparationScript) {
                        CreateModalFromTemplate("templates/ml/prediction-model/interactive-scoring-conflict-dialog.html", $scope, null, function(newScope) {
                            newScope.showTips = true;
                            newScope.pasteAnyway = () => {
                                newScope.dismiss();
                                $scope.bucket = [];
                                resolve();
                            }
                            newScope.cancel = () => {
                                newScope.dismiss();
                                reject();
                            };
                        });
                } else {
                    resolve();
                }
            });
        }

        function safelyPasteInBucket(pastedItems) {
            const needPreparationScript = pastedItems[0].applyPreparationScript;
            protectBucketFromConflicts(needPreparationScript).then(() => {
                pasteInBucket(pastedItems);
            });
        }

        // Add current
        function addToBucket() {
            const newItem = {
                name: '',
                score: $scope.score,
                explanation: $scope.explanations,
                features: $scope.getFeatures(),
                applyPreparationScript: $scope.uiState.applyPreparationScript,
            };
            // ignore any keys that aren't in the model's list of features
            //const features = $scope.uiState.features.map(feature => feature.name);
            newItem.features = formatFeatures(newItem.features);
            $scope.scrollToLastItem = true;
            $scope.bucket.push(newItem);
            ActivityIndicator.success("Added to comparator.");
            $scope.$apply();
        }

        $scope.safelyAddToBucket = function() {
            const needPreparationScript = $scope.uiState.applyPreparationScript;
            protectBucketFromConflicts(needPreparationScript).then(() => {
                addToBucket();
            });
        };

        $scope.removeFromBucket = function(index) {
            $scope.bucket[index].removing = true;

            $timeout(() => {
                $scope.bucket.splice(index, 1);
            }, 500);
        };

        $scope.removeAllFromBucket = function() {
            Dialogs.confirm($scope, "Clear all items", "Are you sure you want to clear all items in the comparator?").then(function () {
                $scope.bucket = [];
                ActivityIndicator.success('All items cleared from comparator.');
            });
        };

        // comparator item formatting
        function formatItemsForLocalStorage(items) {
            let formatedItems = angular.copy(items);

            return formatedItems.map(item => {
                Object.keys(item).forEach(key => {
                    if (key.startsWith('$')) {
                        delete item[key];
                    }
                });
                item.features = formatFeatures(item.features);

                // Do not save model's predictions which can become invalid in case the model threshold is changed at some point.
                item.score = undefined;
                item.explanation = undefined;

                return item;
            });
        }

        // Generates prediction chart under each item in the comparator
        function getPredictionChartData(predictions, maxClasses) {
            if (!$scope.isClassification()) return;

            // assign a color to each prediction
            let classes = $scope.modelData.classes.filter(pc => `${scorePrefix}${pc}` in predictions);
            let colorPalette = $scope.colors.slice(0, classes.length);
            let chartPredictions = classes.map((pc, index) => ({
                name: pc,
                value: predictions[`${scorePrefix}${pc}`],
                color: colorPalette[index]
            }));

            if ($scope.isBinaryClassification()) {
                chartPredictions = [chartPredictions[1], chartPredictions[0]];
            } else {
                chartPredictions.sort((p1, p2) => p2.value - p1.value);

                if (chartPredictions.length > maxClasses) {
                    chartPredictions = chartPredictions.slice(0, maxClasses - 1);
                    const othersPercentage = 1 - chartPredictions.reduce((total, prediction) => total + prediction.value, 0);
                    chartPredictions.push({
                        name: 'Others',
                        value: othersPercentage,
                        color: OTHERS_COLOR
                    });
                }
            }

            return chartPredictions;
        }

        $scope.$watch('bucket', function() {
            localStorageService.set(LOCAL_STORAGE_BUCKET_KEY, formatItemsForLocalStorage($scope.bucket));
            if ($scope.isClassification()) {
                $scope.bucketCharts = $scope.bucket.map(item => item.score ? getPredictionChartData(item.score, 3) : null);
            }

            if ($scope.bucket.length > 1) {
                const listOfFeatureValues = $scope.bucket[0].features.map((_, colIndex) => $scope.bucket.map(row => row.features[colIndex].value));
                const allEqual = arr => arr.every(v => v === arr[0]);
                for (const [i, featureValues] of listOfFeatureValues.entries()) {
                    for (const item of $scope.bucket) {
                        item.features[i].greyed = allEqual(featureValues);
                    }
                }
            } else if ($scope.bucket.length === 1) {
                $scope.bucket[0].features.forEach(f => f.greyed = false);
            }
        }, true);

        $scope.$watch("uiState", function() {
            localStorageService.set(LOCAL_STORAGE_UI_STATE_KEY, $scope.uiState);
        }, true);

        $scope.$watch("selection.orderQuery", $scope.featuresSortingService.updateSortOrderAfterOptionChange.bind($scope.featuresSortingService));

        $scope.$on('$destroy', function() {
            localStorageService.set(LOCAL_STORAGE_BUCKET_KEY, $scope.bucket);
        });

        $scope.$watchCollection('userFeatures', setUserDefinedOrdersIfNecessary);

        // ----- Copy/paste

        const copyType = 'interactive-scoring';

        $scope.copyValues = function(items) {
            let copy = {
                type: copyType,
                version: $scope.$root.appConfig.version.product_version,
                samples: items.map(item => ({
                    name: item.name,
                    features: formatFeatures(item.features),
                    applyPreparationScript: item.applyPreparationScript,
                }))
            };

            ClipboardUtils.copyToClipboard(JSON.stringify(copy, null, 2), `Copied ${items.length} item${items.length === 1 ? '': 's'} to clipboard.`);
        };

        $scope.disableAllFeatures = function() {
            $scope.getFeatures().forEach(feature => $scope.changeEditMode(feature, "UNSET", false));
            onFeatureChange();
        };

        $scope.resetAllFeaturesToDefault = function() {
            $scope.getFeatures().forEach(feature => {
                feature.value = feature.defaultValue;
                feature.editMode = feature.defaultEditMode;
            });
            onFeatureChange();
        };

        $scope.openPasteDialog = function(pasteType) {
            let newScope = $scope.$new();

            CreateModalFromTemplate("/templates/ml/prediction-model/interactive_scoring_paste_modal.html", newScope, 'PasteModalController', function(modalScope) {
                modalScope.showTips = true;
                modalScope.copyType = 'interactive-scoring';
                modalScope.itemKey = 'samples';
                modalScope.pasteSingle = pasteType === WhatIfView.MAIN;
                modalScope.pasteItems = pasteType === WhatIfView.MAIN ? pasteFeatures : safelyPasteInBucket;
                modalScope.validateData = validatePastedData;
                modalScope.applyGenericFormat = formatCopiedData;
            });
        };

        function formatCopiedData(data) {
            if (Array.isArray(data.samples)) {
                return data;
            } else {
                return { // May come from "Copy rows as JSON" in Dataset explore
                    type: copyType,
                    version: $scope.$root.appConfig.version.product_version,
                    samples: [{
                        name: "",
                        features: Object.entries(data).map(([name, value]) => { return {name , value} }),
                        applyPreparationScript: null,
                    }]
                }
            }
        }

        // immediately show preview state since we've already pasted
        $scope.openPasteModalFromKeydown = function(data) {
            try {
                data = JSON.parse(data);
            } catch (e) { /* Nothing for now */ }

            if (data && !Array.isArray(data.samples)) { // May come from "Copy rows as JSON" from Dataset explore
                data = formatCopiedData(data);
                if (!validatePastedData(data.samples)) {
                    return;
                }
            }

            if (data && data.samples && data.samples.length && data.type === copyType) {
                let newScope = $scope.$new();

                CreateModalFromTemplate("/templates/ml/prediction-model/interactive_scoring_paste_modal.html", newScope, 'PasteModalController', function(modalScope) {
                    modalScope.uiState.editMode = false;
                    modalScope.uiState.items = data.samples;
                    modalScope.pasteSingle = whatIfRouter.getCurrentView() === WhatIfView.MAIN;
                    modalScope.pasteItems = whatIfRouter.getCurrentView() === WhatIfView.MAIN ? pasteFeatures : safelyPasteInBucket;
                    modalScope.validateData = validatePastedData;
                });
            }
        };

        $scope.getCurrentItem = function() {
            return {
                name: '',
                features: $scope.getFeatures(),
                applyPreparationScript: $scope.uiState.applyPreparationScript,
            };
        };

        $scope.keydownCopy = function(event) {
            if (whatIfRouter.getCurrentView() === WhatIfView.MAIN && isViewComputing(WhatIfView.MAIN)) {
                $scope.copyValues([$scope.getCurrentItem()]);

                event.currentTarget.focus();
            }
        };

        function validatePastedData(pastedItems) {
            const allFeatures = $scope.getFeatures();

            // make sure at least one feature in each sample matches features used in the model
            return pastedItems.every(item => {
                return item.features.some(feature => {
                    return allFeatures.find(f => feature.name === f.name);
                });
            });
        }

        function pasteFeatures(pastedItems) {
            if (!pastedItems.length) return;
            const firstItem = pastedItems[0];
            $scope.uiState.applyPreparationScript = firstItem.applyPreparationScript;
            $scope.getFeatures().forEach(feature => {
                // take first set of features from list
                const pastedFeature = firstItem.features.find(f => feature.name === f.name);
                const pastedValue = pastedFeature ? pastedFeature.value : null;

                feature.editMode = getNewEditMode(feature, pastedValue);
                feature.value = pastedValue;
            });

            onFeatureChange();

            WT1.event('interactive-scoring-paste', {
                nbItems: pastedItems.length,
                type: 'features'
            });
        }

        // find the appropriate edit mode for the new value when pasted
        function getNewEditMode(feature, newValue) {
            if (newValue !== null) {
                if (feature.type === 'NUMERIC') {
                    if (newValue < feature.min || newValue > feature.max) {
                        return 'RAW';
                    }
                } else if (feature.type === "CATEGORY") {
                    if (!feature.possibleValues.includes(newValue)) {
                        return 'RAW';
                    }
                }

                return 'DOMAIN';
            }

            return 'UNSET';
        }

        function pasteInBucket(pastedItems) {
            const featureNames = $scope.getFeatures().map(feature => feature.name);
            const newItems = [];
            pastedItems.forEach(item => {
                newItems.push({
                    name: item.name,
                    features: featureNames.map(featureName => {
                        const matchedFeature = item.features.find(f => f.name === featureName) || {}
                        return {
                            name: featureName,
                            value: matchedFeature.value,
                            type: matchedFeature.type,
                        }
                    }),
                    applyPreparationScript: item.applyPreparationScript
                });
            });
            const newBucket = $scope.bucket.concat(angular.copy(newItems));
            computeAll(newBucket, null, () => {
                $scope.bucket = newBucket;
                $scope.scrollToLastItem = whatIfRouter.getCurrentView() === WhatIfView.COMPARATOR;
                ActivityIndicator.success(`${newItems.length} item${newItems.length === 1 ? '': 's'} successfully pasted.`);
            });

            WT1.event('interactive-scoring-paste', {
                nbItems: pastedItems.length,
                type: 'comparator'
            });
        }


        // ----- Export

        $scope.exportComparator = function() {
            // We supposed here that each item in the bucket have the same columns (theoretically ensure when pasting or adding)
            const featureColumns = $scope.bucket[0].features.map((feature) => ({ name: feature.name, type: $scope.uiState.featuresStorageType[feature.name] || "string" }));
            const nameColumn = {
                name: 'score_name',
                type: 'string'
            };

            let predictionColumns = [];
            if ($scope.isClassification()) {
                predictionColumns = [...$scope.modelData.classes.map(c => ({
                    name: `${scorePrefix}${c}`,
                    type: 'string'
                })), {
                    name: 'prediction',
                    type: 'string'
                }];
            } else if ($scope.isRegression()) {
                predictionColumns = [{
                    name: 'prediction',
                    type: $scope.uiState.featuresStorageType[$scope.modelData.coreParams.target_variable]
                }];
            }

            const explanationColumns = [{
                name: 'explanations',
                type: 'string'
            }, {
                name: 'explanation_method',
                type: 'string'
            }];

            const withOverrideInfo = $scope.bucket.some(item => item.score.override);

            // for each bucket, 1) loop through the columns and find the associated value (if it exists)
            // and 2) add prediction information, and 3) explanations if present
            const data = $scope.bucket.map(item => {
                const featureValues = featureColumns.map(column => {
                    const feature = item.features.find(f => f.name === column.name);

                    return feature ? feature.value : null;
                });
                let predictions = [item.score ? item.score.prediction[0] : null];
                // add proba for each class
                if ($scope.isClassification()) {
                    predictions = [...$scope.modelData.classes.map(c => item.score ? item.score[`${scorePrefix}${c}`] : null), ...predictions];
                }

                let values = [...featureValues, item.name, ...predictions]
                if ($scope.getExplanationIncompatibility() == null) {
                    const listOfExplanations = item.explanation.map(f => { return { [f.feature]: f.value }});
                    const explanationsStr = JSON.stringify(Object.assign({}, ...listOfExplanations));
                    values = [...values, explanationsStr, $scope.explanationParams.method]
                }

                if (withOverrideInfo) {
                    values.push(JSON.stringify(item.score.override));
                }

                return values;
            });

            let columns = [...featureColumns, nameColumn, ...predictionColumns];
            if ($scope.getExplanationIncompatibility() == null) {
                columns = [...columns, ...explanationColumns ]
            }

            if (withOverrideInfo) {
                columns.push({
                    name: 'override',
                    type: 'string'
                });
            }

            ExportUtils.exportUIData($scope, {
                name : "Interactive scoring for model: " + $scope.modelData.userMeta.name,
                columns,
                data
            }, 'Export Interactive Scoring');

            WT1.event('interactive-scoring-export-button', {
                explanationMethod: $scope.explanationParams.method,
                nbItems: data.length
            });
        };

        // ----- Backend computing

        // ----- Backend handling and computing

        function canCompute() {
            return !($scope.uiState.applyPreparationScript && $scope.uiState.couldNotRetrieveSchema)
                && !$scope.allFeaturesEmpty() && $scope.allEnabledImageFeaturesFilled();
        }

        function compute(actionFn, params, view, computationType, callback) {
            if (!isViewComputing(view, computationType)) {
                currentComputation[view][computationType].status = computationStatus.COMPUTING;

                $scope.errors[computationType] = null;
                currentComputation[view][computationType].nextComputation = null;

                actionFn(params).then(function(data) {
                    if (!isViewAborted(view, computationType)) { 
                        
                        currentComputation[view][computationType].status = computationStatus.DONE;
                        if (currentComputation[view][computationType].nextComputation) { // if another computation was launched during current compute
                            compute(actionFn, currentComputation[view][computationType].nextComputation, view, computationType, callback);
                        } else {
                            callback(data);
                            $scope.$digest();   // Debounce uses setTimeout, angular might have skipped some updates
                        }
                    }
                    // Note: we do not compute nextComputation if current computation was aborted.
                    
                }).catch((error) => {
                    $scope.errors[computationType] = error;
                    currentComputation[view][computationType].status = computationStatus.ERROR;

                    if (currentComputation[view][computationType].nextComputation) { // if another computation was launched during current compute
                        compute(actionFn, currentComputation[view][computationType].nextComputation, view, computationType, callback);
                    }
                });
            } else {
                currentComputation[view][computationType].nextComputation = $scope.uiState;
            }
        }

        function makeComputationPromise(apiCall, view, computationType) {
            return new Promise(function(resolve, reject) {
                function err(...args) {
                    currentComputation[view][computationType].runningJobId = null;
                    setErrorInScope.apply($scope, args);
                    reject(...args);
                }
                apiCall.noSpinner().success(function(initialResponse) {
                    if (initialResponse.hasResult) {
                        resolve(prepareOverrideInfoFromResults(initialResponse.result));
                    } else {
                        currentComputation[view][computationType].runningJobId = initialResponse.jobId;
                        FutureWatcher.watchJobId(initialResponse.jobId)
                            .success(doneResponse => {
                                currentComputation[view][computationType].runningJobId = null; 
                                resolve(prepareOverrideInfoFromResults(doneResponse.result));
                            })
                            .error(err);
                    }
                }).error(err);
            });
        }

        function computeScore() {
            const view = whatIfRouter.getCurrentView();
            const promiseFn = params => {
                const computationParams = {
                    applyPreparationScript: params.applyPreparationScript,
                    type: InteractiveModelCommand.SCORING
                };
                return makeComputationPromise(DataikuAPI.interactiveModel.compute(
                    fullModelId,
                    computationParams,
                    formatFeatureDomains([getFeaturesFromUiState(params)])
                ), view, computationParams["type"]);
            };

            return compute(promiseFn, $scope.uiState, view, InteractiveModelCommand.SCORING, (data) => {
                if ($scope.allFeaturesEmpty()) { // features may have been disabled during compute
                    $scope.score = null;
                } else {
                    $scope.score = data[0] === null ? null : data[0].score;
                    generateChartData();
                }
            });
        }

        function computeExplanations() {
            const currView =  whatIfRouter.getCurrentView();
            const promiseFn = params => {
                const computationParams = {
                    applyPreparationScript: params.applyPreparationScript,
                    explanationMethod: $scope.explanationParams.method,
                    nExplanations: $scope.explanationParams.nbExplanations,
                    type: InteractiveModelCommand.EXPLANATIONS
                };
                return makeComputationPromise(DataikuAPI.interactiveModel.compute(
                    fullModelId,
                    computationParams,
                    formatFeatureDomains([getFeaturesFromUiState(params)])
                ), currView, computationParams["type"]);
            }

            return compute(promiseFn, $scope.uiState, currView, InteractiveModelCommand.EXPLANATIONS, function(data) {
                const allExplanations = sortAndFormatExplanations(data)
                if ($scope.allFeaturesEmpty()) { // features may have been disabled during compute
                    $scope.explanations = null;
                } else {
                    $scope.explanations = (allExplanations !== null && allExplanations.length) ? allExplanations[0] : [];
                }
            });
        }

        async function computeAll(bucketItems, currentFeatures, callback) {
            const allFeatures = bucketItems.map(item => item.features);
            if (currentFeatures) {
                allFeatures.push(currentFeatures);
            }
            
            const view =  whatIfRouter.getCurrentView();
            const computationParams = {
                applyPreparationScript: $scope.uiState.applyPreparationScript
            };
            let computePromise;
            const records = formatFeatureDomains(allFeatures);
            if ($scope.getExplanationIncompatibility()) {
                computationParams["type"] = InteractiveModelCommand.SCORING;
                computePromise = makeComputationPromise(DataikuAPI.interactiveModel.compute(
                    fullModelId,
                    computationParams,
                    records
                ), view, computationParams["type"]);
            } else {
                computationParams["explanationMethod"] = $scope.explanationParams.method;
                computationParams["nExplanations"] = $scope.explanationParams.nbExplanations;
                computationParams["type"] = InteractiveModelCommand.EXPLANATIONS;
                computePromise = makeComputationPromise(DataikuAPI.interactiveModel.compute(
                    fullModelId,
                    computationParams,
                    records
                ), view, computationParams["type"]);
            }
            currentComputation[view][computationParams["type"]].status = computationStatus.COMPUTING;
            if (whatIfRouter.getCurrentView() === WhatIfView.COMPARATOR) {
                SpinnerService.lockOnPromise(computePromise);
            }
            $q.when(computePromise)
            .then((results) => {
                if (!isViewAborted(view, computationParams["type"])) { 
                    currentComputation[view][computationParams["type"]].status = computationStatus.DONE;

                    const scores = results.map(result => result === null ? null: result.score);
                    let explanations = sortAndFormatExplanations(results);
                    if (currentFeatures) {
                        $scope.score = scores.pop();
                        $scope.explanations = explanations ? explanations.pop() : null;
                    }
                    bucketItems.forEach((item, index) => {
                        item.explanation = explanations ? explanations[index] : null;
                        item.score = scores[index];
                    });
                    if (callback) {
                        callback();
                    }
              }
            }).catch((error) => {
                currentComputation[view][computationParams["type"]].status = computationStatus.ERROR;
                setErrorInScope.bind($scope)(error.data, error.status, error.headers);
            });
        }
        
        function abortComputationIfRunning(view, computationType) {
            const jobId = currentComputation[view][computationType].runningJobId;
            if (jobId !== null) {
                // set computation as aborted & clear results only if the jobs are stillrunning (we want to keep results for completed computations)
                if (view === WhatIfView.MAIN) {
                    computationType === InteractiveModelCommand.SCORING? $scope.score = null: $scope.explanations=null; // make sure not to display outdated scores.
                }

                DataikuAPI.futures.abort(jobId).success(function(data) {
                    currentComputation[view][computationType].runningJobId = null;
                    currentComputation[view][computationType].status = computationStatus.ABORTED;
                    currentComputation[view][computationType].nextComputation = null;
                });
            }
        }

        function prepareOverrideInfoFromResults(results) {
            results.forEach(result => {
                if (result && result.score && result.score.override) {
                    result.score.override = JSON.parse(result.score.override);
                }
            });
            return results;
        }

        $scope.onExplanationParamsChange = function() {
            localStorageService.set(LOCAL_STORAGE_EXPLANATION_PARAMS_KEY, $scope.explanationParams);
            $scope.explanations = undefined;
            computeAll($scope.bucket, $scope.getFeatures(), 
            () => {
                if (whatIfRouter.getCurrentView() === WhatIfView.MAIN) { 
                    // recompute chart data just in case it was previously aborted:
                    generateChartData(); 
                }  
            $scope.sendBasicWT1Event();
            });
        }

        $scope.explanationDisableToggled = function() {
            localStorageService.set(LOCAL_STORAGE_EXPLANATION_PARAMS_KEY, $scope.explanationParams);
            if ($scope.explanationParams.enabled) {
                computeExplanations();
            } else {
                // Removing all information about explanations (results, error) as we consider it deactivated
                $scope.explanations = null;
                $scope.errors[InteractiveModelCommand.EXPLANATIONS] = null;
            }
            $scope.sendBasicWT1Event();
        };

        function onFeatureChange() {
            if (canCompute()) {
                // make sure not to display outdated values:
                $scope.explanations = undefined;
                $scope.score = undefined;
                
                computeFeature(true);
            }
            else{
                $scope.explanations = null;
                $scope.score = null;
            }
        }

        async function computeFeature(withDebounce) {
            if (withDebounce) {
                computeScoreDebounced();
                if (!$scope.getExplanationIncompatibility() && $scope.explanationParams.enabled) {
                    computeExplanationsDebounced();
                }
            } else {
                computeScore();
                if (!$scope.getExplanationIncompatibility() && $scope.explanationParams.enabled) {
                    computeExplanations();
                }
            }
        }

        $scope.extractFeatureFromBucketItem = (item, featureName) => item.features.find(f => f.name === featureName);

        // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
        $scope.puppeteerHook_elementContentLoaded = true;
    });

    app.component('interactiveScoringFeatureValue', {
        template: `
            <div ng-class="{'interactive_scoring__greyed-feature': $ctrl.feature.greyed}">
                <h5 class="font-weight-semi-bold mbot0 mx-textellipsis">{{ $ctrl.feature.name }}</h5>
                <span ng-if="isNull"><i>ignored</i></span>
                <span ng-if="!isNull">
                    <span ng-if="isEmptyOrUndefined"><i>none</i></span>
                    <span ng-if="!isEmptyOrUndefined">
                        <span ng-if="$ctrl.feature.type !== 'IMAGE'">{{ $ctrl.feature.value }}</span>
                        <img ng-if="$ctrl.feature.type === 'IMAGE'" data-ng-src="data:image/png;base64,{{$ctrl.feature.value}}"/>
                    </span>
                </span>
            </div>`,
        bindings: { feature: '<' },
        controller: function($scope) {
            this.$onInit = function() {
                $scope.isNull = this.feature.value === null;
                $scope.isEmptyOrUndefined = this.feature.value === undefined || this.feature.value === '';
            }
        }
    });

    app.component('whatIfSortingWidget', {
        template: `
            <div class="std-list-search-box horizontal-flex h-auto {{$ctrl.additionalClass}}">
                <span ng-if="!$ctrl.searchBarOnly" class="horizontal-flex mright8">
                    <sort-order-button class="interactive-scoring__reverse-sort-btn" value="$ctrl.selection.orderReversed"></sort-order-button>
                    <select dku-bs-select class="input-group-btn interactive-scoring__sort-select" ng-model="$ctrl.selection.orderQuery" ng-options="opt.value as opt.label for opt in $ctrl.featureFilterOptions"></select>
                </span>
                <span class="interactive-scoring__search-bar-group">
                    <span class="add-on"><i class="icon-dku-search padleftright4"></i></span>
                    <input class="interactive-scoring__search-bar" type="search" ng-model="$ctrl.selection.filterQuery.userQuery" placeholder="Filter..."/>
                </span>
            </div>`,
        bindings: { searchBarOnly: '<', selection: '<', featureFilterOptions: '<', additionalClass: '<' }
    });

})();
