(function() {
'use strict';

const app = angular.module('dataiku.recipes');

app.controller("_NLPLLMRecipeControllerBase", function($scope, $stateParams, $q, DataikuAPI, PartitionDeps, ComputableSchemaRecipeSave, SmartId, PromptUtils) {
    $scope.onLLMChange = function() {
        if ($scope.desc && $scope.desc.llmId && $scope.availableLLMs) {
            $scope.activeLLM = $scope.availableLLMs.find(l => l.id === $scope.desc.llmId);
            if (!$scope.activeLLM) {
                $scope.retrievableKnowledge = undefined;
                return;
            }
            $scope.temperatureRange = PromptUtils.getTemperatureRange($scope.activeLLM);
            $scope.topKRange = PromptUtils.getTopKRange($scope.activeLLM);
            if ($scope.activeLLM.type === "RETRIEVAL_AUGMENTED") {
                const smartId = SmartId.resolve($scope.activeLLM.savedModelSmartId);
                DataikuAPI.savedmodels.get(smartId.projectKey, smartId.id).then(function({data}) {
                    DataikuAPI.retrievableknowledge.get(smartId.projectKey, data.inlineVersions[0].ragllmSettings.kbRef).then(function({data}) {
                        $scope.retrievableKnowledge = data;
                    })
                }).catch(setErrorInScope.bind($scope));
            } else {
                $scope.retrievableKnowledge = undefined;
            }
        }
    };

    $scope.$watch('recipe.inputs.main', function(nv, ov) {
        if (!nv) return;

        $scope.inputRef = ($scope.recipe.inputs.main.items || [{}])[0].ref;
        if (!$scope.inputRef) return;
                
        var mainRoleAcceptDataset = true;
        if($scope.recipeDesc && $scope.recipeDesc.inputRoles){
            mainRoleAcceptDataset = $scope.recipeDesc.inputRoles.find(role => role.name == 'main').acceptsDataset; // support for folder as main role (embed documents)
        } 

        if(mainRoleAcceptDataset){

            const resolvedRef = SmartId.resolve($scope.inputRef, $stateParams.projectKey);
            DataikuAPI.datasets.get(resolvedRef.projectKey, resolvedRef.id, $stateParams.projectKey).then(response => {
                const schema = response.data.schema;
                $scope.inputDatasetColumns = schema.columns.map(column => column.name);
                $scope.inputDatasetSchema = schema;
            });
        }
    });

    const deregister = $scope.$watch("script", function() {
        if (!$scope.script) return;

        $scope.desc = JSON.parse($scope.script.data);

        $scope.onLLMChange();
        deregister();
    });

    $scope.loadLLMs = function(purpose) {
        DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, purpose).success(function(data) {
            $scope.availableLLMs = data.identifiers;
            $scope.onLLMChange();
        }).error(setErrorInScope.bind($scope));
    };

    $scope.hooks.recipeIsDirty = function() {
        if (!$scope.recipe) return false;
        if ($scope.creation)  return true;

        const dirtyRecipe = !angular.equals($scope.recipe, $scope.origRecipe);
        const origDesc = JSON.parse($scope.origScript.data);
        const dirtyDesc = !angular.equals(origDesc, $scope.desc);
        return dirtyRecipe || dirtyDesc;
    };

    $scope.hooks.save = function(){
        const recipeSerialized = angular.copy($scope.recipe);
        PartitionDeps.prepareRecipeForSerialize(recipeSerialized);

        const payload = angular.toJson($scope.desc);
        const deferred = $q.defer();
        ComputableSchemaRecipeSave.handleSave($scope, recipeSerialized, payload, deferred);
        $scope.script.data = payload;
        return deferred.promise;
    };
});


app.controller("NLPRecipeConvertToPromptRecipeModal", function($scope, $stateParams, $state, DataikuAPI) {
    DataikuAPI.flow.recipes.nlp.getPromptForNLPRecipe($stateParams.projectKey, $scope.recipe.name).success(function(data) {
        $scope.prompt = data;
    }).error(setErrorInScope.bind($scope));

    $scope.convert = function() {
        if (!$scope.recipe || !$scope.recipe.name) return;

        DataikuAPI.flow.recipes.nlp.convertToPromptRecipe($stateParams.projectKey, $scope.recipe.name).success(function() {
            $state.reload();
        }).error(setErrorInScope.bind($scope));
    };
});


app.controller("GenAIBaseEvaluationRecipeCreationController", function($scope, evaluatedType, DataikuAPI, ActiveProjectKey, DatasetUtils, $controller, RecipeComputablesService) {
    $controller("_RecipeCreationControllerBase", {$scope:$scope});

    $scope.recipe = {
        projectKey : ActiveProjectKey.get(),
        type: "nlp_" + evaluatedType + "_evaluation",
        inputs : {},
        outputs : {},
        params: {}
    };

    $scope.forceMainLabel = true;
    $scope.uiState = {inputDs: null} // needed to replace 'io' (we don't want to use _RecipeOutputNewManagedBehavior), so that binding inputDs to the dataset-selector works across scopes


    addDatasetUniquenessCheck($scope, DataikuAPI, ActiveProjectKey.get());
    fetchManagedDatasetConnections($scope, DataikuAPI);

    $scope.$watch("uiState.inputDs", function(nv, ov) {
        if (nv) {
            $scope.recipe.name = evaluatedType + "_evaluate_" + nv.replace(/[A-Z]*\./, "");
        }
        if ($scope.uiState.inputDs) {
            $scope.recipe.inputs.main = {items:[{ref:$scope.uiState.inputDs}]}; // for the managed dataset creation options
        } else {
            $scope.recipe.inputs.main = {items:[]}; // for the managed dataset creation options
        }
    });


    DatasetUtils.listDatasetsUsabilityInAndOut(ActiveProjectKey.get(), $scope.recipe.type).then(function(data){
        $scope.availableInputDatasets = data[0];
    });

    RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map){
        $scope.setComputablesMap(map);
    });

    $scope.hasMain = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.main && outputs.main.items && outputs.main.items.length > 0 && outputs.main.items[0].ref
    }
    $scope.hasMetrics = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.metrics && outputs.metrics.items && outputs.metrics.items.length > 0 && outputs.metrics.items[0].ref
    }
    $scope.hasEvaluationStore = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.evaluationStore && outputs.evaluationStore.items && outputs.evaluationStore.items.length > 0 && outputs.evaluationStore.items[0].ref
    }

    $scope.canCreate = function(){
        return $scope.recipe.name
            && $scope.recipe.name.length > 0
            && $scope.recipe.outputs
            && !$scope.shouldDisplayOutputExplanation()
            && !($scope.newRecipeForm.$invalid);
    }

    $scope.shouldDisplayOutputExplanation = function () {
        return !$scope.hasMain() && !$scope.hasMetrics() && !$scope.hasEvaluationStore();
    };

    $scope.generateOutputExplanation = function () {
        const requiredOutputRoles = [];
        $scope.recipeDesc.outputRoles.forEach((role, outputRoleidx) => {
            requiredOutputRoles.push(role.name === "main" ? '"Output Dataset"' : '"' + (role.label || role.name) + '"');
        });
        const message = "This recipe requires at least one output in: "
            + requiredOutputRoles.slice(0, -1).join(', ')
            + (requiredOutputRoles.length === 2 ? ' or ' : ', or ')
            + requiredOutputRoles.slice(-1) + ".";
        return message;
    };

    $scope.doCreateRecipe = function() {
        $scope.creatingRecipe = true;
        const settings = {
            zone: $scope.zone
        }

        return DataikuAPI.flow.recipes.generic.create($scope.recipe, settings);
    };
});

app.controller(
    "GenAIBaseEvaluationRecipeEditorController",
    function($scope, evaluatedType, $rootScope, $controller, $stateParams, ActiveProjectKey, DataikuAPI, FutureProgressModal, DOCUMENT_SPLITTING_METHOD_MAP,
             ModelLabelUtils, EmbeddingUtils, PromptUtils, LLMEvaluationMetrics, GenAiEvaluationMetrics) {
    $controller("_NLPLLMRecipeControllerBase", {$scope: $scope});

    $scope.NO_COLUMN_SELECTED = "None - no column selected";
    $scope.NO_LLM_SELECTED = {"id": "None", "friendlyName": "None - no model selected", type: "None"}
    $scope.DOCUMENT_SPLITTING_METHOD_MAP = DOCUMENT_SPLITTING_METHOD_MAP;
    $scope.COLUMNS_NEEDED_BY_PROMPT_RECIPE = ["llm_raw_query", "llm_raw_response"];
    $scope.COLUMNS_NEEDED_BY_DATAIKU_ANSWERS = ["question", "answer", "sources"];

    $scope.$watch("desc", (nv, ov) => {
        if (angular.equals(nv, ov)) { // only happens on init
            $scope.onInit();
        }
    });

    $scope.missingColumnsWarning = "";

    $scope.$watch("inputDatasetColumns", (nv, ov) => {
        if (nv) {
            $scope.canPromptRecipe = $scope.COLUMNS_NEEDED_BY_PROMPT_RECIPE.every(col => nv.includes(col));
            $scope.canDataikuAnswers = $scope.COLUMNS_NEEDED_BY_DATAIKU_ANSWERS.every(col => nv.includes(col));
            $scope.updateMissingColumnsWarning($scope.desc.inputFormat);
        }
    });

    $scope.updateMissingColumnsWarning = function(inputFormat) {
        if ($scope.inputDatasetColumns) {
            if (inputFormat === "PROMPT_RECIPE" && !$scope.canPromptRecipe) {
                $scope.missingColumnsWarning = "Input dataset lacks some required columns: " + $scope.COLUMNS_NEEDED_BY_PROMPT_RECIPE.filter(col => !$scope.inputDatasetColumns.includes(col)) + ". Make sure \"Raw query output mode\" and \"Raw response output mode\" in your Prompt recipe are not set to \"None\".";
            } else if (inputFormat === "DATAIKU_ANSWERS" && !$scope.canDataikuAnswers) {
                $scope.missingColumnsWarning = "Input dataset lacks some required columns: " + $scope.COLUMNS_NEEDED_BY_DATAIKU_ANSWERS.filter(col => !$scope.inputDatasetColumns.includes(col)) + ". Make sure \"Retrieval Method\" is set to 'Use knowledge bank retrieval'.";
            } else {
                $scope.missingColumnsWarning = "";
            }
        }
    }

    DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "TEXT_EMBEDDING_EXTRACTION").success(function(data) {
        $scope.availableEmbeddingLLMs = [$scope.NO_LLM_SELECTED].concat(data.identifiers);
        $scope.refreshEmbeddingLLM();
    }).error(setErrorInScope.bind($scope));


    DataikuAPI.pretrainedModels.listAvailableLLMConnections().success(function(data) {
        $scope.localHuggingFaceConnectionNames = Object.values(data)
            .filter(connection => connection.type === 'HuggingFaceLocal')
            .map(connection => connection.name);
    });

    $scope.refreshEmbeddingLLM = function() {
        if ($scope.desc && $scope.desc.embeddingLLMId && $scope.availableEmbeddingLLMs) {
            $scope.activeEmbeddingLLM = $scope.availableEmbeddingLLMs.find(l => l.id === $scope.desc.embeddingLLMId);
        }
    }

    DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").success(function(data) {
        $scope.availableCompletionLLMs = [$scope.NO_LLM_SELECTED].concat(data.identifiers);
        $scope.refreshCompletionLLM();
    }).error(setErrorInScope.bind($scope));

    $scope.refreshCompletionLLM = function() {
        if ($scope.desc && $scope.desc.completionLLMId && $scope.availableCompletionLLMs) {
            $scope.activeCompletionLLM = $scope.availableCompletionLLMs.find(l => l.id === $scope.desc.completionLLMId);
        }

        if ($scope.activeCompletionLLM && $scope.activeCompletionLLM.id !== $scope.NO_LLM_SELECTED.id) {
            $scope.temperatureRange = PromptUtils.getTemperatureRange($scope.activeCompletionLLM);
            $scope.topKRange = PromptUtils.getTopKRange($scope.activeCompletionLLM);
        }
    }

    $scope.getMaxTokenLimit = function() {
        return $scope.activeEmbeddingLLM ? $scope.activeEmbeddingLLM.maxTokensLimit : null;
    }

    $scope.shouldWarnAboutChunkSize = function() {
        return EmbeddingUtils.shouldWarnAboutChunkSize($scope.activeEmbeddingLLM, $scope.desc.embeddingSettings.documentSplittingMode, $scope.desc.embeddingSettings.chunkSizeCharacters);
    };

    $scope.defaultRagasMaxWorkers = $rootScope.appConfig.defaultRagasMaxWorkers;
    $scope.mayRunCustomMetrics = $scope.mayWriteSafeCode();

    $scope.setRagasMaxWorkers = function(value) {
        if (!value) {
            $scope.desc.ragasMaxWorkers = null;
        }
    }

    $scope.hasAllRequiredOutputs = function() {
        if (!$scope.recipe || !$scope.recipe.outputs) {
            return false;
        }
        const out = $scope.recipe.outputs;
        // at least one of the outputs is needed
        if (out.main && out.main.items && out.main.items.length) {
            return true;
        } else if (out.evaluationStore && out.evaluationStore.items && out.evaluationStore.items.length) {
            return true;
        } else if (out.metrics && out.metrics.items && out.metrics.items.length) {
            return true;
        } else {
            return false;
        }
    };

    $scope.codeEnvWarning = {
        enabled: false,
        envName: "",
        reason: ""
    }

    DataikuAPI.codeenvs.listWithVisualMlPackages($stateParams.projectKey).success(function(data) {
        $scope.envsCompatibility = data;
        updateCodeEnvWarning();
    }).error(setErrorInScope.bind($scope));

    $scope.$watchGroup(['recipe.params.envSelection.envMode', 'recipe.params.envSelection.envName'], (nv, ov) => {
        updateCodeEnvWarning();
    });

    function updateCodeEnvWarning() {
        if ($scope.envsCompatibility === undefined) {
            return;
        }

        const compatibilityInfo = filterCodeEnvCompatibilityInfo($scope.envsCompatibility);
        let compat;
        if (evaluatedType === 'llm') {
            compat =  compatibilityInfo.llmEvaluation
        } else if (evaluatedType === 'agent') {
            compat =  compatibilityInfo.agentEvaluation
        } else {
            return;
        }

        // LLM agent evaluation compatibility info for now is always the same, but we need the distinction in the backend in order to have different text
        // in the incompatibility reasons. Plus, this might change in the future.
        $scope.codeEnvWarning.enabled = !compat.compatible;
        $scope.codeEnvWarning.envName = compatibilityInfo.envName ? compatibilityInfo.envName : "DSS builtin env"
        if (compat.reasons.length > 0) {
            $scope.codeEnvWarning.reason = compat.reasons[0];
        } else {
            $scope.codeEnvWarning.reason = "";
        }
    }

    function filterCodeEnvCompatibilityInfo(envsCompatibilityInfo) {
        if ($scope.recipe.params.envSelection.envMode === "EXPLICIT_ENV") {
            return envsCompatibilityInfo.envs.filter(env => env.envName == $scope.recipe.params.envSelection.envName)[0]
        }
        if ($scope.recipe.params.envSelection.envMode === "INHERIT" && $scope.envsCompatibility.resolvedInheritDefault) {
            return envsCompatibilityInfo.envs.filter(env => env.envName == $scope.envsCompatibility.resolvedInheritDefault)[0]
        }
        return envsCompatibilityInfo.builtinEnvCompat; // Built-in, or inheriting with no default set at the instance-level
    }

    const baseCanSave = $scope.canSave;
    $scope.canSave = function() {
        return ModelLabelUtils.validateLabels($scope.desc) && baseCanSave();
    };
});


app.service("GenAiEvaluationMetrics", function() {
    const svc = {};

    svc.getCustomMetricNamesFromEvaluationDetails = function(evaluationDetails) {
        let names = [];

        if (evaluationDetails.metrics && evaluationDetails.metrics.customMetricsResults) {
            names = evaluationDetails.metrics.customMetricsResults.map(function(metric) { return metric.metric.name });
        }

        return names;
    }


    return svc;
});


app.component("llmSelector", {
  bindings: {
    availableLlms: "<",
    selectedLlmId: "=",
    onChange: "<",
    onChangeOutput: "&", // refactor to just use onChange with &
    defaultOpened: '<',
    dropdownPosition: '@',
    disableAlertWarning: '<',
    placeholder: '<', // Optional, defaults to "Nothing selected"
    nullValueLabel: '<',
    nullValueId: '<',
    hideCopyButton: '<',
    hideGoToLlmButton: '<',
    disabled: '<',
    clearable: '<'
  },
  template: `
    <basic-select ng-if="$ctrl.availableLlms"
                  items="$ctrl.availableLlms"
                  ng-model="$ctrl.selectedLlmId"
                  bind-value="id"
                  bind-label="friendlyNameShort"
                  group-by-fn="$ctrl.groupBy"
                  bind-group-label-fn="$ctrl.niceModelType"
                  bind-group-annotation="connection"
                  bind-group-annotation-icon-fn="$ctrl.modelConnectionIcon"
                  bind-group-description="connectionDescription"
                  searchable="true"
                  search-fn="$ctrl.searchFunction"
                  placeholder="{{ $ctrl.nullValueLabel || $ctrl.placeholder }}"
                  invalidate-on-ghosts="true"
                  default-opened="$ctrl.defaultOpened"
                  dropdown-position="{{ $ctrl.dropdownPosition }}"
                  ng-disabled="$ctrl.disabled"
                  clearable="$ctrl.clearable"
                  ghost-items-tooltip="LLM not available. It has been disabled, retired, or is not supported for this feature. Choose another LLM or contact your administrator."
                  custom-ng-select-class="ng-dropdown-panel--500 
                                          ng-select--custom-dropdown-height 
                                          ng-select--custom-x-overflow"></basic-select>
    <copy-to-clipboard-icon ng-if="!$ctrl.hideCopyButton" icon="'copy-step'" icon-class="($ctrl.selectedLlmId || $ctrl.nullValueId) ? 'btn btn--text btn--icon btn--secondary' : 'btn btn--text btn--icon btn--secondary disabled'" disabled="!$ctrl.selectedLlmId && !$ctrl.nullValueId" value="$ctrl.selectedLlmId || $ctrl.nullValueId" name="'LLM ID'"></copy-to-clipboard-icon>
    <div class="help-inline padleft0" ng-if="!$ctrl.hideGoToLlmButton && $ctrl.selectedLLMHref">
        <external-link href="{{$ctrl.selectedLLMHref}}" target="_blank">Go to {{ ($ctrl.selectedLlmId || $ctrl.nullValueId || "").startsWith("agent:") ? "agent" : "LLM" }}</external-link>
    </div>
    <div class="mbot0 mtop8 alert alert-warning" ng-if="!$ctrl.isAvailable && !$ctrl.disableAlertWarning">
        LLM not available. It has been disabled, retired, or is not supported for this feature. Choose another LLM or contact your administrator. <strong>{{$ctrl.selectedLlmId}}</strong>
    </div>`,
  controller: function ($scope, $filter, StateUtils, StringUtils) {
    const $ctrl = this;

    $scope.$watch(
      function () {
        return $ctrl.selectedLlmId;
      },
      function (newVal, oldVal) {
        $ctrl.updateAvailability();
        $ctrl.updateLLMSourceLink();
        if (newVal !== oldVal && $ctrl.onChange) {
            $ctrl.onChange();
        }
        if (newVal !== oldVal && $ctrl.onChangeOutput) {
            $ctrl.onChangeOutput();
        }
      }
    );

    $ctrl.$onChanges = () => {
        if ((typeof $ctrl.placeholder === 'undefined' || $ctrl.placeholder === null) && (typeof $ctrl.nullValueLabel === 'undefined' || $ctrl.nullValueLabel === null)) {
            $ctrl.placeholder = "Nothing selected";
        }

        $ctrl.updateAvailability();
        $ctrl.updateLLMSourceLink();
    };

    $ctrl.isAvailable = true;
    $ctrl.updateAvailability = () => {
        if ($ctrl.availableLlms && $ctrl.selectedLlmId) {
            $ctrl.isAvailable = !!$ctrl.availableLlms.find((l) => l.id === $ctrl.selectedLlmId);
        }
    };

    $ctrl.selectedLLMHref = undefined;
    $ctrl.updateLLMSourceLink = () => {
        $ctrl.selectedLLMHref = undefined;
        const targetId = $ctrl.selectedLlmId || $ctrl.nullValueId;
        if ($ctrl.availableLlms && targetId) {
            const selectedLLM = $ctrl.availableLlms.find((l) => l.id === targetId);
            if (selectedLLM && (["SAVED_MODEL_AGENT", "RETRIEVAL_AUGMENTED"].includes(selectedLLM.type) || selectedLLM.type.startsWith("SAVED_MODEL_FINETUNED"))) {
                $ctrl.selectedLLMHref = StateUtils.href.savedModel(selectedLLM.savedModelSmartId, undefined, {}, { redirectToActiveVersion: true });
            }
        }
    };

    $ctrl.groupBy = (modelOption) => {
        return $ctrl.niceModelType(modelOption) + ' - ' + modelOption.connection;
    };

    $ctrl.niceModelType = (modelOption) => {
        return $filter('niceLLMType')(modelOption.type);
    };

    $ctrl.modelConnectionIcon = (modelOption) => {
        return 'dku-icon-plug-disconnected-12';
    }

    $ctrl.searchFunction = (term, llm) => {
        const normalized = StringUtils.normalizeTextForSearch(term);
        return StringUtils.normalizeTextForSearch(llm.friendlyNameShort).includes(normalized)
            || StringUtils.normalizeTextForSearch(llm.id).includes(normalized)
            || StringUtils.normalizeTextForSearch(llm.connection).includes(normalized)
            || StringUtils.normalizeTextForSearch(llm.connectionDescription).includes(normalized);
    }
  },
});

app.component("perCharsSplittingSettings", {
    bindings: {
        chunkSizeCharacters: "=",
        chunkOverlapCharacters: "=",
        embeddingLlm: '<',
        documentSplittingMode: '<',
        defaultChunkSize: '<',
        defaultOverlapSize: '<',

    },
    templateUrl: "templates/recipes/nlp/per-chars-splitting-settings.component.html",
    controller: function ($scope, EmbeddingUtils) {
      const $ctrl = this;
  
        $ctrl.shouldWarnAboutChunkSize = () => {
            return EmbeddingUtils.shouldWarnAboutChunkSize($ctrl.embeddingLlm, $ctrl.documentSplittingMode, $ctrl.chunkSizeCharacters);
        };

        $ctrl.getModelMaxSizeInChars = () => {
            return EmbeddingUtils.tokensToCharsOrDefault($ctrl.embeddingLlm);
        };

        $ctrl.resetToDefaultSplittingSettings = function(){ 
            $ctrl.chunkSizeCharacters = $ctrl.defaultChunkSize;
            $ctrl.chunkOverlapCharacters = $ctrl.defaultOverlapSize;
        }
        
    },
  });

app.service('TraceExplorerService', function ($q, $rootScope, DataikuAPI, Dialogs, PluginsService, DataikuCloudService, $window, ActivityIndicator, SemanticVersionService) {
    const traceExplorerPluginId = 'traces-explorer';
    const requiredPluginVersion = '1.0.4';

    const buildTraceExplorerUrl = (traceExplorerWebApp, trace) => {
        return DataikuAPI.webapps
            .getPublicInfo(traceExplorerWebApp.projectKey, traceExplorerWebApp.webAppId)
            .then((resp) => {
                const publicInfo = resp.data;
                
                try {
                    localStorage.setItem('ls.llm.traceExplorer.trace', JSON.stringify(trace));
                } catch (e) {
                    const message = 'Due to the trace size, Trace Explorer cannot be launched automatically. Please open it manually to copy and paste the trace data.';
                    Dialogs.error($rootScope, 'Unable to auto-open Trace Explorer', message);
                    return $q.reject('LOCAL_STORAGE_ERROR');
                }

                let baseUrl = '';
                if ($rootScope.appConfig.dssExternalURL) {
                    baseUrl += $rootScope.appConfig.dssExternalURL;
                }
                baseUrl += '/webapps/';

                const vanityOrId = publicInfo.securityInfo.vanityURL || `${publicInfo.webapp.projectKey}/${publicInfo.webapp.id}`;
                return `${baseUrl}${vanityOrId}/`;
            })
            .catch((e) => {
                if (e !== 'LOCAL_STORAGE_ERROR') {
                    const message = `The "Trace Explorer" Web App instance ${traceExplorerWebApp.webAppId} in the project ${traceExplorerWebApp.projectKey} does not exist or you don't have read access to the project. Please contact your administrator.`;
                    Dialogs.error($rootScope, 'Permission required', message);
                }
                return $q.reject();
            });
    };

    const handlePluginInstalled = (trace) => {
        return PluginsService.getPluginVersion(traceExplorerPluginId).then((pluginVersion) => {
            if (SemanticVersionService.compareVersions(pluginVersion, requiredPluginVersion) < 0) {
                const message = `The currently installed "Trace Explorer" plugin (version ${pluginVersion}) does not support this feature. To proceed, the plugin must be updated to version ${requiredPluginVersion} or newer. Please contact your administrator.`;
                Dialogs.error($rootScope, 'Plugin update required', message);
                return $q.reject();
            }
            const traceExplorerDefaultWebApp = $rootScope.appConfig?.llmSettings?.traceExplorerDefaultWebApp;

            if (!traceExplorerDefaultWebApp?.projectKey || !traceExplorerDefaultWebApp?.webAppId) {
                const message = '"Trace Explorer" plugin is installed. To finalize the setup, <doclink page="/agents/tracing#accessing-via-explore-trace-shortcut" title="read the documentation"/> or contact your administrator.';
                Dialogs.errorUnsafeHTML($rootScope, 'Configuration required', message);
                return $q.reject();
            }

            return buildTraceExplorerUrl(traceExplorerDefaultWebApp, trace).then((traceExplorerUrl) => {
                $window.open(`${traceExplorerUrl}?readTraceFromLS=true`, '_blank');
            });
        });
    };

    const handlePluginNotInstalled = () => {
        let cloudInfo;

        return DataikuCloudService.getCloudInfo()
            .then((info) => {
                cloudInfo = info;
                let installMessage = 'Enhance your experience by installing the "Trace Explorer" plugin. You can ';
                if (cloudInfo.isDataikuCloud) {
                    installMessage += cloudInfo.isSpaceAdmin ? 'install it on the Launchpad.' : 'ask your administrator to install it.';
                } else {
                    installMessage += `${$rootScope.appConfig.admin ? 'install' : 'request'} it from the store.`;
                }
                return Dialogs.confirm($rootScope, 'Unlock Trace Exploration', installMessage, { positive: true, btnConfirm: 'OK' });
            })
            .then(() => {
                return PluginsService.getPluginStoreLink(cloudInfo, traceExplorerPluginId);
            })
            .then((pluginStoreUrl) => {
                $window.open(pluginStoreUrl, '_blank');
            });
    };

    this.openTraceExplorer = (trace) => {
        PluginsService.checkInstalledStatus('traces-explorer')
            .then((isPluginInstalled) => {
                if (isPluginInstalled) {
                    return handlePluginInstalled(trace);
                } else {
                    return handlePluginNotInstalled();
                }
            })
            .catch((error) => {
                if (error && error.message) {
                    ActivityIndicator.error(error.message, 8000);
                }
            });
    };
});

})();
