(function() {
    'use strict';
    const app = angular.module('dataiku.recipes');

    app.controller("_DocExtractionRecipeControllerBase", function($scope, $state, $rootScope, $q, $controller, $stateParams, DataikuAPI, RequestCenterService, EmbeddingUtils,
                                                          DOCUMENT_SPLITTING_METHOD_MAP, UPDATE_METHOD_MAP) {
        $controller("RAGEmbeddingRecipeEditionController", { $scope: $scope, VECTOR_STORE_UPDATE_METHOD_MAP: UPDATE_METHOD_MAP });
        $controller("_RecipeWithEngineBehavior", { $scope: $scope }); //Controller inheritance // for step status update

        DataikuAPI.flow.recipes.nlp.getDefaultEmbedDocumentRecipeSettings($stateParams.projectKey).success(function(data) {
            if (data?.defaultVLMId) {
                $scope.adminDefaultVLMId = data.defaultVLMId; // admin has defined a default VLM to use on the instance
                $scope.setDefaultVLM();
            }
        }).error(setErrorInScope.bind($scope));

        DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "IMAGE_INPUT").success(function(data) {
            $scope.availableVLMs = data.identifiers;
            $scope.setDefaultVLM();
        }).error(setErrorInScope.bind($scope));

        $scope.setDefaultVLM = function() {
            if (!$scope.recipe.params.defaultVlmId) {
                if ($scope.adminDefaultVLMId && $scope.availableVLMs && $scope.availableVLMs.some(x => x.id === $scope.adminDefaultVLMId)) {
                    $scope.recipe.params.defaultVlmId = $scope.adminDefaultVLMId;
                } else if ($scope.availableVLMs && $scope.availableVLMs.length > 0) {
                    $scope.recipe.params.defaultVlmId = $scope.availableVLMs[0].id;
                }
            }
        };


        $scope.EMBED_DOCS_EXTRACTION_MODES = [ // keep in sync with EmbedDocumentsRecipeParams.java - ExtractionMode enum
            {
                id: 'MANAGED_TEXT_ONLY',
                title: 'Text-first Extraction',
                base_description: "Extracts text from documents and uses headers to divide the content into meaningful extraction units. Supported file formats: {0}."
            },
            {
                id: 'MANAGED_VISUAL_ONLY',
                title: 'Visual-first Extraction',
                base_description: "Uses a Vision Language Model (VLM) to extract information from documents that may include text and visual elements like charts, graphics, and tables. Supported file formats: PDF, DOC, DOCX, PPT, PPTX, ODP, ODT, JPG, JPEG, PNG, TXT, MD."
            },
            {
                id: 'CUSTOM_RULES',
                title: 'Custom rules',
                base_description: "Create your own rules for how text is extracted and choose which documents to apply them to. This gives you full control over the extraction process."
            },
        ];

        $scope.EMBED_DOCS_IMAGE_HANDLING_MODES = { // keep in sync with EmbedDocumentsRecipeParams.java - ExtractionMode enum
            "IGNORE" :
            {
                id: "IGNORE",
                title: "Ignore",
                formats: ["PDF", "PPTX", "DOCX", "HTML", "TXT", "MD"]
            },
            "OCR":
            {
                id: "OCR",
                title: "OCR",
                formats: ["PDF", "PPTX", "DOCX", "HTML", "JPG", "PNG", "TXT", "MD"]
            },
            "VLM_ANNOTATE":
            {
                id: 'VLM_ANNOTATE',
                title: 'Describe images with VLM',
                formats: ["PDF", "PPTX", "DOCX", "HTML", "TXT", "MD"]
            },
        };

        $scope.EMBED_DOCS_ENGINE_MAP = {
            'STRUCTURED': {title: 'Structured text extraction', description: "When possible, the engine uses document headers to divide the content into extraction units. Image handling can be used to leverage images in documents and scanned PDFs." +
                    " Supported file formats are PDF, DOCX, PPTX, HTML, TXT, MD (without OCR), JPG, JPEG, and PNG (with OCR enabled)."},
            'RAW': {title: 'Raw text extraction', description: "Fast but doesn't extract the structure of the document, only the raw text. OCR can be used on images with text." +
                    " Supported file formats are PDF, DOCX, PPTX, HTML, TXT, MD (without OCR), JPG, JPEG, and PNG (with OCR enabled)."},
            'VLM': {title: 'VLM extraction', description: "Uses a Vision Language Model to extract information from documents that may include text and visual elements like charts, graphics, and tables." +
                    " Supported file formats are PDF, DOC, DOCX, ODT, PPT, PPTX, ODP, JPG, JPEG and PNG."},
            'DONOTEXTRACT': {title: 'Ignore (Do not extract)', description: "Ignore matching files."},
        };

        DataikuAPI.flow.recipes.nlp.getDefaultDocExtractionPrompts($stateParams.projectKey, $scope.recipe.name).success(function(data) {
            $scope.promptsTemplates = data;
        }).error(setErrorInScope.bind($scope));

        $scope.getPrompt = function(charsLimit, promptVersion) {
            var promptTemplate;
            if (!$scope.promptsTemplates){
                return ""; // if not yet retrieved
            }
            switch (promptVersion) {
            case "DEFAULT_VLM_EXTRACTION":
                promptTemplate = $scope.promptsTemplates.defaultVlmExtractionTemplate;
                break;
            case "SUMMARY_PROMPT":
                promptTemplate = $scope.promptsTemplates.vlmSummaryExtractionTemplate;
                break;
            case "FULL_CONTENT_PROMPT":
                promptTemplate = $scope.promptsTemplates.vlmFullExtractionTemplate;
                break;
            case "ANNOTATION_PROMPT":
                promptTemplate = $scope.promptsTemplates.vlmAnnotationTemplate;
                break;
            default:
                return ""; // Unsupported
            }
            return promptTemplate.replace($scope.promptsTemplates.limitPlaceholder, charsLimit);
        };

        $scope.getPromptFromRule = function (rule, promptVersion) {
            if (promptVersion) {
                var chunkSize = $scope.defaultChunkSize;
                if (rule?.splittingSettings?.chunkSizeCharacters) { // splitting settings can be null for extract content recipe
                    chunkSize = rule.splittingSettings.chunkSizeCharacters;
                }
                return $scope.getPrompt(chunkSize, promptVersion);
            }
        };

        $scope.overrideVLMAnnotationCustomPromptWith = function (rule) {
            rule.structuredSettings.customVLMPrompt = $scope.getPromptFromRule(rule, "ANNOTATION_PROMPT");
        };

        $scope.overrideVLMExtractionCustomPromptWith = function (rule, promptVersion) {
            rule.vlmSettings.customPrompt = $scope.getPromptFromRule(rule, promptVersion);
        };

        $scope.getActiveVLMfromLLMid = function(vlmId) {
            return $scope.availableVLMs.find(llm => llm.id === vlmId);

        };

        $scope.recipe.params.rules = $scope.recipe.params.rules || [];  // shouldn't be needed as default rules are retrieved from the backend

        $scope.makeDefaultVLMSettings = function(rule) {
            let chunkSize = $scope.defaultChunkSize
            // If we set the chunk size for a structure extraction rule, and then transform
            // it into a VLM rule, we want to keep the chunk size reflected in the summary
            // prompt.
            if (rule?.splittingSettings?.chunkSizeCharacters) { // splitting settings can be null for extract content recipe
                chunkSize = rule.splittingSettings.chunkSizeCharacters;
            }
            return {
                "llmId" : $scope.recipe.params.defaultVlmId === "DSS_NO_SELECTION"? undefined: $scope.recipe.params.defaultVlmId,
                "splitUnit" : "PAGE",
                "customNbPages": 1,
                "customPagesOverlap": 0,
                "useCustomPrompt" : false,
                "customPrompt": $scope.getPrompt(chunkSize, 'DEFAULT_VLM_EXTRACTION'),
            }
        };

        $scope.makeDefaultStructuredSettings = function() {
            let chunkSize = $scope.defaultChunkSize;
            return {
                "splitUnit": "SECTION",
                "maxSectionDepth": 6,
                "imageHandlingMode": "IGNORE",
                "ocrEngine": "AUTO",
                "ocrLanguages": "en",
                "useCustomVLMPrompt" : false,
                "customVLMPrompt": $scope.getPrompt(chunkSize, "ANNOTATION_PROMPT"),
                "enableImageClassificationFiltering": true,
            };
        };

        $scope.makeDefaultRawSettings = function() {
            return {
                "useOCR": false,
                "ocrEngine": "AUTO",
                "ocrLanguages": "en",
            };
        }

        $scope.addNewRule = function() {
            $scope.recipe.params.rules.push({
                "filter": {
                    "enabled": true,
                    "distinct": false,
                    "uiData": {
                        "mode": "||",
                        "conditions": [
                            {
                                "input": "file extension",
                                "operator": "== [string]i" // equals case insensitive
                            }
                        ]
                    }
                },

                "actionToPerform": "STRUCTURED",
                "rawSettings": $scope.makeDefaultRawSettings(),
                "structuredSettings": $scope.makeDefaultStructuredSettings(),
                "splittingSettings": $scope.makeDefaultSplittingSettings(),
                "storeInMultimodalColumn": "FULL_CONTENT"
            });
            $scope.uiState.showAdvancedSettings.push(false);
        }

        $scope.onRuleActionChange = function(idx) {
            let ruleToBeUpdated = idx == -1 ? $scope.recipe.params.allOtherRule : $scope.recipe.params.rules[idx];

            switch (ruleToBeUpdated.actionToPerform) {
                case 'VLM':
                    ruleToBeUpdated.vlmSettings = ruleToBeUpdated.vlmSettings || $scope.makeDefaultVLMSettings(ruleToBeUpdated);
                    ruleToBeUpdated.storeInMultimodalColumn = "IMAGES";
                    break;
                case 'STRUCTURED':
                    ruleToBeUpdated.structuredSettings = ruleToBeUpdated.structuredSettings || $scope.makeDefaultStructuredSettings();
                    ruleToBeUpdated.storeInMultimodalColumn = $scope.shouldStoreDetectedImages(ruleToBeUpdated) ? "IMAGES" : "FULL_CONTENT";
                    break;
                case 'RAW':
                    ruleToBeUpdated.rawSettings = ruleToBeUpdated.rawSettings || $scope.makeDefaultRawSettings();
                    ruleToBeUpdated.storeInMultimodalColumn = "FULL_CONTENT";
                    break;
            }
        }

        $scope.clearAll = function() { // remove all rules except 'all other files' rule
            // eslint-disable-next-line for-direction
            for (var i = $scope.recipe.params.rules.length - 1; i >= 0; i--) {
                $scope.removeRule(i);
            }
        };

        $scope.removeRule = function(idx) {
            // only 'classic' (not 'allOtherFiles') rules can be deleted
            $scope.recipe.params.rules.splice(idx, 1);
            $scope.uiState.showAdvancedSettings.splice(idx, 1);
        };

        $scope.uiState = {
            currentStep: 'strategy',
            showAdvancedSettings: [],
            showAdvancedSettingsAllOtherFiles: false,
            selectAllMetadataColumns: true,
            requestSent: false,
        };

        void function() { // to hide all advanced settings by default
            for (var i = 0; i < $scope.recipe.params.rules.length; i++) {
                $scope.uiState.showAdvancedSettings.push(false);
            }
        }();

        /******  recipe related *****/
        $scope.hooks.getPayloadData = function() {
            return angular.toJson($scope.desc);
        };

        $scope.hooks.updateRecipeStatus = function() {
            let deferred = $q.defer();
            let payload = $scope.hooks.getPayloadData();
            $scope.updateRecipeStatusBase(false, payload).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) return deferred.reject();
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };
        $scope.hasAllRequiredOutputs = () => {
            if (!$scope.recipeStatus || !$scope.recipeStatus.output || $scope.recipeStatus.output.fatal) {
                return false;
            }
            return true;
        };

        $scope.hasMissingOutputFolder = () => {

            if ($scope.recipe.outputs.images && !! $scope.recipe.outputs.images.items.length){
                return false; // folder is set
            }

            return !!$scope.recipeStatus && !!$scope.recipeStatus.output && $scope.recipeStatus.output.messages?.some(
                msg => msg.code === "ERR_RECIPE_DOC_EXTRACTION_MISSING_FOLDER"
            );
        };

        $scope.showWarningOnDocumentExtractionCodeEnv = () => {
            if ($scope.recipeStatus?.strategy) {
                return $scope.recipeStatus.strategy.messages?.some(
                    msg => msg.code === "WARN_RECIPE_CODE_ENV_NOT_AVAILABLE"
                );
            }
            return false;
        };

        $scope.internalCodeEnvsHRef = function() {
            if ($scope.appConfig.isAutomation) {
                return $state.href("admin.codeenvs-automation.internal");
            } else {
                return $state.href("admin.codeenvs-design.internal");
            }
        };


        $scope.sendDocExtractionInternalCodeEnvRequest = function() {
            DataikuAPI.requests.createInternalCodeEnvRequest("INSTALL_CODE_ENV", "INTERNAL_document_extraction", "Please install document extraction internal code env", 'MANUAL').success(function(data) {
                RequestCenterService.WT1Events.onRequestSent("CODE_ENV", null, "INTERNAL_document_extraction", "Please install document extraction internal code env", data.id);
            }).error(setErrorInScope.bind($scope));
            $scope.uiState.requestSent = true;
        };

        $scope.canChangeEngine = function() {
            return false;  // no need, there is a single DSS engine for now
        };

        $scope.updateMethodDisabledReason = function(method) {
            return ""; // no use case where we need to disable update methods for now
        };

        $scope.baseRuleSchema = {
            "columns": [
                {
                    "name": "file extension",
                    "type": "string",
                    "$$groupKey": ""
                },
                {
                    "name": "file name", // Used in EmbedDocumentsRule.java, keep in sync to keep the form clean at creation time.
                    "type": "string",
                    "$$groupKey": ""
                },
                {
                    "name": "file path",
                    "type": "string",
                    "$$groupKey": ""
                }
            ],
            "userModified": false
        };

        $scope.ruleSchema = undefined;

        $scope.getRulesSchema = function() {
            return $scope.ruleSchema;
        };

        $scope.createCustomRulesFrom = function(extractionMode) {
            if ($scope.recipe.params) {
                const llmId = $scope.retrievableKnowledge ? $scope.retrievableKnowledge.embeddingLLMId: null;
                    DataikuAPI.flow.recipes.nlp.getDefaultDocExtractionRules(extractionMode.id, llmId, $scope.recipe.params.defaultVlmId, $stateParams.projectKey, $scope.recipe.params.defaultImageHandlingMode, $scope.recipe.name).success(function(data) {
                    if (data) {
                        $scope.recipe.params.rules = data.rules;
                        Object.assign($scope.recipe.params.allOtherRule, data.allOtherRule); // only copy properties without changing reference
                        $scope.recipe.params.extractionMode = 'CUSTOM_RULES';
                    }
                }).error(setErrorInScope.bind($scope));
            }
        }

        $scope.overrideCustomPromptWith = function (rule, promptVersion) {
            if (promptVersion && $scope.VLM_EXTRACTION_PROMPTS[promptVersion]) {
                let chunkSize = $scope.defaultChunkSize;
                if (rule?.splittingSettings?.chunkSizeCharacters) { // splitting settings can be null for extract content recipe
                    chunkSize = rule.splittingSettings.chunkSizeCharacters;
                }
                rule.vlmSettings.customPrompt = $scope.getPrompt(chunkSize, promptVersion);
            }
        };

        $scope.overrideVLMAnnotationCustomPromptWith = function (rule) {
            rule.structuredSettings.customVLMPrompt = $scope.getVLMAnnotationPrompt(rule.splittingSettings.chunkSizeCharacters);
        };

        $scope.getAppropriateImagePath = function () {
            if ($scope.recipe.params.extractionMode === 'MANAGED_TEXT_ONLY' && $scope.recipe.params.defaultImageHandlingMode !== 'IGNORE') {
                return "/static/dataiku/images/recipes/"+ $scope.recipe.type+ "/"+ $scope.recipe.params.extractionMode + "_WITH_IMAGE_HANDLING.svg"
            }
            return "/static/dataiku/images/recipes/"+ $scope.recipe.type+ "/"+ $scope.recipe.params.extractionMode + ".svg"
        };


        $scope.shouldStoreDetectedImages = function(rule){
            if(rule.actionToPerform === 'STRUCTURED' ){
                if(($scope.recipe.type === 'embed_documents' && rule.structuredSettings.imageHandlingMode === 'VLM_ANNOTATE' || $scope.recipe.type === 'extract_content' && $scope.desc.storeImages)){
                    return true;
                }
            }
            return false;
        };

        $scope.onImageHandlingChange = function(rule){
            if(['STRUCTURED', 'RAW'].contains(rule.actionToPerform)) {
                if ($scope.shouldStoreDetectedImages(rule)) {
                    rule.storeInMultimodalColumn = 'IMAGES';
                } else {
                    rule.storeInMultimodalColumn = 'FULL_CONTENT';
                }
            }
        };

    });

    app.component("ocrEngine", {
        bindings: {
            engine: "=",
            languages: "=",
        },
        templateUrl: "templates/recipes/doc-extraction-common/ocr-engine.component.html",
    });

})();