(function(){
'use strict';

const app = angular.module('dataiku.semanticmodels', ['platypus.utils']);

app.controller("SemanticModelsListController", function($scope, $controller, $stateParams, TopNav, DataikuAPI, CreateModalFromTemplate, StateUtils, PluginsService) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    TopNav.setNoItem();
    TopNav.setLocation(TopNav.TOP_SEMANTIC_MODELS, TopNav.LEFT_SEMANTIC_MODELS, TopNav.TABS_NONE, null);

    $scope.sortBy = [
        { value: 'name', label: 'Name' },
        { value: '-lastModifiedOn', label: 'Last modified'},
    ];

    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            }
        },
        filterParams: {
            userQueryTargets: ["name", "tags"],
            propertyRules: {tag: 'tags'},
        },
        orderQuery: "-lastModifiedOn",
        orderReversed: false
    }, $scope.selection || {});

    $scope.sortCookieKey = 'semantic-models';
    $scope.maxItems = 20;

    $scope.list = function() {
        DataikuAPI.semanticModels.listHeads($stateParams.projectKey).success(function(data) {
            $scope.listItems = data;
            $scope.restoreOriginalSelection();
        }).error(setErrorInScope.bind($scope));
    };

    $scope.newSemanticModel = function() {
        CreateModalFromTemplate("/templates/semantic-models/semantic-model-creation-modal.html", $scope, null, function(newScope) {
            newScope.uiState = { name: '' };
            newScope.creating = false;
            newScope.create = function() {
                newScope.creating = true;
                DataikuAPI.semanticModels.create($stateParams.projectKey, newScope.uiState.name).success(function(data) {
                    newScope.creating = false;
                    StateUtils.go.semanticModel(data.id, $stateParams.projectKey);
                    newScope.dismiss();
                }).error(setErrorInScope.bind(newScope));
            };
        });
    };

    const semanticModelsLabPluginId = 'semantic-models-lab';

    $scope.isPluginInstalled = function (pluginId) {
        return PluginsService.checkInstalledStatus(pluginId).catch(setErrorInScope.bind($scope));
    };

    $scope.isPluginInstalled(semanticModelsLabPluginId).then((result) => {
                $scope.semanticModelsLabPluginInstalled = result;
    });
    $scope.semanticModelsLabPluginUrl = StateUtils.href.pluginStore() + semanticModelsLabPluginId;

    $scope.list();
});

app.controller("SemanticModelController", function($scope, $state, $stateParams, DataikuAPI, TopNav, ActiveProjectKey, ActivityIndicator, VirtualWebApp, WebAppsService) {
    TopNav.setLocation(TopNav.TOP_SEMANTIC_MODELS, TopNav.LEFT_SEMANTIC_MODELS, TopNav.TABS_NONE, null);
    $scope.uiState = { dirty: false };

    function load() {
        DataikuAPI.semanticModels.get($stateParams.projectKey, $stateParams.semanticModelId).success(function(data) {
            $scope.uiState.semanticModel = data;
            $scope.uiState.dirty = false;
            TopNav.setItem(TopNav.ITEM_SEMANTIC_MODEL, $stateParams.semanticModelId, {
                name: data.name,
            });

            WebAppsService.getSkins('SEMANTIC_MODEL', null, {});

            $scope.enableSkin();

        }).error(setErrorInScope.bind($scope));
    }

    $scope.markDirty = function() {
        $scope.uiState.dirty = true;
    };

    $scope.isOnEditorPage = function() {
        return $state && $state.current && $state.current.name === "projects.project.semanticmodels.semanticmodel.edit";
    }

    $scope.enableSkin = function() {
        VirtualWebApp.changeSkin($scope, 'SEMANTIC_MODEL', {
            "ownerPluginId": "semantic-models-lab",
            "webappType": "webapp_semantic-models-lab_semantic-model-editor"
            }, $scope.uiState, 'skin-holder', $scope.uiState.semanticModel.id, null, true);
    }

        $scope.getBackendLogURL = function(projectKey, webAppId) {
            return DataikuAPI.webapps.getBackendLogURL(projectKey, webAppId);
        };


    // For Raw Editor
    $scope.saveSemanticModel = function() {
        return DataikuAPI.semanticModels.save($scope.uiState.semanticModel)
            .success(function() {
                $scope.uiState.dirty = false;
                ActivityIndicator.success("Saved");
            }).error(setErrorInScope.bind($scope));
    };

    $scope.projectKey = ActiveProjectKey.get();
    load();
});

app.controller("SemanticModelEditController", function($controller, $scope) {

    $scope.enableSkin();
});

app.controller("SemanticModelPageRightColumnActions", function($controller, $scope, $rootScope, DataikuAPI, $stateParams, ActiveProjectKey, TopNav) {
    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.selection = {};

    DataikuAPI.semanticModels.get(ActiveProjectKey.get(), $stateParams.semanticModelId).success((data) => {
        $scope.selection = {
            selectedObject: data,
            confirmedItem: data,
        };
        $scope.selection.selectedObject.nodeType = "SEMANTIC_MODEL";

        TopNav.setItem(TopNav.ITEM_SEMANTIC_MODEL, $stateParams.semanticModelId, {
            name: data.name,
        });
    }).error(setErrorInScope.bind($scope));

    $scope.renameObjectAndSave = function(newName) {
        $scope.selection.selectedObject.name = newName;
        return DataikuAPI.semanticModels.save($scope.selection.selectedObject);
    };

    function updateUserInterests() {
        DataikuAPI.interests.getForObject($rootScope.appConfig.login, "SEMANTIC_MODEL", ActiveProjectKey.get(), $scope.selection.selectedObject.id).success(function(data) {
            $scope.selection.selectedObject.interest = data;
            $scope.semanticModelData.interest = data;
        }).error(setErrorInScope.bind($scope));
    }

    const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
    $scope.$on("$destroy", interestsListener);
});

app.directive('semanticModelRightColumnSummary', function($controller, $state, $rootScope, DataikuAPI, CreateModalFromTemplate, QuickView,
    ActiveProjectKey, ActivityIndicator, TopNav, WT1, GlobalProjectActions) {

    return {
        templateUrl :'/templates/semantic-models/right-column-summary.html',

        link : function(scope) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});
            $controller('_TaggableObjectPageRightColumnActions', {$scope: scope});

            scope.refreshData = function() {
                const projectKey = scope.selection.selectedObject.projectKey;
                const name = scope.selection.selectedObject.name;

                DataikuAPI.semanticModels.getFullInfo(ActiveProjectKey.get(), scope.selection.selectedObject.id).then(function({data}){
                    if (!scope.selection.selectedObject || scope.selection.selectedObject.projectKey != projectKey || scope.selection.selectedObject.name != name) {
                        return;
                    }
                    scope.semanticModelData = data;
                    scope.semanticModel = data.semanticModel;
                    scope.selection.selectedObject.interest = data.interest;
                }).catch(setErrorInScope.bind(scope));
            };

            function save() {
                return DataikuAPI.semanticModels.save(scope.semanticModel, {summaryOnly: true})
                .success(function() {
                    ActivityIndicator.success("Saved");
                }).error(setErrorInScope.bind(scope));
            }

            scope.$on("objectSummaryEdited", save);

            scope.$watch("selection.selectedObject", function() {
                if (scope.selection.selectedObject != scope.selection.confirmedItem) {
                    scope.semanticModel = null;
                    scope.objectTimeline = null;
                }
            });

            scope.$watch("selection.confirmedItem", function(nv) {
                if (!nv) return;
                if (!nv.projectKey) {
                    nv.projectKey = ActiveProjectKey.get();
                }
                scope.refreshData();
            });

            scope.deleteSemanticModel = function() {
                GlobalProjectActions.deleteTaggableObject(scope, 'SEMANTIC_MODEL', scope.selection.selectedObject.id, scope.selection.selectedObject.name);
            };
        }
    };
});

})();
