
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('GroupedColumnsUtils', GroupedColumnsUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/column.js
     */
    function GroupedColumnsUtils(SVGUtils, ChartMeasure, ChartAxesUtils, ChartsStaticData, CHART_VARIANTS, UnaggregatedMeasureComputeModes, CHART_LABELS) {
        const computeUAValueData = function(genericData, chartData, measure, measureIndex, hasLogScale) {
            const getBarBase = (total) => {
                if (total === 0 && hasLogScale) {
                    return 1;
                }
                return total;
            };
            const result = {
                ...genericData,
                isUnaggregated: true,
                uaComputeMode: measure.uaComputeMode
            };
            const uaValues = chartData.aggr(measureIndex).get();
            if (measure.uaComputeMode === UnaggregatedMeasureComputeModes.STACK) {
                let positiveTotal = 0;
                let negativeTotal = 0;
                result.data = uaValues.map(function(uaValue, idx) {
                    let base = uaValue >= 0 ? positiveTotal : negativeTotal;
                    let top = base + uaValue;

                    if (uaValue >= 0) {
                        base = getBarBase(base);
                        top = getBarBase(top);
                        positiveTotal = top;
                    } else {
                        negativeTotal = top;
                    }
                    return {
                        isUnaggregated: true,
                        isStacked: true,
                        ...genericData,
                        base,
                        top,
                        uaValueIdx: idx
                    };
                });
            } else {
                // overlay mode
                const positiveData = [];
                const negativeData = [];
                uaValues.forEach(function(uaValue, idx) {
                    const commonData = {
                        isUnaggregated: true,
                        base: hasLogScale ? 1 : 0,
                        top: uaValue,
                        ...genericData,
                        uaValueIdx: idx
                    };
                    if (uaValue >= 0) {
                        positiveData.push(commonData);
                    } else {
                        negativeData.push(commonData);
                    }
                });
                result.data = [
                    /*
                     * in Overlay mode, we display one bar per ua value, to handle tooltip and hover effect
                     * but we color only the biggest bar for positive values and for negative values, other bars are transparent by default
                     */
                    ...positiveData.map(function(data, idx) {
                        return {
                            ...data,
                            transparent: idx > 0
                        };
                    }),
                    ...negativeData.reverse().map(function(data, idx) {
                        return {
                            ...data,
                            transparent: idx > 0
                        };
                    })
                ];
            }

            return result;
        };

        const computeWaterfallTotals = function(chartData, animationLabels, facetLabels) {
            animationLabels.forEach(function(_, a) {
                chartData.fixAxis('animation', a);
                facetLabels.forEach(function(_, f) {
                    chartData.fixAxis('facet', f);
                    const isLastIteration = a === animationLabels.length - 1 && f === facetLabels.length - 1;
                    chartData.computeTotals(a, f, isLastIteration);
                });
            });
        };


        return {
            prepareData: function(chartDef, chartData, measureFilter, ignoreLabels = new Set()) {
                const xLabels = chartData.getAxisLabels('x');
                const colorLabels = chartData.getAxisLabels('color') || [null];
                const facetLabels = chartData.getAxisLabels('facet') || [null];
                const animationLabels = chartData.getAxisLabels('animation') || [null];
                const hasLogScale = chartDef.xAxisFormatting.isLogScale || ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, ChartsStaticData.LEFT_AXIS_ID);
                const getBarBase = (total) => {
                    if (total === 0 && hasLogScale) {
                        return 1;
                    }
                    return total;
                };

                // Waterfall chart, first pass to compute all totals
                if (chartDef.variant === CHART_VARIANTS.waterfall) {
                    computeWaterfallTotals(chartData, animationLabels, facetLabels);
                }

                const animationData = { frames: [], maxTotal: 0, minTotal: 0 };
                animationLabels.forEach(function(animationLabel, a) {
                    chartData.fixAxis('animation', a);

                    const frameData = { facets: [], maxTotal: 0, minTotal: 0 };
                    facetLabels.forEach(function(facetLabel, f) {
                        chartData.fixAxis('facet', f);

                        const facetData = { groups: [], maxTotal: 0, minTotal: 0 };
                        let runningTotal = 0; // only usefull for waterfall
                        xLabels.forEach(function(xLabel, x) {
                            chartData.fixAxis('x', x);
                            const columns = [];
                            if (xLabel && ignoreLabels.has(xLabel.label)) {
                                return;
                            }

                            let cIndex = 0;
                            let nonEmptyIdx = 0;
                            colorLabels.forEach(function(colorLabel, c) {
                                chartData.fixAxis('color', c);
                                if (colorLabel && ignoreLabels.has(colorLabel.label)) {
                                    return;
                                }
                                const isTotalBar = !!chartData.isTotalBar?.(c, x);
                                if (isTotalBar) {
                                    // for total bar, reset the total to start from the x axis
                                    runningTotal = 0;
                                }
                                chartDef.genericMeasures.forEach(function(measure, measureIndex) {
                                    if (measureFilter && !measureFilter(measure)) {
                                        return;
                                    }
                                    const measureData = {
                                        color: chartData.getColorOfValue(c, x),
                                        measure: measureIndex,
                                        x: x,
                                        tooltipOverrides: chartData.getTooltipOverrides(cIndex, x),
                                        colorScaleIndex: chartData.getColorOfValue(cIndex, x)
                                    };

                                    let subTextElementsData = [];
                                    if (chartData.isBinMeaningful(measure.function, null, measureIndex)) {
                                        if (ChartMeasure.isRealUnaggregatedMeasure(measure)) {
                                            // in that case the value is a double array instead of a single value
                                            Object.assign(measureData, computeUAValueData(measureData, chartData, measure, measureIndex, hasLogScale));
                                        } else {
                                            subTextElementsData = SVGUtils.getLabelSubTexts(chartDef, measureIndex, measure.valuesInChartDisplayOptions, false).map(function(subTextElementData) {
                                                return {
                                                    ...subTextElementData,
                                                    // add measure data at sub text level also, to retrieve it easily
                                                    ...measureData
                                                };
                                            });
                                            const value = chartData.aggr(measureIndex).get({ x: x, c: c });
                                            const newRunningTotal = runningTotal + value;
                                            measureData.data = [{ ...measureData, base: getBarBase(runningTotal), top: getBarBase(newRunningTotal) }];
                                            facetData.maxTotal = Math.max(facetData.maxTotal, newRunningTotal);
                                            facetData.minTotal = Math.min(facetData.minTotal, newRunningTotal);
                                            runningTotal = chartData.shouldRestartFrom0(c, x) ? 0 : newRunningTotal;
                                        }
                                        measureData.nonEmptyIdx = nonEmptyIdx;
                                        nonEmptyIdx += 1;
                                    } else {
                                        // we don't want to display a bar for non meaningful values
                                        measureData.data = [{ ...measureData, base: getBarBase(runningTotal), top: getBarBase(runningTotal) }];
                                    }

                                    columns.push({
                                        ...measureData,
                                        textsElements: subTextElementsData,
                                        valuesInChartDisplayOptions: measure.valuesInChartDisplayOptions
                                    });
                                });
                                cIndex++;
                            });
                            // compute non empty bins here, we use them for the "Hide empty bins" option
                            const nbNonEmptyBins = nonEmptyIdx;
                            columns.forEach(col => {
                                col.nbNonEmptyBins = nbNonEmptyBins;
                            });

                            facetData.groups.push({ x: x, columns, nbNonEmptyBins });

                        });
                        frameData.facets.push(facetData);
                        frameData.maxTotal = Math.max(frameData.maxTotal, facetData.maxTotal);
                        frameData.minTotal = Math.min(frameData.minTotal, facetData.minTotal);
                    });
                    animationData.frames.push(frameData);
                    animationData.maxTotal = Math.max(animationData.maxTotal, frameData.maxTotal);
                    animationData.minTotal = Math.min(animationData.minTotal, frameData.minTotal);
                });

                if (chartDef.variant === CHART_VARIANTS.waterfall) {
                    const customXLabels = chartData.getCustomAxisLabels('x');
                    if (customXLabels) {
                        // for waterfall with second breakdown, we have to reorganize bars, as they are displayed individually, not by group like for standard vertical bars
                        for (let index = 0; index < animationData.frames.length; index++) {
                            const frameData = animationData.frames[index];
                            const frameFacets = frameData.facets;
                            frameFacets.forEach(facetData => {
                                const facetGroups = facetData.groups;
                                const finalGroupColumns = new Array(customXLabels.length).fill(null);
                                facetGroups.forEach(groupData => {
                                    groupData.columns.forEach(col => {
                                        const position = customXLabels.findIndex(lbl =>
                                            lbl?.coordinates?.x === col.x && lbl?.coordinates?.color === col.tooltipOverrides.color);
                                        if (position >= 0) {
                                            finalGroupColumns[position] = { x: col.x, columns: [col], nbNonEmptyBins: col.nbNonEmptyBins };
                                        }
                                    });
                                });
                                facetData.groups = finalGroupColumns;
                            });
                        }
                    }
                }

                return animationData;
            }
        };
    }
})();
