/* jshint loopfunc: true*/
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/maps/scatter.js
    app.factory('ScatterMapChart', function(ChartFormatting, ChartPropertiesService, ChartUADimension, _MapCharts, ChartLabels, UaChartsCommon, ChartLegendUtils, ColorUtils, ChartColorUtils, BuiltinMapBackgrounds) {
        return function($container, chartDef, data, chartHandler) {

            const elt = _MapCharts.getOrCreateMapContainer($container);

            let layerGroup, colorScale;

            // Build color scale
            const hasUAColor = UaChartsCommon.hasUAColor(chartDef);
            if (hasUAColor) {
                colorScale = UaChartsCommon.makeColorScale(chartDef, data, chartHandler);
            } else {
                var resultingColor = UaChartsCommon.makeSingleColor(chartDef);
            }

            // Build legend
            if (hasUAColor && (ChartUADimension.isAlphanumLike(chartDef.uaColor[0]) || ChartUADimension.isDiscreteDate(chartDef.uaColor[0]))) {
                const baseFadeColor = BuiltinMapBackgrounds.getBackgrounds().find(b => b.id === chartDef.mapOptions.tilesLayer).fadeColor || '#333';
                const fadeColor = ColorUtils.toRgba(baseFadeColor, .5 * chartDef.colorOptions.transparency);

                const legend = {
                    type: 'COLOR_DISCRETE',
                    items: data.values.color.str.sortedMapping.map(function(_, v) {
                        return {
                            label: {
                                ...data.values.color.str.sortedMapping[v],
                                colorId: ChartColorUtils.getColorId(chartDef.genericMeasures, { data }, v)
                            },
                            color: colorScale(v),
                            focusFn: function() {
                                layerGroup.getLayers().forEach(function(layer) {
                                    const opts = layer.options;
                                    if (!opts.actualFillColor) {
                                        opts.actualFillColor = opts.fillColor;
                                    }

                                    if (opts.colorIdx !== v) {
                                        opts.fillColor = fadeColor;
                                    } else {
                                        opts.fillColor = opts.actualFillColor;
                                    }

                                    layer.setStyle(opts);
                                });
                            },
                            unfocusFn: function() {
                                layerGroup.getLayers().forEach(function(layer) {
                                    const opts = layer.options;
                                    opts.fillColor = opts.actualFillColor;
                                    layer.setStyle(opts);
                                });
                            },
                            focused: false
                        };
                    })
                };

                chartHandler.legendsWrapper.deleteLegends();
                chartHandler.legendsWrapper.pushLegend(legend);
            } else {
                if (colorScale) {
                    colorScale.type = 'MEASURE';
                }
                ChartLegendUtils.initLegend(chartDef, null, chartHandler.legendsWrapper, colorScale);
                if (colorScale) {
                    if (ChartUADimension.isDateRange(chartDef.uaColor[0])) {
                        chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForDate();
                    } else {
                        chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForLegend(chartDef.uaColor[0], colorScale.innerScale.domain());
                    }
                }
            }

            // Draw legend, then map
            ChartLegendUtils.drawLegend(chartDef, chartHandler, $container).then(function() {
                _MapCharts.adjustLegendPlacement(chartDef, $container);

                const map = _MapCharts.createMapIfNeeded(elt, chartHandler.chartSpecific, chartDef);
                if (elt.data('leaflet-data-layer')) {
                    map.removeLayer(elt.data('leaflet-data-layer'));
                }
                _MapCharts.repositionMap(map, elt, data);

                const hasUASize = UaChartsCommon.hasUASize(chartDef);
                if (hasUASize) {
                    var sizeScale = UaChartsCommon.makeSizeScale(chartDef.bubblesOptions.defaultRadius, chartDef.uaSize[0], data, 1);
                }
                const uaLFn = ChartLabels.getUaLabel;

                layerGroup = L.layerGroup();
                const colorCache = {};

                data.xAxis.num.data.forEach(function(x, i) {
                    const y = data.yAxis.num.data[i];
                    let radius = UaChartsCommon.makeSize(chartDef, data, i, sizeScale);
                    const color = UaChartsCommon.makeColor(chartDef, data, i, colorScale, resultingColor, colorCache);

                    if (radius < 0) {
                        radius = 0; // ensure radius is non-negative in case of data mismatch
                    }

                    const options = {
                        radius,
                        fillOpacity: 1
                    };

                    // Used for the legend to highlight everything with the same color label
                    if (ChartUADimension.isAlphanumLike(chartDef.uaColor[0]) || ChartUADimension.isDiscreteDate(chartDef.uaColor[0])) {
                        options.colorIdx = data.values.color.str.data[i];
                    }

                    if (chartDef.bubblesOptions.singleShape == 'EMPTY_CIRCLE') {
                        options.stroke = true;
                        options.fill = false;
                        options.color = color;
                    } else {
                        options.stroke = false;
                        options.fill = true;
                        options.fillColor = color;
                    }

                    // LatLng
                    const pointLayer = L.circleMarker([y, x], options);

                    if (ChartPropertiesService.showTooltip(chartDef.tooltipOptions, chartHandler.chartTileProperties)) {

                        let html = '';
                        html += 'Lon: <strong>' + ChartFormatting.getForIsolatedNumber()(x) + '</strong><br />';
                        html += 'Lat: <strong>' + ChartFormatting.getForIsolatedNumber()(y) + '</strong><br />';
                        if (hasUAColor) {
                            html += sanitize(uaLFn(chartDef.uaColor[0])) + ': <strong>' +
                                sanitize(UaChartsCommon.formattedColorVal(chartDef, data, i)) + '</strong><br />';
                        }
                        if (hasUASize && (!hasUAColor || (chartDef.uaSize[0].column !== chartDef.uaColor[0].column || chartDef.uaColor[0].dateMode !== 'RANGE'))) {
                            html += sanitize(uaLFn(chartDef.uaSize[0])) + ': <strong>' +
                                sanitize(UaChartsCommon.formattedSizeVal(chartDef, data, i)) + '</strong><br />';
                        }

                        if (chartDef.uaTooltip.length > 0) {
                            html += '<hr/>';
                        }

                        chartDef.uaTooltip.forEach(function(ua, j) {
                            html += sanitize(uaLFn(ua)) + ': <strong>' + sanitize(UaChartsCommon.formattedVal(data.values['tooltip_' + j], ua, i)) + '</strong><br/>';
                        });

                        //Classname is a locator for the integration tests - Leaflet API doesn't allow us to add an id
                        pointLayer.bindPopup(html, { className: 'qa-chart-tooltip' });
                    }

                    layerGroup.addLayer(pointLayer);
                });

                layerGroup.addTo(map);
                elt.data('leaflet-data-layer', layerGroup);
            }).finally(function() {
                /*
                 * Signal to the callee handler that the chart has been loaded.
                 * Dashboards use it to determine when all insights are completely loaded.
                 */
                if (typeof (chartHandler.loadedCallback) === 'function') {
                    chartHandler.loadedCallback();
                }
            });
        };
    });
})();
