(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/stats/boxplots.js
    app.factory('BoxplotsChart', function(ChartAxesUtils, D3ChartAxes, ChartsStaticData, ChartDataWrapperFactory, ChartColorUtils, ChartColorScales, ChartLegendUtils, $timeout, ChartContextualMenu, ChartFormatting, ChartStoreFactory, ChartTooltips, ChartLabels, ChartYAxisPosition, CHART_MODES, ChartDimension, SVGUtils, ChartDrilldown, ChartHierarchyDimension, ChartPropertiesService) {
        return function(element, chartDef, data, chartHandler, axesDef) {
            function prepareData(chartDef, chartData, measureFilter) {
                const xLabels = chartData.getAxisLabels('x') || [];
                const colorLabels = chartData.getAxisLabels('color') || [null];
                const groupsData = [];
                let numberOfBins = 0;
                xLabels.forEach(function(xLabel, x) {
                    const boxplots = [];
                    colorLabels.forEach(function(colorLabel, c) {
                        chartDef.boxplotValue.forEach(function(measure, m) {
                            if (measureFilter && !measureFilter(measure)) {
                                return;
                            }
                            numberOfBins++;
                            boxplots.push({ color: c, measure: m, x: x });
                        });
                    });
                    groupsData.push({ x: x, boxplots });
                });
                return { groupsData, numberOfBins };
            }

            const BOX_WIDTH = 40;
            const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef);
            const { groupsData, numberOfBins } = prepareData(chartDef, chartData);
            const xLabels = (chartData.getAxisLabels('x') || []).map(v => ({ ...v, label: ChartFormatting.getForOrdinalAxis(v.label) }));
            const colorLabels = chartData.getAxisLabels('color') || [];
            const hasColorDimension = ChartColorUtils.getColorDimensionOrMeasure(chartDef) !== undefined;
            const colorSpec = { type: 'DIMENSION', name: 'color', dimension: chartDef.genericDimension1[0] };
            const yAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT);
            const yExtent = ChartAxesUtils.initCustomAxisExtent(ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, ChartsStaticData.LEFT_AXIS_ID), [data.global.min, data.global.max]);
            const breakdownDimension = ChartDimension.getBreakdownDimension(chartDef);
            const xSpec = { type: 'DIMENSION', dimension: breakdownDimension, name: 'x', customExtent: chartDef.xAxisFormatting.customExtent, mode: CHART_MODES.COLUMNS };
            const ySpec = { id: yAxisID, type: 'MEASURE', extent: yExtent, name: 'y', customExtent: ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, yAxisID), position: ChartYAxisPosition.LEFT };
            const colorScale = ChartColorScales.createColorScale({ chartData, colorOptions: chartDef.colorOptions, defaultLegendDimension: chartDef.boxplotValue, colorSpec, chartHandler, theme: chartHandler.getChartTheme() });

            $(element).children().remove();

            const div = $(d3.selectAll(element).append('div')[0][0]);
            div.addClass('mainzone');
            const mainZone = $('<div style="width:100%; height: 100%;" />');
            $(div).append(mainZone);

            mainZone.addClass('horizontal-flex chart-wrapper');

            const globalSVG = d3.selectAll(mainZone).append('svg').attr('class', 'noflex')[0][0];
            const scrollableDiv = $(d3.selectAll(mainZone).append('div')[0][0]);

            scrollableDiv.addClass('flex');
            scrollableDiv.addClass('oa');
            const axisSpecs = {
                x: xSpec,
                [yAxisID]: ySpec
            };

            chartDef.$axisSpecs = axisSpecs;

            const xAxis = xSpec.dimension ? D3ChartAxes.createAxis(chartData, xSpec, false, chartDef.xAxisFormatting.isLogScale, undefined, chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting, chartDef) : null;
            const includeZero = ChartAxesUtils.shouldIncludeZero(chartDef, yAxisID);
            const yAxis = D3ChartAxes.createAxis(chartData, ySpec, false, ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, ySpec.id), includeZero, ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, ySpec.id), chartDef);
            const yAxes = [yAxis];
            const { store, id } = ChartStoreFactory.getOrCreate(chartDef.$chartStoreId);
            chartDef.$chartStoreId = id;
            // this is done normally in the ChartManager but since we don't use it, we need to do it manually
            store.setAxisSpecs(axisSpecs);

            ChartLegendUtils.createLegend(element, chartDef, chartData, chartHandler, colorSpec, colorScale).then(() => {
                $timeout(() => {

                    const mainSVG = d3.selectAll(scrollableDiv).append('svg')[0][0];
                    const $svg = $(globalSVG);
                    const BOX_PADDING = 15;
                    const INTER_BIN_PADDING = 40; // distance between boxes in different x bins will be 2 * BOX_PADDING + INTER_BIN_PADDING
                    const BOX_TOTAL_WIDTH = BOX_WIDTH + 2 * BOX_PADDING;
                    const leftAxisFormatting = chartDef.yAxesFormatting[0];
                    const fontSize = ChartPropertiesService.showYAxis(leftAxisFormatting, chartHandler.chartTileProperties) ? leftAxisFormatting.axisValuesFormatting.axisTicksFormatting.fontSize : 0;
                    // FIXME: Dirty, normally, we should have the option on the axis.
                    let { margins } = D3ChartAxes.getMarginsAndAxesWidth(chartHandler, chartDef, $svg, xAxis, yAxes);
                    const mAxisWidth = (BOX_WIDTH + BOX_PADDING) * numberOfBins + (BOX_PADDING + INTER_BIN_PADDING) * numberOfBins / (colorLabels.length || 1);

                    // Adjust bottom margin for x axis
                    if (xAxis) {
                        xAxis.ordinalScale.rangeBands([0, mAxisWidth]); // Needed so we can calculate a correct usedBand
                        if (ChartPropertiesService.showXAxis(chartDef.xAxisFormatting, chartHandler.chartTileProperties)) {
                            margins = D3ChartAxes.adjustBottomMargin(chartDef.xAxisFormatting.axisValuesFormatting.axisTicksFormatting, margins, $svg, xAxis);
                        }
                    }
                    const $mainZone = $(mainZone);
                    // Prevent the left margins to be too big
                    margins.left = Math.min(margins.left, $mainZone.width() / 6);
                    /*
                     * We have calculated the margins but boxplot has a space between content and axis
                     * whereas bar start directly on the 0, so add that little margin
                     */
                    margins.bottom += ChartsStaticData.CHART_BASE_MARGIN;
                    const globalBoxplotWidth = margins.left + BOX_TOTAL_WIDTH + ChartsStaticData.CHART_BASE_MARGIN;
                    const availableHeight = $mainZone.height() - margins.bottom - margins.top;

                    const globalG = d3.select(globalSVG)
                        .style('width', globalBoxplotWidth + 'px')
                        .append('g')
                        .attr('transform', `translate(${margins.left}, ${margins.top})`);

                    const mChartWidth = mAxisWidth + margins.left;
                    const mainG = d3.select(mainSVG)
                        .style('width', mChartWidth + 'px')
                        .append('g')
                        .attr('transform', `translate(${ChartsStaticData.CHART_BASE_MARGIN}, ${margins.top})`);

                    const dataZoneHeight = availableHeight;
                    const measureFormatters = ChartFormatting.createMeasureFormatters(chartDef, chartData, Math.max(availableHeight, mChartWidth));
                    const tooltips = ChartTooltips.create(element, chartHandler, chartData, chartDef, measureFormatters);
                    if (element.data('previous-tooltip')) {
                        element.data('previous-tooltip').destroy();
                    }
                    element.data('previous-tooltip', tooltips);
                    angular.extend(chartHandler.tooltips, tooltips);

                    mainZone.on('click', function(evt) {
                        if (evt.target.hasAttribute('data-legend') || evt.target.hasAttribute('tooltip-el')) {
                            return;
                        }
                        tooltips.resetColors();
                        tooltips.unfix();
                    });
                    const contextualMenu = ChartContextualMenu.create(chartData, chartDef, store);
                    const yscale = yAxes[0].scale();
                    yscale.range([dataZoneHeight, 0]).domain(yExtent);

                    function drawBoxplot(g, boxplot, yscale, plotMin, plotMax, color = '#FFF', opacity = 1, coords) {
                        g.selectAll('line.center')
                            .data([boxplot])
                            .enter().append('line')
                            .attr('class', 'center')
                            .attr('y1', function(d) {
                                return yscale(d.lowWhisker);
                            })
                            .attr('x1', BOX_WIDTH / 2)
                            .attr('y2', function(d) {
                                return yscale(d.highWhisker);
                            })
                            .attr('x2', BOX_WIDTH / 2)
                            .style('opacity', 1)
                            .style('stroke', '#666');
                        g.selectAll('rect.box')
                            .data([boxplot])
                            .enter().append('rect')
                            .attr('class', 'box')
                            .attr('x', 0)
                            .attr('width', BOX_WIDTH)
                            .attr('y', function(d) {
                                return yscale(d.pc75);
                            })
                            .attr('height', function(d) {
                                return Math.max(yscale(d.pc25) - yscale(d.pc75), 0);
                            })
                            .attr('fill', color)
                            .attr('stroke', '#666')
                            .style('opacity', opacity)
                            .each(function() {
                                const chartDefPath = 'boxplotValue[0]';
                                tooltips.registerEl(this, coords ? angular.extend({}, coords) : undefined, coords ? 'fill' : null, undefined, {
                                    additionalMeasures: [
                                        { displayLabel: ChartLabels.COUNT_OF_RECORDS_LABEL, value: boxplot.nbVAlid, chartDefPath },
                                        { displayLabel: 'Mean', value: boxplot.mean, chartDefPath },
                                        { displayLabel: 'Std. dev.', value: boxplot.stddev, chartDefPath },
                                        { displayLabel: 'Min', value: boxplot.min, chartDefPath },
                                        { displayLabel: '1st quartile', value: boxplot.pc25, chartDefPath },
                                        { displayLabel: 'Median', value: boxplot.median, chartDefPath },
                                        { displayLabel: '3rd quartile', value: boxplot.pc75, chartDefPath },
                                        { displayLabel: 'Max', value: boxplot.max, chartDefPath }
                                    ]
                                });
                                contextualMenu.addContextualMenuHandler(this, angular.extend({}, coords));
                            });
                        g.selectAll('line.median')
                            .data([boxplot])
                            .enter().append('line')
                            .attr('class', 'median')
                            .attr('val', boxplot.median)
                            .attr('x1', 0)
                            .attr('y1', yscale(boxplot.median))
                            .attr('x2', BOX_WIDTH)
                            .attr('y2', yscale(boxplot.median))
                            .style('opacity', 1)
                            .style('stroke', '#666');
                        g.selectAll('line.whisker')
                            .data([boxplot.lowWhisker, boxplot.highWhisker])
                            .enter().append('svg:line')
                            .attr('class', 'whisker')
                            .attr('val', function(d) {
                                return d;
                            })
                            .attr('x1', BOX_WIDTH * 0.3)
                            .attr('y1', function(d) {
                                return yscale(d);
                            })
                            .attr('x2', BOX_WIDTH * 0.7)
                            .attr('y2', function(d) {
                                return yscale(d);
                            })
                            .style('stroke', '#666');
                        if (plotMin !== undefined && plotMax !== undefined) {
                            // Min and max of this modality
                            g.selectAll('circle.minmax').data([plotMin, plotMax]).enter()
                                .append('circle').attr('class', 'minmax')
                                .attr('transform', function(d) {
                                    return 'translate(' + BOX_WIDTH / 2 + ',' + yscale(d) + ')';
                                })
                                .attr('r', 2)
                                .attr('fill', '#999');
                        }
                    }

                    drawBoxplot(globalG.append('g').attr('transform', `translate(${BOX_PADDING}, 0)`), data.global, yscale, data.global.min, data.global.max);
                    /* X Axis */
                    let xAxisG;

                    const allNegative = yAxes.every(yAxis => {
                        const currentYExtent = D3ChartAxes.getCurrentAxisExtent(yAxis);
                        return !currentYExtent || currentYExtent[1] < 0;
                    });
                    if (xLabels.length) {
                        /*
                         * This is OK since we always create an ordinal scale on boxplot
                         * Cannot use xAxis.setScaleRange since it's not doing exactly the same thing
                         */

                        if (ChartPropertiesService.showXAxis(chartDef.xAxisFormatting, chartHandler.chartTileProperties)) {
                            xAxisG = mainG.insert('g', ':first-child');
                            xAxisG.attr('class', 'x axis')
                                .attr('transform', `translate(0, ${dataZoneHeight + ChartsStaticData.CHART_BASE_MARGIN})`);
                            xAxis.orient('bottom');
                            xAxisG.call(xAxis);
                            if (xAxisG) {
                                xAxisG.selectAll('.tick text')
                                    .attr('fill', chartDef.xAxisFormatting.axisValuesFormatting.axisTicksFormatting.fontColor)
                                    .attr('font-size', chartDef.xAxisFormatting.axisValuesFormatting.axisTicksFormatting.fontSize);
                            };
                            if (xAxis.labelAngle) {
                                if (!allNegative) {
                                    xAxisG.selectAll('text')
                                        .attr('transform', (xAxis.labelAngle == Math.PI / 2 ? 'translate(-13, 9)' : 'translate(-10, 0)') + ' rotate(' + xAxis.labelAngle * -180 / Math.PI + ', 0, 0)')
                                        .style('text-anchor', 'end');
                                } else {
                                    xAxisG.selectAll('text')
                                        .attr('transform', 'translate(10, 0) rotate(' + xAxis.labelAngle * -180 / Math.PI + ', 0, 0)')
                                        .style('text-anchor', 'start');
                                }
                            }
                        }
                    }

                    groupsData.forEach(function(group) {
                        group.boxplots.forEach((coords) => {
                            const bp = chartData.aggr(coords.measure).get(coords);
                            const start = xAxis.scale().range()[coords.x] + INTER_BIN_PADDING / 2;
                            const g = mainG.append('g').attr('transform', 'translate(' + (start + BOX_PADDING + (coords.color * (BOX_WIDTH + BOX_PADDING))) + ', 0)');
                            if (bp != null) {
                                bp.label = xLabels[coords.x].label;
                                if (hasColorDimension) {
                                    bp.labelColor = colorLabels[coords.color].label;
                                }
                                drawBoxplot(g, bp, yscale, bp.min, bp.max, colorScale(coords.color), chartDef.colorOptions.transparency, coords);
                            }
                        });
                    });

                    /* Y Axis */
                    if (ChartPropertiesService.showYAxis(leftAxisFormatting, chartHandler.chartTileProperties)) {
                        const yAxis = yAxes[0];
                        yAxis.orient('left');
                        globalG.insert('g', ':first-child')
                            .attr('class', 'y axis').call(yAxis);
                        const yAxisG = d3.select(globalSVG).select('g.y.axis');
                        if (yAxisG) {
                            yAxisG.selectAll('.tick text')
                                .attr('fill', leftAxisFormatting.axisValuesFormatting.axisTicksFormatting.fontColor)
                                .attr('font-size', fontSize);

                            if (chartDef.gridlinesOptions.horizontal.show && D3ChartAxes.shouldDrawGridlinesForAxis(chartDef.type, yAxis, chartDef.gridlinesOptions.horizontal.displayAxis)) {
                                let gridlinesPosition = { x1: 0, x2: ChartsStaticData.CHART_BASE_MARGIN + BOX_TOTAL_WIDTH, y1: 0, y2: 0 };
                                D3ChartAxes.drawGridlines(yAxisG, 'y', chartDef.gridlinesOptions.horizontal.lineFormatting, gridlinesPosition, mainG);

                                // directly appending gridlines in the main zone as they are not connected to the y axis ticks
                                const linesG = d3.select(mainSVG).insert('g', ':first-child');
                                linesG.attr('transform', `translate(0, ${margins.top})`);
                                gridlinesPosition = { x1: 0, x2: mChartWidth,
                                    y1: function(d) {
                                        return yscale(d);
                                    },
                                    y2: function(d) {
                                        return yscale(d);
                                    }
                                };
                                D3ChartAxes.appendGridlines(linesG.selectAll('.hline').data(yscale.ticks()).enter(), gridlinesPosition,
                                    chartDef.gridlinesOptions.horizontal.lineFormatting, 'hline');
                            }
                        }
                    }

                    if (chartDef.gridlinesOptions.vertical.show && xAxisG) {
                        const currentYExtent = D3ChartAxes.getCurrentAxisExtent(yAxes[0]);
                        const allNegative = currentYExtent && currentYExtent[1] < 0;
                        const gridlinesPosition = { x1: 0, x2: 0, y1: 0, y2: allNegative ? dataZoneHeight + margins.top : -(dataZoneHeight + margins.top) };
                        D3ChartAxes.drawGridlines(xAxisG, 'x', chartDef.gridlinesOptions.vertical.lineFormatting, gridlinesPosition, mainG);
                    }

                    ChartLegendUtils.adjustLegendPlacement(chartDef, element, { top: margins.top, bottom: margins.bottom, left: BOX_WIDTH, right: ChartsStaticData.CHART_BASE_MARGIN });

                    if (ChartHierarchyDimension.getCurrentHierarchyLevel(chartDef) > 0) {
                        const anchor = SVGUtils.addPlotAreaContextMenuAnchor(mainG, mAxisWidth, dataZoneHeight);
                        contextualMenu.addChartContextualMenuHandler(anchor.node(), undefined, () => ChartDrilldown.getDrillupActions(chartDef));
                    }

                    // Signal to the callee handler that the chart has been successfully loaded. Dashboards use it to determine when all insights are completely loaded.
                    if (typeof (chartHandler.loadedCallback) === 'function') {
                        chartHandler.loadedCallback();
                    }
                });
            });
        };
    });

})();
