// @ts-check
(function() {
    'use strict';
    /** @typedef {import("../../types").ChartDef} ChartDef */
    /** @typedef {import("../../types").GeneratedSources.MeasureDef} MeasureDef */
    /** @typedef {import("../../types").GeneratedSources.DimensionDef} DimensionDef */

    angular.module('dataiku.charts')
        .factory('HierarchicalChartsUtils', function(ChartFormatting, ChartFeatures, ChartDimension) {

            return {

                /**
                 * Compute cell formatter properties.
                 * @returns {{ cellFormatters: ((value: number) => string)[], isPercentageOnly: boolean }}
                 */
                computeFormatterProperties(chartDef) {
                    let isPercentageOnly = true;
                    const cellFormatters = [];
                    const displayEmpty = chartDef.pivotTableOptions && chartDef.pivotTableOptions.displayEmptyValues;
                    for (let i = 0; i < chartDef.genericMeasures.length; i++) {
                        const measure = chartDef.genericMeasures[i];
                        cellFormatters.push(ChartFormatting.getForIsolatedNumber({ ...measure, displayEmpty }));

                        if (isPercentageOnly && (measure.computeMode !== 'PERCENTAGE')) {
                            isPercentageOnly = false;
                        }
                    }
                    // second pass to add additional measures after all generic measures
                    for (let i = 0; i < chartDef.genericMeasures.length; i++) {
                        const measure = chartDef.genericMeasures[i];
                        if (measure.valuesInChartDisplayOptions && measure.valuesInChartDisplayOptions.additionalMeasures) {
                            measure.valuesInChartDisplayOptions.additionalMeasures.forEach(additionalMeasure => {
                                cellFormatters.push(ChartFormatting.getForIsolatedNumber({ ...additionalMeasure, displayEmpty }));
                            });
                        }
                    }
                    return { cellFormatters, isPercentageOnly };
                },

                /**
                 * Calls `visitCallback` with every possible coordinates of aggregations and subtotals composing a dimension list.
                 * @example With `dimensionIds` set as [dim1, dim2], considering each of these dimensions has 2 values besides the subtotal bin of index 0
                 * Call successively `visitCallback` with {dim1: 1}, {dim1: 1, dim2: 1}, {dim1: 1, dim2: 2}, {dim1: 2}, {dim1: 2, dim2: 1}, {dim1: 2, dim2: 2}
                 * @param {*} chartData
                 * @param {string []} dimensionIds
                 * @param {boolean} visitSubtotals
                 * @param {(coordDict: Record<string, number>) => void} visitCallback
                 */
                visitAggregationTree: function(chartData, dimensionIds, visitSubtotals, visitCallback) {
                    if (!chartData.hasSubtotalCoords()) {
                        throw new Error('Option `computeSubTotals` must be enabled to fetch subtotals in the tensor request');
                    }
                    const [headDimId, ...tailDimIds] = dimensionIds;
                    /** @type {{coordDict: Record<string, number>, tailDimIds: string []} []} */
                    const stack = [];
                    chartData.getAxisLabels(headDimId).forEach((axisLabel, index) => {
                        if (index === chartData.getSubtotalLabelIndex(headDimId)) {
                            return;
                        }
                        /** @type {Record<string, number>} */
                        const coordDict = {};
                        coordDict[headDimId] = index;
                        stack.push({ coordDict, tailDimIds });
                    });
                    while (stack.length) {
                        const current = stack.pop();
                        const isSubTotal = current.tailDimIds.length > 0;
                        if (!isSubTotal || visitSubtotals) {
                            visitCallback(current.coordDict);
                        }
                        if (current.tailDimIds && current.tailDimIds.length) {
                            const [childHeadDimId, ...childTailDimIds] = current.tailDimIds;
                            chartData.getAxisLabels(childHeadDimId).forEach((axisLabel, index) => {
                                if (index === chartData.getSubtotalLabelIndex(childHeadDimId)) {
                                    return;
                                }
                                /** @type {Record<string, number>} */
                                const coordDict = { ...current.coordDict };
                                coordDict[childHeadDimId] = index;
                                stack.push({ coordDict, tailDimIds: childTailDimIds });
                            });
                        }
                    }

                },
                /**
                 * Filters the set of tensor indexes that should be considered by the color scale to compute the min & max values.
                 * It keeps only the entries inside the chart that are not the Grand total.
                 * @param {*} chartData
                 * @param {string[]} xDimensionIds
                 * @param {string[]} yDimensionIds
                 * @param {number} colorAggrIdx
                 * @param {ChartDef} chartDef
                 * @returns {Set<number>} tensor indexes of the grand total and the subtotals that should be included in the color scale
                 */
                getBinsToIncludeInColorScale: function(chartData, xDimensionIds, yDimensionIds, colorAggrIdx, chartDef, hasSubtotalXDim, hasSubtotalYDim) {
                    const binsToInclude = new Set();
                    const aggregation = chartData.data.aggregations[colorAggrIdx];

                    const colorMeasureFn = this.getColorMeasureAggFunction(chartDef, colorAggrIdx);

                    if (xDimensionIds.length && yDimensionIds.length) { // When dimensions are set as yDimensions & xDimensions, only the crossing of those dimensions should be considered
                        /** @type {Record<string, number> []} */
                        const xCoordsList = [];
                        this.visitAggregationTree(chartData, xDimensionIds, hasSubtotalXDim, coordDict => xCoordsList.push(coordDict));

                        xCoordsList.forEach(xCoords => {
                            this.visitAggregationTree(chartData, yDimensionIds, hasSubtotalYDim, yCoords => {
                                const tensorSubtotalIndex = chartData.getCoordsLoc(aggregation, chartData.getSubTotalCoordsArray({ ...xCoords, ...yCoords }));
                                if (chartData.isBinMeaningful(colorMeasureFn, tensorSubtotalIndex, colorAggrIdx)) {
                                    binsToInclude.add(tensorSubtotalIndex);
                                }
                            });
                        });
                    } else { // When dimensions are set only as yDimensions or only as xDimensions, only the dimension tree should be considered
                        const dimensionIds = xDimensionIds.length ? xDimensionIds : yDimensionIds;
                        const hasSubtotals = xDimensionIds.length ? hasSubtotalXDim : hasSubtotalYDim;
                        this.visitAggregationTree(chartData, dimensionIds, hasSubtotals, coords => {
                            const tensorSubtotalIndex = chartData.getCoordsLoc(aggregation, chartData.getSubTotalCoordsArray(coords));
                            if (chartData.isBinMeaningful(colorMeasureFn, coords, colorAggrIdx)) {
                                binsToInclude.add(tensorSubtotalIndex);
                            }
                        });
                    }
                    return binsToInclude;
                },

                /**
                 * Checks if we should ignore empty bins for color dimension breakdown
                 * @param {*} chartDef
                 */
                shouldIgnoreEmptyBinsForColorDimension: function(chartDef) {
                    if (!ChartFeatures.canIgnoreEmptyBinsForColorDimension(chartDef)) {
                        return false;
                    }
                    return ChartDimension.getGenericDimensions(chartDef).length === 1 && chartDef.genericDimension1.length === 1 && chartDef.genericDimension1[0].ignoreEmptyBins;
                },

                /**
                 * Get the aggregation function of the measure being used to color a cell in a pivot table
                 * @param {ChartDef} chartDef
                 * @param {number} colorMeasureIndex
                 * @returns {string} Aggregation function used for this group/pivot table
                 */
                getColorMeasureAggFunction: function(chartDef, colorMeasureIndex) {
                    if (chartDef.colorMode !== 'COLOR_GROUPS' || !ChartFeatures.canHaveConditionalFormatting(chartDef.type)) {
                        // unique color scale
                        return chartDef.colorMeasure[0].function;
                    }
                    if (colorMeasureIndex < chartDef.genericMeasures.length) {
                        // color group based on the values themselves
                        return chartDef.genericMeasures[colorMeasureIndex].function;
                    }

                    // color group based on another column
                    const group_index = colorMeasureIndex - chartDef.genericMeasures.length;

                    return chartDef.colorGroups[group_index].colorMeasure[0].function;
                }
            };
        });
})();
