/* global describe, it, beforeEach, expect, ChartFeaturesTestUtils, inject, module, PivotTableTestUtils, beforeAll */

/**
 * For all representative configurations, ensure proper behavior of the headers dipslay depending on existing options
 */
describe('chart pivot table - headers', function () {

    const chartType = 'pivot_table';
    const chartVariant = 'normal';

    /* Initialize each test */
    beforeEach(() => {
        ChartFeaturesTestUtils.initEachTest();
        ChartFeaturesTestUtils.initChartType(chartType, chartVariant);
    });

    it('hides row and column headers in 0 - 1 - 2', function () {
        function expectRowMainHeadersToBeVisibleFor(gridOptions, showColumnMainHeaders = true) {
            const index = showColumnMainHeaders ? 0 : 1;
            expect(gridOptions.columnDefs[index].headerName).toBe('Value');
        }

        function expectRowMainHeadersToBeHiddenFor(gridOptions, showColumnMainHeaders = true, showRowHeaders = true) {
            const index = showColumnMainHeaders ? 0 : 1;
            if (showRowHeaders) {
                expect(gridOptions.columnDefs[index].headerName).toBe('');
            } else {
                expect(gridOptions.columnDefs[index].children).toBeUndefined();
            }
        }

        function expectColumnMainHeadersToBeVisibleFor(gridOptions) {
            expect(gridOptions.columnDefs[1].headerName).toBe('Platinium member');
        }

        function expectColumnMainHeadersToBeHiddenFor(gridOptions, showColumnHeaders = true) {
            if (showColumnHeaders) {
                expect(gridOptions.columnDefs[0].headerName).toBe('FALSE');
            } else {
                expect(gridOptions.columnDefs[0].headerName).toBe('');
            }
        }

        inject(function (AgGridConverter, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper, ChartDimension) {
            const { axesDef, chartDef, data } = PivotTableTestUtils.getPivotDataWith1ColumnDim2Measures();
            const { $container, chartData, chartHandler, chartStore, tooltips, contextualMenu } = PivotTableTestUtils.initPivotTableContext(chartDef, axesDef, data, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartDimension, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper);

            let gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();

            // In this mode, tree data is always false, yet the first column looks like a tree data structure when we list the measure names under the "Value" column.
            
            // Assert that there's no row headers by default
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide row main headers
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide row and column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide row headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders, chartDef.pivotTableOptions.tableFormatting.showRowHeaders);
            expectColumnMainHeadersToBeVisibleFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide row headers + column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders, chartDef.pivotTableOptions.tableFormatting.showRowHeaders);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide column headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToHaveOnlyOneRowFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders && chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders, chartDef.pivotTableOptions.tableFormatting.showRowHeaders);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide column headers + row main headers => everything's hidden except value names (row subheaders)
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders && chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders, chartDef.pivotTableOptions.tableFormatting.showRowHeaders);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide column and row headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders && chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders, chartDef.pivotTableOptions.tableFormatting.showRowHeaders);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);
        });
    });

    it('hides row and column headers in 0 - 2 - 1', function () {

        function expectColumnMainHeadersToBeVisibleFor(gridOptions) {
            expect(gridOptions.columnDefs[0].headerName).toBe('Platinium member');
        }

        function expectColumnMainHeadersToBeHiddenFor(gridOptions) {
            // Only subheaders (dimension values) are displayed
            expect(gridOptions.columnDefs[0].headerName).toBe('FALSE');
        }

        inject(function (AgGridConverter, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartDimension, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper) {
            const { axesDef, chartDef, data } = PivotTableTestUtils.getPivotDataWith2ColumnDim1Measure();
            const { $container, chartData, chartHandler, chartStore, tooltips, contextualMenu } = PivotTableTestUtils.initPivotTableContext(chartDef, axesDef, data, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartDimension, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper);

            let gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();

            // Assert that there's no row headers by default
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide column main headers
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions);

            // Hide column headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
        });
    });

    it('hides row and column headers in 0 - 2 - 2', function () {

        function expectRowMainHeadersToBeVisibleFor(gridOptions, showColumnMainHeaders = true) {
            const index = showColumnMainHeaders ? 0 : 1;
            expect(gridOptions.columnDefs[index].headerName).toBe('Value');
        }

        function expectRowMainHeadersToBeHiddenFor(gridOptions, showColumnMainHeaders = true, showRowHeaders = true) {
            const index = showColumnMainHeaders ? 0 : 1;
            if (showRowHeaders) {
                expect(gridOptions.columnDefs[index].headerName).toBe('');
            } else {
                expect(gridOptions.columnDefs[index].children).toBeUndefined();
            }
        }

        function expectColumnMainHeadersToBeVisibleFor(gridOptions) {
            expect(gridOptions.columnDefs[1].headerName).toBe('Platinium member');
        }

        function expectColumnMainHeadersToBeHiddenFor(gridOptions, showColumnHeaders = true) {
            if (showColumnHeaders) {
                expect(gridOptions.columnDefs[0].headerName).toBe('FALSE');
            } else {
                expect(gridOptions.columnDefs[0].headerName).toBe('');
            }
        }

        inject(function (AgGridConverter, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper, ChartDimension) {
            const { axesDef, chartDef, data } = PivotTableTestUtils.getPivotDataWith2ColumnDim2Measures();
            const { $container, chartData, chartHandler, chartStore, tooltips, contextualMenu } = PivotTableTestUtils.initPivotTableContext(chartDef, axesDef, data, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartDimension, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper);

            let gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();

            // In this mode, tree data is always false, yet the first column looks like a tree data structure when we list the measure names under the "Value" column.
            
            // Assert that there's no row headers by default
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide row main headers
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide row and column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide row headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders, chartDef.pivotTableOptions.tableFormatting.showRowHeaders);
            expectColumnMainHeadersToBeVisibleFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide row headers + column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders, chartDef.pivotTableOptions.tableFormatting.showRowHeaders);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide column headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToHaveOnlyOneRowFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders && chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders, chartDef.pivotTableOptions.tableFormatting.showRowHeaders);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide column headers + row main headers => everything's hidden except value names (row subheaders)
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders && chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders, chartDef.pivotTableOptions.tableFormatting.showRowHeaders);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide column and row headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders && chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders, chartDef.pivotTableOptions.tableFormatting.showRowHeaders);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);
        });
    });

    it('hides row and column headers in 2 - 0 - 1', function () {
        function expectRowMainHeadersToBeVisibleFor(gridOptions) {
            expect(gridOptions.autoGroupColumnDef.headerName).toBe('Platinium member');
            expect(gridOptions.columnDefs[0].headerName).toBe('Continent');
        }

        function expectRowMainHeadersToBeHiddenFor(gridOptions) {
            expect(gridOptions.autoGroupColumnDef.headerName).toBe('');
            expect(gridOptions.columnDefs[0].headerName).toBe('');
        }

        function expectColumnMainHeadersToBeVisibleFor(gridOptions) {
            expect(gridOptions.columnDefs[1].headerName).toBe('nb_visit');
        }

        function expectColumnMainHeadersToBeHiddenFor(gridOptions) {
            expect(gridOptions.columnDefs[1].headerName).toBe('');
        }

        inject(function (AgGridConverter, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper, ChartDimension) {
            const { axesDef, chartDef, data } = PivotTableTestUtils.getPivotDataWith2RowDim1MeasureAggregAsRows();
            const { $container, chartData, chartHandler, chartStore, tooltips, contextualMenu } = PivotTableTestUtils.initPivotTableContext(chartDef, axesDef, data, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartDimension, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper);

            let gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();

            // Assert both headers and main headers are displayed from the beginning
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide row main headers
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions);

            // Hide both row and column main headers => top header should be hidden as well as the main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions);

            // Hide row headers => no tree data, the rest is displayed
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide row headers and column main headers => everything is hidden
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);

            // Hide column headers => everything should be there except the column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions);

            // Hide column headers and row main headers => top header should be hidden as well as the main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions);

            // Hide both headers => everything should be hidden
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
        });
    });

    it('hides row and column headers in 2 - 2 - 1', function () {
        function expectRowMainHeadersToBeVisibleFor(gridOptions) {
            expect(gridOptions.autoGroupColumnDef.headerName).toBe('gender');
            expect(gridOptions.columnDefs[0].headerName).toBe('Content');
        }

        function expectRowMainHeadersToBeHiddenFor(gridOptions) {
            expect(gridOptions.autoGroupColumnDef.headerName).toBe('');
            expect(gridOptions.columnDefs[0].headerName).toBe('');
        }

        function expectColumnMainHeadersToBeVisibleFor(gridOptions, measuresDisplayedAsColumns) {
            expect(gridOptions.columnDefs[1].headerName).toBe('Continent');
            expect(gridOptions.columnDefs[1].children[0].children[0].headerName).toBe('Platinium member');
            if (measuresDisplayedAsColumns) {
                expect(gridOptions.columnDefs[1].children[1].children[0].children[0].children[0].headerName).toBe('nb_visit');
            }
        }

        function expectColumnMainHeadersToBeHiddenFor(gridOptions, showColumnMainHeaders = true, measuresDisplayedAsColumns) {
            if (showColumnMainHeaders) {
                expect(gridOptions.columnDefs[1].headerName).toBe('Amerique');
                expect(gridOptions.columnDefs[1].children[0].headerName).toBe('FALSE');
                if (measuresDisplayedAsColumns) {
                    expect(gridOptions.columnDefs[1].children[1].children[0].headerName).toBe('nb_visit');
                }
            } else {
                expect(gridOptions.columnDefs[1].headerName).toBe('');
                expect(gridOptions.columnDefs[1].children[0].headerName).toBe('');
                if (measuresDisplayedAsColumns) {
                    expect(gridOptions.columnDefs[1].children[1].children[0].headerName).toBe('');
                }
            }
        }

        inject(function (AgGridConverter, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper, ChartDimension) {
            // MEASURES AS ROWS
            const { axesDef, chartDef, data } = PivotTableTestUtils.getPivotDataWith2RowDim2ColumnDim1MeasureWithAggregationAsRows();
            const { $container, chartData, chartHandler, chartStore, tooltips, contextualMenu } = PivotTableTestUtils.initPivotTableContext(chartDef, axesDef, data, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartDimension, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper);

            let gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();

            // Assert that there's everything by default
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide row main headers
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions);

            // Hide both row and column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions);

            // Hide row headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide row headers + column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions);

            // Hide column headers => top Header still there but only contains row main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToHaveOnlyOneRowFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide column headers and row main headers => top Header hidden
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // Hide row and column headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders);

            // MEASURES AS COLUMNS 
            const { axesDef: axesDefAsColumns, chartDef: chartDefAsColumns, data: dataAsColumns } = PivotTableTestUtils.getPivotDataWith2RowDim2ColumnDim1MeasureWithAggregationAsColumns();
            const { $container: $containerAsColumns, chartData: chartDataAsColumns, chartHandler: chartHandlerAsColumns, chartStore: chartStoreAsColumns, tooltip: tooltipAsColumns, contextualMenu: contextualMenuAsColumns } = PivotTableTestUtils.initPivotTableContext(chartDefAsColumns, axesDefAsColumns, dataAsColumns, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartDimension, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper);
            const measuresDisplayedAsColumns = true;
            let gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();

            // Assert that there's everything by default
            expectTopHeaderToBeVisibleFor(gridOptionsAsColumns);
            expectTreeDataToBeVisibleFor(gridOptionsAsColumns);
            expectRowMainHeadersToBeVisibleFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeVisibleFor(gridOptionsAsColumns, measuresDisplayedAsColumns);

            // Hide row main headers
            chartDefAsColumns.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeVisibleFor(gridOptionsAsColumns);
            expectTreeDataToBeVisibleFor(gridOptionsAsColumns);
            expectRowMainHeadersToBeHiddenFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeVisibleFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide column main headers
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeVisibleFor(gridOptionsAsColumns);
            expectTreeDataToBeVisibleFor(gridOptionsAsColumns);
            expectRowMainHeadersToBeVisibleFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeHiddenFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide row and column main headers
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeVisibleFor(gridOptionsAsColumns);
            expectTreeDataToBeVisibleFor(gridOptionsAsColumns);
            expectRowMainHeadersToBeHiddenFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeHiddenFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide row headers
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showRowHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeVisibleFor(gridOptionsAsColumns);
            expectTreeDataToBeHiddenFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeVisibleFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide row headers + column main headers
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showRowHeaders = false;
            chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeVisibleFor(gridOptionsAsColumns);
            expectTreeDataToBeHiddenFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeHiddenFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide column headers => only one row for top header
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToHaveOnlyOneRowFor(gridOptionsAsColumns);
            expectTreeDataToBeVisibleFor(gridOptionsAsColumns);
            expectRowMainHeadersToBeVisibleFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeHiddenFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide column headers and row main headers => no more top header
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            chartDefAsColumns.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeHiddenFor(gridOptionsAsColumns);
            expectTreeDataToBeVisibleFor(gridOptionsAsColumns);
            expectRowMainHeadersToBeHiddenFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeHiddenFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide row and column headers => everything's hidden
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            chartDefAsColumns.pivotTableOptions.tableFormatting.showRowHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeHiddenFor(gridOptionsAsColumns);
            expectTreeDataToBeHiddenFor(gridOptionsAsColumns);
        });
    });

    it('hides row and column headers in 2 - 2 - 2', function () {
        function expectRowMainHeadersToBeVisibleFor(gridOptions, measuresDisplayedAsRows) {
            expect(gridOptions.autoGroupColumnDef.headerName).toBe('gender');
            expect(gridOptions.columnDefs[0].headerName).toBe('Content');
            if (measuresDisplayedAsRows) {
                expect(gridOptions.columnDefs[1].headerName).toBe('Value');
            }
        }

        function expectRowMainHeadersToBeHiddenFor(gridOptions, measuresDisplayedAsRows) {
            expect(gridOptions.autoGroupColumnDef.headerName).toBe('');
            expect(gridOptions.columnDefs[0].headerName).toBe('');
            if (measuresDisplayedAsRows) {
                expect(gridOptions.columnDefs[1].headerName).toBe('');
            }
        }

        function expectColumnMainHeadersToBeVisibleFor(gridOptions, measuresDisplayedAsColumns) {
            const index = measuresDisplayedAsColumns ? 1 : 2;
            expect(gridOptions.columnDefs[index].headerName).toBe('Continent');
            expect(gridOptions.columnDefs[index].children[0].children[0].headerName).toBe('Platinium member');
            if (measuresDisplayedAsColumns) {
                expect(gridOptions.columnDefs[index].children[1].children[0].children[0].children[0].headerName).toBe('nb_visit_2');
                expect(gridOptions.columnDefs[index].children[1].children[0].children[0].children[1].headerName).toBe('nb_visit');
            }
        }

        function expectColumnMainHeadersToBeHiddenFor(gridOptions, showColumnHeaders = true, measuresDisplayedAsColumns) {
            const index = measuresDisplayedAsColumns ? 1 : 2;
            if (showColumnHeaders) {
                expect(gridOptions.columnDefs[index].headerName).toBe('Europe');
                expect(gridOptions.columnDefs[index].children[0].headerName).toBe('TRUE');
                if (measuresDisplayedAsColumns) {
                    expect(gridOptions.columnDefs[index].children[1].children[0].headerName).toBe('nb_visit_2');
                    expect(gridOptions.columnDefs[index].children[1].children[1].headerName).toBe('nb_visit');
                }
            } else {
                expect(gridOptions.columnDefs[index].headerName).toBe('');
                expect(gridOptions.columnDefs[index].children[0].headerName).toBe('');
                if (measuresDisplayedAsColumns) {
                    expect(gridOptions.columnDefs[index].children[1].children[0].headerName).toBe('');
                }
            }
        }

        inject(function (AgGridConverter, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper, ChartDimension) {
            // MEASURES AS ROWS
            const { axesDef, chartDef, data } = PivotTableTestUtils.getPivotDataWith2RowDim2ColumnDim2MeasureAggregationAsRows();
            const { $container, chartData, chartHandler, chartStore, tooltips, contextualMenu } = PivotTableTestUtils.initPivotTableContext(chartDef, axesDef, data, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartDimension, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper);

            let gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            let measuresDisplayedAsRows = true;
            let measuresDisplayedAsColumns = false;

            // Assert that there's everything by default
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions, measuresDisplayedAsRows);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide row main headers
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions, measuresDisplayedAsRows);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions, measuresDisplayedAsRows);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide both row and column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions, measuresDisplayedAsRows);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide row headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeVisibleFor(gridOptions);

            // Hide row headers + column main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeVisibleFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions);

            // Hide column headers => top Header still there but only contains row main headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToHaveOnlyOneRowFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeVisibleFor(gridOptions, measuresDisplayedAsRows);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide column headers and row main headers => top Header hidden
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeVisibleFor(gridOptions);
            expectRowMainHeadersToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide row and column headers
            resetHeadersFor(chartDef);
            chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = false;
            gridOptions = AgGridConverter.getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $container).convert();
            expectTopHeaderToBeHiddenFor(gridOptions);
            expectTreeDataToBeHiddenFor(gridOptions);
            expectColumnMainHeadersToBeHiddenFor(gridOptions, chartDef.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // MEASURES AS COLUMNS 
            const { axesDef: axesDefAsColumns, chartDef: chartDefAsColumns, data: dataAsColumns } = PivotTableTestUtils.getPivotDataWith2RowDim2ColumnDim2MeasureAggregationAsColumns();
            const { $container: $containerAsColumns, chartData: chartDataAsColumns, chartHandler: chartHandlerAsColumns, chartStore: chartStoreAsColumns, tooltip: tooltipAsColumns, contextualMenu: contextualMenuAsColumns } = PivotTableTestUtils.initPivotTableContext(chartDefAsColumns, axesDefAsColumns, dataAsColumns, ChartFormatting, ChartTooltips, ChartContextualMenu, ChartDimension, ChartStoreFactory, ChartDataWrapperFactory, ChartLegendsWrapper);

            measuresDisplayedAsRows = false;
            measuresDisplayedAsColumns = true;
            let gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();

            // Assert that there's everything by default
            expectTopHeaderToBeVisibleFor(gridOptionsAsColumns);
            expectTreeDataToBeVisibleFor(gridOptionsAsColumns);
            expectRowMainHeadersToBeVisibleFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeVisibleFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide row main headers
            chartDefAsColumns.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeVisibleFor(gridOptionsAsColumns);
            expectTreeDataToBeVisibleFor(gridOptionsAsColumns);
            expectRowMainHeadersToBeHiddenFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeVisibleFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide column main headers
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeVisibleFor(gridOptionsAsColumns);
            expectTreeDataToBeVisibleFor(gridOptionsAsColumns);
            expectRowMainHeadersToBeVisibleFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeHiddenFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide row and column main headers
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeVisibleFor(gridOptionsAsColumns);
            expectTreeDataToBeVisibleFor(gridOptionsAsColumns);
            expectRowMainHeadersToBeHiddenFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeHiddenFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide row headers
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showRowHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeVisibleFor(gridOptionsAsColumns);
            expectTreeDataToBeHiddenFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeVisibleFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide row headers + column main headers
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showRowHeaders = false;
            chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnMainHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeVisibleFor(gridOptionsAsColumns);
            expectTreeDataToBeHiddenFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeHiddenFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide column headers => only one row for top header
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToHaveOnlyOneRowFor(gridOptionsAsColumns);
            expectTreeDataToBeVisibleFor(gridOptionsAsColumns);
            expectRowMainHeadersToBeVisibleFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeHiddenFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide column headers and row main headers => no more top header
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            chartDefAsColumns.pivotTableOptions.tableFormatting.showRowMainHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeHiddenFor(gridOptionsAsColumns);
            expectTreeDataToBeVisibleFor(gridOptionsAsColumns);
            expectRowMainHeadersToBeHiddenFor(gridOptionsAsColumns);
            expectColumnMainHeadersToBeHiddenFor(gridOptionsAsColumns, chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders, measuresDisplayedAsColumns);

            // Hide row and column headers => everything's hidden
            resetHeadersFor(chartDefAsColumns);
            chartDefAsColumns.pivotTableOptions.tableFormatting.showColumnHeaders = false;
            chartDefAsColumns.pivotTableOptions.tableFormatting.showRowHeaders = false;
            gridOptionsAsColumns = AgGridConverter.getAgGridConverter(chartDefAsColumns, chartStoreAsColumns, chartDataAsColumns, dataAsColumns, chartHandlerAsColumns, tooltipAsColumns, contextualMenuAsColumns, $containerAsColumns).convert();
            expectTopHeaderToBeHiddenFor(gridOptionsAsColumns);
            expectTreeDataToBeHiddenFor(gridOptionsAsColumns);
        });
    });

    function expectTopHeaderToBeVisibleFor(gridOptions) {
        expect(gridOptions.headerHeight).toBeUndefined();
    }

    function expectTopHeaderToBeHiddenFor(gridOptions) {
        expect(gridOptions.headerHeight).toBe(0);
    }

    function expectTopHeaderToHaveOnlyOneRowFor(gridOptions) {
        expect(gridOptions.headerHeight).toBeUndefined();
        expect(gridOptions.groupHeaderHeight).toBe(0);
    }

    function expectTreeDataToBeVisibleFor(gridOptions) {
        expect(gridOptions.treeData).toBe(true);
    }

    function expectTreeDataToBeHiddenFor(gridOptions) {
        expect(gridOptions.treeData).toBe(false);
    }

    function resetHeadersFor(chartDef) {
        chartDef.pivotTableOptions.tableFormatting.showRowHeaders = true;
        chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = true;
        chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = true;
        chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = true;
    }
});
