class ContextLibCompat:

    @staticmethod
    def nullcontext():
        if ContextLibCompat.can_use_nullcontext():
            from contextlib import nullcontext
            return nullcontext()
        else:
            return ContextLibCompat.DummyContext()

    @staticmethod
    def can_use_nullcontext():
        import sys
        print(sys.version_info)
        return sys.version_info >= (3, 7)

    class DummyContext(object):
        """
        Context manager that does no additional processing.
        """

        def __init__(self, enter_result=None):
            self.enter_result = enter_result

        def __enter__(self):
            return self.enter_result

        def __exit__(self, *excinfo):
            pass

class ImpCompat:
    """
    imp was deprecated in python 3.4, and finally removed in 3.12
    Here, we currently keep using it in all cases except 3.12, where we use various methods from importlib
    """

    @staticmethod
    def load_source(name, filename):
        if ImpCompat.can_use_imp():
            return ImpCompat._imp_load_source(name, filename)
        else:
            return ImpCompat._importlib_load_source(name, filename)

    @staticmethod
    def find_and_load_module(package_name, path):
        if ImpCompat.can_use_imp():
            return ImpCompat._imp_find_and_load_module(package_name, path)
        else:
            return ImpCompat._importlib_find_and_load_module(package_name, path)

    @staticmethod
    def can_use_imp():
        import sys

        # imp support was deprecated in 3.4, but finally removed in 3.12
        return not (sys.version_info[0] == 3 and sys.version_info[1] >= 12)

    @staticmethod
    def _imp_find_and_load_module(package_name, path):
        import imp

        fp, pathname, description = imp.find_module(package_name, path)
        try:
            return imp.load_module(package_name, fp, pathname, description)
        finally:
            if fp:
                fp.close()

    @staticmethod
    def _importlib_find_and_load_module(package_name, path):
        from importlib.machinery import PathFinder
        spec = PathFinder.find_spec(package_name, path)

        if spec is not None:
            return spec.loader.load_module(package_name)

    @staticmethod
    def _imp_load_source(name, filename):
        import imp
        return imp.load_source(name, filename)

    @staticmethod
    def _importlib_load_source(name, filename):
        from importlib.machinery import SourceFileLoader
        return SourceFileLoader(name, filename).load_module()
