import os
import re
import subprocess


def grant_permissions(path, dirs_permission="u+rwx,go+rx", files_permission="u+rw,go+r", force=False):
    """
    Recursively grant permissions to directories and files in a given path, in a UIF setup.
    :param str path: relative or absolute path
    :param str dirs_permission: permission to add to directories, in symbolic chmod notation (e.g. u+rwx,go+rx)
    :param str files_permission: permission to add to files, in symbolic chmod notation (e.g. u+rw,go+r)
    :param bool force: if True, apply even if UIF is disabled (e.g. running in an API node)
    :return: True if successful, None if not in a UIF setup and check is not bypassed
    """

    if not force and os.environ.get("DKU_IMPERSONATION_ENABLED", "0") == "0":
        return

    _check_permission(dirs_permission)
    _check_permission(files_permission)

    for root, dirs, files in os.walk(path):
        for directory in dirs:
            subprocess.check_call(['chmod', dirs_permission, os.path.join(root, directory)])
        for file in files:
            subprocess.check_call(['chmod', files_permission, os.path.join(root, file)])

    return True


permission_add_regex = re.compile(r"^([augo]+\+[rwx]+,?)+$")


def _check_permission(permission):
    """
    Check if a permission string corresponds to a chmod addition in symbolic notation (e.g. u+rwx,go+rx)
    """
    if permission_add_regex.match(permission) is None:
        raise ValueError("'{}' does not correspond to a chmod addition in symbolic notation".format(permission))
