# coding: utf-8
from __future__ import unicode_literals

import inspect
import json

import dataiku
from dataikuapi import DSSClient

"""
Main entry point of Deployment hooks execution engine implementation
This is a server implementing commands defined in the DeploymentHookKernelProtocol Java class
"""

import logging
import sys

from dataiku.base.socket_block_link import JavaLink, parse_javalink_args
from dataiku.base.utils import watch_stdin, get_argspec
from dataiku.core import debugging

logger = logging.getLogger(__name__)


def python2_friendly_exec(code, ctx_global, ctx_local):
    exec(code, ctx_global, ctx_local)


class LoadProjectMetadata(object):
    def __init__(self, data):
        self._data = data

    @property
    def requesting_user(self):
        return self._data["requestingUser"]

    @property
    def deployment_id(self):
        return self._data["deploymentId"]

    @property
    def deployment_report(self):
        return self._data.get("deploymentReport", None)

    @property
    def automation_node_url(self):
        return self._data.get("automationNodeUrl", None)

    @property
    def automation_node_api_key(self):
        return self._data.get("automationNodeApiKey", None)

    @property
    def automation_node_connection_infos(self):
        return self._data.get("automationConnectionInfos", None)

    @property
    def trust_all_ssl_certificates(self):
        return self._data["trustAllSSLCertificates"]

    @property
    def deploying_user(self):
        return self._data["deployingUser"]

    @property
    def deployed_project_key(self):
        return self._data["deployedProjectKey"]


class LoadApiMetadata(object):
    def __init__(self, data):
        self._data = data

    @property
    def requesting_user(self):
        return self._data["requestingUser"]

    @property
    def deployment_id(self):
        return self._data["deploymentId"]

    @property
    def deployment_report(self):
        return self._data.get("deploymentReport", None)


class ExecuteHook(object):
    def __init__(self, data):
        self._data = data

    @property
    def hook(self):
        return self._data["hook"]

    def __str__(self):
        return json.dumps(self.hook, indent=4)


class DeploymentHookProtocol(object):
    def __init__(self, link):
        self.link = link
        self.is_python3 = sys.version_info >= (3, 0)
        self._requesting_user = None
        self._deploying_user = None
        self._deployment_id = None
        self._deployer_client = None
        self._automation_client = None
        self._automation_clients = None
        self._deployed_project_key = None
        self._deployment_report = None

    def _load_project_metadata(self, load_metadata):
        """
        :param load_metadata: LoadProjectMetadata
        :return: Nothing
        """
        self._requesting_user = load_metadata.requesting_user
        self._deployment_id = load_metadata.deployment_id
        self._deployment_report = load_metadata.deployment_report
        self._deployer_client = dataiku.api_client()
        if load_metadata.automation_node_url is not None and load_metadata.automation_node_api_key is not None:
            self._automation_client = DSSClient(load_metadata.automation_node_url, load_metadata.automation_node_api_key,
                                                no_check_certificate=load_metadata.trust_all_ssl_certificates)
        if load_metadata.automation_node_connection_infos is not None:
            self._automation_clients = {}
            for connection_info in load_metadata.automation_node_connection_infos:
                self._automation_clients[connection_info["automationNodeId"]] = DSSClient(connection_info["url"], connection_info["adminAPIKey"],
                                                                                          no_check_certificate=load_metadata.trust_all_ssl_certificates)
        self._deploying_user = load_metadata.deploying_user
        self._deployed_project_key = load_metadata.deployed_project_key
        self.link.send_json({"type": "LoadProjectMetadataResult"})

    def _load_api_metadata(self, load_metadata):
        """
        :param load_metadata: LoadApiMetadata
        :return: Nothing
        """
        self._requesting_user = load_metadata.requesting_user
        self._deployment_id = load_metadata.deployment_id
        self._deployment_report = load_metadata.deployment_report
        self._deployer_client = dataiku.api_client()
        self.link.send_json({"type": "LoadApiMetadataResult"})

    def _handle_execute_hook(self, execute_hook):
        """
        :param execute_hook: ExecuteHook
        :return: Nothing
        """
        class HookResult:
            @staticmethod
            def success(message):
                return {"status": "SUCCESS", "message": message}

            @staticmethod
            def warning(message):
                return {"status": "WARNING", "message": message}

            @staticmethod
            def error(message, sensitive_data=""):
                return {"status": "ERROR", "message": message, "sensitiveData": sensitive_data}

        namespace = {}
        error_message, sensitive_data = self._load_hook_code_in_namespace(namespace, execute_hook)
        if error_message is not None:
            logger.error(error_message + ". " + sensitive_data)
            self._send_hook_result(HookResult.error(error_message, sensitive_data))
            return

        namespace["HookResult"] = HookResult
        namespace["_requesting_user"] = self._requesting_user
        namespace["_deployment_id"] = self._deployment_id
        namespace["_deployment_report"] = self._deployment_report
        namespace["_deployer_client"] = self._deployer_client
        expected_args = ["requesting_user", "deployment_id", "deployment_report", "deployer_client"]
        if self._automation_client is not None:
            namespace["_automation_client"] = self._automation_client
            expected_args.append("automation_client")
        if self._automation_clients is not None:
            namespace["_automation_clients"] = self._automation_clients
            expected_args.append("automation_clients")
        if self._automation_client is not None or self._automation_clients is not None:
            namespace["_deploying_user"] = self._deploying_user
            namespace["_deployed_project_key"] = self._deployed_project_key
            expected_args.append("deploying_user")
            expected_args.append("deployed_project_key")
        error_message, sensitive_data = self._check_execute_signature(namespace, expected_args)
        if error_message is not None:
            self._send_hook_result(HookResult.error(error_message, sensitive_data))
            return

        execute_named_params = ", ".join(["{0}=_{0}".format(param) for param in expected_args])
        python2_friendly_exec("""
try:
    result = execute(%s)
except Exception as e:
    import traceback
    import logging
    error_message = str(e)
    sensitive_data = "The stacktrace is:\\n{0}".format(traceback.format_exc())
""" % execute_named_params, namespace, namespace)
        if namespace.get("error_message") is not None:
            logger.error(namespace["error_message"] + namespace["sensitive_data"])
            self._send_hook_result(HookResult.error(namespace["error_message"], namespace["sensitive_data"]))
            return

        result = namespace["result"]
        if isinstance(result, dict) and all(k in result.keys() for k in ["status", "message"]):
            self._send_hook_result(result)
        else:
            error_message = "The deployment hook did not return a valid result"
            sensitive_data = "Returned value: {0}".format(str(result))
            logger.error(error_message + sensitive_data)
            self._send_hook_result(HookResult.error(error_message, sensitive_data))

    @staticmethod
    def _load_hook_code_in_namespace(namespace, execute_hook):
        try:
            python2_friendly_exec(execute_hook.hook["code"], namespace, namespace)
        except Exception as e:
            return "Failed parsing the deployment hook code", "Got following error: {0}. The deployment hook code is:\n{1}".format(e, execute_hook.hook["code"])
        if "execute" not in namespace:
            return "Custom hook execute function not defined", "The deployment hook code is:\n{0}".format(execute_hook.hook["code"])
        return None, None

    def _check_execute_signature(self, namespace, expected_args):
        if self.is_python3:
            sig = inspect.signature(namespace["execute"]).parameters
            args = list(sig.keys())
            kwargs_in_signature = any(parameter.kind == inspect.Parameter.VAR_KEYWORD for parameter in sig.values())
        else:
            args, _, kwargs, _ = get_argspec(namespace["execute"])
            kwargs_in_signature = kwargs is not None
        if any([arg not in args for arg in expected_args]):
            if kwargs_in_signature:
                logger.info("The execute function defined in hook is missing a required named parameter, it has been passed as kwarg. "
                            "You should update the function signature. Expected: %s; actual: %s" % (", ".join(expected_args), ", ".join(args)))
            else:
                return "The execute function defined in hook is missing a required named parameter and does not accept additional kwargs", \
                    "Expected: {0}; actual: {1}".format(", ".join(expected_args), ", ".join(args))
        return None, None

    def _send_hook_result(self, result):
        self.link.send_json({"type": "ExecuteHookResult", "result": result})

    def start(self):
        try:
            while True:
                command = self.link.read_json()
                if command["type"] == "LoadProjectMetadata":
                    load_metadata_command = LoadProjectMetadata(command)
                    self._load_project_metadata(load_metadata_command)
                if command["type"] == "LoadApiMetadata":
                    load_metadata_command = LoadApiMetadata(command)
                    self._load_api_metadata(load_metadata_command)
                elif command["type"] == "ExecuteHook":
                    execute_command = ExecuteHook(command)
                    self._handle_execute_hook(execute_command)
        except EOFError:
            logger.info("Connection with the deployment hook client closed")


def serve(port, secret, server_cert=None):
    link = JavaLink(port, secret, server_cert=server_cert)
    link.connect()
    protocol_handler = DeploymentHookProtocol(link)
    try:
        protocol_handler.start()
    finally:
        link.close()


if __name__ == "__main__":
    logging.basicConfig(level=logging.INFO,
                        format='[%(asctime)s] [%(process)s/%(threadName)s] [%(levelname)s] [%(name)s] %(message)s')
    debugging.install_handler()

    watch_stdin()
    port, secret, server_cert = parse_javalink_args()
    serve(port, secret, server_cert=server_cert)
