import logging
import numpy as np
import pandas as pd

from dataiku.core import dkujson
from dataiku.doctor.deephub.deephub_evaluation import ObjectDetectionPerformanceResults

logger = logging.getLogger(__name__)


class ObjectDetectionPerformanceComputer(object):
    IOU_THRESHOLD_STEP = 0.05
    IOU_THRESHOLDS = np.around(np.arange(.5, 1. + IOU_THRESHOLD_STEP, IOU_THRESHOLD_STEP), 3)
    CONFIDENCE_THRESHOLD_STEP = 0.05
    CONFIDENCE_THRESHOLDS = np.around(np.arange(0, 1. + CONFIDENCE_THRESHOLD_STEP, CONFIDENCE_THRESHOLD_STEP), 3)

    IOU_BASED_METRICS = [{
        "iou": 0.5,
        "metric_name": "AVERAGE_PRECISION_IOU50",
        "field_name": "averagePrecisionIOU50"
    }, {
        "iou": 0.75,
        "metric_name": "AVERAGE_PRECISION_IOU75",
        "field_name": "averagePrecisionIOU75"
    }]
    ALL_IOU_METRIC = {
        "metric_name": "AVERAGE_PRECISION_ALL_IOU",
        "field_name": "averagePrecisionAllIOU"
    }

    def __init__(self, target_remapping, origin_index, ground_truth_df, detections_df, metric_params):
        """
        In all the following computations, we use "category" for category index (i.e. an integer between 0 and
        the number of categories) for simplicity in the naming.

        We only put back the actual category (thanks to the `target_remapping`) at the end when serializing
        results in the predicted data
        """
        self.target_remapping = target_remapping
        self.categories = list(range(len(target_remapping)))
        self.origin_index = origin_index
        self.ground_truth_df = ground_truth_df
        self.detections_df = detections_df
        self.metric_params = metric_params

    @staticmethod
    def keep_rows_not_in_ref_df(df, ref_df, columns):
        """
        Filter `df`, keeping only rows that are not in `ref_df` with respect to `columns`

        :param df: dataframe to filter rows on
        :param ref_df: reference dataframe
        :param columns: columns to match df an ref_df on
        :return: filtered df
        """
        filtered_df = df.merge(ref_df.loc[:, columns], how="left", left_on=columns, right_on=columns, indicator=True)
        filtered_df = filtered_df[filtered_df["_merge"] == "left_only"]
        del filtered_df["_merge"]
        return filtered_df

    @staticmethod
    def compute_iou(iou_df):
        """
        Computes IOU on dataframe and keep only rows for which IOU > 0

        :param iou_df: dataframe
        :return: dataframe with IOU computed and filtered on iou > 0
        """
        input_cols = iou_df.columns

        # Only keep rows where actual overlap between detection and ground truth so that IOU > 0
        # create intermediate columns to ease computations (lower right corner of bboxes)
        iou_df["x1_gt"] = iou_df["x0_gt"] + iou_df["w_gt"]
        iou_df["y1_gt"] = iou_df["y0_gt"] + iou_df["h_gt"]
        iou_df["x1_det"] = iou_df["x0_det"] + iou_df["w_det"]
        iou_df["y1_det"] = iou_df["y0_det"] + iou_df["h_det"]

        iou_df = iou_df[
            (iou_df["x0_det"] <= iou_df["x1_gt"]) & (iou_df["x0_gt"] <= iou_df["x1_det"]) &
            (iou_df["y0_det"] <= iou_df["y1_gt"]) & (iou_df["y0_gt"] <= iou_df["y1_det"])
            ].copy()

        # compute IOU
        iou_df["x0_inter"] = iou_df.loc[:, ["x0_det", "x0_gt"]].max(axis=1)
        iou_df["x1_inter"] = iou_df.loc[:, ["x1_det", "x1_gt"]].min(axis=1)
        iou_df["y0_inter"] = iou_df.loc[:, ["y0_det", "y0_gt"]].max(axis=1)
        iou_df["y1_inter"] = iou_df.loc[:, ["y1_det", "y1_gt"]].min(axis=1)
        iou_df["area_inter"] = (iou_df["x1_inter"] - iou_df["x0_inter"]) * (iou_df["y1_inter"] - iou_df["y0_inter"])
        iou_df["area_det"] = (iou_df["x1_det"] - iou_df["x0_det"]) * (iou_df["y1_det"] - iou_df["y0_det"])
        iou_df["area_gt"] = (iou_df["x1_gt"] - iou_df["x0_gt"]) * (iou_df["y1_gt"] - iou_df["y0_gt"])
        iou_df["area_union"] = iou_df["area_gt"] + iou_df["area_det"] - iou_df["area_inter"]
        iou_df["iou"] = iou_df["area_inter"] / iou_df["area_union"]

        return iou_df[list(input_cols) + ["iou"]]

    @staticmethod
    def pair_best_ground_truth_detection(iou_df):
        """
        Pairing (ground_truth, detection), leveraging the COCO approach.

        For each image:
            * we sort the results by confidence
            * then we iterate over detections
            * we take the first available gt_id with the max IOU that has not been matched yet
            * we match it with the corresponding dt_id

        This step is very iterative and seems complex to vectorize, so we resort to good old python loops for now.
        This should be acceptable because we are only considering overlapping (ground_truths, detections)

        :param iou_df: dataframe to pair (ground_truth, detection) on
        :return: a dataframe with columns: ["image_id", "det_id", "gt_id", "iou"]
        """
        iou_list = []

        iou_df = iou_df.loc[:, ["image_id", "iou", "det_id", "gt_id", "confidence"]]
        # Sorting is preserved by group by
        iou_df.sort_values(by=["image_id", "confidence", "det_id", "iou"],
                           ascending=[True, False, True, False],
                           inplace=True)

        matched_img_id_gt_id = set()
        for ((image_id, det_id), grouped_by_det_df) in iou_df.groupby(["image_id", "det_id"]):
            for row in grouped_by_det_df.itertuples(index=False):
                if (image_id, row.gt_id) not in matched_img_id_gt_id:
                    iou_list.append({"image_id": image_id, "det_id": det_id, "gt_id": row.gt_id, "iou": row.iou})
                    matched_img_id_gt_id.add((image_id, row.gt_id))
                    break

        best_iou_df = pd.DataFrame(iou_list, columns=["image_id", "det_id", "gt_id", "iou"])
        return best_iou_df

    @staticmethod
    def compute_best_iou(ground_truth_df, detections_df):
        """
        Computes best Intersection Over Union (IOU) between ground truths & detections

        Follows COCO approach to pair (ground truths, detections)

        For efficiency reasons, only returns (ground_truth, detections) for which IOU > 0

        :param ground_truth_df: dataframe containing ground truths,
                                must have following columns: ["bbox", "image_id", "gt_id"]
        :type ground_truth_df: pd.DataFrame
        :param detections_df: dataframe containing detections
                              must have following columns: ["bbox", "image_id", "det_id"]
        :type detections_df: pd.DataFrame
        :return: dataframe with columns ["image_id", "det_id", "iou"], where IOU > 0
        :rtype: (pd.DataFrame, pd.DataFrame)
        """

        # First, create a column for each coordinate:
        #  * (x0, y0) is the top left corner of the bbox
        #  * (width, height) are the dimensions of the bbox
        ground_truth_df[["x0", "y0", "w", "h"]] = pd.DataFrame(ground_truth_df['bbox'].tolist(),
                                                               columns=["x0", "y0", "w", "h"],
                                                               dtype=float)
        detections_df[["x0", "y0", "w", "h"]] = pd.DataFrame(detections_df['bbox'].tolist(),
                                                             columns=["x0", "y0", "w", "h"],
                                                             dtype=float)

        # As a first, match only detections and ground truths with the same category, joining by (image_id, category)
        iou_df = detections_df.merge(ground_truth_df, how="inner", suffixes=("_det", "_gt"),
                                     left_on=["image_id", "category"], right_on=["image_id", "category"])

        iou_df = ObjectDetectionPerformanceComputer.compute_iou(iou_df)
        best_iou_df = ObjectDetectionPerformanceComputer.pair_best_ground_truth_detection(iou_df)

        # As a second step, try to match the remaining unmatched detections, ground truths to be able to
        # build the confusion matrix.
        # To do so, restrict detections & ground truths that are unmatched
        unmatched_det_df = ObjectDetectionPerformanceComputer.keep_rows_not_in_ref_df(detections_df, best_iou_df,
                                                                                      ["image_id", "det_id"])
        unmatched_gt_df = ObjectDetectionPerformanceComputer.keep_rows_not_in_ref_df(ground_truth_df, best_iou_df,
                                                                                     ["image_id", "gt_id"])
        unmatched_iou_df = unmatched_det_df.merge(unmatched_gt_df, how="inner", suffixes=("_det", "_gt"),
                                                  left_on=["image_id"], right_on=["image_id"])

        # Keeping only rows for which categories are different, all potential matches for same category
        # were already done in first step
        unmatched_iou_df = unmatched_iou_df[unmatched_iou_df["category_gt"] != unmatched_iou_df["category_det"]]

        unmatched_iou_df = ObjectDetectionPerformanceComputer.compute_iou(unmatched_iou_df)
        best_unmatched_iou_df = ObjectDetectionPerformanceComputer.pair_best_ground_truth_detection(unmatched_iou_df)

        return best_iou_df, best_unmatched_iou_df

    @staticmethod
    def compute_per_image_pairing_df(iou_df):
        """
        Aggregate the (detection, ground_truth) pairing at the image level, to save them for further use in the
        result UI.

        The pairing computation is quite complex, so we prefer doing it only once.

        Only images for which there is a pairing are represented in this dataframe.

        The result is a dataframe that looks like:
           image_id                                                                                 pairing
                0            [{"gt_id": 0, "det_id": 1, "iou": 0.4}, {"gt_id": 1, "det_id": 0, "iou": 0.8}]
                2                                                   [{"gt_id": 0, "det_id": 0, "iou": 0.6}]

        :param iou_df: dataframe containing all the pairings, built from `compute_best_iou`
        :type: pd.DataFrame
        """
        if iou_df.empty:
            # can happen if no image has an iou > 0. Make sure to still return both columns
            return pd.DataFrame(columns=['image_id', 'pairing'])

        per_image_series = iou_df.loc[:, ["image_id", "gt_id", "det_id", "iou"]].groupby("image_id") \
            .apply(lambda group: group.drop("image_id", axis=1).to_dict(orient="records"))
        per_image_series.name = "pairing"
        return per_image_series.reset_index()

    @staticmethod
    def get_series_as_flattened_list(series):
        """
        Return the content of a series as a list of list containing:
            * the element of the index
            * the value of the series

        Example: for the following multi-index series

                        index0  index1
                             0       2         10
                             1       2         20
                                     3         30

                the result will be:
                    [[0, 2, 10], [1, 2, 20], [1, 3, 30]]

        :type series: pd.Series
        :rtype: list of list
        """
        if len(series) == 0:
            return []

        index_array = np.array(series.index.tolist())  # needed for multi-index series
        values = series.values.reshape(-1, 1)
        return np.hstack([index_array, values]).tolist()

    @staticmethod
    def compute_confusion_matrix_data(iou_df, detections_df, ground_truth_df, categories):
        """
        Compute data required to build the confusion matrix in the frontend.

        NB: this requires pandas >= 0.24 due to the usage of `pd.Series.droplevel`.

        This data needs to be split:
        * by IOU threshold
        * by confidence score

        so that we can see the impact of moving those variables on the confusion matrix

        Besides, we also need the information of:
        * not detected object (ground truth without pairing)
        * not an object (detection not paired with a ground truth)

        To do so, we build 3 objects (and return them as a dict):
        * "groundTruthsCounts": number of ground truths per category
        * "perConfidenceScoresDetectionsCount": number of detections per category, split by confidence score
                                                (between 0. and 1 by CONFIDENCE_THRESHOLD_STEP)
        * "confusionMatrices": number of (detection_category, ground_truth_category) split by IOU and by
                               confidence_score.
                               This is a (num_IOU, num_confidence_score) matrix, each cell
                               containing the sparse representation of the corresponding confusion matrix

        From that, we can rebuild the confusion matrix for each (IOU, confidence score)

        The result looks like
        {
            "groundTruthsCounts": [gt_count_for_cat_0, gt_count_for_cat_1, ...],
            "perConfidenceScoreDetectionsCount: [
                [count for cat 0 and confidence_score=0.0, count for cat 0 and confidence_score=0.05,
                 count for cat 0 and confidence_score=0.10, ...],
                [count for cat 1 and confidence_score=0.0, count for cat 1 and confidence_score=0.05,
                 count for cat 1 and confidence_score=0.10, ...],
                [results for cat 2],
              ...
            ],
            "confusionMatrices": [
                [
                    [  # confusion matrix for IOU=0.5, confidence_score=0.0
                        [0, 0, 22], # det_category=0 gt_category=0 count=22
                        [1, 1, 34], # det_category=1 gt_category=1 count=34
                        ...
                    ],
                    [ confusion matrix for IOU=0.5, confidence_score=0.05 ],
                    [ confusion matrix for IOU=0.5, confidence_score=0.1 ],
                    ...
                ],
                [
                    [ confusion matrix for IOU=0.55, confidence_score=0.0 ],
                    ...
                ],
                ...
            ]
        }

        :param iou_df: dataframe with (at least) columns ["image_id", "det_id", "gt_id", "iou"]
        :type iou_df: pd.DataFrame
        :param detections_df: dataframe with (at least) columns ["image_id", "det_id", "confidence", "category"]
        :type detections_df: pd.DataFrame
        :param ground_truth_df: pd.DataFrame with (at least) columns ["image_id", "gt_id", "category"]
        :type ground_truth_df: pd.DataFrame
        :param categories: list of categories in the object detection task
        :type categories: list
        :rtype: dict
        """
        detections_df_copy = detections_df.loc[:, ["image_id", "det_id", "category", "confidence"]].copy()

        # clipping confidence to each CONFIDENCE_THRESHOLD_STEP for further aggregation, using the column
        # `confidence_index`
        step = ObjectDetectionPerformanceComputer.CONFIDENCE_THRESHOLD_STEP
        detections_df_copy["confidence_index"] = np.floor(detections_df_copy["confidence"] / step).astype(int)
        nb_confidence_steps = ObjectDetectionPerformanceComputer.CONFIDENCE_THRESHOLDS.size

        consolidated_iou_df = iou_df.merge(detections_df_copy, how="left", left_on=["image_id", "det_id"],
                                           right_on=["image_id", "det_id"])
        consolidated_iou_df.rename(columns={"category": "det_category"}, inplace=True)

        consolidated_iou_df = consolidated_iou_df.merge(ground_truth_df.loc[:, ["image_id", "gt_id", "category"]],
                                                        how="left", left_on=["image_id", "gt_id"],
                                                        right_on=["image_id", "gt_id"])
        consolidated_iou_df.rename(columns={"category": "gt_category"}, inplace=True)

        # Add back potentially missing categories for alignment purpose
        per_category_num_ground_truths = (ground_truth_df.groupby("category").size().astype(int)
                                          .reindex(categories, fill_value=0).tolist())

        # Computing number of detections per category, split by confidence score
        # groupby preserves sorting
        detections_df_copy.sort_values(by=["category", "confidence_index"], ascending=[True, False], inplace=True)

        per_category_det_series = (detections_df_copy
                                   .groupby(["category", "confidence_index"], sort=False)
                                   .size()

                                   # Adding back missing confidence index. Because we have a Multi-index here, 
                                   # we need to : * unstack the "confidence_index", i.e. project it on the columns. 
                                   # This will create a dataframe with index ["category"] for the rows, 
                                   # and "confidence_index" for the columns * reindex the columns to ensure having 
                                   # all confidence index in them. This is required only if some column_index are not
                                   # found in any (category) * fill_na with 0. * re-stack the "confidence_index" to get 
                                   # back a Series with index ["category", "confidence_index"] 
                                   .unstack("confidence_index").reindex(columns=np.flip(np.arange(nb_confidence_steps)))
                                                               .fillna(0)
                                                               .stack("confidence_index")
    
                                   .groupby("category").cumsum()
    
                                   # Return results as a list, but need to reverse it first as it is ordered by
                                   # decreasing confidence_index
                                   .groupby("category").agg(lambda g: np.flip(g.values.astype(int)).tolist()))

        # Add back potentially missing categories for alignment purpose
        per_category_det = per_category_det_series.reindex(categories)
        # Replacing NaN by None, cannot be done at reindex time because pandas will put NaN, even if specifying None as
        # filling value
        per_category_det = [elem if not (isinstance(elem, float) and pd.isna(elem)) else None
                            for elem in per_category_det]

        # Compute confusion matrix, for each possible IOU
        consolidated_iou_df.sort_values(by=["det_category", "gt_category", "confidence_index"],
                                        ascending=[True, True, False],
                                        inplace=True)  # groupby preserves sorting

        confusion_matrices = []
        for iou in ObjectDetectionPerformanceComputer.IOU_THRESHOLDS:
            consolidated_iou_df = consolidated_iou_df[consolidated_iou_df["iou"] >= iou]

            if consolidated_iou_df.shape[0] == 0:
                confusion_matrices.append(None)
                continue

            grouped_series = (consolidated_iou_df.loc[:, ["det_category", "gt_category", "confidence_index"]]
                              .groupby(["det_category", "gt_category", "confidence_index"], sort=False).size()

                              # Adding back missing confidence index
                              # For more details on the following lines, please refer to above computation of
                              # `per_category_det_series` that uses the same approach
                              .unstack("confidence_index").reindex(columns=np.flip(np.arange(nb_confidence_steps)))
                                                          .fillna(0)
                                                          .stack("confidence_index")

                              .groupby(["det_category", "gt_category"]).cumsum())

            confusion_matrix = (grouped_series
                                .groupby(level="confidence_index", sort=True)
                                .apply(lambda g: ObjectDetectionPerformanceComputer.get_series_as_flattened_list(
                                                        # only keep positive counts
                                                        g[g > 0].astype(int).droplevel("confidence_index")))
                                .tolist())

            confusion_matrices.append(confusion_matrix)

        confusion_matrix_data = {
            "groundTruthsCounts": per_category_num_ground_truths,
            "perConfidenceScoreDetectionsCount": per_category_det,
            "confusionMatrices": confusion_matrices
        }

        return confusion_matrix_data

    @staticmethod
    def prune_curve(x, y, distance_threshold=0.05):
        """ Given numpy arrays (x,y) representing curve points, remove points until there is no segment
            (x_i, y_i) , (x_(i+1), y_(i+1)) that are smaller than distance_threshold

            This is a rewrite of `dataiku.doctor.prediction.scoring_base.trim_curve` in a more numpy-ish approach
            we should harmonize them at some point (there is a slight difference, in this function, we always keep
            the last point of the curve, even if not at distance_threshold from previous point)

            :param x: array containing x axis values
            :type x: np.ndarray
            :param y: array containing y axis values
            :type y: np.ndarray
            :type distance_threshold: float
            :return (x pruned, y pruned, index of kept points)
            :rtype: tuple
        """
        if x.shape[0] == 0:
            # no detection is from the testset classes, so we have no point to show in the curve
            return x, y, np.empty((0,))

        # First compute distance between each element
        distance = np.sqrt(np.power(np.diff(x), 2) + np.power(np.diff(y), 2))
        # add a fake distance for first element that we want to insert
        distance = np.insert(distance, 0, distance_threshold + 1)
        # Also ensure that we add last element
        distance[-1] = distance_threshold + 1

        # Then only keep elements such that cumulated distance increased enough since last point
        cum_distance = distance.cumsum()
        rounded_cum_distance = np.around(cum_distance * (1 / distance_threshold)) * distance_threshold
        _, index = np.unique(rounded_cum_distance, return_index=True)

        return x[index], y[index], index

    @staticmethod
    def divide_array(arr1, arr2, value_if_arr2_is_zero):
        """
        Divide 2 np arrays, and put `value_if_arr2_is_zero` when division is not defined (arr2 value is 0)

        :type arr1: np.ndarray
        :type arr2: np.ndarray
        :type value_if_arr2_is_zero: float
        :return: array representing division
        """
        ret = np.full(arr1.shape, value_if_arr2_is_zero)
        np.divide(arr1, arr2, out=ret, where=(arr2 != 0.))
        return ret

    @staticmethod
    def compute_precision_recall_curve(precision_recall_df):
        """
        Compute Precision - Recall Curve

        :param precision_recall_df: dataframe containing precision & recall, sorted by confidence, should be
                                    result of `ObjectDetectionPerformanceComputer.compute_precision_and_recall`
        """

        precision = precision_recall_df["acc_precision"].values
        recall = precision_recall_df["acc_recall"].values
        confidence = precision_recall_df["confidence"].values

        # Prune results so that we don't keep too many points
        recall, precision, kept_index = ObjectDetectionPerformanceComputer.prune_curve(recall, precision)

        # if no detection is kept, precision-recall curve will have no point to display
        has_detections = kept_index.shape[0] > 0

        # f1 may be ill-defined (e.g. equal to NaN) if precision & recall are 0.
        # In that case we set it to 0.0 (same behaviour as sklearn)
        f1 = ObjectDetectionPerformanceComputer.divide_array(2 * precision * recall, precision + recall, 0.)
        confidence = confidence[kept_index] if has_detections else np.empty((0,))

        # Computing confidences that maximize each metric
        confidence_max_precision = confidence[np.argmax(precision)] if has_detections else 0
        confidence_max_recall = confidence[np.argmax(recall)] if has_detections else 0
        confidence_max_f1 = confidence[np.argmax(f1)] if has_detections else 0

        return {
            "curve": {
                "recall": recall.tolist(),
                "precision": precision.tolist(),
                "f1": f1.tolist(),
                "confidence": confidence.tolist(),
            },
            "confidenceOfBest": {
                "recall": confidence_max_recall,
                "precision": confidence_max_precision,
                "f1": confidence_max_f1
            }
        }

    @staticmethod
    def compute_every_point_average_precision(per_category_df):
        """
        All data points average precision (PASCAL VOCVOC2010-2012 challenge)

        The goal is to compute an approximation of the area under the Precision - Recall curve
        To do so, the formula is the following:
            average_precision = sum _{r_i in all recalls} (r_(i+1) - r_i) * max_precision_for_recall_above(r_(i+1))

        Because the first term is 0 when the recall does not change, we can only apply this formula to distinct values
        of recall

        We do it as follows:
         * for each recall r_i, compute the maximum precision over all the recall r_j >= r_i ("max_precision_above")
         * for each distinct r_i, compute the max "max_precision_above", and keep only one row per distinct r_i
         * apply the formula on the remaining rows

        :type per_category_df: pd.DataFrame
        :return: the every point average precision
        :rtype: float
        """

        max_precision_df = per_category_df.copy()
        max_precision_df.sort_values("acc_recall", ascending=False, inplace=True)
        max_precision_df["max_precision_above"] = max_precision_df["acc_precision"].cummax()

        distinct_recall_df = max_precision_df.groupby("acc_recall")["max_precision_above"].max().reset_index()
        distinct_recall_df.sort_values("acc_recall", ascending=True, inplace=True)

        # Compute diff between recall (r_(i+1) - r_i) & artificially adds the first recall value for alignment purpose
        recall_values = distinct_recall_df["acc_recall"].values
        distinct_recall_df["recall_diff"] = np.insert(np.diff(recall_values), 0, recall_values[0])

        average_precision = (distinct_recall_df["recall_diff"] * distinct_recall_df["max_precision_above"]).sum()
        return average_precision

    @staticmethod
    def compute_precision_and_recall(df, num_ground_truth):
        df = df.copy()
        df = df.sort_values("confidence", ascending=False).reset_index(drop=True)
        df["acc_TP"] = np.cumsum(df["TP"].values).astype(float)
        df["acc_FP"] = np.cumsum(df["FP"].values).astype(float)

        # As recall is the proportion of relevant items detected, we set recall=0 if num_ground_truth=0 (can happen if
        # a category is detected by the model but no sample with this class exists in the testset ground truth)
        df["acc_recall"] = 1.0 * df["acc_TP"] / num_ground_truth if num_ground_truth > 0 else 0.
        df["acc_precision"] = np.divide(df["acc_TP"], (df["acc_TP"] + df["acc_FP"]))
        return df

    @staticmethod
    def compute_per_category_perf(per_category_df, num_ground_truths_for_cat):
        per_category_df = ObjectDetectionPerformanceComputer.compute_precision_and_recall(per_category_df,
                                                                                          num_ground_truths_for_cat)
        return {
            "averagePrecision": ObjectDetectionPerformanceComputer.compute_every_point_average_precision(per_category_df)
        }

    def compute_performance(self):

        orig_detections_df = self.detections_df.copy()
        exploded_ground_truth_df = self.explode(self.ground_truth_df, "target", "gt_id")
        exploded_detections_df = self.explode(self.detections_df, "prediction", "det_id")

        if exploded_ground_truth_df.shape[0] == 0 or exploded_detections_df.shape[0] == 0:
            logger.info("No object in data or no object detected, cannot compute performance")
            return ObjectDetectionPerformanceResults.empty()

        matched_iou_df, different_categories_iou_df = self.compute_best_iou(exploded_ground_truth_df,
                                                                            exploded_detections_df)
        perf = {}
        all_iou_df = pd.concat([matched_iou_df, different_categories_iou_df])
        confusion_matrix = self.compute_confusion_matrix_data(all_iou_df, exploded_detections_df,
                                                              exploded_ground_truth_df, self.categories)
        perf["confusion_matrix"] = confusion_matrix
        per_image_pairing_df = self.compute_per_image_pairing_df(all_iou_df)

        predicted_df = self.build_predicted_df(orig_detections_df, per_image_pairing_df)

        # iou_df contains only non-zero IOUs, so we need to merge it back to detections_df to retrieve all detections
        # with no match (IOU=0) so that they are taken into account as False Positive (FP)
        detections_df_with_iou = exploded_detections_df.merge(matched_iou_df, how="left",
                                                              left_on=["image_id", "det_id"],
                                                              right_on=["image_id", "det_id"])
        detections_df_with_iou["iou"].fillna(0, inplace=True)

        # gt_categories might be different than target_remapping categories if ground_truth_df is from a test set which
        # doesn't contain at least 1 sample from each category
        gt_categories = pd.unique(exploded_ground_truth_df["category"])
        logger.info("Category indices in target map: {}".format(list(self.categories)))
        logger.info("Category indices found in ground truth: {}".format(gt_categories))
        # Keep only interesting fields & categories (categories not in ground truth shouldn't impact global metrics)
        logger.info("Categories indices found in detections: {}".format(pd.unique(detections_df_with_iou["category"])))
        detections_df_with_iou = detections_df_with_iou.loc[detections_df_with_iou.category.isin(gt_categories),
                                                            ["category", "confidence", "iou"]]

        # Computing perf for each IOU
        per_iou_perf = []
        for iou in ObjectDetectionPerformanceComputer.IOU_THRESHOLDS:
            iou_perf = {"iou": iou}
            # make a df copy per iou to recompute all the metrics according to this fixed IOU:
            all_categories_fixed_iou_df = detections_df_with_iou.copy()
            all_categories_fixed_iou_df["TP"] = (all_categories_fixed_iou_df["iou"] >= iou).astype(np.uint8)
            all_categories_fixed_iou_df["FP"] = (1 - all_categories_fixed_iou_df["TP"]).astype(np.uint8)

            precision_recall_df = self.compute_precision_and_recall(all_categories_fixed_iou_df,
                                                                    exploded_ground_truth_df.shape[0])
            precision_recall_curve = self.compute_precision_recall_curve(precision_recall_df)
            iou_perf["optimalConfidenceScoreThreshold"] = self.get_optimal_threshold(precision_recall_curve["confidenceOfBest"])

            # Initialize with perf when category not found in the detections (or not found in the ground truth)
            per_category = [{"averagePrecision": 0.} for _ in self.categories]

            for category, per_category_fixed_iou_df in all_categories_fixed_iou_df.groupby("category"):
                category_perf = self.compute_per_category_perf(
                    per_category_fixed_iou_df,
                    np.count_nonzero(exploded_ground_truth_df["category"] == category)
                )
                per_category[category] = category_perf

            iou_perf["precisionRecallCurve"] = precision_recall_curve

            # Categories not in ground truth shouldn't impact average precision of the model over all classes)
            iou_perf["global"] = {
                "averagePrecision": np.mean([e["averagePrecision"]
                                             for category, e in enumerate(per_category)
                                             if category in gt_categories])
            }

            iou_perf["perCategory"] = per_category
            per_iou_perf.append(iou_perf)

        perf["IOUThresholds"] = ObjectDetectionPerformanceComputer.IOU_THRESHOLDS
        perf["confidenceScoreThresholds"] = ObjectDetectionPerformanceComputer.CONFIDENCE_THRESHOLDS
        perf["perIOU"] = per_iou_perf
        perf["optimalConfidenceScoreThreshold"] = self.get_global_optimal_threshold(per_iou_perf)
        perf["perCategoryMetrics"] = self.compute_per_category_metrics(perf["perIOU"])

        perf["metrics"] = {
            self.ALL_IOU_METRIC["field_name"]: np.mean([m["global"]["averagePrecision"] for m in perf["perIOU"]])
        }
        for iou_based_metric in self.IOU_BASED_METRICS:
            iou_global_metrics = self.get_iou_key(per_iou_perf, iou_based_metric["iou"], "global")
            perf["metrics"][iou_based_metric["field_name"]] = (iou_global_metrics["averagePrecision"]
                                                               if iou_global_metrics is not None else 0.)

        return ObjectDetectionPerformanceResults(perf, predicted_df, raw_predictions=orig_detections_df["prediction"])

    @staticmethod
    def get_iou_key(per_iou_perf, iou, key, default_value=None):
        for iou_metrics in per_iou_perf:
            if iou_metrics["iou"] == iou:
                return iou_metrics[key]

        return default_value

    @staticmethod
    def explode(df, exploded_col, exploded_index_name):
        """
        Explode dataframe with a column containing list of dict (with the same keys):
        * each list item is put in a separate row
        * each dict is split into a separate column
        * rows with empty list are filtered out from the results

        NB: this requires pandas >=1.0 due to the usage of `pd.DataFrame.explode`

        Example:
                   image_id                                                                         target
                0         0         [{"bbox": [0, 1, 2, 3], "category": 0}, {"bbox": [1, 2, 3, 4], "category": 1}]
                1         1                                                [{"bbox": [2, 2, 3, 4], "category": 3}]

          with flattened_col="target" and exploded_index_name="gt_id", will be exploded to:

                   image_id     gt_id                 bbox     category
                0         0         0         [0, 1, 2, 3]            0
                0         0         1         [1, 2, 3, 4]            1
                0         1         0         [2, 2, 3, 4]            3


        :param df: input dataframe
        :param exploded_col: column to be exploded
        :param exploded_index_name: name to give to the column containing the index of the item within the list
        :return: the exploded dataframe
        """
        exploded_df = df[df[exploded_col].str.len() != 0].explode(exploded_col)
        exploded_df.index.rename(exploded_index_name, inplace=True)
        exploded_df.reset_index(inplace=True)
        exploded_df[exploded_index_name] = exploded_df.groupby(exploded_index_name).cumcount()
        exploded_df = pd.concat([exploded_df, pd.json_normalize(exploded_df[exploded_col])], axis=1)
        del exploded_df[exploded_col]

        return exploded_df

    def build_predicted_df(self, detections_df, per_image_pairing_df):
        predicted_df = detections_df.merge(per_image_pairing_df, how="left", left_on=["image_id"],
                                           right_on=["image_id"])
        predicted_df.sort_values(by="image_id", inplace=True)

        # Fill NaN rows with empty list (not possible to use fillna with empty list directly)
        predicted_df["pairing"] = predicted_df["pairing"].apply(lambda val: val if isinstance(val, list) else [])

        def map_to_actual_category_and_serialize(detections):
            if not isinstance(detections, list):
                detections = []

            for detection in detections:
                detection["category"] = self.target_remapping.get_category(detection["category"])
            return dkujson.dumps(detections)

        predicted_df["prediction"] = predicted_df["prediction"].apply(map_to_actual_category_and_serialize)
        predicted_df["pairing"] = predicted_df["pairing"].apply(dkujson.dumps)

        predicted_df = predicted_df.set_index("image_id")
        predicted_df = predicted_df.reindex(self.origin_index)
        return predicted_df

    def get_optimal_threshold(self, best_confidence_per_metric):
        metric_to_optimize = self.metric_params["confidenceScoreThresholdOptimMetric"].lower()
        if metric_to_optimize not in best_confidence_per_metric:
            raise ValueError("Unknown metric to optimize threshold: '{}'".format(metric_to_optimize))
        return round_to_decimal(best_confidence_per_metric[metric_to_optimize], 0.05)

    def get_global_optimal_threshold(self, per_iou_perf):
        eval_metric = self.metric_params["evaluationMetric"]
        for iou_based_metric in self.IOU_BASED_METRICS:
            if eval_metric == iou_based_metric["metric_name"]:
                return self.get_iou_key(per_iou_perf, iou_based_metric["iou"], "optimalConfidenceScoreThreshold",
                                        default_value=0.)
        if eval_metric == self.ALL_IOU_METRIC["metric_name"]:
            return round_to_decimal(np.mean([m["optimalConfidenceScoreThreshold"] for m in per_iou_perf]), 0.05)
        else:
            raise ValueError("Unknown evaluation metric: '{}'".format(eval_metric))

    def compute_per_category_metrics(self, per_iou_perf):
        per_category = [{self.ALL_IOU_METRIC["field_name"]: []} for _ in range(len(self.categories))]

        for iou_perf in per_iou_perf:
            for category_index, category_perf in enumerate(iou_perf["perCategory"]):

                for iou_based_metrics in self.IOU_BASED_METRICS:
                    if iou_perf["iou"] == iou_based_metrics["iou"]:
                        per_category[category_index][iou_based_metrics["field_name"]] = category_perf["averagePrecision"]

                per_category[category_index][self.ALL_IOU_METRIC["field_name"]].append(category_perf["averagePrecision"])

        for category_metrics in per_category:
            category_metrics[self.ALL_IOU_METRIC["field_name"]] = np.mean(category_metrics[self.ALL_IOU_METRIC["field_name"]])

        return per_category


def round_to_decimal(value, step=0.05):
    return np.around(1. * value / step) * step
