# encoding: utf-8

"""
Execute an evaluation recipe in PyRegular mode
Must be called in a Flow environment
"""
import logging
import sys

from dataiku.base.folder_context import build_folder_context
from dataiku.doctor.evaluation.evaluation_errors import ERR_MSG_NO_EVAL_FOLDER, ERR_MSG_NO_REF_FOLDER
from dataiku.modelevaluation.drift.embedding_drift_settings import TextDriftSettings, ImageDriftSettings
from dataiku.doctor.prediction.custom_standalone_evaluation_scoring import calculate_custom_standalone_evaluation_metrics, ModelParametersForCustomMetric
from dataiku.doctor.utils.gpu_execution import log_nvidia_smi_if_use_gpu
from dataikuapi.dss.analysis import RemoveRowsStepBuilder, FilterOnBadTypeStepBuilder

import dataiku
import numpy as np
import pandas as pd
from dataiku import Dataset
from dataiku.base.remoterun import read_dku_env_and_set
from dataiku.base.utils import ErrorMonitoringWrapper
from dataiku.core import debugging, schema_handling
from dataiku.core import dkujson as dkujson
from dataiku.core import doctor_constants
from dataiku.doctor import step_constants
from dataiku.doctor.diagnostics import default_diagnostics
from dataiku.doctor.diagnostics import diagnostics
from dataiku.doctor.evaluation.base import add_statistics_to_evaluation, recreate_preds_from_probas, \
    recreate_preds_from_probas_and_threshold, \
    add_custom_metrics_to_perf, is_classification, is_regression, is_binary_classification, \
    proba_definitions_from_probas, handle_drift_failure
from dataiku.doctor.evaluation.base import compute_drift
from dataiku.doctor.evaluation.base import format_data_drift_column_handling
from dataiku.doctor.evaluation.base import load_input_dataframe
from dataiku.doctor.evaluation.base import run_binary_scoring
from dataiku.doctor.evaluation.base import run_multiclass_scoring
from dataiku.doctor.evaluation.base import run_regression_scoring
from dataiku.doctor.evaluation.base import sample_and_store_dataframe
from dataiku.doctor.exception import EmptyDatasetException
from dataiku.doctor.prediction.classification_scoring import save_classification_statistics
from dataiku.doctor.prediction.decisions_and_cuts import DecisionsAndCuts
from dataiku.doctor.prediction.regression_scoring import save_regression_statistics
from dataiku.doctor.preprocessing_collector import PredictionPreprocessingDataCollector
from dataiku.doctor.utils import normalize_dataframe
from dataiku.doctor.utils.listener import DiagOnlyContext
from dataiku.doctor.utils.listener import ProgressListener
from dataiku.modelevaluation.data_types import cast_as_numeric
from dataiku.modelevaluation.data_types import cast_as_string
from dataikuscoring.utils.prediction_result import ClassificationPredictionResult
from dataikuscoring.utils.prediction_result import PredictionResult

logger = logging.getLogger(__name__)


class StandaloneEvaluateRecipe(object):
    prediction_type = None

    prediction_column_name = None

    has_model = None
    dont_compute_performance = None
    target_column_name = None
    weight_column_name = None

    resolved_preprocessing_params = None
    target_mapping = None
    is_proba_aware = None
    proba_definition = None

    modeling_params = {}

    input_dataset_smartname = None
    preparation_output_schema = None
    reference_dataset_smartname = None
    diagnostics_folder = None
    recipe_desc = None
    reference_dataset_schema = None

    should_replace_empty_target_with_default_class = False

    def __init__(self,
                 input_dataset_smartname, managed_folder_smart_id, managed_folder_smart_id_ref,
                 recipe_desc,
                 preparation_output_schema,
                 evaluation_store_folder,
                 reference_dataset_smartname,
                 reference_dataset_schema,
                 diagnostics_folder):
        """
        This initialization is made solely to read and format the input of the recipes and to create objects that will
        be used the compute method for better readability

        :param str input_dataset_smartname: The smart name of the primary dataset to evaluate.
        :param str managed_folder_smart_id: The smart ID of the managed folder for evaluation data.
        :param str managed_folder_smart_id_ref: The smart ID of the managed folder for reference data.
        :param dict recipe_desc: The recipe's parameters (StandaloneEvaluationRecipePayloadParams).
        :param dict preparation_output_schema: The schema of the prepared dataset (Schema).
        :param str evaluation_store_folder: The path to the Model Evaluation Store folder.
        :param str reference_dataset_smartname: The smart name of the reference dataset.
        :param dict reference_dataset_schema: The schema of the reference dataset (Schema).
        :param str diagnostics_folder: The path to the folder storing diagnostic files.
        """
        self.evaluation_store_folder_context = build_folder_context(evaluation_store_folder)
        diagnostics_folder_context = build_folder_context(diagnostics_folder)
        evaluation = self.evaluation_store_folder_context.read_json("_evaluation.json")

        rppp_file_filename ='rpreprocessing_params.json'
        if self.evaluation_store_folder_context.isfile(rppp_file_filename):
            self.resolved_preprocessing_params = self.evaluation_store_folder_context.read_json(rppp_file_filename)
        else:
            raise Exception("'preprocessing_params.json' file not found in the evaluation store folder")

        self.has_model = recipe_desc.get('hasModel', True)
        self.has_custom_evaluation_metrics = any(recipe_desc.get("customEvaluationMetrics", []))

        if not self.has_model:
            logger.info("No model has been selected in the recipe configuration. Will only compute drift")
            self.target_column_name = None
            self.weight_column_name = None
            self.is_proba_aware = None
            self.prediction_column_name = None

        else:
            self.prediction_type = evaluation["predictionType"]
            self.dont_compute_performance = recipe_desc.get('dontComputePerformance', False)
            self.prediction_column_name = recipe_desc.get('predictionVariable', None)

            if self.dont_compute_performance:
                logging.info("Will not compute performance")

            self.target_mapping = {
                c['sourceValue']: int(c['mappedValue'])
                for c in self.resolved_preprocessing_params.get('target_remapping', [])
            }

            self.is_proba_aware = recipe_desc.get('isProbaAware', False) and is_classification(self.prediction_type)
            self.proba_definition = proba_definitions_from_probas(recipe_desc.get('probas', []))

            if self.is_proba_aware and len(self.proba_definition) <= 1:
                raise Exception(
                    "You must define a map between classes and probabilities in your recipe configuration. Otherwise, you can uncheck the 'Proba aware' option'.")

            if self.is_proba_aware and is_binary_classification(self.prediction_type) and len(self.proba_definition) != 2:
                raise Exception(
                    "Because the prediction type is 'Two-class classification', you have to specify the mapping between exactly two classes and two probability columns for this recipe to run successfully.")

            self.proba_columns = [kv['value'] for kv in self.proba_definition]

            if self.is_proba_aware:
                manual_threshold_set = not recipe_desc.get("autoOptimizeThreshold", False)
                # When not computing performance, we can't optimize the threshold
                auto_optimize_threshold = False if self.dont_compute_performance else not manual_threshold_set
                forced_classifier_threshold = recipe_desc.get("activeClassifierThreshold", 0.5) if manual_threshold_set else 0.5
            else:
                auto_optimize_threshold = False
                forced_classifier_threshold = 0.5  # FIXME: threshold has no effect if not proba_aware, can it be set to None?

            # rebuild a fake modeling_params
            self.modeling_params = {
                'algorithm': 'EVALUATED',
                'metrics': evaluation.get('metricParams', {}),
                'autoOptimizeThreshold': auto_optimize_threshold,
                'forcedClassifierThreshold': forced_classifier_threshold
            }

            if not self.dont_compute_performance:

                self.target_column_name = recipe_desc.get('targetVariable', None)
                self.weight_column_name = recipe_desc.get('weightsVariable', None)
                if not self.is_proba_aware and not self.prediction_column_name:
                    raise Exception("Prediction column not set")

                if not self.target_column_name:
                    raise Exception(
                        "No labels column set. Please provide one or disable the performance metrics computation.")

            else:
                logging.info("Will not compute performance")
                self.target_column_name = None
                self.weight_column_name = None

            logger.info("target=%s prediction=%s weights=%s" % (
                self.target_column_name, self.prediction_column_name, self.weight_column_name))

            if self.prediction_type == doctor_constants.MULTICLASS and not self.is_proba_aware and recipe_desc.get("emptyPredictionClass"):
                self.should_replace_empty_target_with_default_class = True

        self.listener = ProgressListener(context=DiagOnlyContext(diagnostics_folder_context))
        default_diagnostics.register_evaluation_callbacks()

        self.input_dataset_smartname = input_dataset_smartname
        self.preparation_output_schema = preparation_output_schema
        self.reference_dataset_smartname = reference_dataset_smartname
        self.recipe_desc = recipe_desc
        self.reference_dataset_schema = reference_dataset_schema

        self.text_drift_settings, self.image_drift_settings = (
            self._prepare_embedding_drift_settings(managed_folder_smart_id, managed_folder_smart_id_ref))


    def compute(self):
        # --- Fetch input dataset schema and fix target/prediction dtypes
        names, dtypes, parse_date_columns = Dataset.get_dataframe_schema_st(
            self.preparation_output_schema["columns"], parse_dates=True, infer_with_pandas=False, bool_as_str=True,
            use_nullable_integers=True)

        dtypes = self.adjust_dtypes(dtypes)

        # --- Fetch the dataset
        with self.listener.push_step(step_constants.ProcessingStep.STEP_LOADING_EVALUATION):
            logger.info("Read with dtypes=%s" % dtypes)
            input_dataset = dataiku.Dataset(self.input_dataset_smartname)  # Evaluation dataset
            has_filtered_valid_rows = False
            # In this case, input_dataset.read_schema() is strictly equal to the names and dtypes
            # Change in the preparation_output_schema must see change in the preparation_requested_output_schema
            input_dataset.preparation_requested_output_schema = {"columns": input_dataset.read_schema()}
            logger.info("Preparation requested output schema =%s" % input_dataset.preparation_requested_output_schema)
            input_dataset.preparation_steps = []

            non_nullable_columns = []
            if self.prediction_column_name and not self.is_proba_aware and not self.should_replace_empty_target_with_default_class:
                logger.info("Will read the dataset without rows with empty prediction")
                non_nullable_columns.append(self.prediction_column_name)
            if self.target_column_name:
                logger.info("Will read the dataset without rows with empty target")
                non_nullable_columns.append(self.target_column_name)
            if self.weight_column_name:
                logger.info("Will read the dataset without rows with empty weight")
                non_nullable_columns.append(self.weight_column_name)
            if self.is_proba_aware:
                non_nullable_columns += self.proba_columns
                logger.info("Will read the dataset without empty and invalid (non decimal) probabilities for %s " % ", ".join(
                    self.proba_columns))

                input_dataset.preparation_steps.append(FilterOnBadTypeStepBuilder()
                                                       .with_columns(*self.proba_columns)
                                                       .with_meaning("DoubleMeaning")
                                                       .with_column_selection_mode("COLUMNS")
                                                       .build())
                has_filtered_valid_rows = True

            if non_nullable_columns:
                input_dataset.preparation_steps.append(RemoveRowsStepBuilder()
                                                       .with_columns(*non_nullable_columns)
                                                       .with_column_selection_mode("COLUMNS" if len(non_nullable_columns) > 1 else "SINGLE_COLUMN")
                                                       .build())
                has_filtered_valid_rows = True

            input_df_valid_rows = load_input_dataframe(
                input_dataset=input_dataset,
                sampling=self.recipe_desc.get('selection', {"samplingMethod": "FULL"}),
                columns=names,
                dtypes=dtypes,
                parse_date_columns=parse_date_columns,
            )

            self.check_input_df_consistency(input_df_valid_rows, non_nullable_columns)

            logger.info("Normalizing evaluation dataset")
            input_df_valid_rows = normalize_dataframe(input_df_valid_rows, self.resolved_preprocessing_params["per_feature"])

            diagnostics.on_load_evaluation_dataset_end(df=input_df_valid_rows, target_column=self.target_column_name, prediction_type=self.prediction_type)

        if self.has_model:
            # --- Extract target, weight, preds, and if applicable, probas
            target = input_df_valid_rows[self.target_column_name] if self.target_column_name else None
            weight = input_df_valid_rows[self.weight_column_name] if self.weight_column_name else None
            if weight is not None:
                assert weight.values.min() > 0, "Sample weights must be positive"
            preds = input_df_valid_rows[self.prediction_column_name] if self.prediction_column_name else None
            probas = input_df_valid_rows[self.proba_columns].values if self.is_proba_aware else None

            # --- For proba-aware classifications, recreate the prediction column from the probabilities
            if self.is_proba_aware:
                preds, threshold = recreate_preds_from_probas(probas, self.target_column_name, target, self.target_mapping, self.prediction_type,
                                                   self.modeling_params)

            # --- For multi-class, interpret the empty predictions as ...
            if self.should_replace_empty_target_with_default_class:
                classes = self.recipe_desc.get('classes', [])
                emptyPredictionClass = self.recipe_desc.get('emptyPredictionClass')
                if emptyPredictionClass not in classes:
                    raise Exception('The class for empty predictions "%s" must be one of the defined classes : %s' % (emptyPredictionClass, classes))
                preds.fillna(self.recipe_desc.get('emptyPredictionClass'), inplace=True)

            # --- Score the predictions
            if not self.dont_compute_performance:
                current_mapping = self.target_mapping
                if is_classification(self.prediction_type):
                    unmapped_preds = np.zeros(preds.shape, np.int64)
                    unmapped_target = np.zeros(target.shape, np.int64)
                    if self.target_mapping:
                        for k, v in self.target_mapping.items():
                            v = int(v)
                            mask = preds.astype('str') == k  # because k is always str
                            unmapped_preds[mask.values] = v
                        for k, v in self.target_mapping.items():
                            v = int(v)
                            mask = target.astype('str') == k  # because k is always str
                            unmapped_target[mask.values] = v
                    else:
                        auto_mapping = np.unique(np.concatenate((preds, target)))
                        current_mapping = {v: index[0] for index, v in np.ndenumerate(auto_mapping)}
                        for k, v in current_mapping.items():
                            mask_preds = preds == k
                            mask_target = target == k
                            unmapped_preds[mask_preds.values] = v
                            unmapped_target[mask_target.values] = v

                    unmapped_target = pd.Series(unmapped_target, dtype=int, name=target.name)

                    if is_binary_classification(self.prediction_type):
                        # `decisions_and_cuts` don't need to be computed by a ScorableModel, because since
                        # it's a fake clf, it implies that it has no overrides and pred = argmax(proba)
                        decisions_and_cuts = DecisionsAndCuts.from_probas_or_unmapped_preds(probas, unmapped_preds,
                                                                                            self.target_mapping)
                        run_binary_scoring(self.modeling_params, decisions_and_cuts, unmapped_target, current_mapping,
                                           weight, self.evaluation_store_folder_context,
                                           treat_metrics_failure_as_error=self.recipe_desc.get("treatPerfMetricsFailureAsError", True))
                    else:
                        prediction_result = ClassificationPredictionResult(self.target_mapping, probas=probas,
                                                                           preds=preds.values,
                                                                           unmapped_preds=unmapped_preds.astype(int))
                        run_multiclass_scoring(self.modeling_params, prediction_result, unmapped_target,
                                               current_mapping,
                                               weight, self.evaluation_store_folder_context,
                                               treat_metrics_failure_as_error=self.recipe_desc.get("treatPerfMetricsFailureAsError", True))

                elif is_regression(self.prediction_type):
                    prediction_result = PredictionResult(preds)
                    run_regression_scoring(self.modeling_params, prediction_result, target, weight,
                                           self.evaluation_store_folder_context)

            # --- Saving the predictions
            if is_classification(self.prediction_type):
                target_map = None
                if self.is_proba_aware:
                    classes = [kv['key'] for kv in self.proba_definition]
                    target_map = {value: idx for idx, value in enumerate(classes)}
                save_classification_statistics(preds,
                                               probas=probas,
                                               target_map=target_map,
                                               sample_weight=weight,
                                               base_folder_context=self.evaluation_store_folder_context)
            elif is_regression(self.prediction_type):
                save_regression_statistics(preds, self.evaluation_store_folder_context)

            # --- Compute diagnostics on classes
            if is_classification(self.prediction_type):
                with self.listener.push_step(step_constants.ProcessingStep.STEP_COLLECTING):
                    if self.is_proba_aware:
                        user_defined_classes = [kv['key'] for kv in self.proba_definition]
                    else:
                        user_defined_classes = self.recipe_desc.get('classes', None)
                    prediction_column_values = np.unique(preds)
                    target_column_values = np.unique(target) if target is not None else None
                    diagnostics.on_class_definition(prediction_column_values=prediction_column_values,
                                                    target_column_values=target_column_values,
                                                    classes=user_defined_classes)

        # --- Compute statistics on the evaluated data
        # the nb of evaluation rows is the one of the input_df before filtering
        # We want to make the drift computation on all the rows, even the invalid ones
        if not has_filtered_valid_rows:
            input_df = input_df_valid_rows
        else:
            input_dataset.preparation_steps = None
            input_df = load_input_dataframe(
                input_dataset=input_dataset,
                sampling=self.recipe_desc.get('selection', {"samplingMethod": "FULL"}),
                columns=names,
                dtypes=dtypes,
                parse_date_columns=parse_date_columns,
            )
            normalize_dataframe(input_df, self.resolved_preprocessing_params["per_feature"])

        add_statistics_to_evaluation(input_df, self.evaluation_store_folder_context)

        # --- Compute the collector_data on the sample (not the full evaluated data)
        sample_df = sample_and_store_dataframe(self.evaluation_store_folder_context, input_df,
                                               {'columns': schema_handling.get_schema_from_df(input_df)},
                                               filename='sample_scored.csv.gz',
                                               schema_filename='sample_scored_schema.json',
                                               limit_sampling=self.recipe_desc.get('limitSampling', True))

        if is_binary_classification(self.prediction_type) and self.is_proba_aware:
            probas = sample_df[self.proba_columns].values
            preds = recreate_preds_from_probas_and_threshold(probas, self.target_mapping, self.prediction_type, threshold)
            sample_df[self.prediction_column_name] = preds

        stats_sample_df = sample_df.copy()  # Cast the dataset with the types guessed (and/or overriden in config) for data collector. Drift computation does it itself
        for col in self.resolved_preprocessing_params["per_feature"]:
            if self.resolved_preprocessing_params["per_feature"][col]["type"] == "NUMERIC":
                stats_sample_df[col] = cast_as_numeric(stats_sample_df[col])
            else:
                stats_sample_df[col] = cast_as_string(stats_sample_df[col])

        collector = PredictionPreprocessingDataCollector(stats_sample_df, self.resolved_preprocessing_params)
        collector_data = collector.build()
        self.evaluation_store_folder_context.write_json("collector_data.json", collector_data)

        # --- Compute drift (if a reference dataset has been provided)
        ref_df = None  # Reference dataset
        if self.reference_dataset_smartname:
            logger.info("Computing data drift using %s as reference." % self.reference_dataset_smartname)
            (names, dtypes, parse_date_columns) = Dataset.get_dataframe_schema_st(
                self.reference_dataset_schema["columns"], parse_dates=True, infer_with_pandas=False, bool_as_str=True,
                use_nullable_integers=True)

            ref_df = load_input_dataframe(
                input_dataset=dataiku.Dataset(self.reference_dataset_smartname),
                sampling=self.recipe_desc.get('referenceDatasetSelection', {"samplingMethod": "FULL"}),
                columns=names,
                dtypes=dtypes,
                parse_date_columns=parse_date_columns
            )

            if ref_df.empty:
                raise EmptyDatasetException(
                    "The reference dataset can not be empty. Check the input dataset or the recipe sampling configuration.")

            logger.info("Normalizing reference dataset")
            ref_df = normalize_dataframe(ref_df, self.resolved_preprocessing_params["per_feature"],
                                         missing_columns='SKIP')

            sample_and_store_dataframe(self.evaluation_store_folder_context, ref_df, self.reference_dataset_schema,
                                       filename="drift_reference.csv.gz",
                                       schema_filename="drift_reference.json",
                                       limit_sampling=self.recipe_desc.get('limitSampling', True))
            drift_column_settings = {}

            if self.recipe_desc.get('treatDataDriftColumnHandling'):
                drift_column_settings = format_data_drift_column_handling(
                    self.recipe_desc.get('dataDriftColumnHandling'))
            compute_drift(sample_df, sample_df, self.resolved_preprocessing_params,
                          self.evaluation_store_folder_context,
                          self.recipe_desc.get('treatDriftFailureAsError', False),
                          self.prediction_type,
                          drift_column_settings,
                          self.recipe_desc.get('driftConfidenceLevel', 0.95),
                          prediction_column_name=self.prediction_column_name,
                          text_drift_settings=self.text_drift_settings,
                          image_drift_settings=self.image_drift_settings)

        if self.has_custom_evaluation_metrics:
            model_parameters = ModelParametersForCustomMetric(self.prediction_type, self.recipe_desc, self.modeling_params)

            custom_metrics_def = self.recipe_desc.get("customEvaluationMetrics")
            custom_eval_metrics = calculate_custom_standalone_evaluation_metrics(
                custom_metrics_def,
                input_df, # normalized
                ref_df, # normalized
                model_parameters,
                treat_metrics_failure_as_error=self.recipe_desc.get("treatPerfMetricsFailureAsError", True))
            add_custom_metrics_to_perf(custom_eval_metrics, self.evaluation_store_folder_context, self.prediction_type)

    def check_input_df_consistency(self, input_df, non_nullable_columns):
        if input_df.empty:
            message = "The evaluation dataset can not be empty. Check the input dataset or the recipe sampling configuration."
            if len(non_nullable_columns) > 0:
                message += " Only rows with non-empty values for the columns (%s) are considered for evaluation." % ', '.join(
                    non_nullable_columns)
            raise EmptyDatasetException(message)

        if self.weight_column_name and self.weight_column_name not in input_df:
            raise Exception("Weight column '%s' is not in the input" % self.weight_column_name)
        if self.target_column_name and self.target_column_name not in input_df:
            raise Exception("Target column '%s' is not in the input" % self.target_column_name)
        if self.prediction_column_name and self.prediction_column_name not in input_df:
            raise Exception("Prediction column '%s' is not in the input" % self.prediction_column_name)

        logger.info("Got a dataframe : %s" % str(input_df.shape))
        logger.info("dtypes : %s" % str(input_df.dtypes))

    def adjust_dtypes(self, dtypes):
        if self.is_proba_aware:
            for kv in self.proba_definition:
                proba_column_name = kv['value']
                dtypes[proba_column_name] = np.float64
                logger.info("proba column %s in current cast as float" % proba_column_name)
        if self.target_column_name in dtypes:
            if is_classification(self.prediction_type):
                logger.info("using target_mapping=%s" % self.target_mapping)
                # force the target to string so that we don't have pandas converting int columns to floats
                dtypes[self.target_column_name] = str
                logger.info("setting target %s as string" % self.target_column_name)
            elif is_regression(self.prediction_type):
                dtypes[self.target_column_name] = np.float64
                logger.info("setting target %s as float" % self.target_column_name)
        if self.weight_column_name in dtypes:
            # Forcing the type of weight to float in order to avoid any issues in the standalone evaluation recipe where int columns are loaded as pd.Int*Dtype.
            # Not forcing to float makes pandas load the weight column as a pd.IntegerArray, which may cause issues afterward depending on the version of pandas
            dtypes[self.weight_column_name] = np.float64
            logger.info("setting weight column %s as float" % self.weight_column_name)
        if self.prediction_column_name:
            if is_classification(self.prediction_type):
                # force the prediction to string so that we don't have pandas converting int columns to floats
                dtypes[self.prediction_column_name] = str
                logger.info("setting prediction column %s in current cast as string" % self.prediction_column_name)
            elif is_regression(self.prediction_type):
                dtypes[self.prediction_column_name] = np.float64
        return dtypes

    def _prepare_embedding_drift_settings(self, managed_folder_smart_id, managed_folder_smart_id_ref):
        """Prepares and validates the settings for text and image embedding drift."""
        treat_drift_failure_as_error = self.recipe_desc.get('treatDriftFailureAsError', False)

        text_drift_settings = TextDriftSettings(self.recipe_desc.get('hasTextDrift', False))
        if text_drift_settings.should_be_computed:
            text_drift_settings = text_drift_settings.with_params(self.recipe_desc.get('textDriftParams', None))

        image_drift_settings = ImageDriftSettings(self.recipe_desc.get('hasImageDrift', False))
        if image_drift_settings.should_be_computed:
            if not managed_folder_smart_id:
                handle_drift_failure(ERR_MSG_NO_EVAL_FOLDER, treat_drift_failure_as_error)

            if not managed_folder_smart_id_ref:
                handle_drift_failure(ERR_MSG_NO_REF_FOLDER, treat_drift_failure_as_error)

            image_drift_settings = image_drift_settings.with_params(
                self.recipe_desc.get('imageDriftParams', None), managed_folder_smart_id, managed_folder_smart_id_ref)

        return text_drift_settings, image_drift_settings

if __name__ == "__main__":
    debugging.install_handler()
    logging.basicConfig(level=logging.INFO, format='%(asctime)s %(levelname)s %(message)s')
    read_dku_env_and_set()

    with ErrorMonitoringWrapper():
        reference_dataset_schema = dkujson.load_from_filepath(sys.argv[8]) if sys.argv[8] != "" else None
        ser_runner = StandaloneEvaluateRecipe(sys.argv[1], sys.argv[2],sys.argv[3],
                                              dkujson.load_from_filepath(sys.argv[4]),
                                              dkujson.load_from_filepath(sys.argv[5]), sys.argv[6], sys.argv[7],
                                              reference_dataset_schema, sys.argv[9])
        log_nvidia_smi_if_use_gpu(recipe_desc=ser_runner.recipe_desc)
        ser_runner.compute()
