from abc import ABCMeta
from abc import abstractmethod
from enum import Enum

import numpy as np
from six import add_metaclass


class DistanceName(Enum):
    GOWER = "gower"
    EUCLIDEAN = "euclidean"


@add_metaclass(ABCMeta)
class BaseFeatureHandler(object):
    def __init__(self, feature_domain):
        self.feature_domain = feature_domain

    @abstractmethod
    def distance(self, x1, x2):
        """
        Compute distance between 2 elements
        """

    @abstractmethod
    def fit(self, x, y=None):
        """
        Prepare the handler using the distribution of the feature
        """


@add_metaclass(ABCMeta)
class CFFeatureHandlerMixin(object):
    @abstractmethod
    def generate_cf_values(self, x_ref, y_ref, radii):
        """
        :param x_ref: value from the reference point
        :param y_ref: prediction for the reference
        :param np.ndarray radii: floats between -1. and +1. Generated values will generally be:
            - close to x_ref if the radii are close to zero
            - different from x_ref if the radii are close to 1 or -1
        :return: one value for each radius that was given in input
        :rtype: np.ndarray
        """


@add_metaclass(ABCMeta)
class OOFeatureHandlerMixin(object):
    @abstractmethod
    def generate_from_replicates(self, x, std_factor, n_replicates):
        """
        Generate new samples that resemble the `x` parameter. `n_replicates` samples will be
        generated for each sample in `x`, and the lower `std_factor` will be, the more similar
        the samples will be to `x`.

        :param np.ndarray x: initial values around which we will generate the new values
        :param float std_factor: to control how far should be the generated standard values from `x`
        :param int n_replicates: nb. of new variations of `x` using the standard drawing method
        :return: `(x.size, n_replicates)` new values
        :rtype: np.ndarray
        """

    @abstractmethod
    def generate_global_uniform_values(self, n):
        """
        :param int n: size of the global uniform drawing
        :return: `(n,)` new values
        :rtype: np.ndarray
        """


class FrozenFeatureHandler(BaseFeatureHandler, CFFeatureHandlerMixin, OOFeatureHandlerMixin):
    def fit(self, x, y=None):
        return self  # nothing to do

    def distance(self, x1, x2):
        return 0.

    def generate_cf_values(self, x_ref, y_ref, radii):
        """
        :param x_ref: value from the reference point
        :param y_ref: prediction for the reference
        :param np.ndarray radii: floats between -1. and +1. Generated values will generally be:
            - close to x_ref if the radii are close to zero
            - different from x_ref if the radii are close to 1 or -1
        :return: one value for each radius that was given in input
        :rtype: np.ndarray
        """
        return np.repeat(self.feature_domain.reference, radii.shape[0])

    def generate_from_replicates(self, x, std_factor, n_replicates):
        """
        Generate new samples that resemble the `x` parameter. `n_replicates` samples will be
        generated for each sample in `x`, and the lower `std_factor` will be, the more similar
        the samples will be to `x`.

        :param np.ndarray x: initial values around which we will generate the new values
        :param float std_factor: to control how far should be the generated standard values from `x`
        :param int n_replicates: nb. of new variations of `x` using the standard drawing method
        :return: `(x.size, n_replicates)` new values
        :rtype: np.ndarray
        """
        return np.tile(self.feature_domain.reference, (x.shape[0], n_replicates))

    def generate_global_uniform_values(self, n):
        """
        :param int n: size of the global uniform drawing
        :return: `(n,)` new values
        :rtype: np.ndarray
        """
        return np.repeat(self.feature_domain.reference, n)
