from dataiku.eda.types import Literal

from dataiku.core.scipycompat import binom_test

from dataiku.eda.computations.computation import UnivariateComputation
from dataiku.eda.computations.context import Context
from dataiku.eda.computations.immutable_data_frame import ImmutableDataFrame
from dataiku.eda.exceptions import UnknownObjectType
from dataiku.eda.types import AlternativeHypothesis, SignTest1SampModel, SignTest1SampResultModel


class SignTest1Samp(UnivariateComputation):
    def __init__(self, column: str, hypothesized_median: float, alternative: AlternativeHypothesis):
        super(SignTest1Samp, self).__init__(column)
        self.hypothesized_median = hypothesized_median
        self.alternative = alternative

    @staticmethod
    def get_type() -> Literal["sign_test_1samp"]:
        return "sign_test_1samp"

    @staticmethod
    def build(params: SignTest1SampModel) -> 'SignTest1Samp':
        return SignTest1Samp(
            params['column'],
            params['hypothesizedMedian'],
            params['alternative']
        )

    def apply(self, idf: ImmutableDataFrame, ctx: Context) -> SignTest1SampResultModel:
        series = idf.float_col_no_missing(self.column)

        nb_larger = (series > self.hypothesized_median).sum()
        nb_smaller = (series < self.hypothesized_median).sum()
        nb_equal = len(series) - nb_larger - nb_smaller

        # Values equal to 'hypothesized_median' are discarded
        total = nb_smaller + nb_larger

        dku_to_scipy_alternatives_mapping = {
            'TWO_SIDED': 'two-sided',
            'LOWER': 'less',
            'GREATER': 'greater'
        }

        if self.alternative not in dku_to_scipy_alternatives_mapping:
            raise UnknownObjectType("Alternative must be one of {}".format(", ".join(dku_to_scipy_alternatives_mapping.keys())))

        pvalue = binom_test(nb_larger, total, p=0.5, alternative=dku_to_scipy_alternatives_mapping[self.alternative])

        return {
            "type": self.get_type(),
            "pvalue": pvalue,
            "nbLarger": nb_larger,
            "nbSmaller": nb_smaller,
            "nbEqual": nb_equal
        }
