from dataiku.eda.types import Literal

import numpy as np

from dataiku.eda.computations.computation import UnivariateComputation
from dataiku.eda.computations.context import Context
from dataiku.eda.computations.immutable_data_frame import ImmutableDataFrame
from dataiku.eda.exceptions import NoDataError
from dataiku.eda.types import VarianceModel, VarianceResultModel


class Variance(UnivariateComputation):
    @staticmethod
    def get_type() -> Literal["variance"]:
        return "variance"

    @staticmethod
    def build(params: VarianceModel) -> 'Variance':
        return Variance(params['column'])

    def apply(self, idf: ImmutableDataFrame, ctx: Context) -> VarianceResultModel:
        series = idf.float_col_no_missing(self.column)

        if len(series) == 0:
            # Variance of nothing does not make sense
            raise NoDataError()
        elif len(series) == 1:
            # The unbiased variance estimator is not defined when N=1 (since it divides by (N-1))
            # Per convention, the variance is set to 0 when there is only one element.
            var = 0.0
        else:
            var = np.var(series, ddof=1)

        return {"type": self.get_type(), "value": var}
