from typing import Dict, List

from statsmodels.stats.multitest import multipletests

from dataiku.eda.exceptions import UnknownObjectType
from dataiku.eda.types import PValueAdjustmentMethod


def multitest_correction(pvalues: List[float], adjustment_method: PValueAdjustmentMethod) -> List[float]:
    """
        Adjust p-values when multiple hypothesis are tested at the same time
    """

    if adjustment_method == 'NONE':
        return pvalues

    dku_to_sm_adjustment_methods: Dict[PValueAdjustmentMethod, str] = {
        'FDR_BH': 'fdr_bh',
        'BONFERRONI': 'bonferroni',
        'HOLM_BONFERRONI': 'holm'
    }

    if adjustment_method not in dku_to_sm_adjustment_methods:
        raise UnknownObjectType("Unknown adjustment method")

    _, adjusted_pvalues, _, _ = multipletests(pvalues, method=dku_to_sm_adjustment_methods[adjustment_method])
    return list(adjusted_pvalues)
