import logging
from typing import List, Any

from dataikuapi.dss.llm import DSSLLMStreamedCompletionFooter, DSSLLMStreamedCompletionChunk
from dataikuapi.dss.llm_tracing import SpanBuilder

from .. import SequenceContext, BlockHandler, NextBlock

logger = logging.getLogger("dku.agents.blocks_graph")

_MAX_CHAIN_LENGTH = 50

####
# TODO: Ability to configure multiple scratchpad keys that will be copied to the top-level scratchpad wrapped into a list

class ForEachBlockHandler(BlockHandler):
    def __init__(self, turn, sequence_context, block_config):
        super().__init__(turn, sequence_context, block_config)

    def process_stream(self, trace: SpanBuilder):
        logger.info("ForEach block starting with config %s", self.block_config)

        source_expr = self.block_config["sourceExpression"]
        source_iterable = self.standard_cel_engine().evaluate(source_expr)

        if source_iterable is None:
            source_iterable = []
        if not isinstance(source_iterable, list):
            raise Exception("ForEach block expects source expression %s to return a list, got %s" % (source_expr, source_iterable))

        inner_block_id = self.block_config["blockIdToRepeat"]
        if inner_block_id == self.block_config.get("id"):
            raise Exception("ForEach block cannot execute itself")

        input_key = self.block_config["forEachInputKey"]
        if not input_key:
            raise Exception("ForEach block input variable name is required")

        results: List[Any] = []

        for index, item in enumerate(source_iterable):
            with trace.subspan("DKU_AGENT_FOR_EACH_ITERATION") as iteration_trace:
                logger.info("ForEach iteration %s/%s", index + 1, len(source_iterable))
                iteration_trace.attributes["block_id"] = inner_block_id
                iteration_trace.attributes["iterationNumber"] = index + 1

                iteration_sc = SequenceContext()
                iteration_sc.set_custom_variable(input_key, item)

                iteration_output, pending_external = yield from self._run_iteration(iteration_trace, inner_block_id, iteration_sc)

                if pending_external:
                    logger.info("Inner block requested validation/external action, stopping ForEach execution")
                    return

                results.append(iteration_output)

        output_location = self.block_config["generatedOutputStorageLocation"]
        output_key = self.block_config["targetOutputKey"]

        if output_location == "SCRATCHPAD":
            self.sequence_context.scratchpad[output_key] = results
        elif output_location == "STATE":
            self.turn.state_set(output_key, results)

        yield NextBlock(id=self.block_config.get("nextBlock"))

    def _run_iteration(self, trace: SpanBuilder, starting_block_id: str, iteration_sequence_context: SequenceContext):
        current_block_id = starting_block_id
        blocks_traversed = 0
        pending_external = False

        last_output = None

        while current_block_id:
            blocks_traversed += 1
            if blocks_traversed > _MAX_CHAIN_LENGTH:
                raise Exception("ForEach iteration exceeded maximum block chain length (possible loop)")

            block_handler = self.turn.build_block_handler(current_block_id, iteration_sequence_context)
            logger.info("ForEach iteration running block %s", current_block_id)
            next_block_id = None

            with trace.subspan("DKU_AGENT_FOR_EACH_SUB_BLOCK") as block_trace:
                block_trace.attributes["block_id"] = current_block_id
                for chunk in block_handler.process_stream(block_trace):
                    if isinstance(chunk, DSSLLMStreamedCompletionChunk):
                        yield chunk
                    elif isinstance(chunk, DSSLLMStreamedCompletionFooter):
                        finish_reason = chunk.data.get("finishReason") if hasattr(chunk, "data") else None
                        if finish_reason == "tool_validation_requests":
                            pending_external = True
                            # TODO: @structured-visual-agents: implement HITL in foreach blocks
                            raise Exception("Tool call requires human validation. Currently not supported in foreach blocks.")
                        yield chunk
                    elif isinstance(chunk, NextBlock):
                        next_block_id = chunk.id
                    else:
                        yield chunk

            if pending_external:
                break

            current_block_id = next_block_id

        last_output = iteration_sequence_context.last_text_output
        return last_output, pending_external
