import json
import logging
import random
import string
from typing import List

from dataiku.generated_sources.com.dataiku.dip.llm.online.llm_client.function_tool_call import FunctionToolCall
from dataiku.llm.python.blocks_graph import NextBlock
from dataiku.llm.python.blocks_graph.blocks.single_tool_call import SingleToolCallHandler
from dataiku.llm.python.blocks_graph.utils import _validate_and_parse_tool_call
from dataiku.llm.python.types import ChatMessage
from dataikuapi.dss.llm_tracing import SpanBuilder


logger = logging.getLogger("dku.agents.blocks_graph")

class ManualToolCallBlockHandler(SingleToolCallHandler):
    def __init__(self, turn, sequence_context, block_config):
        super().__init__(turn, sequence_context, block_config)

    def process_stream(self, trace: SpanBuilder):
        logger.info("Manual tool call block starting with config %s", self.block_config)

        cel_engine = self.standard_cel_engine()

        # 1. Load and Validate Tool
        tool_config = self.block_config["tool"]
        tool_def = self.turn.agent.load_or_get_tool(tool_config)
        if tool_def is None:
            raise ValueError(f"Manual tool call block requires a configured tool. Tool '{tool_config}' not found.")
        if len(tool_def.llm_tools) != 1:
            raise ValueError(f"Manual tool call block requires a single tool. No single tool found for '{tool_config}'.")
        llm_tool = tool_def.llm_tools[0]

        # 2. Parse Tool Call arguments
        tool_arguments = {}
        for tool_arg in tool_config.get("setArgs",[]):
            tool_arguments[tool_arg["key"]] = cel_engine.evaluate(tool_arg["value"])

        logger.debug("Evaluated tool call arguments: %s", tool_arguments)

        ftc: FunctionToolCall = {
            "type": "function",
            "id": "fc_"  +''.join(random.choice(string.ascii_lowercase + string.digits) for _ in range(50)),
            "index": 1,
            "function": {
                "name": llm_tool.llm_tool_name,
                "arguments": json.dumps(tool_arguments),
            }
        }
        tool_call = _validate_and_parse_tool_call(ftc)
        
        # 3. Execute Tool
        with trace.subspan("DKU_AGENT_TOOL_CALLS") as tools_trace:
            yield from self._call_tool(tool_call, llm_tool, tools_trace)

        # 4. Update Context & Transition
        yield NextBlock(id=self.block_config.get("nextBlock", None))
