import logging
import inspect

from dataikuapi.dss.llm import DSSLLMStreamedCompletionFooter, DSSLLMStreamedCompletionChunk
from dataiku.llm.python.blocks_graph import NextBlock, BlockHandler


logger = logging.getLogger("dku.agents.blocks_graph")


class PythonCodeBlockHandler(BlockHandler):
    def __init__(self, turn, sequence_context, block_config):
        super().__init__(turn, sequence_context, block_config)

    def process_stream(self, trace):
        logger.info("Python Code block starting with config %s" % self.block_config)

        # Execute code
        local_vars = self.python_context()
        exec(self.block_config["code"], local_vars, local_vars)

        functions = [k for k, v in local_vars.items() if inspect.isfunction(v)]
        user_function_name = self.block_config["functionName"]
        if user_function_name not in functions:
            raise ValueError(f"Function '{user_function_name}' not found in code, found: {functions}")
        user_function = local_vars[user_function_name]

        is_generator = inspect.isgeneratorfunction(user_function)

        logger.info(f"Starting to run Python function {user_function_name} (is_generator: {is_generator})")

        next_block = self.block_config.get("defaultNextBlock")
        if is_generator:
            for ichunk in user_function(trace):
                if isinstance(ichunk, DSSLLMStreamedCompletionChunk):
                    if ichunk.text is not None:
                        yield {"chunk": {"text": ichunk.text}}
                elif isinstance(ichunk, DSSLLMStreamedCompletionFooter):
                    yield ichunk
                elif isinstance(ichunk, NextBlock):
                    next_block = ichunk.id
                elif isinstance(ichunk, dict) and ichunk.get("chunk") is not None:
                    yield ichunk
                elif isinstance(ichunk, str):
                    yield {"chunk": {"text": ichunk}}
                else:
                    raise Exception("Unknown chunk yielded by Python code block generator (type:%s): %s" % (type(ichunk), ichunk))

        else:
            output = user_function(trace)

            if output is None:
                pass
            elif isinstance(output, NextBlock):
                next_block = output.id
            elif isinstance(output, str):
                yield {"chunk": {"text": output}}
            else:
                raise Exception("Don't know how to handle output of type %s: %s" % (type(output), output))

        yield NextBlock(next_block)
