import ast
import operator
import re
import json

# A micro "CEL-like" language evaluator using Python's AST module.
# This implementation supports:
#  - Nested attribute lookup similar to CEL
#  - Basic arithmetic, boolean operations, comparisons
#  - Built-in functions like upper, lower, len, sum, matches
#
# WARNING: This has been entirely written by AI.
#
# Since it uses Python AST, it automatically supports operator precedence and parentheses.
# However, it means that everything must be expressed in a Python-compatible syntax.

class MicroCelVisitor(ast.NodeVisitor):

    OP_MAP = {
        ast.Add: operator.add, ast.Sub: operator.sub, ast.Mult: operator.mul,
        ast.Div: operator.truediv, ast.Mod: operator.mod, ast.Pow: operator.pow,
        ast.Eq: operator.eq, ast.NotEq: operator.ne,
        ast.Lt: operator.lt, ast.LtE: operator.le,
        ast.Gt: operator.gt, ast.GtE: operator.ge
    }

    def __init__(self, engine):
        self.variables = engine.variables
        self.functions = engine._functions

    # --- Visitor Methods for Evaluation ---

    # Node to handle the root of the expression tree
    def visit_Expression(self, node):
        return self.visit(node.body)

    # Handle Literals (Numbers, Strings, Booleans, None)
    # ast.Constant is used for literals in Python 3.8+
    def visit_Constant(self, node):
        return node.value

    # Handle Binary Operations (a + b, a - b, etc.)
    def visit_BinOp(self, node):
        left = self.visit(node.left)
        right = self.visit(node.right)
        op_func = self.OP_MAP.get(type(node.op))
        if op_func is None:
            raise TypeError(f"Unsupported binary operator: {type(node.op).__name__}")
        return op_func(left, right)

    # Handle Boolean Operations (a and b, a or b)
    def visit_BoolOp(self, node):
        # Python uses ast.And and ast.Or as the operator types
        op_type = type(node.op)
        if op_type is ast.And:
            # Short-circuiting AND: if any is false, return false
            result = True
            for value in node.values:
                result = self.visit(value)
                if not result:
                    return False
            return bool(result) # Return the last evaluated value's boolean state

        elif op_type is ast.Or:
            # Short-circuiting OR: if any is true, return true
            for value in node.values:
                if self.visit(value):
                    return True
            return False

        else:
            raise TypeError(f"Unsupported boolean operator: {op_type.__name__}")

    # Handle Unary Operations (!a, -a)
    def visit_UnaryOp(self, node):
        operand = self.visit(node.operand)
        op_type = type(node.op)
        if op_type is ast.Not:
            return not operand
        elif op_type is ast.USub:
            return -operand
        else:
            raise TypeError(f"Unsupported unary operator: {op_type.__name__}")

    # Handle Comparisons (a == b, a > b)
    def visit_Compare(self, node):
        left = self.visit(node.left)

        # We enforce single comparisons for simplicity
        if len(node.ops) != 1 or len(node.comparators) != 1:
            raise SyntaxError("Chained comparisons are not supported in Micro-CEL.")

        right = self.visit(node.comparators[0])
        op_func = self.OP_MAP.get(type(node.ops[0]))
        if op_func is None:
            raise TypeError(f"Unsupported comparison operator: {type(node.ops[0]).__name__}")

        return op_func(left, right)

    # Handle Function Calls (e.g., upper(d.sub1))
    def visit_Call(self, node):
        # We expect node.func to be an ast.Name (e.g., 'upper')
        if not isinstance(node.func, ast.Name):
            raise SyntaxError("Function calls must use simple names (e.g., upper(...)).")

        func_name = node.func.id

        if func_name not in self.functions:
            raise NameError(f"Function '{func_name}' is not defined.")

        # Evaluate all arguments recursively
        args = [self.visit(arg) for arg in node.args]

        return self.functions[func_name](*args)

    # Handle Simple Variable Name Lookup (e.g., 'a', 'is_admin')
    def visit_Name(self, node):
        # If the name is being loaded (i.e., used as a variable, not a function definition)
        if isinstance(node.ctx, ast.Load):
            try:
                # Resolve the simple variable immediately
                return self.variables[node.id]
            except KeyError:
                # If it's a variable lookup failure, raise a specific error
                raise NameError(f"Variable '{node.id}' is not defined in the input variables.")

        # If the context is not 'Load', it might be the function name inside visit_Call
        # In this case, we just return the ID and let visit_Call handle it.
        return node.id

    # Handle Attribute Access (e.g., d.sub1)
    def visit_Attribute(self, node):
        # 1. Evaluate the base object (e.g., 'd' in 'd.sub1')
        base_obj = self.visit(node.value)
        # 2. Get the attribute name (e.g., 'sub1')
        attr_key = node.attr

        # 3. Access the attribute/key
        try:
            return base_obj[attr_key]
        except (KeyError, TypeError) as e:
            # Handle cases where base_obj might be a primitive or missing key
            raise AttributeError(f"Object '{base_obj}' has no key '{attr_key}'. Error: {e}")

    # Handle Subscript/Indexing (e.g., d.sub2[1])
    def visit_Subscript(self, node):
        # 1. Evaluate the base object (e.g., d.sub2)
        base_obj = self.visit(node.value)

        # 2. Evaluate the index (e.g., 1)
        index = self.visit(node.slice)

        # 3. Access the element
        try:
            return base_obj[index]
        except (TypeError, IndexError, KeyError) as e:
            raise IndexError(f"Invalid index '{index}' for object '{base_obj}'. Error: {e}")

    # Handle List Literals (e.g., [1, 2, 3] or [{"v": x}])
    def visit_List(self, node):
        return [self.visit(elt) for elt in node.elts]

    # Handle Dict Literals (e.g., {"key": value, "other": x.y})
    def visit_Dict(self, node):
        keys = [self.visit(k) for k in node.keys]
        values = [self.visit(v) for v in node.values]
        return dict(zip(keys, values))

    # Fallback/Security: Ensure we only visit supported nodes
    def generic_visit(self, node):
        """Disallows any unhandled AST node types."""
        if not isinstance(node, (ast.Expression, ast.Load, ast.Store, ast.Slice)):
            raise TypeError(f"Unsupported expression node: {type(node).__name__}")


class MicroCelEngine:

    def __init__(self, variables: dict):
        self.variables = variables
        self._functions = {
            # String functions
            "upper": str.upper,
            "lower": str.lower,
            "trim": str.strip,
            "matches": self._matches,

            # Collection functions
            "len": len,
            "length": len,
            "sum": sum,

            # Conversion functions
            "string": str,  # official function in the CEL language https://github.com/google/cel-spec/blob/master/doc/langdef.md#functions
            "str": str,  # for convenience
            "int": int,

            # JSON functions
            "to_json": lambda x : json.dumps(x),
            "parse_json": lambda x : json.loads(x),
        }

    def _matches(self, text, pattern):
        """Custom function: Checks if 'text' matches the regex 'pattern'."""
        if not isinstance(text, str) or not isinstance(pattern, str):
            return False
        # Note: We strip whitespace from the text for typical CEL behavior
        return bool(re.match(pattern, text.strip()))

    def evaluate(self, expression: str):
        """
        Parses the expression string into an AST and evaluates it using the visitor.
        """
        try:
            # ast.parse with mode='eval' expects a single expression.
            tree = ast.parse(expression, mode='eval')

            visitor = MicroCelVisitor(self)

            # Start the evaluation at the root value node of the expression.
            return visitor.visit(tree.body)

        except (SyntaxError, TypeError, NameError, AttributeError, ValueError, IndexError, RuntimeError) as e:
            raise RuntimeError(f"Micro-CEL Evaluation Error for '{expression}': {e}")


# --- Example Usage

# input_variables = {
#     "a" : 17,
#     "b": "  stuff  ",
#     "c": ["A", "array"],
#     "d": {
#         "sub1" : "x",
#         "sub2" : [5, 2]
#     },
#     "is_admin": True,
#     "is_active": False,
#     "threshold": 10
# }
#
# engine = MicroCelEngine(input_variables)
#
# print("--- AST-Based Evaluation Results (FIXED) ---")
#
# expressions = [
#     # Simple Lookups
#     "a", "d.sub1", "d.sub2[1]",
#     # Functions
#     "upper(d.sub1)", "len(c)",
#     # Math & Comparison
#     "a + d.sub2[0] * d.sub2[1]", "a > threshold",
#     # Boolean Logic
#     "is_admin && is_active", "not is_active",
#     # Combination
#     "len(c) == 2 && a > 10",
#     "len(c) == 2 && (a < 10 || b == \"  stuff  \")",
#     # Trim + Matches
#     "matches(b, \".*tu.*\")",
#     "matches(b, \"^tu.*\")"
# ]
#
# for expr in expressions:
#     try:
#         result = engine.evaluate(expr)
#         print(f"Expr: '{expr}' -> Result: {result} (Type: {type(result).__name__})")
#     except RuntimeError as e:
#         print(f"Expr: '{expr}' -> Error: {e}")
