import json
import re

from dataiku.llm.python.blocks_graph import SequenceContext
from dataiku.llm.python.blocks_graph.microcel import MicroCelEngine
from dataiku.llm.python.blocks_graph.types import ToolCallWithPotentialValidation
from dataiku.generated_sources.com.dataiku.dip.llm.online.llm_client.function_tool_call import FunctionToolCall

def _validate_and_parse_tool_call(ftc: FunctionToolCall) -> ToolCallWithPotentialValidation:
    id = ftc.get("id")
    if not id:
        raise Exception(f"Tool call missing id: {ftc}")

    function = ftc.get("function") or {}
    name = function.get("name")
    if not name:
        raise Exception(f"Tool call missing name: {ftc}")
    raw_args = function.get("arguments") or "{}"
    try:
        args = json.loads(raw_args, strict=False)
    except json.JSONDecodeError:
        raise Exception(f"Invalid tool call arguments format: {ftc}")

    return ToolCallWithPotentialValidation(ftc, id, name, args)

def default_if_blank(input_string, default):
    if input_string is None or len(input_string.strip()) == 0:
        return default
    else:
        return input_string

def interpolate_cel(engine: MicroCelEngine, prompt: str) -> str:
    """
    Finds all {{ }} blocks in the prompt, evaluates their contents as MicroCel
    expressions using the provided engine, and replaces them with the results.
    """
    def replace_match(match):
        expression = match.group(1).strip()
        result = engine.evaluate(expression)
        return str(result)

    return re.sub(r'\{\{\s*(.*?)\s*\}\}', replace_match, prompt)


def tool_has_been_called(messages, tool_name: str) -> bool:
    pattern = rf"^{re.escape(tool_name)}__[A-Za-z0-9]+$"
    for message in messages:
        for call in message.get("toolCalls", []):
            called_tool_name = call.get("function", {}).get("name")
            if called_tool_name and re.match(pattern, called_tool_name):
                return True
    return False

