#! /bin/bash

# Pushing to Google Artifact Registry requires the gcloud client to be configured as a Docker credentials helper. This script automates this configuration and
# also ensures the target Artifact Registry repository exists, creating it if necessary.

# --- INPUTS ---
# $1: DOCKER_REGISTRY - The full path to the docker registry, e.g. europe-west1-docker.pkg.dev/my-project/my-registry
# $2: PROXY_SETTINGS_JSON - A JSON string with proxy settings (optional), e.g. '{"host":"proxy.example.com", "port":"8080", "username":"user", "password":"pwd"}'
DOCKER_REGISTRY=$1
PROXY_SETTINGS_JSON=$2

proxy_username=$(echo $PROXY_SETTINGS_JSON | jq -r '.username // ""')
proxy_password=$(echo $PROXY_SETTINGS_JSON | jq -r '.password // ""')
proxy_host=$(echo $PROXY_SETTINGS_JSON | jq -r '.host // ""')
proxy_port=$(echo $PROXY_SETTINGS_JSON | jq -r '.port // ""')

if [[ -z "$DOCKER_REGISTRY" ]]; then
    echo "ERROR: DOCKER_REGISTRY (argument 1) is not set." >&2
    exit 1
fi

if [[ -z $(command -v gcloud) ]]; then
  echo "ERROR: required $(gcloud) binary is not in PATH" >&2
  exit 1
fi

GCLOUD_VERSION=$(gcloud version 2>&1)
echo "gcloud cli version = $GCLOUD_VERSION"

# --- PROXY CONFIGURATION ---

if [[ -n "$proxy_host" && -n "$proxy_port" ]]; then
  echo "INFO: setup proxy address $proxy_host:$proxy_port in gcloud config"
  gcloud config set proxy/type http
  gcloud config set proxy/address $proxy_host
  gcloud config set proxy/port $proxy_port
  if [[ -n "$proxy_username" ]]; then
    echo "INFO: setup proxy authentication in gcloud config"
    gcloud config set proxy/username $proxy_username
    gcloud config set proxy/password $proxy_password
  fi
else
  echo "INFO: removing proxy settings from gcloud config"
  gcloud config unset proxy/type
  gcloud config unset proxy/address
  gcloud config unset proxy/port
fi

DOCKER_HOST=$(echo "$DOCKER_REGISTRY" | cut -d'/' -f1)

echo "INFO: setup docker credential helper for host $DOCKER_HOST"
yes | gcloud auth configure-docker $DOCKER_HOST

# --- ARTIFACT REGISTRY REPOSITORY CREATION ---
# Only try to create the artifact repository if the registry is in the expected Artifact Repository format.
# e.g. REGION-docker.pkg.dev/PROJECT_ID/REPOSITORY_NAME
ARTIFACT_REPOSITORY_PATTERN="^[a-z0-9-]+-docker\.pkg\.dev\/[a-zA-Z0-9:._-]+\/[a-zA-Z0-9_-]+$"

if [[ "$DOCKER_REGISTRY" =~ $ARTIFACT_REPOSITORY_PATTERN ]]; then
  echo "INFO: Google Artifact Registry format detected. Verifying repository exists..."

  REGION=${DOCKER_HOST%-docker.pkg.dev}
  PROJECT_ID=$(echo "$DOCKER_REGISTRY" | cut -d'/' -f2)
  REPOSITORY_NAME=$(echo "$DOCKER_REGISTRY" | cut -d'/' -f3)

  echo "INFO: Checking if repository '$REPOSITORY_NAME' exists in project '$PROJECT_ID' in region '$REGION'."
  gcloud artifacts repositories describe "$REPOSITORY_NAME" --project="$PROJECT_ID" --location="$REGION" >/dev/null 2>&1

  if [[ $? -ne 0 ]]; then
    echo "INFO: Repository '$REPOSITORY_NAME' does not exist. Creating it now..."
    gcloud artifacts repositories create "$REPOSITORY_NAME" \
        --project="$PROJECT_ID" \
        --location="$REGION" \
        --repository-format="docker" \
        --description="Docker repository auto-created by pre-push script"

    if [[ $? -ne 0 ]]; then
      echo "ERROR: Failed to create Artifact Registry repository '$REPOSITORY_NAME'." >&2
      exit 1
    else
      echo "INFO: Successfully created Artifact Registry repository '$REPOSITORY_NAME'."
    fi
  else
    echo "INFO: Repository '$REPOSITORY_NAME' already exists."
  fi
fi

if (($?)); then
  echo 'ERROR: could not configure docker credential helper for gcr' >&2
fi
