CREATE FUNCTION govern_drop_constraints(input_table_name text, input_column_names text[], input_constraint_type text)
   RETURNS void 
   LANGUAGE plpgsql
  AS
$$
BEGIN
	EXECUTE (
		WITH search_constraints AS (
			SELECT 
				constraint_catalog, 
				constraint_schema, 
				constraint_name, 
				table_name,
				array_agg(k.column_name ORDER BY k.ordinal_position) AS constraint_columns
			FROM information_schema.key_column_usage k
			JOIN information_schema.table_constraints t USING (
				constraint_catalog, constraint_schema, constraint_name, table_name
			)
			WHERE k.table_name = input_table_name
			AND k.table_schema = (SELECT CURRENT_SCHEMA)
			AND t.constraint_type = input_constraint_type
			GROUP BY constraint_catalog, constraint_schema, constraint_name, table_name
		)
		SELECT 'ALTER TABLE '|| table_name ||' DROP CONSTRAINT '|| constraint_name ||';'
		FROM search_constraints 
		WHERE constraint_columns::text[] = input_column_names
	);
END;
$$;

-- Add missing index on
CREATE INDEX time_series_relationships_artifact_id_time_series_id_index ON time_series_relationships (time_series_relationship_artifact_id, time_series_relationship_time_series_id);

-- Fix json handling for configuration
ALTER TABLE govern_configuration DROP COLUMN configuration_item_id;
ALTER TABLE govern_configuration ADD COLUMN configuration_item_id TEXT GENERATED ALWAYS AS ((configuration_item_json ->> 'type')::TEXT) STORED PRIMARY KEY;

-- Add missing NOT NULL annotations
ALTER TABLE govern_configuration ALTER COLUMN configuration_item_json SET NOT NULL;
ALTER TABLE blueprints ALTER COLUMN blueprint_json SET NOT NULL;
ALTER TABLE blueprint_versions ALTER COLUMN blueprint_version_json SET NOT NULL;
ALTER TABLE blueprint_version_traces ALTER COLUMN blueprint_version_trace_json SET NOT NULL;
ALTER TABLE artifacts ALTER COLUMN artifact_json SET NOT NULL;
ALTER TABLE reference_relationships ALTER COLUMN reference_relationship_json SET NOT NULL;
ALTER TABLE uploaded_files ALTER COLUMN uploaded_file_json SET NOT NULL;
ALTER TABLE uploaded_file_relationships ALTER COLUMN uploaded_file_relationship_json SET NOT NULL;
ALTER TABLE time_series ALTER COLUMN time_series_json SET NOT NULL;
ALTER TABLE time_series_relationships ALTER COLUMN time_series_relationship_json SET NOT NULL;
ALTER TABLE custom_pages ALTER COLUMN custom_page_json SET NOT NULL;
ALTER TABLE user_mappings ALTER COLUMN user_mapping_json SET NOT NULL;
ALTER TABLE group_mappings ALTER COLUMN group_mapping_json SET NOT NULL;
ALTER TABLE global_api_key_mappings ALTER COLUMN global_api_key_mapping_json SET NOT NULL;
ALTER TABLE roles ALTER COLUMN role_json SET NOT NULL;
ALTER TABLE blueprint_role_assignments ALTER COLUMN blueprint_role_assignments_json SET NOT NULL;
ALTER TABLE blueprint_permissions ALTER COLUMN blueprint_permissions_json SET NOT NULL;

-- Remove DROP CASCADE. Non-specified entities are already in `ON DELETE NO ACTION`

-- Artifact Signoffs -> Artifact
SELECT govern_drop_constraints('artifact_signoffs', ARRAY['artifact_signoffs_artifact_id'], 'FOREIGN KEY');
ALTER TABLE artifact_signoffs ADD CONSTRAINT artifact_signoffs_artifact_signoffs_artifact_id_fkey FOREIGN KEY (artifact_signoffs_artifact_id) REFERENCES artifacts(artifact_id) ON DELETE NO ACTION;

-- Reference relationships -> Artifact
SELECT govern_drop_constraints('reference_relationships', ARRAY['reference_relationship_artifact_id_from'], 'FOREIGN KEY');
ALTER TABLE reference_relationships ADD CONSTRAINT reference_relationships_reference_relationship_artifact_id_fkey FOREIGN KEY (reference_relationship_artifact_id_from) REFERENCES artifacts(artifact_id) ON DELETE NO ACTION;

SELECT govern_drop_constraints('reference_relationships', ARRAY['reference_relationship_artifact_id_to'], 'FOREIGN KEY');
ALTER TABLE reference_relationships ADD CONSTRAINT reference_relationships_reference_relationship_artifact_i_fkey1 FOREIGN KEY (reference_relationship_artifact_id_to) REFERENCES artifacts(artifact_id) ON DELETE NO ACTION;

-- Time Series relationships -> Artifact
SELECT govern_drop_constraints('time_series_relationships', ARRAY['time_series_relationship_artifact_id'], 'FOREIGN KEY');
ALTER TABLE time_series_relationships ADD CONSTRAINT time_series_relationships_time_series_relationship_artifac_fkey FOREIGN KEY (time_series_relationship_artifact_id) REFERENCES artifacts(artifact_id) ON DELETE NO ACTION;

-- User Mappings -> Artifact
SELECT govern_drop_constraints('user_mappings', ARRAY['user_mapping_artifact_id'], 'FOREIGN KEY');
ALTER TABLE user_mappings ADD CONSTRAINT user_mappings_user_mapping_artifact_id_fkey FOREIGN KEY (user_mapping_artifact_id) REFERENCES artifacts(artifact_id) ON DELETE NO ACTION;

-- Group Mappings -> Artifact
SELECT govern_drop_constraints('group_mappings', ARRAY['group_mapping_artifact_id'], 'FOREIGN KEY');
ALTER TABLE group_mappings ADD CONSTRAINT group_mappings_group_mapping_artifact_id_fkey FOREIGN KEY (group_mapping_artifact_id) REFERENCES artifacts(artifact_id) ON DELETE NO ACTION;

-- Global API Key Mappings -> Artifact
SELECT govern_drop_constraints('global_api_key_mappings', ARRAY['global_api_key_mapping_artifact_id'], 'FOREIGN KEY');
ALTER TABLE global_api_key_mappings ADD CONSTRAINT global_api_key_mappings_global_api_key_mapping_artifact_id_fkey FOREIGN KEY (global_api_key_mapping_artifact_id) REFERENCES artifacts(artifact_id) ON DELETE NO ACTION;

-- Uploaded File relationships -> Artifact
SELECT govern_drop_constraints('uploaded_file_relationships', ARRAY['uploaded_file_relationship_artifact_id'], 'FOREIGN KEY');
ALTER TABLE uploaded_file_relationships ADD CONSTRAINT uploaded_file_relationships_uploaded_file_relationship_art_fkey FOREIGN KEY (uploaded_file_relationship_artifact_id) REFERENCES artifacts(artifact_id) ON DELETE NO ACTION;

-- Uploaded File relationships -> Uploaded Files
SELECT govern_drop_constraints('uploaded_file_relationships', ARRAY['uploaded_file_relationship_uploaded_file_id'], 'FOREIGN KEY');
ALTER TABLE uploaded_file_relationships ADD CONSTRAINT uploaded_file_relationships_uploaded_file_relationship_upl_fkey FOREIGN KEY (uploaded_file_relationship_uploaded_file_id) REFERENCES uploaded_files(uploaded_file_id) ON DELETE NO ACTION;

-- Artifacts -> Blueprint Versions
SELECT govern_drop_constraints('artifacts', ARRAY['artifact_blueprint_id', 'artifact_version_id'], 'FOREIGN KEY');
ALTER TABLE artifacts ADD CONSTRAINT artifacts_artifact_blueprint_id_artifact_version_id_fkey FOREIGN KEY (artifact_blueprint_id, artifact_version_id) REFERENCES blueprint_versions(blueprint_version_blueprint_id, blueprint_version_version_id) ON DELETE NO ACTION;

-- Blueprint Versions Traces -> Blueprint Versions
SELECT govern_drop_constraints('blueprint_version_traces', ARRAY['blueprint_version_trace_blueprint_id', 'blueprint_version_trace_version_id'], 'FOREIGN KEY');
ALTER TABLE blueprint_version_traces ADD CONSTRAINT blueprint_version_traces_blueprint_version_trace_blueprint_fkey FOREIGN KEY (blueprint_version_trace_blueprint_id, blueprint_version_trace_version_id) REFERENCES blueprint_versions(blueprint_version_blueprint_id, blueprint_version_version_id) ON DELETE NO ACTION;

-- Blueprint Versions Signoff Configuration -> Blueprint Versions
SELECT govern_drop_constraints('blueprint_version_signoff_configurations', ARRAY['blueprint_version_signoff_configurations_blueprint_id', 'blueprint_version_signoff_configurations_version_id'], 'FOREIGN KEY');
ALTER TABLE blueprint_version_signoff_configurations ADD CONSTRAINT blueprint_version_signoff_con_blueprint_version_signoff_co_fkey FOREIGN KEY (blueprint_version_signoff_configurations_blueprint_id, blueprint_version_signoff_configurations_version_id) REFERENCES blueprint_versions(blueprint_version_blueprint_id, blueprint_version_version_id) ON DELETE NO ACTION;

-- Blueprint Versions -> Blueprints
SELECT govern_drop_constraints('blueprint_versions', ARRAY['blueprint_version_blueprint_id'], 'FOREIGN KEY');
ALTER TABLE blueprint_versions ADD CONSTRAINT blueprint_versions_blueprint_version_blueprint_id_fkey FOREIGN KEY (blueprint_version_blueprint_id) REFERENCES blueprints(blueprint_id) ON DELETE NO ACTION;

-- Blueprint Permissions -> Blueprints
SELECT govern_drop_constraints('blueprint_permissions', ARRAY['blueprint_permissions_blueprint_id'], 'FOREIGN KEY');
ALTER TABLE blueprint_permissions ADD CONSTRAINT blueprint_permissions_blueprint_permissions_blueprint_id_fkey FOREIGN KEY (blueprint_permissions_blueprint_id) REFERENCES blueprints(blueprint_id) ON DELETE NO ACTION;

-- Blueprint Role Assignments -> Blueprints
SELECT govern_drop_constraints('blueprint_role_assignments', ARRAY['blueprint_role_assignments_blueprint_id'], 'FOREIGN KEY');
ALTER TABLE blueprint_role_assignments ADD CONSTRAINT blueprint_role_assignments_blueprint_role_assignments_blue_fkey FOREIGN KEY (blueprint_role_assignments_blueprint_id) REFERENCES blueprints(blueprint_id) ON DELETE NO ACTION;

DROP FUNCTION govern_drop_constraints;
