/**
 * This script use dashboardExportToolbox object that expose a set of functions needed for this feature:
 * - checkLoading() to check whether a dashboard is loading.
 * - getPagesToExport() to get the indexes of pages to export (that are visible) of the current dashboard.
 * - clearDashboard() to hide fullscreen button, bottom footer and navigation arrows.
 * - goToFirstPage() to go to dashboard first page.
 * - goToNextPage() to go to dashboard next page.
 * - goToPage(pageIndex) to go to dashboard specific page.
 * - getVerticalBoxesCount(pageIndex) to retrieve the number of grid boxes of a dahsboard page.
 *
 * To access this object, a dummy span has been added with id dashboard-export-toolbox-anchor so we can access the scope.
 *
 * If you want to do modifications or understand the code better, dashboardExportToolbox object is defined in dashboardPage directive.
 *
 * Usage
 * node export.js magicHeadlessAuth scriptConfigFile
 *
 * Arguments
 * - magicHeadlessAuth (String): API Key generated by DSS that allows authentification.
 * - scriptConfigFile (String): Path to the JSON file containing the arguments of the script (see DashboardExportScriptRunner.Config Java class for details)
 */

'use strict';

const puppeteer = require('puppeteer');
const fs = require('fs');
const utils = require('./utils');
const log = require('./log');

// ========================
// Entry point
// ========================
const magicHeadlessAuth = process.argv[2].toString();

const scriptConfigFile = process.argv[3].toString();
log.info("Reading Dashboard export script configuration from " + scriptConfigFile);
const config = JSON.parse(fs.readFileSync(scriptConfigFile));
log.info(JSON.stringify(config));

const urlsRequiringAuthentication = config.urlsRequiringAuthentication;
const enforceSandboxing = config.browserSandBoxing;
const pageDefaultTimeout = config.pageDefaultTimeout;
const outputDirectory = config.outputDirectory;
const fileType = config.fileType;

const dashboards = config.dashboards.map(function (d) {
    return {
        id: d.id,
        url: d.url,
        pageSectionFormat: d.pageSectionFormat,
        slideIndex: (d.slideIndex === "ALL") ? undefined : parseInt(d.slideIndex),
        // Filters can be encoded or decoded so we make sure here to decode any possibly encoded filter.
        queryParams: {
            filtersBySlide: (d.filtersBySlide || []).map(filter => {
                // Any decoded non-empty filter contains at least one ":".
                const isFilterDecoded = filter.length > 0 && filter.includes(':');

                return isFilterDecoded ? filter : decodeURIComponent(filter);
            })
        }
    };
});

try {
    log.info("Dashboard export script started.");

    utils.createBrowser(enforceSandboxing).then(function (browser) {
        return exportDashboards(browser, dashboards).then(function () {
            log.info("Closing browser");
            return browser.close();
        });
    }).then(function () {
        log.info("Done exporting dashboards");
    }).catch(function (err) {
        utils.exit(utils.ERR_GENERIC, "Error while running export script", err);
    });
} catch (err) {
    utils.exit(utils.ERR_GENERIC, "Error while running export script", err);
}

/**
 * Exports the supplied dashboards
 *
 * @return {Promise.<Void>}
 */
async function exportDashboards(browser, dashboards) {
    for (let i = 0; i < dashboards.length; i++) {
        await exportDashboard(browser, dashboards[i]);
    }
}

/**
 * Export a dashboard (single or multiple slides)
 *
 * @return {Promise.<Void>}
 */
async function exportDashboard(browser, dashboard) {
    const dashboardDirectory = createDashboardDirectory(outputDirectory, dashboard.id);
    const page = await loadDashboardPage(browser, dashboard);
    const toolbox = await getToolbox(page);
    if (dashboard.slideIndex === undefined || dashboard.slideIndex === null || dashboard.slideIndex === -1) {
        await exportAllDashboardSlides(page, dashboardDirectory, dashboard.id, dashboard.queryParams, dashboard.pageSectionFormat);
    } else {
        await exportSingleDashboardSlide(page, toolbox, dashboardDirectory, dashboard.id, dashboard.slideIndex, dashboard.queryParams, dashboard.pageSectionFormat);
    }
    // Beware: Do not modify or remove the following line as it is used by the backend to report on the progress of the script.
    log.info("Successfully exported dashboard " + dashboard.id);
}

/**
 * Create a new browser page and navigate to the specified dashboard
 *
 * @return {Promise.<Page>}
 */
async function loadDashboardPage(browser, dashboard) {
    const page = await utils.newBrowserPage(
        browser,
        dashboard.pageSectionFormat.xInPx,
        dashboard.pageSectionFormat.yInPx,
        urlsRequiringAuthentication,
        magicHeadlessAuth,
        pageDefaultTimeout
    )

    await utils.navigateTo(page, dashboard.url)

    await clearDashboard(page)
    log.info("New browser page for dashboard " + dashboard.id + " created");

    return page;
}

/**
 * Exports a single dashboard, the promise of this function indicate us if an export of a single dashboard has been done.
 *
 * @return {Promise.<Void>}
 */
async function exportAllDashboardSlides(page, dashboardDirectory, dashboardId, queryParams, pageSectionFormat) {
    log.info("Exporting all slides of dashboard " + dashboardId);

    const firstVisibleSlideIdx = await getFirstVisibleSlideIdx(page);
    await goToSlide(page, firstVisibleSlideIdx, queryParams.filtersBySlide[firstVisibleSlideIdx] || '', pageSectionFormat);

    try {
        await captureDashboard(page, fileType, dashboardDirectory, queryParams, pageSectionFormat)
        log.info("Multiple pages export for dashboard " + dashboardId + " done");
        return page.close();
    } catch(err) {
        utils.exit(utils.ERR_GENERIC, "Multiple pages export for dashboard " + dashboardId + " failed", err);
    }
}

/**
 * Exports a single slide from a dashboard
 *
 * @return {Promise.<Void>}
 */
async function exportSingleDashboardSlide(page, toolbox, dashboardDirectory, dashboardId, slideIndex, queryParams, pageSectionFormat) {
    log.info("Exporting slide " + slideIndex + " from dashboard " + dashboardId);

    await goToSlide(page, slideIndex, queryParams.filtersBySlide[slideIndex], pageSectionFormat)

    try {
        await captureDashboardSlide(page, toolbox, fileType, dashboardDirectory, slideIndex, pageSectionFormat);
        log.info("Single page export for dashboard " + dashboardId + " done");
        return page.close();
    } catch (err) {
        utils.exit(utils.ERR_GENERIC, "Single Page export for dashboard " + dashboardId + " failed", err);
    };
}

/**
 * Iterate through all its slides of the dashboard.
 */
async function captureDashboard(page, fileType, outputDirectory, queryParams, pageSectionFormat) {
    const indexes = await getVisibleSlideIndexes(page);
    log.info("Capture " + indexes.length + " pages");

    if (indexes.length) {

        for (let i = 0; i < indexes.length; i++) {
            // Retrieve the toolbox here to make sure that we have the one corresponding to the current slide.
            log.info("Exporting slide " + indexes[i]);

            const toolbox = await getToolbox(page);
            await captureDashboardSlide(page, toolbox, fileType, outputDirectory, indexes[i], pageSectionFormat);

            toolbox.dispose();

            log.info("getVisibleSlideIndexes " + indexes + " i " + i);

            if (i < indexes.length - 1) {
                await goToSlide(page, indexes[i + 1], queryParams.filtersBySlide[indexes[i + 1]] || '', pageSectionFormat);
            }
        }

    } else {
        const toolbox = await getToolbox(page);
        await page.setViewport({ width: pageSectionFormat.xInPx, height: pageSectionFormat.yInPx });
        await captureScreen(page, fileType, outputDirectory, 0);
        toolbox.dispose()
    }
}

/**
 * Iterate through all parts of a slide based on the viewport setted with initialWidth and initialHeight (= to the dimensions that the user has passed
 * as parameters to the script). Each part of a slide will be exported as a file. If a slide has a title, we add its height to the first iteration.
 */
async function captureDashboardSlide(page, toolbox, fileType, outputDirectory, slideIndex, pageSectionFormat) {
    const pageExportInfo = await page.evaluate(
        function (toolbox, slideIndex) {
            return toolbox.getPageExportInfo(slideIndex);
        },
        toolbox,
        slideIndex,
    );

    if (pageExportInfo.pageGridHostHeight <= pageSectionFormat.yInPx) {

        await page.setViewport({ width: pageSectionFormat.xInPx, height: pageSectionFormat.yInPx });

        await captureScreen(page, fileType, outputDirectory, (slideIndex + 1));

    } else {
        // The dashboard height is larger than the pageHeight, we need to scroll and take multiple screenshots.
        // ...but we don't want to cut in the middle of a grid cell.
        await captureDashboardSlidePart(page, toolbox, pageExportInfo, outputDirectory, slideIndex, pageSectionFormat);
    }

    // Resetting viewport to initial dimensions before going to next page
    return page.setViewport({ width: pageSectionFormat.xInPx, height: pageSectionFormat.yInPx });
}

async function captureDashboardSlidePart(page, toolbox, pageExportInfo, outputDirectory, slideIndex, pageSectionFormat) {
    const { separatorsPositionsTopPx } = pageExportInfo;

    const imageParts = [...separatorsPositionsTopPx].reduce((prev, curr, index) => {
        return [
            ...prev,
            {
                partIndex: index + 1,
                top: (prev.length >= 1 ? prev[index - 1].bottom : 0),
                bottom: curr,
            }
        ];
    }, []);

    for (const imagePart of imageParts) {
        await captureImagePart(page, toolbox, outputDirectory, slideIndex, imagePart, pageSectionFormat, 2000);
    }
}

async function captureImagePart(page, toolbox, outputDirectory, slideIndex, imagePart, pageSectionFormat, waitTimeAfterReady = 2000) {
    log.info("Capturing image part " + imagePart.partIndex + " > top:" + imagePart.top + ", bottom:" + imagePart.bottom + ", width:" + pageSectionFormat.xInPx);

    // Width and height must be integers in `setViewport`
    await page.setViewport({ width: Math.ceil(pageSectionFormat.xInPx), height: Math.floor(imagePart.bottom - imagePart.top) })

    await page.evaluate(function (toolbox, top) {
        return toolbox.scroll(Math.floor(top));
    }, toolbox, imagePart.top)

    return captureScreen(page, fileType, outputDirectory, (slideIndex + 1) + "_Part-" + imagePart.partIndex, waitTimeAfterReady);
}

/**
 * Captures the current viewport into a single PDF/PNG/JPEG file.
 *
 * @return Promise<Void>
 */
function captureScreen(page, fileType, outputDirectory, slideIndex = "", waitTimeAfterReady = 2000) {
    return utils.captureScreen(page, fileType, getToolbox, outputDirectory, 'Slide-' + slideIndex, waitTimeAfterReady);
}

async function goToSlide(page, slideIndex, filtersQueryParam, pageSectionFormat) {
    if (slideIndex < 0) {
        // All slides are hidden, we don't need to go to a specific slide
        return;
    }

    // WARNING: Puppeteer does not prevent Chrome from garbage-collecting variables
    // used by functions executed inside page.evaluate.
    // Make sure to await your Promise both in the frontend code and here.
    const toolbox = await getToolbox(page)

    await page.evaluate(async function (toolbox, slideIndex) {
        return slideIndex === 0 ? toolbox.goToFirstPage() : toolbox.goToPage(slideIndex);
    }, toolbox, slideIndex);

    // The dashboard page has changed so we need the new toolbox
    const toolboxAfterGoToPage = await getToolbox(page)

    await utils.waitForPageToLoad(page, getToolbox)

    await page.evaluate(function (toolboxAfterGoToPage, pageSectionFormat) {
        return toolboxAfterGoToPage.preparePageFormat({
            orientation: pageSectionFormat.orientation,
            formatName: pageSectionFormat.formatName,
            xInPx: pageSectionFormat.xInPx,
            yInPx: pageSectionFormat.yInPx,
            dpi: pageSectionFormat.dpi,
        });
    }, toolboxAfterGoToPage, pageSectionFormat);

    await page.evaluate(async function (toolboxAfterGoToPage, filtersQueryParam) {
        return toolboxAfterGoToPage.applyFilters(filtersQueryParam);
    }, toolboxAfterGoToPage, filtersQueryParam);

    await Promise.all([
        waitForAllTilesToBeLoaded(page),
        waitForFiltersToBeApplied(page),
        waitForPageFormatPrepared({page, pageSectionFormat})
    ]);
}

function waitForPageFormatPrepared({ page, pageSectionFormat }) {

    let separatorsSelector = (
        "dashboard-page-section-separators"
        + `[data-format-name='${pageSectionFormat.formatName}']`
        + `[data-format-orientation='${pageSectionFormat.orientation}']`
    )

    if (pageSectionFormat.formatName === 'CUSTOM') {
        separatorsSelector += (
            `[data-format-x-in-px='${pageSectionFormat.xInPx}']`
            + `[data-format-y-in-px='${pageSectionFormat.yInPx}']`
        )
    }

    const waitForSeparators = page.waitForSelector(
        separatorsSelector,
        { timeout: 5 * 60 * 1000 }
    );
    const waitForZoomMode = page.waitForSelector(
        "ng2-dashboard-page-zoom [data-zoom-mode='FIT_TO_WIDTH']",
        { timeout: 5 * 60 * 1000 }
    );

    const waitForRoundedPage = page.waitForSelector('.page-grid-rounded', { timeout: 5 * 60 * 1000 });

    return Promise.all([waitForZoomMode, waitForSeparators, waitForRoundedPage])
}

function waitForAllTilesToBeLoaded(page) {
    return page.waitForSelector('.tile-content.loading', { hidden: true, timeout: 5 * 60 * 1000 });
}

async function waitForFiltersToBeApplied(page) {
    const filtersLoadingIndicatorSelector = '.tile-header__actions .dku-loader, .tile-header .dku-loader';
    try {
        // Wait for the filters spinner to appear
        await page.waitForSelector(filtersLoadingIndicatorSelector, { timeout: 4000 });
    } finally {
        // Wait for the filters spinner to disappear
        return page.waitForSelector(filtersLoadingIndicatorSelector, { hidden: true, timeout: 5 * 60 * 1000 });
    }
}

/**
 * Hides fullscreen button, bottom footer and navigation arrows.
 *
 * @param page Browser page
 * @return {Promise.<Void>}
 */
async function clearDashboard(page) {
    await utils.waitForPageToLoad(page, getToolbox)

    const toolbox = await getToolbox(page);

    await page.evaluate(function (toolbox) {
        return toolbox.clearDashboard();
    }, toolbox);
}

async function getVisibleSlideIndexes(page) {

    const toolbox = await getToolbox(page);

    return page.evaluate(function (toolbox) {
        return toolbox.getPagesToExport();
    }, toolbox);
}

async function getFirstVisibleSlideIdx(page) {
    const toolbox = await getToolbox(page);

    return page.evaluate(function (toolbox) {
        return toolbox.getFirstVisiblePageIdx();
    }, toolbox);
}

function getToolbox(page) {
    return page.evaluateHandle(function () {
        let toolboxAnchor = document.querySelector('#dashboard-export-toolbox-anchor');
        return angular.element(toolboxAnchor).scope().dashboardExportToolbox;
    });
}

/**
 * If two dashboards have the same name in a case of mass export, we use the id to differentiate them
 * (so we don't merge into a pdf two dashboards with the same name)
 */
function createDashboardDirectory(outputDirectory, dashboardId) {
    let dashboardDirectory = outputDirectory + "/" + dashboardId;
    try {
        fs.mkdirSync(dashboardDirectory);
    } catch (err) {
        utils.exit(utils.ERR_GENERIC, "Unable to create directory", err);
    }
    return dashboardDirectory;
}
