from __future__ import print_function

import gzip
from os import path as osp
import random

import string
import os, stat
import json
import subprocess
import sys

import install_config

if sys.version_info >= (3,6):
    import secrets

class DipHome:

    def __init__(self, path):
        self.path = path

    def version_filepath(self,):
        return osp.join(self.path, "dss-version.json")

    def version_data(self):
        return json_loadf(self.version_filepath())
    
    def user_data(self):
        return json_loadf(osp.join(self.path, "config", "user-data.json"))

    def has_conf_version(self):
        return "conf_version" in self.version_data()

    def get_conf_version(self):
        return int(self.version_data()["conf_version"])

    def get_dss_version(self):
        return self.version_data()["product_version"]

    def get_install_id(self):
        return self.get_install_config().getOption("general", "installid", "notattributed")

    def set_version(self, conf_version, product_version, product_commitid):
        try:
            assert type(conf_version) == int
            print("[+] Writing version metadata conf=%s product=%s revision=%s" % (conf_version, product_version, product_commitid))
            new_data = {
                "conf_version": "%d" % conf_version,
                "product_version": product_version,
                "product_commitid": product_commitid
            }
            version_file = open(self.version_filepath(), 'w')
            version_file.write(json.dumps(new_data, indent=2))
        except IOError as e:
            raise e

    def get_install_config(self):
        return install_config.InstallConfig(self)

    def get_supervisor_key(self):
        with open(osp.join(self.path, "install-support", "supervisord.key")) as f:
            return f.read()

    def get_cloud(self):
        return self.user_data()["cloud"]



def json_dumpf(path, obj, with_gzip=False):
    if with_gzip:
        with gzip.open(path, 'wb') as f:
            f.write(json.dumps(obj).encode("utf-8"))
    else:
        with open(path, "w") as f:
            json.dump(obj, f, indent=4)


def json_loadf(path, with_gzip=False):
    if not with_gzip:
        with open(path, "r") as f:
            return json.load(f)
    else:
        with gzip.open(path, 'rt', encoding='utf-8') as f:
            content = f.read()
            if not content:
                return None
            else:
                return json.loads(content)

def _sanity_check():
    if "DKUINSTALLDIR" not in os.environ:
        print("migrate.py needs $DKUINSTALLDIR", file=sys.stderr)
        sys.exit(1)
    if "DKUJAVABIN" not in os.environ:
        print("migrate.py needs $DKUJAVABIN", file=sys.stderr)
        sys.exit(1)

def generate_password_string(N):
    if sys.version_info >= (3,6):
        return ''.join(secrets.choice(string.ascii_letters + string.digits) for _ in range(N))
    # fallback to random string if its python <3.6
    return generate_random_string(N)

def generate_random_string(N):
    return ''.join(random.choice(string.ascii_uppercase + string.digits) for _ in range(N))

def generate_random_id(N):
    return ''.join(random.choice(string.ascii_uppercase + string.ascii_lowercase + string.digits) for _ in range(N))

def generate_supervisor_key(dip_home):
    key = generate_random_string(16)

    if not osp.isdir(osp.join(dip_home.path, "install-support")):
        os.makedirs(osp.join(dip_home.path, "install-support"))

    with open(osp.join(dip_home.path, "install-support", "supervisord.key"), "w") as f:
        if not is_os_windows():
            os.fchmod(f.fileno(), stat.S_IRUSR | stat.S_IWUSR)
        f.write(key)

def _link_binary(dip_home, install_dir, binary):
    # We can't use symlink in Windows, so we create a PS Script that calls the linked binary
    # We setup env variables correctly, that the script will retrieve
    if is_os_windows():
        tgt = osp.join(dip_home.path, "bin", binary + ".cmd")
        with open(tgt, "w") as f:
            f.write("@echo off\n")
            f.write("set BINDIR=%~dp0\n")
            f.write("set DIP_HOME=%~dp0\\..\\\n")
            # This commands cd to current folder (syntax is complicated but that's what is does)
            f.write("powershell.exe -ExecutionPolicy Bypass -File %s\\scripts\\linked\\%s.ps1 %%*" % (install_dir, binary))
    else:
        src = osp.join(install_dir, "scripts", "linked", binary)
        tgt = osp.join(dip_home.path, "bin", binary)
        if osp.lexists(tgt):
            os.remove(tgt)
        os.symlink(src, tgt)

def link_dss_binaries(dip_home, install_dir):
    """After this method, DKU & co commands can be called and use the new binaries"""
    if is_os_windows():
        binaries = ["dss", "dku", "jek", "fek", "cak", "dssadmin", "dsscli"]
    else:
        binaries = ["dss", "dku", "jek", "fek", "cak", "hproxy", "dssadmin", "dsscli", "cde"]
    for binary in binaries:
        _link_binary(dip_home, install_dir, binary)

def link_apinode_binaries(dip_home, install_dir):
    """After this method, DKU & co commands can be called and use the new binaries"""
    for binary in ["dss", "dssadmin", "dku", "apinode-admin"]:
        _link_binary(dip_home, install_dir, binary)

def link_fm_binaries(dip_home, install_dir):
    for binary in ["fm", "fmadmin", "internal-fmcli"]:
        _link_binary(dip_home, install_dir, binary)

def link_govern_binaries(dip_home, install_dir):
    for binary in ["dss", "dkugovern", "govern-admin"]:
        _link_binary(dip_home, install_dir, binary)

def create_govern_folders(dip_home):
    """Create the minimal data structures required to run the registration process"""

    with open(osp.join(dip_home.path, "config", "dip.properties"), "w") as f:
        f.write("# Internal Dataiku Govern properties\n")
        f.write("psql.jdbc.url=jdbc:postgresql://my_host:my_port/my_db?currentSchema=my_schema\n")
        f.write("psql.jdbc.user=my_user\n")
        f.write("psql.jdbc.password=my_password\n")

def create_dss_folders(dip_home):
    """Create the minimal data structures required to run the registration process"""

    for folder in ["config/projects", "jobs", "exports", "uploads", "tmp", "config/ipython_notebooks"]:
        tgt = osp.join(dip_home.path, folder)
        if not osp.isdir(tgt):
            os.makedirs(tgt)

    with open(osp.join(dip_home.path, "config", "dip.properties"), "w") as f:
        f.write("# Internal DSS properties\n")
        f.write("logging.limit.s3.ignoredPath=100\n")
        f.write("logging.limit.s3.ignoredFile=100\n")
        f.write("logging.limit.filePartitioner.noMatch=100\n")

def ensure_required_dss_files(dip_home):
    publickeys = osp.join(dip_home.path, "config", "public-apikeys.json")
    if not osp.isfile(publickeys):
        json_dumpf(publickeys, [])


def create_apinode_folders(dip_home):
    """Create the minimal data structures required to run the registration process"""
    for folder in ["config", "services", "run", "bin", "tmp"]:
        tgt = osp.join(dip_home.path, folder)
        if not osp.isdir(tgt):
            os.makedirs(tgt)

    if not osp.isfile(osp.join(dip_home.path, "config", "dip.properties")):
        with open(osp.join(dip_home.path, "config", "dip.properties"), "w") as f:
            f.write("# Internal DSS properties")


def create_dir_if_needed(path):
    if not osp.isdir(path):
        os.makedirs(path)
        return True
    return False

def is_os_windows():
    return sys.platform.startswith("win32")

_is_git_installed = None

# Makes sense only for Windows, not used for Unix systems
def is_git_installed():
    global _is_git_installed
    if _is_git_installed is None:
        try:
            subprocess.run(['git', '--version'])
            _is_git_installed = True
        except:
            _is_git_installed = False

    return _is_git_installed

if __name__ == "__main__":
    def usage():
        print("""Usage:
        base.py generate_password_string LENGTH
        """, file=sys.stderr)
        sys.exit(1)

    if len(sys.argv) == 3 and sys.argv[1] == "generate_password_string" and sys.argv[2].isdigit():
        print(generate_password_string(int(sys.argv[2])))
    else:
        usage()
