import logging
import warnings
import sys
import importlib.util
from importlib.abc import MetaPathFinder, Loader
from importlib.machinery import ModuleSpec
from threading import Lock


_ALIAS_FINDER = None
_REGISTER_LOCK = Lock()

logger = logging.getLogger(__name__)


def register_python_import_alias(old, new):
    """ Register an alias for a Python module to allow importing it from its old location
    """
    global _ALIAS_FINDER

    if old in sys.modules:
        raise RuntimeError(
            "Cannot register alias for '" + old + "': module is already loaded.\n" +
            "You must register aliases before the module is imported anywhere."
        )

    with _REGISTER_LOCK:
        if old in sys.modules:
            raise RuntimeError("Cannot register alias for '" + old + "': module was just loaded by another thread.")

        if _ALIAS_FINDER is None:
            existing = next((x for x in sys.meta_path if isinstance(x, AliasedModuleFinder)), None)
            if existing:
                _ALIAS_FINDER = existing
            else:
                _ALIAS_FINDER = AliasedModuleFinder()
                # sys.meta_path is read sequentially
                sys.meta_path.insert(0, _ALIAS_FINDER)
        _ALIAS_FINDER.register_alias(old, new)


class AliasLoader(Loader):
    """ Alias a module with another.

    Example use-case: langchain.hub doesn't exist anymore, it's now langchain_classic.hub
    """
    def __init__(self, target_name, original_name):
        self.target_name = target_name
        self.original_name = original_name

    def create_module(self, spec):
        return None

    def exec_module(self, module):
        try:
            target = importlib.import_module(self.target_name)
        except ImportError as e:
            raise ImportError("Could not import alias target '" + self.target_name + "': {e}")

        # copy everything from the target to the alias module
        module.__dict__.update(target.__dict__)
        module.__name__ = module.__spec__.name
        # handle submodules (like langchain.hub)
        if hasattr(target, '__path__'):
            module.__path__ = target.__path__

        warnings.warn(self.original_name + " should now be imported from "  + self.target_name, DeprecationWarning)


class MergeLoader(Loader):
    """ Loads the old module, then finds the new one and injects missing attributes into the old module.

    Example use-case: langchain.agents still exists but some classes like AgentExecutor are now in langchain_classic.agents
    """
    def __init__(self, real_loader, real_spec, old_name, new_name):
        self.real_loader = real_loader
        self.real_spec = real_spec
        self.old_name = old_name
        self.new_name = new_name

    def create_module(self, spec):
        # let the real loader create the module (essential for C-extensions or special types)
        if hasattr(self.real_loader, 'create_module'):
            return self.real_loader.create_module(spec)
        return None

    def exec_module(self, module):
        if hasattr(self.real_loader, 'exec_module'):
            self.real_loader.exec_module(module)
        else:
            self.real_loader.load_module(module.__name__)

        try:
            new_name = module.__name__.replace(self.old_name, self.new_name, 1)
            new_module = importlib.import_module(new_name)

            # inject missing attributes
            for attr_name, attr_value in new_module.__dict__.items():
                if not hasattr(module, attr_name):
                    setattr(module, attr_name, attr_value)
                else:
                    logger.warning("attribute " + attr_name + " already exists in module " + module.__name__ + ", not overwriting it")

            # this allows finding submodules that only exist in the new module
            if hasattr(module, '__path__') and hasattr(new_module, '__path__'):
                current_paths = list(module.__path__)
                for p in new_module.__path__:
                    if p not in current_paths:
                        current_paths.append(p)
                module.__path__ = current_paths

            warnings.warn(module.__name__ + " should now be imported from "  + new_name, DeprecationWarning)
        except ImportError:
            pass

    # proxy standard checks to the real loader
    def is_package(self, fullname):
        return self.real_loader.is_package(fullname)

    def get_code(self, fullname):
        return self.real_loader.get_code(fullname)

    def get_source(self, fullname):
        return self.real_loader.get_source(fullname)


class AliasedModuleFinder(MetaPathFinder):
    """ Look for module at their aliases path and either replace them or merge them when needed. Ignores modules that are not registered with `register_alias`.
    """
    def __init__(self):
        self.aliases = {}
        self._search_lock = Lock()

    def register_alias(self, old, new):
        self.aliases[old] = new

    def find_spec(self, fullname, path, target=None):
        for key, value in self.aliases.items():
            if fullname.startswith(key):
                break
        else:
            # if no alias matched, we can just exit now
            return None

        if not self._search_lock.acquire(blocking=False):
            return None

        try:
            # try to find the real module to add missing submodules and classes to it
            for finder in sys.meta_path:
                if finder is self:
                    continue

                try:
                    spec = finder.find_spec(fullname, path, target)
                    if not spec or not spec.loader:
                        continue
                except AttributeError:
                    continue

                spec.loader = MergeLoader(spec.loader, spec, key, value)
                return spec

            # if the module doesn't exist, use the aliased one instead
            real_name = fullname.replace(key, value, 1)
            for finder in sys.meta_path:
                if finder is self:
                    continue

                try:
                    real_spec = importlib.util.find_spec(real_name)
                    if not real_spec or not real_spec.loader:
                        continue
                except ModuleNotFoundError:
                    continue

                loader = AliasLoader(real_name, fullname)
                spec = ModuleSpec(name=fullname, loader=loader, origin=real_spec.origin)
                if real_spec.submodule_search_locations is not None:
                    spec.submodule_search_locations = real_spec.submodule_search_locations
                return spec

            return None
        finally:
            self._search_lock.release()
